'use strict';
define(function() {
  /**
   * Enfin un service por rassembler le nouveau code commun à querywidget et quickquerywidget
   * @return {{replaceWhereClauseQuotedText: (function(*): *)}}
   * @constructor
   */
  const QueryService = (
    FeatureTypeFactory,
    $rootScope, 
    QueryFactory,
    gcPopup,
    gaDomUtils, 
    gaJsUtils, 
    $filter, 
    ngDialog,
    SelectManager,
    gcRestrictionProvider) => {
    /**
     * Découpe la requête à chaque occurrence de "AND" ou "OR" puis lance le remplacement des apostrophes
     * (sauf la première et la dernière occurrence de chaque partie du découpage)
     * @param request texte de la requête du widget
     * @return {string} texte de la requête où les apostrophes contenues à l'intérieur des valeurs sont remplacées par des doubles apostrophes.
     * ex. "(COMMUNE = 'SORDES L'ABBAYE' OR COMMUNE = 'PONT L'ABBE') AND DOMANIALITE = 'T'" => "(COMMUNE = 'SORDES L''ABBAYE' OR COMMUNE = 'PONT L''ABBE') AND DOMANIALITE = 'T'"
     */
    const replaceWhereClauseQuotedText = (request) => {

      // split avec l'expression régulière /\s+(AND|OR)\s+/,
      // qui correspond à un ou plusieurs espaces suivis de "AND" ou "OR" suivis de un ou plusieurs espaces.
      // Cela nous permet de diviser la chaîne en plusieurs parties en fonction des occurrences de "AND" ou "OR",
      // tout en supprimant les espaces autour de ces mots-clés.

      // la string "(COMMUNE = 'SORDES L'ABBAYE' OR COMMUNE = 'PONT L'ABBE') AND DOMANIALITE = 'T'"
      // devient le tableau suivant : ["(COMMUNE = 'SORDES L'ABBAYE'", "OR COMMUNE = 'PONT L'ABBE')", "AND DOMANIALITE = 'T'"]
      const criterias = request.split(/\s+(AND|OR)\s+/);

      const processedCriterias = criterias.map(criteria => replaceWhereClausePartQuotedText(criteria));

      return processedCriterias.join(' ');
    };

    /**
     * Remplace toutes les occurrences de l'apostrophe par deux apostrophes, sauf la première et la dernière occurrence.
     * @param criteria extrait de la clause where
     * @return {string} extrait de la clause
     */
    const replaceWhereClausePartQuotedText = (criteria) => {

      // Recherche de la première occurrence de l'apostrophe
      const firstIndex = criteria.indexOf("'");

      // Si aucune apostrophe n'est trouvée, retournez la chaîne inchangée
      if (firstIndex === -1) {
        return criteria;
      }

      // Recherche de la dernière occurrence de l'apostrophe en partant de la fin
      const lastIndex = criteria.lastIndexOf("'");

      // Séparez la chaîne en trois parties : avant, entre, et après les apostrophes. Ex. "OR COMMUNE = 'PONT L'ABBE')"
      const beforeQuote = criteria.slice(0, firstIndex + 1);            // Jusqu'à la première apostrophe incluse. Ex. "OR COMMUNE = '"
      const betweenQuotes = criteria.slice(firstIndex + 1, lastIndex);  // Entre les apostrophes. Ex. "PONT L'ABBE"
      const afterQuote = criteria.slice(lastIndex);                     // Après la dernière apostrophe incluse. Ex. "')"

      // Remplacez toutes les apostrophes dans la partie "entre" par deux apostrophes. Ex. PONT L''ABBE
      const replacedBetweenApostrophes = betweenQuotes.replace(/'/g, "''");

      // Concaténez les trois parties pour former la chaîne résultante.
      return beforeQuote + replacedBetweenApostrophes + afterQuote;     // Ex. "OR COMMUNE = '" + "PONT L''ABBE" + "')"
    };

    const DATE_TYPES = [
      'java.util.Date',
      'java.sql.Timestamp',
      'java.sql.Time',
      'java.sql.Date',
    ];
    const getWhereClause = (spatialClause, sentWhereClause, scope) =>{
      if (spatialClause !== '') {
        sentWhereClause = sentWhereClause === '' ? 
          spatialClause : '(' + sentWhereClause + ') AND ' + spatialClause;
      } else if (scope.currentSelectionClause !== '') {
        sentWhereClause = sentWhereClause === '' ? 
          scope.currentSelectionClause : '(' + sentWhereClause + ') AND ' + scope.currentSelectionClause;
      }
      return sentWhereClause;
    }
     /**
     * Créé le filtre utilisé pour récupérer les données (QueryFactory.pdata/QueryFactory.getCount)
     * en associant le critère spatial (geolimit) avec le critère alphanumérique (sentWhereClause)
     * @param {string} sentWhereClause clause where écrite dans la textarea du widget
     * @return {string} filtre pour récupérer les objets du requêteur
     */
    const createRequestFilter = (sentWhereClause,scope) =>{
      const wktObj = new ol.format.WKT();

      let spatialClause = '';

      if (scope.geoLimit.value === 'View') {
        const currExtent = scope.map .getView() .calculateExtent(scope.map.getSize());
        const polyExtent = ol.geom.Polygon.fromExtent(currExtent);
        const wktStr = wktObj.writeGeometry(polyExtent);
        spatialClause = 'INTERSECTS(geom, ' + wktStr + ')';
      } else if (scope.geoLimit.value === 'Zone') {
        spatialClause = scope.currentSelectionClause;
      }

      if (scope.geoLimit.value !== 'BD') {
        sentWhereClause = getWhereClause(spatialClause, sentWhereClause,scope);
      }

      // KIS-3145 : gérer la présence d’apostrophe dans les valeurs d’attribut
      return replaceWhereClauseQuotedText(sentWhereClause);
    }

    const mapProjCode = (scope) => {
      return scope.map.getView().getProjection().getCode();
    };

    /**
     * Gestion du cas où aucun objet n'a été trouvé.
     *
     * Dans le cas où la recherche se fait sur la vue courante et que
     * la vue courante est en dehors de la zone de rectricion,
     * un message qui indique cet état de fait est affiché.
    */
    const checkNoResult = (scope) => {
      const currExtent = scope.map.getView().calculateExtent(scope.map.getSize());
      const polyExtent = ol.geom.Polygon.fromExtent(currExtent);
      if (scope.geoLimit.value !== 'View' || !gcRestrictionProvider.hasRestrictionQuery()) {
        // -- Ni cas de la vue courante, ni restriction géographiques
        require('toastr').warning(
            $filter('translate')('quickfilter.no_value')
        );
      }
      else {
        // -- Cas de la vue courante avec restriction géographiques
        const geom = formatGeoJson.writeGeometryObject(polyExtent);
        gcRestrictionProvider.GeometryInRestriction(
            geom, mapProjCode(scope)
        ).then((res) => {
              if (!JSON.parse(res.data)) {
                // -- Rien n'et trouvé parcequ'on est hors de la restriction géographique
                gcRestrictionProvider.showErrorMessage();
              }
              else {
                // -- Rien n'est trouvé, mais on est dans la zone de restriction géographique
                require('toastr').warning(
                    $filter('translate')('quickfilter.no_value')
                );
              }
            },
            (res) => {
              console.error(res.data);
            }
        );
      }
    };
    /**
     * Affichage de la popup info box avec la liste des objets trouvés.
     * Si rien n'est trouvé affichage d'un toastr.
     *
     * @param {*} data GeoJSON des objets trouvés
     * @param {*} polyExtent Etendue de la vue courante
    */
    const showResult = (data, scope) => {
      scope.pop = gcPopup.open({
        scope: scope,
        title: 'Informations : ',
        content: '<div selectfeaturetreewidget></div>',
        showClose: true,
        minimizeMaximize: true,
        resizable: true,
        minWidth: 430,
        className: 'no-default-height',
        onclose: function() {
          SelectManager.clear();
        }
      });
      if (data.totalFeatures === 0) {
        checkNoResult(scope);
      }
    };

    /**
     * Au clic sur le bouton "Exécuter" du panel ou le bouton "Ok" de la popup
     * de requête paramétrée.
     * Exécute la requête lorsque le fti est spatial.
     * @param reset true au clic sur le bouton "Exécuter" du panel
     *              afin d'afficher le titre générique
    */
    const search  = (reset = false, scope) =>{
      let sentWhereClause = scope.whereClause;
      console.log(sentWhereClause);

      gaDomUtils.showGlobalLoader();

      // KIS-3145 : gérer la présence d’apostrophe dans les valeurs d’attribut
      const requestFilter = createRequestFilter(sentWhereClause,scope);

      QueryFactory.pdata(scope.selectedcomponent.uid, requestFilter, mapProjCode(scope)).then(
          (res) => {
            // KIS-2886 BONUS: une notification prévient l’utilisateur
            // lorsque la recherche a bien été faite mais qu’il n’y a pas de résultat
            // remplissant les conditions entrées.
            // Parce que sinon on ne sait pas si la requête a été réalisée ou pas.
            if (res.data && Array.isArray(res.data.features)
                && res.data.features.length === 0) {
              gaDomUtils.hideGlobalLoader();
              require('toastr').warning($filter('translate')('query.noResult'));
            } else {
              SelectManager.addFeaturesFromGeojson(res.data);
              gaDomUtils.hideGlobalLoader();
              if (SelectManager.getExtent()[0]!==Infinity) {
                scope.map.getView()
                .fit(SelectManager.getExtent(), scope.map.getSize());
              }
              if ($rootScope.xgos && $rootScope.xgos.sector === 'hpo' &&
                  $rootScope.xgos.hpo
              ) {
                $rootScope.xgos.hpo.clauseWhere = sentWhereClause;
                if (scope.map && scope.map.getView)
                  $rootScope.xgos.hpo.srid = mapProjCode(scope);
              }

              if (scope.pop) {
                scope.pop.destroy();
              }
              showResult(res.data, scope);
              gaDomUtils.hideGlobalLoader();
            }},
          () => {
            // failed
            gaDomUtils.hideGlobalLoader();
          }
      );
    }
    
    const replaceParamInQuery = (scope) =>{
      if (
        scope.queryParam.value != '' &&
        !(scope.queryParam.value === undefined)
      ) {
        scope.whereClause = scope.whereClause.replace(
          '#' + scope.currtParamName + '#',
          scope.queryParam.value
        );
      }
      scope.parameterValuePopup.close();

      scope.parameterValuePopup = null;

      //Réinitialise la valeur de queryParam pour qu'elle soit pas réaffiché lors de la réouverture de la popup
      scope.queryParam.value = '';
      scope.currtParamName = '';
      let geographic = scope.selectedcomponent.geographic ? 
                                            scope.selectedcomponent.geographic : scope.selectedcomponent.fti.geographic;
      executeQuery(!geographic,scope);
    }
    const readDate = (d) =>{
      const months = d.getMonth() + 1 < 10 ? '0' + (d.getMonth() + 1) : d.getMonth() + 1;
      const days = d.getDate() < 10 ? '0' + d.getDate() : d.getDate();
      return {
        label: days + '/' + months + '/' + d.getFullYear(),
        value: d.getFullYear() + '-' + months + '-' + days
      };
    }
    const getAllValuesAttribut = (scope) =>{
      scope.retrievingAttValues = true;

      let sentWhereClause, fti;
      if (!scope.loadedFav) {
        sentWhereClause = scope.whereClause;
        fti = scope.selectedcomponent.fti.uid;
      }
      else {
        sentWhereClause = scope.loadedFav.where;
        fti = scope.loadedFav.fti;
      }
      let splitedStr = sentWhereClause.split(' ');
      let attvalue = scope.attvalue ? scope.attvalue : splitedStr[0];

      QueryFactory.dataattribute(fti, attvalue).then(function(res) {
        if (
          res.data.length === 0 ||
          (res.data.length === 1 && res.data[0] === '')
        ) {
          require('toastr').info(
            $filter('translate')('model.featuretypes.attributes.no_values')
          );
        }
        else {
          scope.selectedValueTest = '';
          scope.attvaluesTest = [];
          res.data.forEach((att) => {
            if (att !== '') {
              let aDate = { value: att, label: att};
              if (scope.currentAttributeIsDate && att !== null) {
                const d = new Date(att);
                // valid date
                if (d.getTime() > 0) {
                  aDate = readDate(d);
                }
              }
              scope.attvaluesTest.push(aDate);
            }
          });
        }
        if (scope.loadedFav) {
          if (angular.isObject(scope.loadedFav.fti)) {
            scope.selectedcomponent = scope.loadedFav.fti;
          }
          else {
            scope.selectedcomponent = FeatureTypeFactory.getFeatureByUid(
              scope.loadedFav.fti
            );
          }
        }
      }).finally(()=> {
        scope.retrievingAttValues = false;
      });
    }
    const queryWithParameters = (sentWhereClause, reset, scope) =>{
      // KIS-2886 BONUS: on assure la présence d'un caractère espace avant le 1er # rencontré:
      const firstHashtag = sentWhereClause.indexOf('#');
      if (firstHashtag > - 1 && sentWhereClause.charAt(firstHashtag -1) !== ' ') {
        sentWhereClause
          = sentWhereClause.slice(0, firstHashtag) + ' ' + sentWhereClause.slice(firstHashtag);
      }

      const splitedStr = sentWhereClause.split('#');

      if (splitedStr.length > 2) {
        scope.currtParamName = splitedStr[1];
        scope.attvalue = splitedStr[0].split(' ')[
          splitedStr[0].split(' ').length - 5
        ];

        // KIS-2886 BONUS: vérifie le type de l'attribut contenu dans scope.attvalue
        // si date on affichera un datepicker dans la popup
        const fti = scope.loadedFav
          ? FeatureTypeFactory.getFeatureByUid(scope.loadedFav.fti)
          : scope.selectedcomponent.fti;
        if (fti && Array.isArray(fti.attributes)) {
          const attribute = fti.attributes.find(attr => attr.name === scope.attvalue);
          scope.currentAttributeIsDate
            = attribute !== undefined && DATE_TYPES.includes(attribute.type);
        }

        if (scope.parameterValuePopup != null) {
          scope.parameterValuePopup.close();
        }

        getAllValuesAttribut(scope);

        scope.parameterValuePopup = gcPopup.open({
          scope: scope,
          title: scope.request_name
            && !reset ? scope.request_name : $filter('translate')('query.paramRequest'),
          template:
            'js/XG/widgets/mapapp/query/views/queryValueParameter.html',
          showClose: true,
          position: { top: '40vh', left: '40vw'},
          className: 'query-parameter',
          resizable: true,
          minHeight: 175,
          minWidth: 428
        });
      }
    }
    /**
     * Remplace le cararctère singleQuote par deux catactère singleQuote dans la clause where.
     * Permet le fonctionnement de la requête quand la clause where
     * contient une chaine de caractères avec une apostrophe.
     * Evite ainsi une CQLException.
     * @param {string} whereClause clause where de la requête qui va être envoyée à QueryFactory
     * @return {string} whereClause avec quotes échappées pour Postgres (remplace ' par '')
    */
    const escapeSingleQuote = (whereClause) => {
      let index = -1;
      let startIndex = 0;
      const indices = [];
      // recupère les positions de quotes dans la clause where
      while ((index = whereClause.indexOf('\'', startIndex)) > -1) {
        indices.push(index);
        startIndex = index + 1;
      }
      if (indices.length > 2) {
        let whereClauseEscaped = whereClause.slice();
        // ne tient pas compte des quotes englobantes de string
        indices.shift();
        indices.pop();
        // insère une quote supplémentaire à chaque occurence de quote
        for (const indice of indices) {
          whereClauseEscaped
            = whereClauseEscaped.slice(0, indice) + '\'' + whereClauseEscaped.slice(indice);
        }
        return whereClauseEscaped;
      }
      return whereClause;
    };
    /**
     * Au clic sur le bouton "Filtrer" lorsque une table non-spatiale est interrogée,
     * ouvre une datatable avec fonction d'export csv
    */
    const openRequestedTable = (scope) => {
      scope.whereClauseEscaped = escapeSingleQuote(scope.whereClause);
      ngDialog.open({
        template:
            'js/XG/widgets/mapapp/query/views/queryRequestedTable.html',
        className: 'ngdialog-theme-plain large miniclose',
        closeByDocument: false,
        scope: scope,
      });
    }
     /**
     * Au clic sur le bouton "Exécuter" des onglets Recherche et Favoris,
     * Différencie les requêtes paramétrées, les  requêtes sur des tables et les requêtes sur les composants géographiques
     * @param {boolean} isTable est true à l'exécution d'une requête favorite sur une table
     */
    const executeQuery = (isTable, scope) =>{
      if (scope.whereClause.includes('#')) {
        queryWithParameters(scope.whereClause, true, scope);
      } else {
        // KIS-3288: recherche sans clause (tous les objets d'un composant)
        const uid = scope.selectedcomponent.fti ? 
                      scope.selectedcomponent.fti.uid : scope.selectedcomponent.uid;
        const payLoad = {
          groups: [
            {
              filter: createRequestFilter(scope.whereClause,scope),
              operand: 'count',
              ftiUids: [uid]
            }
          ]
        };

        QueryFactory.getCount(payLoad).then(
            res => {
              if (res.data.etat === 'erreur' && res.data.hasOwnProperty('errorList')) {

                // "gestion" KIS de l'erreur serveur
                require('toastr').error(gaJsUtils.errorListToOneString(res.data.errorList));

              } else if (Array.isArray(res.data.objValeur.groups) && 
                          res.data.objValeur.groups.length > 0 && 
                          typeof res.data.objValeur.groups[0].result == 'number') {
                const count = res.data.objValeur.groups[0].result;

                // si la requête s'est bien exécutée
                if (count <= 10000) {
                  // le nombre d'objets à retourner est faible => on poursuit sur la requête des objets
                  ( scope.tablesOnly || isTable ) ? openRequestedTable(scope) : search(true, scope);
                } else {

                  // le nombre d'objets à retourner est important => l'utilisateur doit refaire sa recherche avec un critère
                  swal(
                    {
                      title: $filter('translate')('query.getCountSwalTitle'),
                      text: $filter('translate')('query.dataOverLimit').replace('$1', count),
                      type: 'warning',
                      showCancelButton: false,
                      confirmButtonColor: '#5cb85c',
                      confirmButtonText: $filter('translate')('common.ok'),
                      closeOnConfirm: true
                    }
                  );
                }
              } else {
                require('toastr').error($filter('translate')('query.getCountServerErrorMessage'));
                console.error($filter('translate')('query.getCountServer'));
              }
            }
        ).catch(error => {
          console.error('executeQuery : ', error);
        });
      }
    }
    const gofavoris = (fav, name, scope) => {
      scope.currentAttributeIsDate = false;
      scope.attvaluesTest = [];
      if (fav.attribut && fav.attribut.att) {
        scope.atttype = fav.attribut.att.type;
        scope.attName = fav.attribut.att.name;
        scope.currentAttributeIsDate = DATE_TYPES.includes(scope.atttype);
      }
      scope.loadedFav = fav;
      if (angular.isObject(fav.fti)) {
        scope.selectedcomponent = fav.fti;
      }
      else {
        scope.selectedcomponent = FeatureTypeFactory.getFeatureByUid(
          fav.fti
        );
      }

      if (!scope.selectedcomponent) {
        require('toastr').error('component missing');
        return;
      }
      // KIS-3389: La restriction géographique n'est pas sauvegardé dans les requêtes enregistrées
      if (fav.geoLimit) {
        if(scope.geoLimit.value) {
          scope.geoLimit.value = fav.geoLimit;
        } else {
          scope.geoLimit = fav.geoLimit;
        }
      }
      scope.request_name = name;
      scope.whereClause = fav.where;
      executeQuery(!scope.selectedcomponent.geographic, scope);
    }
    return {
      replaceWhereClauseQuotedText: replaceWhereClauseQuotedText,
      DATE_TYPES: DATE_TYPES,
      getWhereClause: getWhereClause,
      createRequestFilter: createRequestFilter,
      mapProjCode: mapProjCode,
      checkNoResult: checkNoResult,
      showResult: showResult,
      search: search,
      replaceParamInQuery: replaceParamInQuery,
      readDate: readDate,
      getAllValuesAttribut: getAllValuesAttribut,
      queryWithParameters: queryWithParameters,
      openRequestedTable: openRequestedTable,
      executeQuery: executeQuery,
      gofavoris: gofavoris
    };
  };
  QueryService.$inject = [
    'FeatureTypeFactory',
    '$rootScope', 
    'QueryFactory', 
    'gcPopup',
    'gaDomUtils',
    'gaJsUtils', 
    '$filter', 
    'ngDialog',
    'SelectManager',
    'gcRestrictionProvider'
  ];
  return QueryService;
});
