'use strict';
define(function () {
  var defaultfilter = function (FeatureTypeFactory, extendedNgDialog, ngDialog, $filter,
      QueryFactory, ConfigFactory, defaultFiltersFactory, gaDomUtils, $timeout) {
    return {
      templateUrl:
          'js/XG/widgets/mapapp/defaultfilters/views/defaultfilter.html',
      restrict: 'EA',
      scope: {
        filter: '=',
        config: '=',
        cancel: '='
      },
      link: function (scope) {

        /*******************
         *     FILTRE      *
         *******************/

        /**
         * Vérifie si un titre de filtre existe déjà dans la liste des titres de filtres.<br>
         * Exécutée au changement de l'input du titre du filtre
         * <code>titleExists</code> est true quand le titre du filtre existe déjà pour afficher un message d'erreur
         * Si <code>titleExists</code> est true, la classe <code>propertyExists</code> est insérée dans l'input du titre du filtre
         */
        scope.onTitleChange = () => {
          let index = -1;
          if (scope.config.filterTitles && scope.filter.data && scope.filter.data.title) {
            index = scope.config.filterTitles.findIndex(
                filtertitle => filtertitle === scope.filter.data.title);
          }
          const isTitlePresent = index > -1;
          if (scope.titleExists !== isTitlePresent){
            // recalcule la hauteur minimale de la popup quand le message d'erreur apparait/disparaît
            calculeFilterPanelHeightByClausesLength();
          }
          scope.titleExists = isTitlePresent;
        }

        /**
         * Vérifie si un nom de filtre existe déjà dans la liste des noms de filtres.<br>
         * Exécutée au changement de l'input du nom de filtre<br>
         * Si <code>nameExists</code> est true, la classe <code>propertyExists</code> est insérée dans l'input du nom du filtre
         */
        scope.onNameChange = () => {
          let index = -1;
          if (scope.config.filterNames && scope.filter.data && scope.filter.name) {
            index = scope.config.filterNames.findIndex(
                filtername => filtername === scope.filter.name);
          }
          if (scope.nameExists !== index > -1){
            // recalcule la hauteur minimale de la popup quand le message d'erreur apparait/disparaît
            calculeFilterPanelHeightByClausesLength();
          }
          scope.nameExists = index > -1;
        }

        /**
         * Vérifie la validité du nouveau filtre
         * @returns {boolean} <code>true</code> si le nom, le titre et 1 clause sont renseignés
         */
        scope.validNewFilter = () => {
          if (scope.filter) {
            let validClauses = false;
            // check every where clause
            if (scope.filter.data && scope.filter.data.where_clauses
                && scope.filter.data.where_clauses.length > 0
                && scope.filter.data.where_clauses[0].fti) {
              validClauses = true;
            }
            return (
                scope.filter.name !== '' &&
                scope.filter.data.title !== '' &&
                validClauses
            );
          } else {
            return false;
          }
        };

        /**
         * Transmet le filtre au composant parent pour sauvegarde
         * en tant que paramètre du repo.<br>
         * Ferme la popup
         */
        scope.saveFilter = () => {
          scope.$emit('saveFilter', scope.filter);
          if (scope.addEditFilterDialog) {
            scope.addEditFilterDialog.close();
          }
        };

        /**
         * Utilise la méthode du composant parent
         * pour fermer la popup d'ajout/édition du filtre
         */
        scope.cancelFilter = () => {
          if (scope.cancel) {
            scope.cancel();
          }
        };

        /**
         * Au changement de catégorie de l'auto-cocomplete,
         * Si la catégorie sélectionnée est null
         * alors on supprime le thème du filtre courant
         */
        scope.$on('autocomplete-selected', (event, data) => {
          const datatype = data ? data.type : null;
          if (datatype === 'catégorie'){
            const selected = data ? data.selected : null;
            if (!selected) {
              scope.filter.data.theme = null;
              if (scope.filter.theme) {
                delete scope.filter.theme;
              }
            }else if (selected.name && selected.name.length > 0) {
              scope.filter.data.theme = selected;
            }
          }
        });

        /**
         * Limite le redimensionnement vertical de la popup d'ajout/édition de filtre
         */
        const refreshFilterPopupPanelMinHeight = () => {
          const popupPanel = $('.addEditFiltre').next().children()[0];
          $(popupPanel).css('min-height', calculeFilterPanelHeightByClausesLength());
        };

        /**
         * Calcule la valeur de hauteur minimale à donner à la popup d'ajout/édition de filtre
         * 50px barre de titre, 330px le header, 20px les boutons
         * @return {string} valeur numérique suffixée par 'px'
         */
        const calculeFilterPanelHeightByClausesLength = () => {
          let listHeight  = 50 + 330 + 20;
          if (scope.filter){
            if (scope.titleExists || scope.nameExists) {
              if (scope.titleExists && scope.nameExists){
                listHeight += 60;
              } else {
                listHeight += 30;
              }
            }
            if (scope.filter.data && scope.filter.data.where_clauses) {
              listHeight = 47 * scope.filter.data.where_clauses.length + listHeight;
            }
          }
          listHeight = listHeight > 800 ? 800 : listHeight;
          return listHeight + 'px';
        };

        /**********************
         *     CATEGORIES     *
         **********************/

        /**
         * Ré-initialise la catégorie
         * après sauvegarde de la catégorie
         *
         */
        const resetEditCategory = () => {
          // boolean pour éviter d'ouvrir plusieurs popup
          scope.newCategoryFormIsOpen = false;
          // réinitialise l'objet edit_category pour prochain usage
          scope.edit_category = {
            name: ''
          };
        };

        /**
         * Au clic sur le bouton "+"
         * Ouvre la popup de saisie du nom d'une nouvelle catégorie
         */
        scope.openNewCategoryPopup = () => {
          if (!scope.newCategoryFormIsOpen) {
            scope.newCategoryFormIsOpen = true;
            scope.edit_category = {
              name: ''
            };
            ngDialog.open({
              template:
                  'js/XG/widgets/mapapp/defaultfilters/views/modal/modal.defaultfilters.category.add.html',
              className: 'ngdialog-theme-plain width360 miniclose',
              closeByDocument: false,
              title: $filter('translate')('model.clauses.addCategory'),
              scope: scope,
              preCloseCallback: resetEditCategory
            });
          }
        };

        /**
         * Au clic sur le bouton "Enregistrer" de la popup,
         * ajoute la catégorie à la liste de catégorie.
         * Ré-initialise le model de la nouvelle catégorie
         */
        scope.saveCategory = () => {
          if (!scope.config.categories) {
            scope.config.categories = [];
          }
          defaultFiltersFactory.addNewCategory(scope.config.ConfigName, scope.edit_category).then(
              res => {
                scope.config.categories = res;
                scope.edit_category = {
                  name: ''
                };
              },
              () => {
                require('toastr').error(
                    $filter('translate')('model.default_filters.newCategoryError')
                );
              }
          );
        };

        /**
         * Ouvre la popup de suppression d'une catégorie
         * au clic sur le bouton poubelle rouge dans la popup d'ajout/édition de filtre.
         * @see modal.defaultfilters.category.remove.html
         */
        scope.openDeleteCategoryPopup = () => {
          if (!scope.deleteCategoryFormIsOpen) {
            scope.deleteCategoryFormIsOpen = true;
              scope.categoryHasFilter = true;
              scope.edit_category = scope.filter.data.theme;
              const index = scope.config.categories.findIndex(c => c.name === scope.edit_category.name);
              if (index > -1 && scope.config.categories[index].filters && scope.config.categories[index].filters.length > 0){
                scope.edit_category.filters = scope.config.categories[index].filters;
              }
            ngDialog.open({
              template:
                  'js/XG/widgets/mapapp/defaultfilters/views/modal/modal.defaultfilters.category.remove.html',
              className: 'ngdialog-theme-plain width500 category-remove miniclose',
              closeByDocument: false,
              title: $filter('translate')('model.clauses.addCategory'),
              scope: scope,
              preCloseCallback: () =>{
                scope.deleteCategoryFormIsOpen = false;
              }
            });
          }
        };

        /**
         * Au clic sur oui/non, dans la popup de suppression de catégorie,
         * envoie un objet au parent:  {nom catégorie, suppression des filtres}
         * @param deleteAll est true si on supprime aussi les filtres de la catégorie
         * @see defaultFiltersFactory.deleteCotegory
         */
        scope.deleteCategorySubmit = (deleteAll) => {
          const categoryToDelete = {
            name: scope.filter.data.theme.name,
            deleteAll:deleteAll
          }
          gaDomUtils.showGlobalLoader();
          scope.$emit('onDeleteCategory', categoryToDelete);
          scope.deleteCategoryFormIsOpen = false;
        };

        /**
         * Dans l'input de la popup d'ajout de catégorie, à chaque caractère saisi,
         * vérifie si le nom de la nouvelle catégorie est déjà présent dans la liste de catégorie
         * Passe <code>categoryExists</code> à true quand le nom de la nouvelle catégorie est égal à un nom de catégorie existante
         */
        scope.onCategoryChange = () => {
          scope.categoryExists = scope.config.categories.findIndex(
              c => c.name.toLowerCase() === scope.edit_category.name.toLowerCase()) > -1;
        };

        /**
         * Dans la popup d'ajout d'une nouvelle catégorie,
         * adapte la largeur de l'input de nouvelle catégorie au label placé au-dessus
         * @return {string} hauteur de l'élément avec le suffixe 'px'. Ex: '120px'
         */
        scope.adjustNewCategoryInputWidth = () => {
          const labelHtml = document.getElementById('defaultFilters-category-title');
          const inputHtml = document.getElementById('defaultFilters-category-title');
          if (labelHtml && inputHtml){
            return labelHtml.offsetWidth + 'px';
          }
        };

        /**
         * Etend la card de la liste des filtres de la catégorie en cours de suppression
         * @param category objet contenant la liste des filtres de la catégorie
         */
        scope.expandCard = (category) => {
          category.show = !category.show;
        };


        /*******************
         *     CLAUSES     *
         *******************/

        // tableaux des opérateurs possibles suivant le type de l'attribut sélectionné dans la clause
        const stringOperators = ['=', 'LIKE', 'IN'];
        const numericOperators = ['=','>','>=','<','<=', '<>'];

        /**
         * Renvoie le nombre servant à calculer la propriété css 'height' de la table de clauses.<br>
         * On compte 47px par clause et 163px de base<br>
         * Limite la hauteur à 800px
         */
        scope.calculeClauseTableHeightByClausesLength = () => {
          const buttons = 163;
          let listHeight;
          if (scope.filter.data && scope.filter.data.where_clauses) {
            listHeight = 47 * scope.filter.data.where_clauses.length + buttons;
          }
          listHeight = listHeight > 800 ? 800 : listHeight;
          return listHeight + 'px';
        };

        /**
         * Initialise la table des clauses du filtre courant<br>
         * <ol>
         *   <li>scope.currentResources</li>
         *   <li>scope.currentResources[].composant</li>
         * </ol>
         *
         */
        const initListeClauses = () => {
          if (!scope.filter) {
            // initialise une liste de clauses vide si le filtre en input est null
            scope.filter = {};
            scope.currentResources = [];
          } else {
            // clauses du filtre comme lignes de la liste des clauses
            scope.currentResources = scope.filter.data.where_clauses;
            // création de la propriété 'Composant' pour affichage de la colonne dans la liste des clauses
            for (const clause of scope.currentResources) {
              clause.composant = FeatureTypeFactory.getFeatureByUid(clause.fti).alias;
            }
          }
        };

        // exécutée au démarrage
        initListeClauses();

        /**
         * Créé le tableau de fti
         * avant l'ouverture de la popup d'ajout/édition de clause en mode création
         * @see addSelectedFti
         */
        const openAddClauseDialog = () => {
          getFtisThenOpenAddEditClauseDialog(true);
        };

        /**
         * Créé le tableau de fti
         * avant l'ouverture de la popup d'ajout/édition de clause en mode édition
         * @see addSelectedFti
         */
        const openEditClauseDialog = () => {
          getFtisThenOpenAddEditClauseDialog(false);
        };

        /**
         * Ouvre la popup d'édition/ajout d'une clause
         * au clic sur le bouton d'édition de la table des clauses<br>
         * Supprime la variable <code>scope.edit_resource</code> à la fermeture de la popup
         * @see resetEditResource
         */
        const getFtisThenOpenAddEditClauseDialog = (isNew) => {
          if (!scope.addEditClauseDialog) {
            if (!scope.ftis) {
              // à la 1ère exécution après l'initialisation du widget
              FeatureTypeFactory.get().then(
                  res => {
                    // construit le tableau de fti exploitables par l'autocomplete
                    scope.ftis = res.map(fti => {
                      return {
                        name: fti.alias,
                        fti: fti
                      }
                    });
                    // ouverture de la popup
                    openAddEditClauseDialog(isNew, scope.ftis);
                  },
                  (err) => {
                    console.log(err);
                    // ouverture de la popup dans un état non opérationnel
                    openAddEditClauseDialog(isNew);
                  }
              );
            } else {
              // à partir de la 2nde exécution
              openAddEditClauseDialog(isNew, scope.ftis);
            }
          }
        };

        /**
         * Ouvre la popup d'édition de clause en mode ajout/modification:<br>
         * <ul>
         *  <li>Prépare les valeurs de model par défaut (opérateur, opérande)</li>
         *  <li>Initialise l'objet contenant les models de la popup: <code>scope.edit_clause</code></li>
         * </ul>
         * L'objet <code>edit_clause</code> est le conteneur des models de la popup d'ajout/édition de clause
         * @param isNew est true si la popup doit être ouverte en mode ajout
         * @param ftis est un tableau d'objets 'fti': <code>[{name: fti.alias, fti: fti},...]</code>
         */
        const openAddEditClauseDialog = (isNew, ftis) => {
          const title = isNew ? 'model.clauses.newClause' : 'model.clauses.editClause';

          // ré-initialisation de currentAttributeIsDate pour masquer le datepicket
          scope.currentAttributeIsDate = false;
          // initialisation de la configuration si aucune configuration n'existe
          if (!scope.config) {
            scope.config = {};
          }
          if (scope.edit_resource && !isNew) {
            // mode édition
            scope.edit_clause = scope.edit_resource;
            // création d'une propriété 'id' contenant le rang de la clause dans la liste de clauses du filtre (pour permettre la màj)
            scope.edit_clause.id = scope.filter.data.where_clauses.findIndex(
                wc => wc.fti === scope.edit_resource.fti && scope.edit_resource.where === wc.where);
            // récupération du fti de la clause
            if (ftis) {
              scope.edit_clause.selectedFti = ftis.find(f => f.fti.uid === scope.edit_resource.fti);
            }
            // récupération du texte de la clause
            if (scope.edit_resource.where) {
              scope.config.whereClause = scope.edit_resource.where;
            } else {
              scope.config.whereClause = '';
            }
          } else {
            // mode écriture: nouvelle clause
            scope.edit_clause = {};
            scope.config.whereClause = '';
          }

          // valeurs par défaut quelque soit le mode
          scope.edit_clause.operators = stringOperators;
          if (!scope.edit_clause.operand) {
            scope.edit_clause.operand = 'AND';
          }
          scope.edit_clause.editmode = !isNew;

          // active/désactive le select de composant
          scope.lockedClauseFti = !!scope.edit_clause.selectedFti && scope.config.whereClause.length > 0;

          scope.addEditClauseDialog = extendedNgDialog.open({
            template:
                'js/XG/widgets/mapapp/defaultfilters/views/modal/modal.defaultfilters.clause.addEdit.html',
            className:
                'ngdialog-theme-plain width400 nopadding miniclose addEditClause',
            closeByDocument: false,
            scope: scope,
            draggable: true,
            resizable: true,
            title: $filter('translate')(title),
            preCloseCallback: resetEditResource
          });
          // assure de placer la popup d'édition de clause au-dessus de la popup parente
          $timeout(()=> {
            const popupPanel = $('.addEditClause').next().children()[0];
            $(popupPanel).css('min-width', '400px');
            $(popupPanel).css('min-height', '405px');
          },500);

        };

        /**
         * Supprime la resource éditée
         * afin qu'{@link openAddEditClauseDialog} ouvre un formulaire vierge au clic sur le bouton "Ajouter".<br>
         * La variable <code>edit_resource</code> est gérée manuellement par un output ($emit) depuis editList
         * @see editList.js (scope.selectResource ligne 577)
         */
        const resetEditResource = () => {
          scope.edit_resource = undefined;
          scope.addEditClauseDialog = undefined;
          scope.config.whereClause = undefined;
          scope.edit_clause = undefined;
          scope.currentAttributeIsDate = false;
        };

        /**
         * Au changement sur le select des valeurs d'attribut de la popup de la clause:<br>
         * <ul>
         *   <li>Réinitialise la liste des valeurs</li>
         *   <li>Vérifie si l'attribut sélectionné est un type de date</li>
         * <ul>
         */
        scope.onAttributeChange = () => {
          scope.currentAttributeIsDate = false;
          scope.selectedattvalue = '';
          scope.attvalues = [];
          if (scope.edit_clause.selectedAttribute) {
            if ((scope.edit_clause.selectedAttribute.name !== 'fid') &&
                [
                  'java.util.Date',
                  'java.sql.Timestamp',
                  'java.sql.Time',
                  'java.sql.Date'
                ].indexOf(scope.edit_clause.selectedAttribute.type) !== -1
            ) {
              scope.currentAttributeIsDate = true;
            }
            if ([
              'java.lang.Integer',
              'java.lang.Double',
              'java.lang.Float',
              'java.util.Date',
              'java.sql.Timestamp',
              'java.sql.Time',
              'java.sql.Date'
            ].indexOf(scope.edit_clause.selectedAttribute.type) !== -1){
              scope.edit_clause.operators = numericOperators;
            }else {
              scope.edit_clause.operators = stringOperators;
            }
          }
        }

        /**
         * Récupère les valeurs de l'attribut sélectionné du compsant sélectionné
         * Copie adaptée de la méthode onAttributeChange de la directive querywidget
         */
        scope.getAttributeValues = () => {
          if (scope.edit_clause.selectedFti.fti.uid && scope.edit_clause.selectedAttribute.name) {
            let currentAttributeIsDate = false;
            scope.waitAttributeValues = true;
            QueryFactory.dataattribute(
                scope.edit_clause.selectedFti.fti.uid,
                scope.edit_clause.selectedAttribute.name
            ).then((res) => {
                  if ((scope.edit_clause.selectedAttribute.name !== 'fid') &&
                      [
                        'java.util.Date',
                        'java.sql.Timestamp',
                        'java.sql.Time',
                        'java.sql.Date',
                      ].indexOf(scope.edit_clause.selectedAttribute.type) !== -1
                  ) {
                    // cache le select des valeurs et affiche le datepicker
                    currentAttributeIsDate = true;
                  }
                  if (
                      res.data.length === 0 ||
                      (res.data.length === 1 && res.data[0] === '')
                  ) {
                    // l'attribut n'a aucune valeur
                    require('toastr').info(
                        $filter('translate')('model.featuretypes.attributes.no_values')
                    );
                    if (scope.attvalues) {
                      // ré-initialise la liste des valeurs
                      scope.attvalues = [];
                    }
                  } else {
                    // l'attribut contient une liste de valeurs
                    scope.selectedattvalue = '';
                    scope.attvalues = [];
                    // ré-initialise l'objet model du datepicker
                    scope.edit_clause.editvalue = null;

                    // définie deux variables par valeur
                    for (let att of res.data) {
                      if (att !== '') {
                        let label = att;
                        if (currentAttributeIsDate && att !== null) {
                          // transformation de la date en string yyyy-MM-dd
                          const d = new Date(att);
                          // valid date
                          if (d.getTime() > 0) {
                            label =
                                d.getFullYear() +
                                '-' +
                                (d.getMonth() + 1) +
                                '-' +
                                d.getDate();

                            att = d.toISOString();
                          }
                        }

                        // assigne les variables aux propriétés d'un objet inséré dans le tableau des valeurs
                        scope.attvalues.push({
                          value: att,
                          label: label,
                        });
                      }
                    }
                  }
                  scope.waitAttributeValues = false;
                },
                () => {
                  require('toastr').error(
                      $filter('translate')('model.featuretypes.attributes.no_values')
                  );
                  scope.waitAttributeValues = false;
                });
          }
        };

        /**
         * Entoure la valeur d'attributs string ou date de simple quotes
         * après un clic sur le bouton vert '+' pour insérer le critère dans la textarea
         */
        scope.addAttributeValueOptionalQuotes = () => {
          if (scope.edit_clause.selectedattvalue && scope.edit_clause.selectedattvalue.value) {
            // si une valeur d'attribut est sélectionnée
            const isDateOrStringAttribute = scope.edit_clause.selectedAttribute.name !== 'fid' &&
            ['java.util.Date','java.sql.Timestamp','java.sql.Time','java.sql.Date','java.lang.String']
            .indexOf(scope.edit_clause.selectedAttribute.type) !== -1;
            if (isDateOrStringAttribute) {
              // si l'attribut est de type String ou Date on entoure la valeur de quotes
              scope.config.whereClause =
                  scope.config.whereClause + '\'' + scope.edit_clause.selectedattvalue.value + '\' ';
            } else {
              // si l'attribut est numérique
              scope.config.whereClause =
                  scope.config.whereClause + scope.edit_clause.selectedattvalue.value;
            }
          }
        };

        /**
         * Transforme une date en string formaté yyyy-MM-dd
         * @param date date au format Date
         * @return {string|null} renvoie null si la date en entrée n'est pas valide
         */
        const dateToString = (date) => {
          const d = new Date(date);
          // valid date
          if (d.getTime() > 0) {
            return d.getFullYear() +
                '-' +
                (d.getMonth() + 1) +
                '-' +
                d.getDate();
          }
          return null;
        }

        /**
         * Au clic sur le bouton "+" de la popup d'ajout / édition d'une clause
         * Ajoute un nouveau critère dans la textarea
         */
        scope.updateClause = () => {
          // si un attribut est sélectionné
          if (scope.edit_clause.selectedAttribute) {
            // si la clause n'est pas vide
            if (scope.config.whereClause && scope.config.whereClause.length > 0
                && scope.config.whereClause !== "\"\"") {
              if (!scope.config.whereClause.endsWith(' ')) {
                scope.config.whereClause = scope.config.whereClause + ' ';
              }
              // l’opérateur AND ou OR est inséré dans la clause avant l’ajout du nouveau critère.
              scope.config.whereClause = scope.config.whereClause + scope.edit_clause.operand + ' '
                  + scope.edit_clause.selectedAttribute.name;
            } else {
              // si aucun critère n’est encore présent (=clause vide) dans la clause alors l’opérateur AND ou OR n’est pas ajouté dans la clause.
              scope.config.whereClause = scope.edit_clause.selectedAttribute.name;
            }
          }
          if (scope.edit_clause.operator) {
            // si je coche la négation alors le mot clé NOT est inséré avant l’opérateur dans la construction du critère
            scope.config.whereClause = scope.edit_clause.not ? scope.config.whereClause + ' NOT '
                : scope.config.whereClause;
            scope.config.whereClause = scope.edit_clause.operator ? scope.config.whereClause
                + " "+ scope.edit_clause.operator + " ": scope.config.whereClause;

            scope.config.whereClause = scope.edit_clause.operator ==='IN' ? scope.config.whereClause + '(':scope.config.whereClause;

          }
          if (scope.edit_clause.editvalue) {
            // si l'attribut est une date et si une date du calendrier a été sélectionné
              const dateString = dateToString(scope.edit_clause.editvalue);
              scope.config.whereClause = scope.config.whereClause + '\'' + dateString + '\'';
          } else if (scope.edit_clause.selectedattvalue
              && scope.edit_clause.selectedattvalue.value) {
            if (scope.currentAttributeIsDate) {
              const datestring = dateToString(scope.edit_clause.selectedattvalue)
                scope.config.whereClause = scope.config.whereClause + '\'' + datestring + '\'';
            }else{
              //  si l’attribut est de type Chaine de caractères alors la valeur est encapsulée par des simples quotes
              scope.addAttributeValueOptionalQuotes();
            }
          } else if (scope.edit_clause.selectedAttribute
              && ['java.util.Date', 'java.sql.Timestamp', 'java.sql.Time', 'java.sql.Date',
                'java.lang.String']
              .indexOf(scope.edit_clause.selectedAttribute.type) !== -1) {
            // Si l'attribut est de type string et aucune valeur n’a été sélectionnée
            // alors je dois trouver '' dans le critère
            scope.config.whereClause = scope.config.whereClause + '\'\''
          }
          scope.config.whereClause = scope.edit_clause.operator ==='IN' ? scope.config.whereClause + ') ':scope.config.whereClause;
        };

        /**
         * Au clic sur le bouton "Enregistrer" sur la popup d'ajout/édition d'une clause<br>
         * Ajoute ou modifie un élément du tableau 'where_clauses' à partir de la variable <code>scope.config.whereClause</code>
         */
        scope.saveClause = () => {
          if (scope.edit_clause.selectedFti && scope.config.whereClause
              && scope.config.whereClause.length > 0) {

            // créé l'objet clause à partir des models
            const clause = {
              fti: scope.edit_clause.selectedFti.fti.uid,
              where: scope.config.whereClause,
              composant: scope.edit_clause.selectedFti.fti.alias
            }

            // échappe les quotes contenus dans les chaînes de caractères
            clause.where = escapeStringQuotes(clause.where);

            if (scope.edit_clause.editmode) {
              // mode édition
              if (scope.edit_clause.id > -1) {
                // si la clause existe déjà on la remplace
                scope.filter.data.where_clauses[scope.edit_clause.id] = clause;
              } else {
                // si la clause n'existe pas on l'ajoute à la
                scope.filter.data.where_clauses.push(clause);
              }
            } else {
              // mode création
              scope.filter.data.where_clauses.push(clause);
            }
            // met à jour la table des clauses
            scope.currentResources = scope.filter.data.where_clauses;
            scope.edit_resource = undefined;

            refreshFilterPopupPanelMinHeight();

            if (scope.addEditClauseDialog) {
              scope.addEditClauseDialog.close();
            }
          }
        };

        /**
         * Supprime la clause du filtre édité
         * Répercute l'état du filtre dans la table des clauses du filtre
         */
        scope.deleteClause = () => {
          const clauses = scope.filter.data.where_clauses;
          if (clauses && clauses.length > 0) {
            const index = clauses.findIndex(clause => clause.fti === scope.edit_resource.fti)
            if (index > -1) {
              scope.filter.data.where_clauses.splice(index, 1);
              scope.currentResources = scope.filter.data.where_clauses;
              scope.edit_resource = undefined;
            }
          }
          refreshFilterPopupPanelMinHeight();
        };

        /**
         * Ferme la popup d'édition de la clause
         */
        scope.cancelClause = () => {
          if (scope.addEditClauseDialog) {
            scope.addEditClauseDialog.close();
          }
        }

        /**
         * Désactive le bouton d'enregistrement de la clause
         * @return {boolean} true si le textarea du model de la variable scope.config.whereClause n'est pas vide
         */
        scope.validNewClause = () => {
          return scope.edit_clause && scope.edit_clause.selectedFti !== undefined
              && scope.config.whereClause !== undefined && scope.config.whereClause.length > 0;
        };

        /**
         * Renvoie <code>true</code> si le models respectifs de l'attribut et de l'opérateur ont été définis<br>
         * Utilisé pour activer le bouton vert '+' d'ajout de critère
         * @return {boolean}
         */
        scope.canAddClause = () => {
          return scope.edit_clause !== undefined && scope.edit_clause.selectedAttribute
              !== undefined && scope.edit_clause.operator !== undefined;
        };

        /**
         * Au changement du texte de la clause
         * active/désactive le select du composant
         * calcule la hauteur de la textarea du texte de clause
         */
        scope.$watch('config.whereClause', () => {
          // si le texte de clause existe et si un fti est sélectionné
          // alors on désactive l'autocomplete des composants
          if (!!scope.edit_clause && !!scope.edit_clause.selectedFti){
            scope.lockedClauseFti = scope.config.whereClause !== undefined && scope.config.whereClause.length > 0;
          }else{
            scope.lockedClauseFti = false;
          }
          const popupPanel = $('.addEditClause').next().children()[0];

          // calcul de la hauteur de la textarea
          if (scope.config.whereClause && scope.config.whereClause.length > 80) {
            if (scope.config.whereClause.length > 80 && scope.config.whereClause.length < 160) {
              scope.criteriaHeight = 100;
              if (scope.criteriaHeight < 100) {
                $(popupPanel).css('min-height', '+=100');
              }else if (scope.criteriaHeight > 100) {
                $(popupPanel).css('min-height', '-=80');
              }
            } else if (scope.config.whereClause.length > 160 && scope.config.whereClause.length
                < 280) {
              if (scope.criteriaHeight < 140) {
                $(popupPanel).css('min-height', '+=80');
              }else if (scope.criteriaHeight > 140) {
                $(popupPanel).css('min-height', '-=70');
              }
              scope.criteriaHeight = 140;
            } else if (scope.config.whereClause.length > 280) {
              if (scope.criteriaHeight < 200) {
                $(popupPanel).css('min-height', '+=70');
              }
              scope.criteriaHeight = 200;
            }
          } else {
            scope.criteriaHeight = 0;
          }
        });

        /**
         * Echappe les apostrophes contenues dans une clause à l'exception des quotes qui entourent les string et dates.
         * @param clause chaîne de caractères comprenant des noms de champs, des opérateurs et des critères
         * @return {string} la clause avec les quotes échapées
         */
        const escapeStringQuotes = (clause) => {
          for (let i=0; i<clause.length; i++){
            if (clause.charAt(i)==='\''){
              let hasSpaceBeforeOrAfter = false;
              let hasOperatorBefore = false;
              let escapedQuote = false;
              let hasCommaBeforeOrAfter = false;
              let hasParenthesisBeforeOrAfter = false;
              if (i > 0){
                hasSpaceBeforeOrAfter = clause.charAt(i-1) === ' ' || clause.charAt(i+1) === ' ';
                hasOperatorBefore = clause.charAt(i-1) === '=';
                hasCommaBeforeOrAfter = clause.charAt(i-1) === ',' || clause.charAt(i+1) === ',';
                hasParenthesisBeforeOrAfter = clause.charAt(i-1) === '(' || clause.charAt(i+1) === ')';
                escapedQuote = clause.charAt(i-1) ==='\\';
                if (i > 4){
                  hasOperatorBefore = hasOperatorBefore || clause.substring(i-4,i) === 'LIKE';
                }
              }
              const lastCharacter = i === clause.length - 1;
              if (!hasSpaceBeforeOrAfter && !hasOperatorBefore && !lastCharacter && !escapedQuote &&
                !hasCommaBeforeOrAfter && !hasParenthesisBeforeOrAfter) {
                  clause = clause.substring(0, i) + '\\' + '\'' + clause.substring(i + 1);
              }
            }
          }
          return clause;
        };

        /**
         * Configuration de la liste des clauses<br>
         * <code>sort</code> est true pour activer les fonctions de tri sur les colonnes de la table editlist.<br>
         * <code>noHeaderOffset</code> est true pour enlever les décalages des en-têtes de table editlist
         */
        scope.editListCfg = {
          dataModule: 'model',
          resource_type: 'clauses',
          cols: ['composant', 'where'],
          addResourceButton: true,
          addFunction: openAddClauseDialog,
          editFunction: openEditClauseDialog,
          removeFunction: scope.deleteClause,
          removeTemplate:
              'js/XG/widgets/mapapp/defaultfilters/views/modal/modal.defaultfilters.clause.remove.html',
          extraGlobalActions: [],
          extraActions: [],
          warning: 'featuresWarning',
          width: 'width500',
          sort: true,
          noHeaderOffset: true
        };
        // titre de la popup ngDialog chargée de la suppression d'une clause: modal.defaultfilters.clause.remove.html
        scope.deleteModalTitle = $filter('translate')('model.clauses.delete');
        scope.categoryDeleteModalTitle = $filter('translate')('model.default_filters.deleteCategoryModalTitle');
      },
    };
  };

  defaultfilter.$inject = ['FeatureTypeFactory', 'extendedNgDialog', 'ngDialog', '$filter',
    'QueryFactory', 'ConfigFactory', 'defaultFiltersFactory', 'gaDomUtils', '$timeout'];
  return defaultfilter;
});