'use strict';
define(function () {

  var biztopologywidget = function (
    $filter,
    $timeout,
    ConfigFactory,
    FeatureTypeFactory,
    BizTopologyFactory,
    ngDialog,
    gaJsUtils,
    extendedNgDialog,
    gclayers,
    SelectManager,
    $interval,
    processFactory
  ) {
    return {
      templateUrl:
          'js/XG/widgets/mapapp/biztopology/views/biztopologywidget.html',
      restrict: 'AE',
      link: function (scope) {

        //-- Nom du fichier de configuration défini
        //-- à l'ajout du widget par l'utilisateur
        const configName = scope.toolBarWidget.config;

        let stop;

        scope.checkingIfRunning = true;
        if (!scope.themes) {
          scope.themes = [];
        }

        //-- Inialisation de la configuration au démarrage
        //-- pour prévenir les erreurs de template html
        //-- avant l'appel de getConfig()
        if (!scope.config) {
          scope.config = {};
          if (!scope.config.currentView) {
            //-- Si true, alors, l'analyse est limitée aux objets présents
            //-- dans la vue courante
            scope.config.currentView = false;
          }
        }



        /**
         * Facilitateur de récupération de message.
         *
         * @param {*} messageId chemin json d'accés au message
         * @returns message en clair depuis le fichier message
         */
        const translate = (messageId) => {
          return $filter('translate')('biztopology.'+messageId);
        };


        /**
         * Règles à vérifier (Lydec)
         * AJOUTER ICI DES NOUVELLES VERIFICATIONS
         */
        scope.executions = [
          {
            id: 0,
            alias: translate('isolatedObjects'),
            name: 'isolatedObjects',
            target: 'point'
          },
          {
            id: 1,
            alias: translate('pipeEndsWithoutManholes'),
            name: 'pipeEndsWithoutManholes',
            target: 'line'
          },
          {
            id: 2,
            alias: translate('dontSplitPipeObjects'),
            name: 'dontSplitPipeObjects',
            target: 'point'
          },
        ];


        /**
         * Opérations possibles (Lydec)
         * AJOUTER ICI DES NOUVELLES OPERATIONS
         */
        scope.lineOperations = [
          {
            id: 0,
            alias: translate('config.operations.nodeAtEnds')
          },
        ];
        scope.pointOperations = [
          {
            id: 0,
            alias: translate('config.operations.endsOf')
          },
          {
            id: 1,
            alias: translate('config.operations.placedOn')
          },
        ];

        /** Configuration de l'exécution */
        scope.process = {
          // exécution sélectionnée (bouton radio). ie scope.execution
          selected: undefined,
          // couche sur laquelle sera portée la vérification
          layer: undefined,
          tolerance: 0.01,
        };


        /** Onglets "Points/linéaires" */
        scope.tabs = [
          {
            title: translate('config.point')
          },
          {
            title: translate('config.linear')
          },
        ];
        /** Onglet actif */
        scope.assbizrules = {
          activeTab: 0,
        };

        /**
         * Récupère et charge  la configuration depuis le fichier.
         * @returns {undefined}
         */
        scope.getConfig = () => {
          const promise = ConfigFactory.get('widgets', configName);
          promise.then((res) => {
            if (res.data !== '') {
              scope.config = res.data;
              console.info(translate('config.success'));
            }
          })
            .catch((error) =>
              require('toastr').error(
                translate('config.error') + error
              )
            );
          return promise;
        };


        /**
         * Ouvre une fenetre de consultation et de modification
         * de la configuration.
         *
         * @returns {undefined}
         */
        let ngDialogPromise;

        scope.openConfigDialog = () => {
          // initialisation du model du select de composant présent
          // dans chaque onglet pour construire une nouvelle règle
          scope.draft = {};
          // définition de l'onglet actif (sinon le corps de l'onglet
          // n'apparaît pas)
          scope.assbizrules.activeTab = 0;
          ngDialogPromise = ngDialog.openConfirm({
            template:
                'js/XG/widgets/mapapp/biztopology/views/biztopologyconfig.html',
            scope: scope,
            className:
                'ngdialog-theme-plain width800 overflowY nopadding miniclose',
          });
          ngDialogPromise.then(
            (data) => {
              scope.saveconfig();
              require('toastr').info(data);
            },
            (data) => {
              scope.getConfig();
              require('toastr').error(data);
            }
          );
        };


        /**
         * Sauve la configuration dans le fichier 'configName'
         * @returns {undefined}
         */
        scope.saveconfig = () => {
          ConfigFactory.add(scope.config, 'widgets', configName)
            .then((res) => {
              require('toastr').success(res.data);
            })
            .catch((error) => require('toastr').error(error));
        };


        /**
         * Récupération des informations utiles du FTI:
         * - UID pour identifier le composant
         * - NAME pour identifier le composant
         * - ALIAS pour présenter à l'utilisateur
         *
         * @param {*} fti : description du comosant
         * @returns : Informations utiles du composant
         */
        const getFtiNameAliasAndUid = (fti) => {
          const result = {};
          result.uid = fti.uid;
          result.name = fti.name;
          result.alias = fti.alias;
          return result;
        };


        /**
         * Est-ce que la géométrie du composant est de type POINT ?
         *
         * @param {*} ft : description du composant
         * @returns VRAI si le composant est d type POINT, FAUX sinon.
         */
        const isFeaturePoint = (ft) => {
          return ft.typeInfo === 'POINT';
        };


        const isFeatureLine = (ft) => {
          return ft.typeInfo === 'LINE';
        };


        /**
         * Filtre les fti d'une règle (on ne garde que l'uid et le nom du fti)
         * @param {*} ftis
         * @returns
         */
        const getFtiNamesAliasAndUids = (ftis) => {
          const features = [];
          for (const fti of ftis) {
            const feature = getFtiNameAliasAndUid(fti);
            features.push(feature);
          }
          return features;
        };


        /** FONCTIONS POUR FILTRER LES FEATURES */
        const filterByLineAndInRules = (ft) => {
          return ft.typeInfo === 'LINE'
            && scope.config.rules
            && scope.config.rules.lines.find(
              rule => rule.component.uid === ft.uid);
        };


        const filterByPointAndInRules = (ft) => {
          return isFeaturePoint(ft)
          && scope.config.rules
          && scope.config.rules.points.find(
            rule => rule.component.uid === ft.uid);
        };


        const byTheme = (ft) => {
          if (!scope.config.theme) {
            return true;
          }
          const filterByMetier = scope.config.theme;
          return ft.theme === filterByMetier
            || (ft.theme==='undefined' && scope.config.theme==='Default');
        };


        const filterByLineAndTheme = (ft) => {
          return isFeatureLine(ft) && byTheme(ft);
        };


        const filterByPointAndTheme = (ft) => {
          return isFeaturePoint(ft) && byTheme(ft);
        };


        const getFilter = () => {
          //Recherche des objets intersectés par l'intersection
          const extent = scope.map.getView().calculateExtent();
          const leftX = extent[0];
          const bottomY = extent[1];
          const rightX = extent[2];
          const topY = extent[3];
          return 'INTERSECTS(geom, POLYGON((' +
            leftX +' ' + bottomY + ',' + rightX + ' ' + bottomY + ',' +
            rightX + ' ' + topY + ',' + leftX + ' ' + topY + ',' +
            leftX + ' ' + bottomY +
            ')))';
        };


        const checkIfEsri = (features) => {
          if (features && features.length > 0) {
            scope.isEsriFeature = scope.features[0].type === 'esri';
          }
        };


        /**
         * Construit le liste des composants de type ligne ou point
         * (selon les paramétres) et correspondant au métier configuré.
         * A n'appeler que quand on a la liste compélte
         * des composants déjà chargée.
         *
         * @param {*} listName : featuresPoints ou featuresLines
         * @param {*} filter : filterByPointAndMetier ou  filterByLineAndMetier
         */
        const buildFeatureListByType = (listName, filter, fromUpdateList) => {
          scope[listName]
            = FeatureTypeFactory.resources.featuretypes.filter(filter);
          scope[listName] = getFtiNamesAliasAndUids(scope[listName]);
          if (scope.process.selected && !fromUpdateList) {
            scope.updateFeaturesList(scope.process.selected);
          }
        };


        /**
         * Construit le liste des composants de type ligne ou point
         * (selon les paramétres) et correspondant au métier configuré.
         *
         * @param {*} listName : featuresPoints ou featuresLines
         * @param {*} filter : filterByPointAndMetier ou  filterByLineAndMetier
         */
        const getFeatureList = (listName, filter) => {
          // appel api si la liste de features du service est vide
          FeatureTypeFactory.get().then(() => {
            buildFeatureListByType(listName, filter);
          })
            .catch((error) =>
              require('toastr').error(
                translate('config.retrieveError') + error
              )
            );
        };


        /**
         * Construction de laliste des thèmes.
         * La liste des thèmes est stockée dans le scope.
         */
        const buildThemes = () => {
          if (scope.features && scope.features.length > 0) {
            for (const fti of scope.features) {
              if (scope.themes.indexOf(fti.theme) < 0
                && fti.theme !== 'undefined') {
                scope.themes.push(fti.theme);
              }
            }
          }
        };


        const buildFeatureList = (listName,filter,fromUpdateList) => {
          if (FeatureTypeFactory.resources.featuretypes.length > 0) {
            buildFeatureListByType(listName,filter,fromUpdateList);
          }
          else {
            getFeatureList(listName,filter,fromUpdateList);
          }
        };


        /**
         * Appelé à l'initialisatin et lors du choix d'un théme
         * dans la configuration.
         * Initialise et trie les listes de composants
         * selon le type de géométrie et le métier
         * récupère 3 listes (composants, lignes, points)
         */
        scope.retrieveComponents = () => {
          if (FeatureTypeFactory.resources.featuretypes.length > 0) {
            scope.features = FeatureTypeFactory.resources.featuretypes;
            checkIfEsri(scope.features);
            buildThemes();
            scope.features = getFtiNamesAliasAndUids(scope.features);
          }
          else {
            // appel api si la liste de features du service est vide
            FeatureTypeFactory.get().then(() => {
              scope.features = FeatureTypeFactory.resources.featuretypes;
              checkIfEsri(scope.features);
              buildThemes();
              scope.features = getFtiNamesAliasAndUids(scope.features);
            })
              .catch((error) =>
                require('toastr').error(
                  translate('config.retrieveError') + error
                )
              );
          }
          buildFeatureList('configFeaturesLines',filterByLineAndTheme);
          buildFeatureList('configFeaturesPoints',filterByPointAndTheme);
          buildFeatureList('featuresLines',filterByLineAndInRules);
          buildFeatureList('featuresPoints',filterByPointAndInRules);
        };


        /**
         * Configuration
         * Créé une règle à l'ajout d'un composant ( bouton [+])
         */
        scope.createEmptyRule = () => {
          const isOngletPonctuel = scope.assbizrules.activeTab === 0;
          const metier = scope.config.theme;
          const rule = {
            uuid: gaJsUtils.createUuid(),
            geometryType: isOngletPonctuel ? 'POINT' : 'LINE',
            metier: metier,
            component: isOngletPonctuel
              ? scope.draft.selectedPointComponent
              : scope.draft.selectedLineComponent,
            operation: '',
            targetComponents: [],
          };
          if (!scope.config.rules) {
            scope.config.rules = {};
          }
          if (!scope.config.rules.points) {
            scope.config.rules.points = [];
          }
          if (!scope.config.rules.lines) {
            scope.config.rules.lines = [];
          }
          if (isOngletPonctuel && scope.draft.selectedPointComponent) {
            scope.config.rules.points.push(rule);
          } else if (!isOngletPonctuel && scope.draft.selectedLineComponent) {
            scope.config.rules.lines.push(rule);
          }
        };


        /**
         * Configuration
         * Supprime une règle au clic sur [x]
         * @param {*} ruleUuid: string
         */
        scope.removeRule = (ruleUuid) => {
          const isOngletPonctuel = scope.assbizrules.activeTab === 0;
          if (isOngletPonctuel) {
            scope.config.rules.points = scope.config.rules.points.filter(
              (rule) => rule.uuid !== ruleUuid
            );
          } else {
            scope.config.rules.lines = scope.config.rules.lines.filter(
              (rule) => rule.uuid !== ruleUuid
            );
          }
        };


        /**
         * Met à jour la liste de composants du select "couche"
         * suivant le bouton radio
         */
        scope.updateFeaturesList = (execution) => {
          scope.process.selected = execution;
          if (execution && execution.target) {
            //-- Actualiser les listes, la config a pu bouger entre temps.
            buildFeatureList('featuresLines',filterByLineAndInRules, true);
            buildFeatureList('featuresPoints',filterByPointAndInRules, true);
            switch (execution.target) {
              case 'point':
                scope.listFeatures = scope.featuresPoints;
                break;
              case 'line':
                scope.listFeatures = scope.featuresLines;
                break;
            }
          }
        };


        /**
         * Préparation de la vérification de la régle topologique.
         */
        const initLoading = () => {
          scope.loading = true;
          scope.processedObjects = scope.totalObjects =
            scope.initStep = undefined;
          scope.loadingLaunched = true;
          $timeout(() => {
            setProgressBar(0);
          });
        };


        /**
         * La vérification est finie, on affiche l'état de ce dernier traitement.
         */
        const conversionStopped = () => {
          scope.conversionIsRunning = false;
          if (scope.loadingLaunched) {
            scope.loadingLaunched = false;
          }
        };


        /**
         * Lancement de la vérification des règles depuis click utilisateur.
         * "senddata" est l'objet contenant les caractéristiques du traitement
         * à effectuer (scope.process)
         * appname est le nom de l'application
         * (ex. 'MAP' pour lydec, 'SIG' pour Annonay)
         */
        scope.executeProcess = () => {
          const senddata = scope.process;
          const srid = scope.map.getView().getProjection().getCode();
          const cqlFilter = scope.config.currentView ? getFilter() : null;
          initLoading();
          BizTopologyFactory.checkRule(senddata, configName, scope.config.theme,
            srid, cqlFilter).then((res) => {
            if (res.data && res.data.etat) {
              if (res.data.etat !== 'RUNNING') {
                require('toastr').error(res.data.reason);
                scope.conversionIsRunning = false;
                conversionStopped(res.data);
              }
              else {
                scope.conversionIsRunning = true;
                scope.currentProcess = res.data;
                checkProcessInProgress(res.data,750);
              }
            }
          })
            .catch((error) => require('toastr').error(error));
        };


        /**
         * Afficher un message qui indique que tous les objets
         * du composant respectent la règle.
         *
         * @param {*} ftiUid : UID du composant vérifié.
         */
        const swalRuleIsRespected = (ftiUid) => {
          const composant = scope.features.filter(
            (feat) => feat.uid === ftiUid).map((feat) => feat.alias);
          swal(
            translate('topologicalRuleCheckup') + ' ' + scope.config.theme,
            translate('ruleOf') + scope.process.selected.alias
              + '" ("' + composant + translate('isRespected')
          );
        };


        /**
         * Une fois la vérification topologique finie ou interrompue,
         * on affiche les résultats. Le srésultats sont partiels
         * si on a interrompu la vérification.
         *
         * @param {*} result contient la liste  des objets ne vérifiant pas
         *  la régle topologique.
         */
        const showRuleCheckResult = (result) => {

          //-- scope.process ok si on est dans la page depuis laquelle
          //-- on a lancé la vérification. Sinon, on récupére
          //-- les informations depuis la réponse de la requête HTTP
          //-- qui demande où en est la progression de la vérification.
          const ftiUid = scope.process.layer
            ? scope.process.layer : result.coucheId;
          if (gaJsUtils.isJsonParsable(result.result)) {
            if (result.result && result.result.features
              && result.result.features.length === 0) {
              //-- Tous les objets du composant respectent la règle.
              swalRuleIsRespected(ftiUid);
            } else {
              //-- Afficher la liste des objets qui ne respectent pas la règle.
              getResponseFile(result, ftiUid);
            }
          }
          conversionStopped(result);
          scope.loading = false;

          //-- Un petit délai pour laisser le 100% de la barre de progression
          //-- visible un instant.
          $timeout(() => {
            //-- Effacement de la barre de progression
            scope.currentProcess = undefined;
          }, 15000);
        };


        const getExecutionInfo = (resData) => {
          scope.process.selected = scope.executions.find(
            execution => execution.id === resData.operation);
          scope.updateFeaturesList(scope.process.selected);
          scope.process.layer = resData.coucheId;
          scope.process.tolerance = resData.tolerance;
          scope.process.currentView = resData.filetr !== null;
        };


        /**
         * Fonction appelée sur timeout pour interroger le serveur
         * sur l'avancement de la vérification en cours pour cet utilisateur.
         * Cette fonction géré les cas de fin, d'erreur et d'interruption
         * de la vérification.
         *
         * @param {*} process : descripion du processus de vérification
         * @param {*} intervalMsec : délai d'attente pour
         *                           la prochaine interrogation
         * @param {*} existing : si VRAI, la vérification a été lancé par
         *                        cet utilisateur depuis une autre page.
         */
        const checkProcessInProgress2 = (process, intervalMsec, existing) => {
          if (process.creation) {
            process.creation = new Date(process.creation).getTime();
          }
          if (process.end) {
            process.end = new Date(process.end).getTime();
          }
          if (process.both !== undefined) {
            process.both = undefined;
          }
          BizTopologyFactory.getProgressionRuleCheck(process).then(
            (res) => {
              if (existing) {
                //-- Vérification lancé depuis une autre page.
                getExecutionInfo(res.data);
              }
              if (res.data.etat === 'FINISHED' && res.data.progress === 100) {
                scope.totalObjects = res.data.totalObjects;
                if (res.data.messageToProcess !== 'interruption asked') {
                  //-- Ne pas mettre la barre de progresson à 100% quand
                  //-- le calcul est interrompu.
                  setProgressBar(res.data.progress);
                }
                showRuleCheckResult(res.data);
              }
              else if (scope.loadingLaunched && res.data.etat === 'FAILED') {
                ruleCheckFailed(res, stop);
                scope.loading = false;
                scope.currentProcess = undefined;
              }
              else {
                //-- RUNNING
                if (res.data.processedObjects) {
                  scope.processedObjects = res.data.processedObjects;
                  scope.totalObjects = res.data.totalObjects;
                  scope.initStep = res.data.initStep;
                  setProgressBar(res.data.progress);
                }
                if (++scope.intervalCounter > 50 && intervalMsec < 90000) {
                  //-- Au plus c'est long, au moins on va interroger
                  //-- le serveur fréquemment sur l'état d'avancement.
                  intervalMsec *= 2;
                  scope.intervalCounter = 0;
                }
                $timeout(() => {
                  checkProcessInProgress2(process, intervalMsec);
                }, intervalMsec);              }
            },
            () => {
              if (scope.loadingLaunched) {
                require('toastr').error(
                  translate('result.error')
                );
                scope.conversionIsRunning = false;
              }
            }
          );
        };


        /**
         *     Fonction qui appel selon un interval de temps adaptif
         *  le service qui consulte l'état d'avancement de l'intégration.
         *
         *  resdata0: description du processus lancé.
         *  intervalSec:  Interval d'appel du service en MilliSecondes.
         */
        const checkProcessInProgress = (process, intervalMsec, existing) => {
          if (!intervalMsec) {
            intervalMsec = 750;
          }
          scope.intervalCounter = 0;
          if (process.creation) {
            process.creation = $filter('date')(
              new Date(process.creation),
              'dd MMM yyyy HH:MM'
            );
          }
          //-- Ne pas utiliser interval. Avec interval on n'attend pas
          //-- la réponse de l'appel précédent. Du coup on paut se retrouver
          //-- avec 3 appels pour récupérer le résultat final et
          //-- donc 3 fenêtres résultat affichées.
          $timeout(() => {
            checkProcessInProgress2(process, intervalMsec, existing);
          },
          intervalMsec);
        };


        const getResponseFile = (process, ftiUid) => {
          switch (process.etat) {
            case 'FINISHED':
              require('toastr').success(
                translate('result.success'));
              scope.loadingLaunched = false;
              if (gaJsUtils.isJsonParsable(process.result)) {
                scope.geojson = JSON.parse(process.result);
                // affichage de la table s'il y a un objet dans
                // le tableau scope.features[]
                if (scope.geojson && scope.geojson.features) {
                  if (scope.geojson.features.length > 0) {
                    displayResults(process, ftiUid);
                  } else {
                    // aucun résultat
                    require('toastr').warning(
                      translate('result.noResult'));
                    //-- Effacement de la barre de progression sans attendre.
                    scope.currentProcess = undefined;
                  }
                }
              }
              break;
            case 'FAILED':
              require('toastr').error(
                translate('result.displayError'));
              break;
          }
        };


        /**
         * Préparation des propriétés utiles au gcDataTable pour lister
         * les objets qui ne respectent pas la règle topologique.
         * Puis appel de l'affichage des objets dans le gcDataTable.
         *
         * @param {*} process : description du processus de vérification
         * @param {*} ftiUid : UID du composant vérifié
         */
        const displayResults = (process, ftiUid) => {
          scope.attributes = null;
          if (FeatureTypeFactory.resources.featuretypes.length > 0) {
            let ftis = FeatureTypeFactory.resources.featuretypes.filter(
              (featType) => featType.uid === ftiUid);
            if (ftis.length > 0) {
              //-- Paramétres pour gcdatatble
              scope.fti = ftis[0];
              scope.attributes = ftis[0].attributes;
            }
            //-- Ouverture du gcdatatable
            openTable(process);
          } else {
            FeatureTypeFactory.getFeatureByUid(ftiUid).then(
              res => {
                if (res && res.data) {
                  //-- Paramétre pour gcdatatble
                  scope.attributes = res.data.attributes;
                  //-- Ouverture du gcdatatable
                  openTable(process);
                }
              }
            );
          }
        };


        /**
         * Ouverture du gcdatatable dans une "dialog".
         *
         * @param {*} process : description du processus de vérification
         *                      avec le résultat.
         */
        const openTable = (process) => {
          // titre de la popup
          let title = '';
          let operation;
          if (scope.process.selected) {
            operation = scope.process.selected.id;
          }
          else {
            operation = process.operation;
          }
          switch (operation) {
            case 0:
              title = translate('result.isolatedObjects');
              break;
            case 1:
              title = translate('result.pipeEndsWithoutManholes');
              break;
            case 2:
              title = translate('result.dontSplitPipeObjects');
              break;
          }
          if (gaJsUtils.isJsonParsable(process.result)){
            scope.geojson = JSON.parse(process.result);
            // ouverture de la popup
            extendedNgDialog.open({
              template: 'js/XG/widgets/mapapp/biztopology/views/biztopologypopup.html',
              className:
                  'ngdialog-theme-default height500 nopadding miniclose width1100',
              closeByDocument: true,
              scope: scope,
              minimizeMaximize: true,
              scrollable: true,
              resizable: true,
              draggable: {
                title: title,
              },
              preCloseCallback: () => {
                gclayers.getDrawLayer().getSource().clear();
                gclayers.clearhighLightFeatures();
                SelectManager.clear();
              }
            });
          }
        };


        /**
         * Gestion du  cas où la chargement des EDIGEOs a échoué
         * @param {*} res
         * @param {*} stop
         */
        const ruleCheckFailed = (res, stop) => {
          $interval.cancel(stop);
          let mess = translate('severeError');
          if (res.data.errorMessages && res.data.errorMessages.length != 0) {
            mess += '<br><br>'+ res.data.errorMessages[0];
          }
          require('toastr').error(mess);
          conversionStopped(res.data);
        };


        /**
         * Actualisation de la barre de progression.
         *
         * @param {*} value : avancement en pourcentage
         *                    de la barre de progression.
         */
        const setProgressBar = (value) => {
          const iStart = 15;
          if (scope.pbValuePrec === value) {
            scope.pbSameValueSince++;
          } else {
            scope.pbSameValueSince = 0;
          }
          scope.pbValuePrec = value;
          let pbValue = '' + value + '%';
          let pbBackgroundColor = '#8db4ea';
          if (
            ++scope.pbSameValueSince >= iStart &&
                scope.pbSameValueSince <= iStart + 2
          ) {
            pbBackgroundColor = '#4b9df5';
          } else if (
            scope.pbSameValueSince >= iStart + 3 &&
                scope.pbSameValueSince <= iStart + 4
          ) {
            pbBackgroundColor = '#5b5df5';
          } else if (
            scope.pbSameValueSince >= iStart + 5 &&
                scope.pbSameValueSince <= iStart + 6
          ) {
            pbBackgroundColor = '#4b5df5';
          } else if (scope.pbSameValueSince > iStart + 6) {
            pbBackgroundColor = '#6b7df5';
            scope.pbSameValueSince = 0;
          }
          scope.pbStyle = {
            'width': pbValue,
            'background-color': pbBackgroundColor,
            'padding': 0,
            'margin': 0,
            'padding-top': '5px'
          };
        };


        const getExistingProcesses = () => {
          let ts = 0;
          let indOfLast = -1;
          scope.loadingLaunched = false;
          if (!scope.config || !scope.config.theme) {
            scope.checkingIfRunning = false;
          }
          else {
            const theme = scope.config.theme.replace(' ', '_');
            const type = 'rule_check_' + theme;
            processFactory.getProcessByUsertype(type).then((res) => {
              let ind;
              if (res.data && res.data.length > 0) {
                for (ind = 0; ind < res.data.length; ind++) {
                  if (res.data[ind]) {
                    if (res.data[ind].etat === 'RUNNING') {
                      scope.conversionIsRunning = true;
                      if (res.data[ind].both) {
                        delete res.data[ind].both;
                      }
                      scope.currentProcess = res.data[ind];
                      checkProcessInProgress(res.data[ind],750,true);
                      break;
                    } else {
                      if (res.data[ind].creation > ts) {
                        ts = res.data[ind].creation;
                        indOfLast = ind;
                      }
                    }
                  }
                }
                if (!scope.conversionIsRunning && indOfLast !== -1) {
                  conversionStopped(res.data[indOfLast]);
                }
              }
              scope.checkingIfRunning = false;
            });
          }
        };


        /**
         * Appelé sur click utilisateur pour demander l'interruption
         * de la vérification en cours.
         */
        scope.stopProcess = () => {
          scope.interrupted = true;
          if (scope.currentProcess) {
            BizTopologyFactory.stopProcess(scope.currentProcess.uid).then(
              res => {
                if (res && res.data && scope.loading) {
                  scope.loadingLaunched = false;
                  //-- Enlever le message interrompu au bout de 15 secondes.
                  //-- Mais, currentprocess est mis à undefined
                  //--  quand récupération du résultat donc de toute façon
                  //-- la zone est enlevée.Le timeout sert à laisser
                  //-- le message durant l'affichage de la zone de progression/
                  $timeout(() => {
                    scope.interrupted = false;
                  }, 15000);
                }
              }
            ).finally(
              () => {
                scope.loading = false;
              }
            );
          }
        };


        scope.isEmptyThemeAndLayer = () => {
          scope.hasClickLayer = true;
          if (scope.process.selected) {
            scope.emptylayer = false;
            scope.hasClickLayer = false;
          } else {
            scope.emptylayer = true;
          }
          if (scope.config.theme) {
            scope.emptyTheme = false;
            scope.hasClickLayer = false;
          } else {
            scope.emptyTheme = true;
          }
        };


        scope.hasCurrentProcess = () => {
          return scope.currentProcess !== undefined;
        };


        /** APPELS DE FONCTION AU DEMARRAGE */
        scope.getConfig().then(() => {
          scope.retrieveComponents();
          getExistingProcesses();
        });

      },
    };
  };

  biztopologywidget.$inject = [
    '$filter',
    '$timeout',
    'ConfigFactory',
    'FeatureTypeFactory',
    'BizTopologyFactory',
    'ngDialog',
    'gaJsUtils',
    'extendedNgDialog',
    'gclayers',
    'SelectManager',
    '$interval',
    'processFactory'
  ];
  return biztopologywidget;

});
