/*global define, $ */
define(['ol3js', 'angular', 'toastr'], function(ol, angular, toastr) {
  'use strict';

  // 1px = 0.28 mm according to the OGC spec for DPI
  var PX_PER_M = (1000 * (window.devicePixelRatio || 1)) / 0.28;

  /*
   * The Annotations Widget allows to draw OpenLayers features on the map.
   * Features are drawn on the drawLayer's source.
   *
   * TODO as of october 2017:
   * - Make better edition dialogs (needs someone who knows a tad about design)
   * - Allow setting symbol X & Y anchor ('auto' or 'N pixels') & label Y offset ('0-50 pixels')
   * - When we edit a feature and save it, it should stay in place instead of going to the foreground
   * - I failed to do this: have a selection style (ie, highlight selected features)
   */
  var annotationswidget = function(
    $rootScope,
    $filter,
    $timeout,
    $q,
    gclayers,
    gcInteractions,
    ParametersFactory,
    extendedNgDialog
  ) {
    var removeDrawListener,
      deferredDraw = $q.defer();
    var draw = {
      source: null,
      layer: null,
      actions: [],
      name: '',
      enabled: false,
      select: false,
      features: null,
      available: deferredDraw.promise,
    };

    var amountScaled = 0;

    /*
     * Get the draw layer as soon as it is available
     */
    if (gclayers.getAnnotationLayer().getSource) {
      deferredDraw.resolve();
    } else {
      removeDrawListener = $rootScope.$on('gcTechnicalLayerChange', function() {
        if (gclayers.getAnnotationLayer().getSource) {
          removeDrawListener();
          deferredDraw.resolve();
        }
      });
    }

    /* ------------------------------------------------------
     * Methods used by annotations
     */

    /*
     * Generate an OpenLayers style from a style template
     */
    function getDefaultStyle(type, styleTemplate) {
      var style = {};
      switch (type) {
        case 'Point':
        case 'text':
          style.image = new ol.style.Circle({
            radius: styleTemplate.image.radius,
            fill: new ol.style.Fill(styleTemplate.image.fill),
          });
          break;
        case 'Symbol':
          style.image = new ol.style.Icon({
            //anchor: [0.5, 46],
            //anchorXUnits: 'fraction',
            //anchorYUnits: 'pixels',
            opacity: 1,
            src: styleTemplate.image.src,
            scale: styleTemplate.image.scale,
            rotation: styleTemplate.image.rotation,
          });
          break;
        case 'Label':
          style.text = new ol.style.Text({
            text: styleTemplate.text.text,
            font: styleTemplate.text.font,
            //offsetX: styleTemplate.text.offsetX,
            //offsetY: styleTemplate.text.offsetY,
            rotation: styleTemplate.text.rotation,
            fill: new ol.style.Fill(styleTemplate.text.fill),
            stroke: new ol.style.Stroke(styleTemplate.text.stroke),
          });
          break;
        case 'Polygon':
          style.fill = new ol.style.Fill(styleTemplate.fill);
          style.stroke = new ol.style.Stroke(styleTemplate.stroke);
          // console.log(style);

          break;
        case 'LineString':
          style.stroke = new ol.style.Stroke(styleTemplate.stroke);
          break;
      }
      return new ol.style.Style(style);
    }

    /*
     * Provide a style template for the given annotation type.
     * They can be edited with the user interface, can be stored,
     * and transformed into OpenLayers styles when needed.
     * We cannot edit or store OpenLayers styles directly.
     */
    function getDefaultTemplate(type) {
      var templates = {
        Point: {
          image: {
            radius: 5,
            fill: { color: '#d35400' },
          },
        },
        Symbol: {
          image: {
            src: 'img/widget/gsw/gsw_01.png',
            scale: 1,
            rotation: 0,
          },
        },
        LineString: {
          stroke: {
            color: '#d35400',
            width: 3,
          },
        },
        Polygon: {
          stroke: {
            color: '#d35400',
            width: 3,
          },
          fill: { color: 'rgba(255,255,255,0.4)' },
          opacity: 100,
        },
        text: {
          text: '',
          backgroundColor: '#ffffff',
          textColor: '#000000',
          opacity: 100,
          image: {
            radius: 7,
            fill: { color: '#f6c531' },
          },
        },
        Label: {
          text: {
            text: 'Label',
            font: '16px sans-serif',
            //offsetX: 0,
            //offsetY: 0,
            rotation: 0,
            fill: { color: '#000000' },
            stroke: {
              // outline
              color: '#ffffff',
              width: 2,
            },
          },
        },
      };
      if (templates[type]) {
        return templates[type];
      }
    }

    /*
     * Change the opacity of the given style; it is currently the
     * only way of hiding an OpenLayers feature without removing it
     */
    function updateStyleOpacity(style, opacity) {
      function setOpacity(dest, prop) {
        if (!(dest[prop] && dest[prop]() && dest[prop]().getColor())) {
          return; // The property must have a color
        }
        // ol.color.asArray has a cache; we MUST NOT use it directly
        var color = [].concat(ol.color.asArray(dest[prop]().getColor()));
        color[3] = opacity;
        dest[prop]().setColor(color);
      }
      // Update image style if any (type: 'Symbol' or 'Point')
      if (angular.isDefined(style.getImage)) {
        if (style.getImage()) style.getImage().setOpacity(opacity);
      }
      // Update text style if any (type: 'Label')
      if (angular.isDefined(style.getText)) {
        if (style.getText()) {
          var txt = style.getText();
          setOpacity(txt, 'getFill');
          setOpacity(txt, 'getStroke');
        }
      }
      // Update basic styles if any
      setOpacity(style, 'getFill');
      setOpacity(style, 'getStroke');
      return style;
    }

    /* ------------------------------------------------------
     * Classes used by annotations
     */

    /*
     * Create a feature from a feature template
     */
    function createFeature(source) {
      var type = source.properties.featureType;
      var style = source.properties.styleTemplate;

      // Secure the properties
      if (type === 'text' && !source.properties.uid) {
        source.properties.uid = Math.round(Math.random() * 100000000);
      }
      if (type === 'Polygon' && style.opacity === undefined) {
        style.opacity = 100;
      }

      // Instantiate and style the feature
      var reader = new ol.format.GeoJSON();
      var layerFt = reader.readFeature(source);
      if (type !== 'text') {
        layerFt.setStyle(getDefaultStyle(type, style));
      }
      return layerFt;
    }

    /*
     * Feature wrapper, associates a template (KIS data)
     * and an OpenLayers vector feature
     */
    function Feature(template, map) {
      var self = this;

      // Display the overlay for 'text' features
      self.setPopup = function(edit) {
        if (self.overlay) {
          map.removeOverlay(self.overlay);
          self.overlay = null;
        }
        if (!self.visible) {
          return;
        }
        // Visible: create the overlay
        var style = self.properties.styleTemplate;
        var css = [
          'background: ' + style.backgroundColor,
          'color: ' + style.textColor,
          'opacity: ' + style.opacity / 100,
        ].join(';');

        var uniqid = 'textpopup_' + Date.now();
        var tpl =
          '<div id="' +
          uniqid +
          '" class="ol-popup annotationPopup" style="' +
          css +
          '">' +
          '<div class="popup-content" style="display: inline-block;"></div>' +
          '</div>';
        angular
          .element(document)
          .find('body')
          .eq(0)
          .append(tpl);

        var container = document.getElementById(uniqid);

        $(container)
          .find('.popup-content')
          .html(style.text);
        // $(container).find('.ol-popup').css('width', '');
        $timeout(function() {
          var width = style.text.length * 14;
          if (width > 400) width = 400;
          if (width < 100) width = 100;

          if (edit) $(container).addClass('editedFeature');

          $(container).css({ width: width + 'px' });
          var height = $(container).css('height');

          //self.properties.styleTemplate.initialScale = 0;
          self.properties.styleTemplate.widthInPx = width;
          self.properties.styleTemplate.heightInPx = height.substring(
            0,
            height.length - 2
          );
          // self.properties.styleTemplate.zoomLevel = map.getView().getZoom();

          self.properties.styleTemplate.bottom = parseInt(
            $(container).css('bottom')
          );
          self.properties.styleTemplate.left = parseInt(
            $(container).css('left')
          );
          self.properties.styleTemplate.fontSize = 14;
          self.properties.styleTemplate.containerId = uniqid;
        });

        self.overlay = new ol.Overlay({
          element: container,
          autoPan: true,
          autoPanAnimation: { duration: 250 },
        });
        self.overlay.set('origin', 'annotations');
        self.overlay.set('uid', self.properties.uid);
        map.addOverlay(self.overlay);
        self.overlay.setPosition(self.vector.getGeometry().getCoordinates());
      };

      // Set the visibility of a single feature
      self.setVisibility = function() {
        var opacity = Number(!!self.visible);
        // Retro-compatibility: 'text' annotations had no own style (null)
        var style = self.vector.getStyle() || draw.layer.getStyle();

        if (self.properties.featureType === 'text') {
          self.setPopup();
        }
        self.vector.set('visible', self.visible);

        if (
          self.properties.featureType === 'Polygon' &&
          self.visible &&
          typeof self.properties.styleTemplate.opacity === 'number'
        ) {
          opacity = self.properties.styleTemplate.opacity / 100;
        }
        self.vector.setStyle(updateStyleOpacity(style, opacity));
      };

      self.showHide = function() {
        self.visible = !self.visible; // toggle
        self.setVisibility();
      };

      self.zoomTo = function() {
        var extent = self.vector.getGeometry().getExtent();
        map.getView().fit(extent, map.getSize());
      };

      self.source = function() {
        return template;
      };

      self.update = function(newTemplate) {
        template = newTemplate;
        // Clean the template
        delete template.properties.selected;
        if (template.visible === undefined) {
          template.visible = true;
        }

        var found = draw.source.getFeatures().some(function(olFeat) {
          if (self.vector === olFeat) {
            draw.source.removeFeature(olFeat);
            return true;
          }
        });
        angular.extend(self, template);
        self.template = template;

        self.vector = createFeature(template);
        if (found) {
          draw.source.addFeature(self.vector);
          self.setVisibility();
        }
      };

      self.updateScale = function() {
        if (angular.isUndefined(self.template.properties.featureType)) return;
        var type = self.template.properties.featureType;
        var style = angular.copy(self.template.properties.styleTemplate);

        // osef
        if (
          !angular.isDefined(self.template.properties.styleTemplate.zoomLevel)
        )
          return;

        var differentialAmountScaled =
          self.template.properties.styleTemplate.zoomLevel -
          map.getView().getZoom();

        // scale des properties
        if (differentialAmountScaled) {
          var ratio = Math.pow(2, Math.abs(differentialAmountScaled));
          // console.log('%c  ' + ratio, 'background: #f0f; color: #fff');

          // On ne grossit pas, on ne fait que reduire lorsqu'on dezoome
          // (sinon remettre * ratio)

          // update radius (points)
          if (style.image && style.image.radius) {
            style.image.radius =
              differentialAmountScaled > 0
                ? style.image.radius / ratio
                : style.image.radius;
            //console.log('%c  ' + style.image.radius,  'background: #f00; color: #fff');
          }
          // update epaisseur trait
          if (style.stroke && style.stroke.width) {
            style.stroke.width =
              differentialAmountScaled > 0
                ? style.stroke.width / ratio
                : style.stroke.width;
            //console.log('%c  ' + style.stroke.width,  'background: #f00; color: #fff');
          }
          // scale image
          if (style.image && style.image.scale) {
            if (0 && self.template.properties.featureType === 'Symbol') {
              style.image.scale =
                differentialAmountScaled > 0
                  ? style.image.scale - (style.image.scale / ratio) * 10
                  : style.image.scale + (style.image.scale * ratio) / 10;
            } else {
              style.image.scale =
                differentialAmountScaled > 0
                  ? style.image.scale / ratio
                  : style.image.scale * ratio;
            }
            //console.log('%c  ' + style.image.scale,  'background: #f00; color: #fff');
          }

          // scale text
          if (
            style.text &&
            style.text.font &&
            style.text.font.indexOf('px') !== -1
          ) {
            var currFontSize = parseInt(style.text.font);
            var newFontSize =
              differentialAmountScaled > 0
                ? currFontSize / ratio
                : currFontSize;
            style.text.font = style.text.font.replace(
              currFontSize,
              newFontSize
            );
          }
        }

        // console.log(type);
        // console.log(style);

        self.vector.setStyle(getDefaultStyle(type, style));
        self.setVisibility();
      };

      // Instanciation
      if (template instanceof ol.Feature) {
        // The template is the vector feature
        self.vector = template;
        var writer = new ol.format.GeoJSON({});
        template = writer.writeFeatureObject(self.vector);
        template.visible = true;
        angular.extend(self, template);
      } else {
        // The template is a KIS template
        self.update(template);
      }

      // Unselected by default
      self.properties.selected = false;
      self.vector.set('selected', false);
    }

    /*
     * Manage the features of an annotation
     */
    function AnnotationFeatures(featureTemplates, map) {
      var self = this; // We extend Array

      // Remove a feature
      self.remove = function(feat) {
        var index = self.indexOf(feat);
        if (index !== -1) {
          draw.source.getFeatures().some(function(olFeat) {
            if (olFeat === feat.vector) {
              draw.source.removeFeature(feat.vector);
              return true;
            }
          });
          self.splice(index, 1);
        }
        return index;
      };

      self.replace = function(a, b) {
        var index = self.remove(a);
        if (index !== -1) {
          self.splice(index, 0, b);
        }
        return index;
      };

      self.vectors = function() {
        return self.map(function(feat) {
          return feat.vector;
        });
      };

      self.find = function(vector) {
        var index = -1;
        self.some(function(feat, i) {
          if (feat.vector === vector) {
            index = i;
            return true;
          }
        });
        return self[index];
      };

      // Instantiate the OpenLayers features
      featureTemplates.forEach(function(template) {
        self.push(new Feature(template, map));
        var ind = self.length - 1;
        var visible = self[ind].properties.visible;
        if (visible != undefined) self[ind].visible = visible;
      });
    }
    AnnotationFeatures.prototype = new Array();

    /*
     * Manage a selection of features
     */
    function AnnotationSelection(features) {
      var self = this; // We extend Array
      self.allChecked = false;

      self.add = function(feat) {
        if (!feat) {
          return;
        }
        feat.properties.selected = true;
        feat.vector.set('selected', true);
        if (self.indexOf(feat) === -1) {
          self.push(feat);
        }
        if (draw.features.getArray().indexOf(feat.vector) === -1) {
          draw.features.push(feat.vector);
        }
      };

      self.remove = function(feat) {
        var index = self.indexOf(feat);
        if (index !== -1) {
          feat.properties.selected = false;
          feat.vector.set('selected', false);
          self.splice(index, 1);
        }
        draw.features.remove(feat.vector);
      };

      self.toggle = function(feat) {
        if (feat) {
          // If we toggle a single feature
          if (features.indexOf(feat) === -1) {
            return; // Feature doesn't exist
          }
          if (!feat.properties.selected) {
            self.add(feat);
          } else {
            self.remove(feat);
          }
        } else {
          // If we toggle all features
          self.allChecked = !self.allChecked;
          if (self.allChecked) {
            features.forEach(self.add);
          } else {
            self.uncheck();
          }
        }
      };

      self.showHide = function() {
        var visible = !self.isVisible(); // toggle
        self.forEach(function(feat) {
          feat.visible = visible;
          feat.setVisibility();
        });
      };

      self.isVisible = function() {
        var visibleCount = 0;
        self.forEach(function(feat) {
          visibleCount += feat.visible * 2 - 1; // 1 (true) or -1 (false)
        });
        return visibleCount >= 0;
      };

      self.uncheck = function() {
        while (self.length > 0) {
          self.remove(self[0]);
        }
      };

      self.erase = function() {
        self.forEach(function(feat) {
          feat.visible = false;
          feat.setVisibility();
          features.remove(feat);
        });
        self.length = 0;
      };
    }
    AnnotationSelection.prototype = new Array();

    /*
     * Controller for the widget
     * Manages the list of annotations (hence the name 'list')
     */
    function AnnotationsWidgetCtrl($scope) {
      var list = this;
      var map = $scope.map;
      $scope.current = null;
      $scope.annotations = [];
      $scope.draw = draw;
      $scope.Infinity = Infinity;

      /* ------------------------------------------------------
       * 'Annotation' class defined below
       */

      /*
       * Very big object to manage an Annotation
       */
      function Annotation(template) {
        var self = this;

        // Show the annotation
        self.show = function() {
          if (!self.isHidden) {
            return;
          }
          self.isHidden = false;
          self.features.forEach(function(feat) {
            feat.setVisibility();
            draw.source.addFeature(feat.vector);
          });
        };

        // Hide the annotation
        self.hide = function() {
          if (self.isHidden) {
            return;
          }
          self.isHidden = true;
          self.features.forEach(function(feat) {
            if (feat.overlay) {
              map.removeOverlay(feat.overlay);
            }
            draw.source.getFeatures().some(function(olFeat) {
              if (olFeat === feat.vector) {
                draw.source.removeFeature(olFeat);
                return true;
              }
            });
          });
        };

        // Toggle annotation visibility
        self.showHide = function showHide(visible) {
          if (visible !== undefined) {
            self.visible = visible;
          }
          self[self.visible ? 'show' : 'hide']();
        };

        // Control visibility with the given scale
        self.showByScale = function showByScale(scale) {
          if (!self.visible) {
            return;
          }
          if (scale <= self.maxScale && scale >= self.minScale) {
            self.show();
          } else {
            self.hide();
          }
        };

        self.setScaling2 = function(diff) {
          if (diff == undefined) return;
          amountScaled += diff;

          for (var i in self.features) {
            if (!angular.isDefined(self.features[i].properties)) continue;

            var _props = self.features[i].properties.styleTemplate;

            var container = document.getElementById(_props.containerId);
            if (self.features[i].properties.featureType === 'text') {
              if (!angular.isDefined(_props.zoomLevel)) continue;

              var differentialAmountScaled =
                map.getView().getZoom() - _props.zoomLevel;

              console.log('----------------');
              console.log('diff', diff);
              console.log('amountScaled', differentialAmountScaled);

              // differentialAmountScaled = differentialAmountScaled*-1;

              // remove all scaled-* classes
              $(container).removeClass(function(index, className) {
                return (className.match(/(^|\s)scaled-\S+/g) || []).join(' ');
              });
              // addClassses
              if (differentialAmountScaled < -2) {
                $(container).addClass('scaled scaled-hidden');
              } else {
                $(container).addClass(
                  'scaled scaled-' + differentialAmountScaled
                );
              }

              var newScaleX = _props.left,
                newScaleY = _props.bottom,
                newWidthInPx = _props.widthInPx,
                newHeightInPx = _props.heightInPx,
                newFontSize = _props.fontSize;

              if (differentialAmountScaled == 0) {
                if (diff != 0) {
                  // niet
                } else {
                  continue;
                }
              } else {
                var ratio = Math.pow(2, Math.abs(differentialAmountScaled));
                newScaleX =
                  differentialAmountScaled > 0
                    ? _props.left * ratio
                    : _props.left / ratio;
                newScaleY =
                  differentialAmountScaled > 0
                    ? _props.bottom * ratio
                    : _props.bottom / ratio;
                newWidthInPx =
                  differentialAmountScaled > 0
                    ? _props.widthInPx * ratio
                    : _props.widthInPx / ratio;
                newHeightInPx =
                  differentialAmountScaled > 0
                    ? _props.heightInPx * ratio
                    : _props.heightInPx / ratio;
                newFontSize =
                  differentialAmountScaled > 0
                    ? _props.fontSize * ratio
                    : _props.fontSize / ratio;
              }

              console.log(map.getView().getZoom());
              console.log(
                '%c differentialAmountScaled ' + differentialAmountScaled,
                'background: #f00; color: #fff'
              );
              console.log('Calcul de X');
              console.log('valeur actuelle', _props.left);

              console.log('nouvelle valeur', newScaleX);
              console.log('Calcul de width');
              console.log('valeur actuelle', _props.widthInPx);
              console.log('nouvelle valeur', newWidthInPx);

              if (newScaleY > 12) newScaleY = 12;
              $(container).css({ left: newScaleX + 'px' });
              $(container).css({ bottom: newScaleY + 'px' });
              $(container).css({ width: newWidthInPx + 'px' });
              $(container).css({ height: newHeightInPx + 'px' });
              $(container).css({
                'font-size': newFontSize + 'px',
              });
            } else {
              self.features[i].updateScale();
            }
          }
        };

        // Write the annotation data
        self.writeData = function writeData() {
          var writer = new ol.format.GeoJSON({});
          template.data = writer.writeFeaturesObject(self.features.vectors());
          angular.extend(template.data, {
            description: self.description,
            visible: self.visible,
            minScale: self.minScale,
            maxScale: self.maxScale,
          });
          angular.extend(template, {
            id: self.id,
            login: self.login,
            name: self.name,
          });
        };

        self.source = function source() {
          return template;
        };

        // Save the annotation
        self.save = function save() {
          self.writeData();
          angular.extend(template.data, {
            zoom: map.getView().getZoom(),
            center: map.getView().getCenter(),
          }); // Don't write zoom and center anywhere else
          // If the annotation is not already in the list
          if ($scope.annotations.indexOf(self) === -1) {
            $scope.annotations.push(self);
          }
          if (self.id) {
            // Existing annotation, update
            ParametersFactory.getbyid(self.id).then(function(res) {
              res.data.data = template.data;
              ParametersFactory.update(res.data, self.id).then(function() {
                toastr.success($filter('translate')('common.updated'));
                list.fetch();
              });
            });
          } else {
            // New annotation, create (add)
            ParametersFactory.add(template.data, 'Annotation', self.name).then(
              function(res) {
                console.log(res.data);
                if (
                  res.data ===
                  'Ajout impossible: un paramètre avec le même nom existe déjà.'
                ) {
                  toastr.error(res.data);
                  return false;
                }
                toastr.success($filter('translate')('common.saved'));
                self.id = res.data.id;
                list.fetch();
              }
            );
          }
        };

        self.reset = function(state) {
          if (self.features) {
            self.hide();
          }
          self.features = new AnnotationFeatures(
            state.data.features || [],
            map
          );
          self.selection = new AnnotationSelection(self.features);
          draw.features.clear();
          angular.extend(self, {
            id: state.id || false,
            login: state.login || $rootScope.xgos.user.login,
            name: state.name || '',
            description: state.data.description || '',
            visible: state.data.visible || false,
            zoom: state.data.zoom || map.getView().getZoom(),
            center: state.data.center || map.getView().getCenter(),
            minScale: state.data.minScale || 0,
            maxScale: state.data.maxScale || Infinity,
          });
          self.isHidden = true;
          self.writeData();
          self.untouched = angular.copy(self.source());

          // update scale !
        };

        self.reset(template);
      }

      /* ------------------------------------------------------
       * 'list' object
       */

      /*
       * Tell whether the current user can modify the given annotation
       */
      list.isEditable = function(annotation) {
        // Editable if the user authored the annotation
        if ($rootScope.xgos.user.login === annotation.login) {
          return true;
        }
        // Editable if the user is Root
        return $rootScope.xgos.user.roles.some(function(role) {
          return role
            ? role.name === 'rootUser'
            : false;
        });
      };

      /*
       * Open the given annotation
       */
      list.open = function open(annotation) {
        if (!list.isEditable(annotation)) {
          return;
        }
        draw.stop();
        $scope.annotations.forEach(function(annot) {
          annot.hide();
        });
        annotation.show();
        map.getView().animate({
          zoom: annotation.zoom,
          center: annotation.center,
          duration: 700,
        });
        $scope.current = annotation;
      };

      /*
       * Navigate back to the annotation or to the list
       */
      list.back = function back() {
        if ($scope.current && draw.enabled) {
          draw.stop();
          draw.enabled = false;
        } else {
          $scope.current.writeData();
          if (
            !angular.equals($scope.current.source(), $scope.current.untouched)
          ) {
            // Unsaved changes!!
            if (!confirm($filter('translate')('annotations.confirm_unsaved'))) {
              return; // The user canceled
            }
            $scope.current.reset($scope.current.untouched);
          }
          $scope.current.hide();
          $scope.current = null;
          $scope.annotations.forEach(function(annotation) {
            annotation.showHide();
          });

          $timeout(function() {
            var container = document.getElementsByClassName('editedFeature');
            if (container != null) $('.editedFeature').remove();
          });
        }
      };

      /*
       * Create a new annotation
       */
      list.create = function create() {
        var created = new Annotation({ data: { features: [] } });
        list.open(created);
      };

      /*
       * Remove the given annotation from the list
       */
      list.remove = function remove(annotation) {
        var index = $scope.annotations.indexOf(annotation);
        if (index === -1 || !list.isEditable(annotation)) {
          return; // Error, annotation not found
        }
        if (!confirm($filter('translate')('annotations.confirm_delete'))) {
          return; // The user canceled the removal
        }
        if ($scope.current === annotation) {
          $scope.current = null;
        }
        ParametersFactory.remove(annotation.id).then(function() {
          annotation.hide();
          $scope.annotations.splice(index, 1);
        });
      };

      /*
       * Fill in the list
       */
      list.fetch = function fetch() {
        ParametersFactory.getbytype('Annotation',$scope.map
            .getView()
            .getProjection()
            .getCode()).then(function(res) {
          var currName = $scope.current && $scope.current.name;
          $scope.current = null;
          if (Array.isArray(res.data)) {
            // Delete existing annotations (even unsaved ones)
            $scope.annotations.forEach(function(annotation) {
              annotation.hide();
            });
            $scope.annotations = [];
            // Add the new annotations and update the current if any
            res.data.forEach(function(template) {
              var annotation = new Annotation(template);
              $scope.annotations.push(annotation);
              if (currName === template.name) {
                list.open(annotation);
              }
              if (!currName) {
                var scale = map.getView().getResolution() * PX_PER_M;
                annotation.showByScale(scale);
              }

              $timeout(function() {
                annotation.setScaling2(0);
              });
            });
          }

          // Open the annotation specified by 'attrs' if any
          if ($scope.attrName) {
            $scope.annotations.forEach(function(annotation) {
              if (annotation.name === $scope.attrName) {
                list.open(annotation);
              }
            });
          }
        });
      };
      // The widget does nothing while the draw layer is unavailable
      draw.available.then(function() {
        list.fetch();
        // Store the draw layer
        draw.layer = gclayers.getAnnotationLayer();
        draw.source = draw.layer.getSource();
        draw.snap = new ol.interaction.Snap({
          source: draw.source,
        });

        // Configure selection (click & collection)
        draw.click = new ol.interaction.Select({
          layer: draw.layer,
          // Alternative: ol.events.condition.altKeyOnly
          // The Shift key is already used by the Modify interaction
          // If you change the condition, update the locale-xx.json files
          toggleCondition: ol.events.condition.always,
        });
        draw.features = draw.click.getFeatures();

        // Configure selection (dragbox)
        draw.box = new ol.interaction.DragBox({
          condition: ol.events.condition.platformModifierKeyOnly, // Ctrl (Command on Mac)
          style: new ol.style.Style({
            stroke: new ol.style.Stroke({ color: [0, 0, 255, 1] }),
          }),
        });
        draw.box.on('boxstart', function() {
          draw.features.clear();
        });
        draw.box.on('boxend', function() {
          var extent = draw.box.getGeometry().getExtent();
          draw.source.forEachFeatureIntersectingExtent(extent, function(
            feature
          ) {
            if (feature.getProperties().visible) draw.features.push(feature);
          });
        });

        // Synchronize the OpenLayers selection and our UI selection
        draw.features.on('add', function(event) {
          var feat = $scope.current.features.find(event.element);
          if (!feat.properties.selected) {
            $scope.current.selection.add(feat);
            try {
              $scope.$apply();
            } catch(err) {}
          }
        });
        draw.features.on('remove', function(event) {
          if ($scope.current) {
            var feat = $scope.current.features.find(event.element);
            if (feat != undefined && feat.properties.selected) {
              $scope.current.selection.remove(feat);
              try {
                $scope.$apply();
              } catch(err) {}
            }
          }
        });
      });

      /*
       * Manage draw interactions
       */
      draw.stop = function() {
        draw.actions.forEach(function(interaction) {
          map.removeInteraction(interaction);
        });
        draw.actions = [];
        var oldName = draw.name;
        draw.name = '';
        draw.select = false;
        return oldName;
      };
      draw.start = function(name, interactions, selecting) {
        draw.name = name;
        draw.select = selecting;
        draw.actions = interactions;
        draw.actions.forEach(function(interaction) {
          interaction.setProperties({
            gctype: 'kis',
            widget: 'annotations',
            interaction: selecting ? 'select' : 'draw',
          });
          map.addInteraction(interaction);
        });
        gcInteractions.setCurrentToolBar($scope.toolBarWidget);
      };
    }
    AnnotationsWidgetCtrl.$inject = ['$scope'];

    return {
      templateUrl:
        'js/XG/widgets/mapapp/annotations/views/annotationswidget.html',
      restrict: 'A',
      controller: AnnotationsWidgetCtrl,
      controllerAs: 'list',
      // Stuff which was not required in the controller
      link: function(scope, elt, attrs) {
        if (attrs.annotationname) {
          scope.attrName = attrs.annotationname;
        }

        // angular.element(document.querySelector('.ol-popup')).css('width', 'fit-content');

        /*
         * Change selection-based action
         */
        scope.addSelect = function(type) {
          if (type === draw.stop()) {
            return;
          }
          var interactions = [draw.click, draw.box];

          if (type === 'Modify') {
            interactions.push(
              new ol.interaction.Modify({
                features: draw.features,
                deleteCondition: function(mapBrowserEvent) {
                  return (
                    ol.events.condition.shiftKeyOnly(mapBrowserEvent) &&
                    ol.events.condition.singleClick(mapBrowserEvent)
                  );
                },
              })
            );
            interactions.push(draw.snap);
          }
          if (type === 'Translate') {
            var translate = new ol.interaction.Translate({
              features: draw.features,
            });
            interactions.push(translate);
            translate.on('translateend', function(evt) {
              evt.features.forEach(function(feature) {
                var container = document.getElementById(
                  feature.getProperties().styleTemplate.containerId
                );
                container.remove();
                var test = new Feature(feature, scope.map);
                test.setPopup(true);
                //Feature.feature.setPopup()
              });
            });
          }

          draw.start(type, interactions, true);
        };

        /*
         * Add a shape to the current annotation
         */
        scope.addShape = function(type) {
          if (type === draw.stop()) {
            return;
          }
          var action = new ol.interaction.Draw({
            source: draw.source,
            type:
              ['Symbol', 'Label', 'text'].indexOf(type) !== -1 ? 'Point' : type,
          });
          draw.start(type, [action, draw.snap], false);

          /*
           * After drawing, set feature properties
           */
          action.on('drawend', function(event) {
            // @RB : shouldn't we keep it active instead ?
            draw.stop();

            var labelTypes = {
              Point: 'Point',
              LineString: 'Line',
              Polygon: 'Polygon',
              text: 'Text',
              Symbol: 'Symbol',
              Label: 'Label',
            };
            var template = getDefaultTemplate(type);
            template.zoomLevel = scope.map.getView().getZoom();
            event.feature.setStyle(getDefaultStyle(type, template));

            event.feature.setProperties({
              name: labelTypes[type] || 'Unknown',
              selected: false,
              styleTemplate: template,
              featureType: type,
            });
            if (type === 'text') {
              event.feature.setProperties({
                uid: Math.round(Math.random() * 10000000000),
              });
            }

            var feat = new Feature(event.feature, scope.map);
            scope.current.features.push(feat);
            feat.setVisibility();

            $timeout(function() {
              scope.editFeature(feat);
            });
          });
        };

        /*
         * Edit a Feature (pop up)
         */
        var editFeatureDialog;
        scope.editFeature = function(feat) {
          if (editFeatureDialog) {
            editFeatureDialog.close();
          }
          scope.originalFeature = feat;
          scope.editedFeature = angular.copy(feat.source());
          scope.editedStyle = scope.editedFeature.properties.styleTemplate;

          if (scope.editedFeature.properties.featureType === 'Symbol') {
            scope.imageSelected =
              scope.editedFeature.properties.styleTemplate.image.src;
          }

          editFeatureDialog = extendedNgDialog.open({
            template: 'js/XG/widgets/mapapp/annotations/views/editFeature.html',
            className: 'ngdialog-theme-plain width800 nopadding miniclose',
            closeByDocument: false,
            scope: scope,
            draggable: true,
            title: $filter('translate')('annotations.widgetname'),
          });
        };

        /*
         * Apply the modifications to the currently edited feature
         */
        scope.applyFeatureModifications = function() {
          scope.editedFeature.properties.styleTemplate.scale100
            = $rootScope.xgos.scale4Representation;
          scope.editedFeature.properties.styleTemplate.zoomLevel = scope.map
            .getView()
            .getZoom();
          scope.originalFeature.update(scope.editedFeature);
          editFeatureDialog.close();
          scope.originalFeature = null;
          scope.editedFeature = null;
          scope.editedStyle = null;
          editFeatureDialog = null;
        };

        /**
         * Convert input degrees if value, else convert slider to input value (decimal is PI * 2)
         */
        scope.convertDegrees = function(value, fromSlider) {
          // convertedRotation
          if (angular.isDefined(value) && !fromSlider)
            scope.editedStyle.image.rotation = (6.28318530718 * value) / 360;
          else if (angular.isDefined(value) && fromSlider)
            $timeout(function() {
              scope.editedStyle.image.convertedRotation = Math.round(
                scope.editedStyle.image.rotation * (180 / Math.PI)
              );
            });
        };

        /*
         * Set image for annotation
         */
        scope.setImage = function(data) {
          scope.editedFeature.properties.styleTemplate.image.src =
            data.url + '/' + data.img;
          scope.imageSelected = data.url + '/' + data.img;
        };

        /*
         * Show or hide annotations depending on their scale
         */
        scope.map.on('moveend', function() {
          // console.log(scope.map.getView().getZoom());

          var scale = scope.map.getView().getResolution() * PX_PER_M;
          var zoomDiff = baseZoom - scope.map.getView().getZoom();

          scope.annotations.forEach(function(annot) {
            annot.showByScale(scale);
            annot.setScaling2(zoomDiff);
          });
        });

        var baseZoom;
        scope.map.on('rb_movestart', function() {
          baseZoom = scope.map.getView().getZoom();
        });

        $rootScope.$on('newScaleForResolution', function() {
          var scale = scope.map.getView().getResolution() * PX_PER_M;
          var zoomDiff = baseZoom - scope.map.getView().getZoom();

          scope.annotations.forEach(function(annot) {
            annot.showByScale(scale);
            annot.setScaling2(zoomDiff);
          });
        });

        // fire movestart avant ol 4.2.0
        // https://stackoverflow.com/a/31897512
        function onpropertychange() {
          scope.map.dispatchEvent('rb_movestart');
          var view = scope.map.getView();
          view.un('propertychange', onpropertychange);
          scope.map.on('moveend', function() {
            view.on('propertychange', onpropertychange);
          });
        }
        scope.map.getView().on('propertychange', onpropertychange);
      }, // link function
    };
  };

  annotationswidget.$inject = [
    '$rootScope',
    '$filter',
    '$timeout',
    '$q',
    'gclayers',
    'gcInteractions',
    'ParametersFactory',
    'extendedNgDialog',
  ];
  return annotationswidget;
});
