/*	Copyright (c) 2018 Jean-Marc VIGLINO, 
  released under the CeCILL-B license (French BSD license)
  (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt).
*/
/**
* A popup element to be displayed on a feature.
*
* @constructor
* @extends {ol.Overlay.Popup}
* @param {} options Extend Popup options 
*	@param {String} options.popupClass the a class of the overlay to style the popup.
*	@param {bool} options.closeBox popup has a close box, default false.
*	@param {function|undefined} options.onclose: callback function when popup is closed
*	@param {function|undefined} options.onshow callback function when popup is shown
*	@param {Number|Array<number>} options.offsetBox an offset box
*	@param {ol.OverlayPositioning | string | undefined} options.positionning 
*		the 'auto' positioning var the popup choose its positioning to stay on the map.
* @param {*} options.template A template with a list of properties to use in the popup
* @param boolean} options.canFix Enable popup to be fixed 
* @api stable
*/
ol.Overlay.PopupFeature = function (options) {
  options = options || {};
  ol.Overlay.Popup.call(this, options);
  this.setTemplate(options.template);
  this.set('canFix', options.canFix)
  // Bind with a select interaction
  if (options.select && (typeof options.select.on ==='function')) {
    this._select = options.select;
    options.select.on('select', function(e){
      if (!this._noselect) this.show(e.mapBrowserEvent.coordinate, options.select.getFeatures().getArray());
    }.bind(this));
  }
};
ol.inherits(ol.Overlay.PopupFeature, ol.Overlay.Popup);
/** Set the template
 * @param {*} template A template with a list of properties to use in the popup
 */
ol.Overlay.PopupFeature.prototype.setTemplate = function(template) {
  this._template = template || {};
  if (this._template.attributes instanceof Array) {
    var att = {};
    this._template.attributes.forEach(function (a) {
      att[a] = true;
    });
    this._template.attributes = att;
  }
};
/** Show the popup on the map
 * @param {ol.coordinate|undefined} coordinate Position of the popup
 * @param {ol.Feature|Array<ol.Feature>} features The features on the popup
 */
ol.Overlay.PopupFeature.prototype.show = function(coordinate, features) {
  if (!(features instanceof Array)) features = [features];
  this._features = features.slice();
  if (!this._count) this._count = 1;
  // Calculate html upon feaures attributes
  this._count = 1;
  var html = this._getHtml(features[0]);
  this.hide();
  if (html) {
    if (!coordinate) {
      coordinate = features[0].getGeometry().getFirstCoordinate();
    }
    ol.Overlay.Popup.prototype.show.call(this, coordinate, html);
  }
};
/**
 * @private
 */
ol.Overlay.PopupFeature.prototype._getHtml = function(feature) {
  if (!feature) return '';
  var html = ol.ext.element.create('DIV', { className: 'ol-popupfeature' });
  if (this.get('canFix')) {
    ol.ext.element.create('I', { className:'ol-fix', parent: html })
      .addEventListener('click', function(){
        this.element.classList.toggle('ol-fixed');
      }.bind(this));
  }
  if (this._template.title) {
    var title;
    if (typeof this._template.title === 'function') {
      title = this._template.title(feature);
    } else {
      title = feature.get(this._template.title);
    }
    ol.ext.element.create('H1', { html:title, parent: html });
  }
  if (this._template.attributes) {
    var tr, table = ol.ext.element.create('TABLE', { parent: html });
    var atts = this._template.attributes;
    for (var att in atts) {
      var a = atts[att];
      tr = ol.ext.element.create('TR', { parent: table });
      ol.ext.element.create('TD', { html: a.title || att, parent: tr });
      var val = feature.get(att);
      ol.ext.element.create('TD', { 
        html: (a.before||'') + (a.format ? a.format(val) : val) + (a.after||''), 
        parent: tr 
      });
    }
  }
  // Zoom button
  ol.ext.element.create('BUTTON', { className: 'ol-zoombt', parent: html })
    .addEventListener('click', function() {
      this.getMap().getView().fit(feature.getGeometry().getExtent(), { duration:1000 });
    }.bind(this));
  // Counter
  if (this._features.length > 1) {
    var div = ol.ext.element.create('DIV', { className: 'ol-count', parent: html });
    ol.ext.element.create('DIV', { className: 'ol-prev', parent: div })
      .addEventListener('click', function() {
        this._count--;
        if (this._count<1) this._count = this._features.length;
        html = this._getHtml(this._features[this._count-1]);
        ol.Overlay.Popup.prototype.show.call(this, this.getPosition(), html);
      }.bind(this));
    ol.ext.element.create('TEXT', { html:this._count+'/'+this._features.length, parent: div });
    ol.ext.element.create('DIV', { className: 'ol-next', parent: div })
      .addEventListener('click', function() {
        this._count++;
        if (this._count>this._features.length) this._count = 1;
        html = this._getHtml(this._features[this._count-1]);
        ol.Overlay.Popup.prototype.show.call(this, this.getPosition(), html);
      }.bind(this));
  }
  this._noselect = true;
  this._select.getFeatures().clear();
  this._select.getFeatures().push(feature);
  this._noselect = false;
  return html;
};
/** Get a function to use as format to get local string for an attribute
 * if the attribute is a number: Number.toLocaleString()
 * if the attribute is a date: Date.toLocaleString()
 * otherwise the attibute itself
 * @param {string} locales string with a BCP 47 language tag, or an array of such strings
 * @param {*} options Number or Date toLocaleString options
 * @return {function} a function that takes an attribute and return the formated attribute
 */
ol.Overlay.PopupFeature.localString = function (locales , options) {
  return function (a) {
    if (a && a.toLocaleString) {
      return a.toLocaleString(locales , options);
    } else {
      // Try to get a date from a string
      var date = new Date(a);
      if (isNaN(date)) return a;
      else return date.toLocaleString(locales , options);
    }
  };
};
