'use strict';
define(function () {
  const itvSoumValidSelection = function ($q, itvSoumFactory, SelectManager,
    $timeout, $filter, itvDrawPartsFactory, itvUtilsFactory, gaDomUtils,
    ogcFactory, FeatureTypeFactory) {


    const wgs84Sphere = new ol.Sphere(6378137);


    /**
     * Récupére la longueur de la portion inspectée depuis
     * les observations du fichier ITV.
     *
     * @param {*} part : Portion dont on veut connaître la longueur inspectée.
     * @returns Longueur ITV de la portion
     */
    const getLengthFromDetails = (part) => {
      const details = part.objValeur.details;
      let lMaxLength2 = 0.0;
      let lMaxLength = 0.0;
      //-- Parcours de chaque observation
      for (let ind = 1; ind < details.length; ind++) {
        if (details[ind].emplct_lng_vert_number <
          details[ind - 1].emplct_lng_vert_number) {
          //-- La longueur suivante est plus petite que la précédente
          //-- donc il y a eut inspection arrêtée et reprise en sens inverse.
          lMaxLength = details[ind - 1].emplct_lng_vert_number;
          lMaxLength2 =
              details[details.length - 1].emplct_lng_vert_number;
          break;
        }
        else {
          //-- La distance prcourue pour l'observation suivante est supérieure
          //-- à la précédente donc pas de reprise en sens inverse
          //-- pour l'instant.
          lMaxLength = details[ind].emplct_lng_vert_number;
        }
      }
      return lMaxLength + lMaxLength2;
    };


    /**
     * Récupére le susteme de projection des géométries de nos cabamisarions.
     *
     * @param {*} map Carte OpenLayers
     * @param {*} source Source OpenLayers contenant les canalisations
     * @returns Système de projection de la géométrie de nos canalisations
     */
    const getSourceProjection = (map, source) => {
      let projection = source.getProjection();
      if (!projection) {
        projection = map.getView().getProjection();
      }
      return projection;
    };


    /**
     * Vérifie si le système de projection de la géométrie des canalisations
     * est EPSG:4326 ou EPSG:3857, dans quel cas un traitement spécial
     * est nécessaire pour calculer correctement les longueurs.
     *
     * @param {*} sourceProjection : projestion de nos géoémtries canalsiation
     * @returns VRAI si systeme projection est 3857 ou 4326
     */
    const isSourceProj4328Or3857 = (sourceProjection) => {
      const projectionCode = sourceProjection.getCode();
      return projectionCode === 'EPSG:4326' || projectionCode === 'EPSG:3857';
    };


    /**
     * Somme les longueurs des canalisations mises en correspondances
     * avec la portion ITV.
     *
     * @param {*} map : carte OpenLayers
     * @returns somme des longueurs des canalisations mises en correspondance
     */
    const getLengthFromPipes = (map) => {
      let lengthOfPipes = 0;
      const sourceProj
        = getSourceProjection(map, itvUtilsFactory.originSourceCANAL);
      const projIs3857Or4326 = isSourceProj4328Or3857(sourceProj);
      //-- Les canalisation de la portion courante sont dessinées
      //-- sur la carte en vert et sont donc dans la source OpenLayers
      //-- nommée "originSourceCANAL".
      const pipes = itvUtilsFactory.originSourceCANAL.getFeatures();
      //-- Parcours des canalisations de l'ITV mises en correspondance.
      for (const pipe of pipes) {
        //-- Récupération de la géométrie de la canlisation
        let geom = pipe.getGeometry();
        if (projIs3857Or4326) {
          //-- Cas spécial de système de projection sur lesquels
          //-- on ne peut pas récupérer directement la longueur
          //-- car celle - ci serait fausse.
          geom = geom.clone();
          const transformFn = ol.proj.getTransform(sourceProj, 'EPSG:4326');
          geom.applyTransform(transformFn);
          const coordinates = geom.getCoordinates();
          for (let ind = 1; ind < coordinates.length; ++ind) {
            lengthOfPipes += wgs84Sphere.haversineDistance(coordinates[ind - 1], coordinates[ind]);
          }
        }
        else {
          //-- Géométrie en projection locale donc la longueur
          //-- est récupérable directement
          if (geom.getLength) {
            //-- Cas de géométrie LineString (SIAH avec modèle RAEPA)
            lengthOfPipes += geom.getLength();
          }
          else if (geom.getLineString) {
            //-- Cas MultiLineString (Annonay)
            lengthOfPipes += geom.getLineString().getLength();
          }
        }
      }
      return lengthOfPipes;
    };


    /**
     * Suite à correspondance modifiée, stockage/mise à jour sur la portion
     * des informations relatives aux longueurs.
     *
     * @param {*} scope : environnement de travail du widget
     * @param {*} part : portion à actualiser
     */
    const getLengthInfo = (partsOfCurrentItv, map, part) => {
      //-- Portion qui sera enregistrée dans le JSON
      part.objValeur.partLength = getLengthFromDetails(part);
      part.objValeur.gisLength = getLengthFromPipes(map);
      part.objValeur.lengthDelta
        = Math.abs(part.objValeur.partLength - part.objValeur.gisLength);
      //-- Portion affichée dans la liste des portions
      const partOfPartList = partsOfCurrentItv.find(
        thisPart => thisPart.partIndex === part.objValeur.indTroncon);
      if (partOfPartList) {
        partOfPartList.partLength = part.objValeur.partLength;
        partOfPartList.gisLength = part.objValeur.gisLength;
        partOfPartList.lengthDelta = part.objValeur.lengthDelta;
      }
    };


    const getPartsOfCurrentItv = (scope) => {
      if (!scope) {
        return;
      }
      else {
        if (scope.partsOfCurrentItv) {
          return scope.partsOfCurrentItv;
        }
        else {
          return getPartsOfCurrentItv(scope.$parent);
        }
      }
    };

    /**
     * S'assurer que l'objet part a des détails.
     * On vérifie que l'objet part a déjà des détails,
     * et si ce n'est pas le cas, prendre dans le tableau
     * scope.correspPolygons les détails d'un des polygons
     * (les 3 polygons ont le même part  et donc les mêmes détails
     * puisqu'il s'agit d'une seule ITV que l'on en train de traiter).
     *
     * La fonction prend deux paramètres : scope et part.
     * -- scope contient des informations sur l'état actuel de l'application,
     * -- part est l'objet part qui pourrait nécessiter des détails.
     */
    const ensurePartHasDetails = (scope, part) => {
      if (part.objValeur && part.objValeur.details) return;

      for (const polygon of scope.correspPolygons) {
        if (polygon.part.objValeur.details && polygon.part.objValeur.details.length) {
          part.objValeur = part.objValeur || {};
          part.objValeur.details = polygon.part.objValeur.details;
          break;
        }
      }
    };


    const synchronizePartFlags = (scope, part) => {
      const codes = itvUtilsFactory.getCodesOfPart(part);
      if (!codes) {
        console.log('aie aie aie');
      }
      const partIndex = itvUtilsFactory.getIndTronconFrom(part);
      if (codes.indexOf('CAA') != -1) {
        part.validated = scope.corresp.partHeaderCenter.partHeader.validated;
      }
      else {
        const partInList
          = scope.information.impor.itvResponse.objValeur.parts[partIndex];
        partInList.validated = part.validated
          = (scope.corresp.inspectionType === 'branchement'
          || (scope.corresp.partHeaderAmont.partHeader.validated
          && scope.corresp.partHeaderAval.partHeader.validated))
            && scope.corresp.partHeaderCenter.partHeader.validated;
        if (part.objValeur) {
          part.objValeur.validated = part.validated;
        }
      }
      if (part.validated) {
        ensurePartHasDetails(scope, part);
        getLengthInfo(getPartsOfCurrentItv(scope), scope.map, part);
      }
    };


    /**
     * Le but de cette fonction est de vérifier si un objet "polygon" donné
     * possède une propriété "feature" ou "features", et si ce n'est pas le cas,
     * de lui en attribuer une sur la base de l'objet "polygon" correspondant
     * dans l'objet corresp.parts.polygons['part'+partIndex].
     * Cela peut être utile dans une situation où certains objets "polygon"
     * n'ont pas certaines propriétés et doivent être mis à jour sur la base
     * des mêmes objets "polygon" qui possèdent les informations nécessaires.
     *
     * @param {*} polygon Description d'un ouvrage mis en correspondance du schéma de l'UI
     * @param {*} corresp Raccourci pour "scope.information.correspondance"
     * @param {*} partIndex Index de la portion de ce "polygon"
     * @returns
     */
    const checkPolygon = (polygon, corresp, partIndex) => {
      if (!polygon.feature && !polygon.features && corresp.parts.polygons) {
        const correspPolygons = corresp.parts.polygons['part'+partIndex];
        if (correspPolygons) {
          for (const correspPolygon of Object.values(correspPolygons)) {
            if (polygon.name===correspPolygon.name) {
              if (correspPolygon.feature) {
                polygon.feature = correspPolygon.feature;
              }
              else {
                polygon.features = correspPolygon.features;
              }
              return;
            }
          }
        }
      }
    };


    const manageEndOfDrawFeatures = (ws, def, corresp, polygon) => {
      if (++ws.resolved === ws.polygons.length) {
        corresp.parts.currentPolygon = ws.prevCurrentPolygon;
        def.resolve(polygon);
      }
    };


    const vspdDrawFeatures = (scope, parts) => {
      const def = $q.defer();
      const corresp = scope.information.correspondance;
      const workspace = {
        resolved: 0,
        polygons: scope.correspPolygons,
        prevCurrentPolygon: corresp.parts.currentPolygon
      };
      workspace.polygons.map((polygon) => {
        corresp.parts.currentPolygon = polygon;
        if (polygon.name.indexOf('Center') == -1) {
          scope.selected = true;
          checkPolygon(polygon, corresp, scope.currentPart.partIndex);
          const promise = scope.validerSelection(polygon, true);
          if (promise) {
            promise.then(() => {
              manageEndOfDrawFeatures (workspace, def, corresp, polygon);
            });
          }
          else {
            manageEndOfDrawFeatures (workspace, def, corresp, polygon);
          }
        }
        else {
          // -- scope.corresp.partHeaderCenter.partHeader et polygon.partHeader
          // -- sont les mêmes objets donc la validation de l'un est
          // -- la validation de l'autre.
          itvUtilsFactory.partHeaderValidated(polygon.partHeader);
          itvUtilsFactory.selectText(polygon);

          const codeofparthead = polygon.partHeader.code;
          let codes = itvUtilsFactory.getCodesOfPart(polygon.part);
          const indexofheaders = codes.indexOf(codeofparthead);
          polygon.part.objValeur.headers[indexofheaders]
            = angular.copy(polygon.partHeader);
          const indexAAD = codes.indexOf('AAD');
          const indexAAB = codes.indexOf('AAB');
          if (codeofparthead === 'AAD') {
            manageAadAab(scope, polygon, indexAAD, 'AAB');
          }
          else if (codeofparthead === 'AAB') {
            manageAadAab(scope, polygon, indexAAB, 'AAD');
          }

          let thePart;
          if (parts.currentPolygon) {
            thePart = parts.currentPolygon.part;
          }
          else {
            thePart = parts.part[0];
          }
          codes = itvUtilsFactory.getCodesOfPart(thePart);
          const indexInfilescurrent = codes.indexOf(polygon.partHeader.code);
          thePart.objValeur.headers[indexInfilescurrent]
            = angular.copy(polygon.partHeader);

          const itvRespParts
              = scope.information.impor.itvResponse.objValeur.parts;
          const indexInfiles = itvRespParts.findIndex(
            (part) => part.partIndex === polygon.part.objValeur.indTroncon);
          const partinitbobj = itvRespParts[indexInfiles].partHeaders
            .findIndex((header) => header.code === codeofparthead);
          itvRespParts[indexInfiles].partHeaders[partinitbobj]
              = angular.copy(polygon.partHeader);

          const ind = parts.partIndex.indexOf(polygon.part.objValeur.indTroncon);
          parts.part[ind] = angular.copy(polygon.part);
          const key = 'part' + polygon.part.objValeur.indTroncon;
          parts.polygons[key][polygon.name] = angular.copy(polygon);

          const indexindraw
            = itvUtilsFactory.getNameIndex(workspace.polygons, polygon.name);
          if (Object.keys(parts.polygons[key]).length == 3
            || parts.currentPolygon.part.objValeur.branchement) {
            itvRespParts[indexInfiles].matchedAutomatically = true;
            itvRespParts[indexInfiles].validated = true;
            polygon.part.objValeur.validated = true;
            workspace.polygons[indexindraw].__data__ = angular.copy(polygon);
            manageEndOfDrawFeatures (workspace, def, corresp, polygon);
          }
        }
      }
      );
      return def.promise;
    };


    const validerSelectionPartDefined = (scope, partovalidate) => {
      const defer = $q.defer();
      const parts = scope.information.correspondance.parts;
      const itvResponse = scope.information.impor.itvResponse;
      if (!angular.isDefined(parts.currentPolygon)
        || !partovalidate.$hiderow) {
        partovalidate.$hiderow = true;
        scope.getPart(partovalidate).then(() => {
          scope.dontSaveItvPart = true;
          vspdDrawFeatures(scope, parts).then((p) => {
            scope.dontSaveItvPart = false;
            itvSoumFactory.saveItvPart(itvResponse.objValeur, p.part,
              scope.dontSaveItvPart, scope.conf.config.network, scope.conf.config.datastore);
            defer.resolve();
          });
        });
      }
      else {
        scope.dontSaveItvPart = true;
        vspdDrawFeatures(scope, parts).then((p) => {
          scope.dontSaveItvPart = false;
          itvSoumFactory.saveItvPart(itvResponse.objValeur, p.part,
            scope.dontSaveItvPart, scope.conf.config.network, scope.conf.config.datastore);
          defer.resolve();
        });
      }
      return defer.promise;
    };

    const addPolygonOfExtremityNode = (scope, p, feature, res, ftiUid) => {
      p.feature = feature;
      p.crs = res.data.crs;
      p.id = ftiUid;
      p.partHeader.validated = true;
      itvUtilsFactory.setHeaderId(ftiUid, feature, p.partHeader);
      // -- Actualiser aussi "scope.information.correspondance.parts"
      // -- afin de ne pas perdre la correspondance.
      const headerInd = p.part.objValeur.headers.findIndex(
        header => header.code === p.partHeader.code);
      setCorrespPart(p.partHeader, headerInd, p.part.objValeur.indTroncon,
        scope);
      return itvUtilsFactory.addpolygons(scope, p);
    };

    const getPolygon = (scope, polygonName) => {
      return scope.correspPolygons.find(
        polygon => polygon.name === polygonName);
    };

    const cfpnceNextOrResolve = (iPol, p, res, i, filter, def, name) => {
      if (++iPol === p.length) {
        def.resolve();
      }
      else {
        cfpNotCpiExist(iPol, p, res, ++i, filter, def, name);
      }
    };

    const cfpNotCpiExist = (iPol, polygons, res, i, filter, def, name) => {
      const p = polygons[iPol].__data__;
      const pname = p.name;
      if (name === pname) {
        cfpnceNextOrResolve(iPol, polygons, res, i, filter, def, name);
      }
      else {
        if (i === res.data.features.length) {
          $timeout(() => {
            def.resolve();
          });
        }
        else {
          addPolygonOfExtremityNode(p, res.data.features[i],
            res, filter).then(() => {
            cfpnceNextOrResolve(iPol, polygons, res, i, filter, def, name);
          });
        }
      }
    };

    let mapProjCode;
    const getProj = (scope) => {
      if (!mapProjCode) {
        mapProjCode = scope.map.getView().getProjection().getCode();
      }
      return mapProjCode;
    };

    const getBoxFilter = (coords) => {
      return 'INTERSECTS(geom, POLYGON(('
        + (coords[0] - 1) + ' ' + (coords[1] - 1) + ','
        + (coords[0] + 1) + ' ' + (coords[1] - 1) + ','
        + (coords[0] + 1) + ' ' + (coords[1] + 1) + ','
        + (coords[0] - 1) + ' ' + (coords[1] + 1) + ','
        + (coords[0] - 1) + ' ' + (coords[1] - 1)
        + ')))';
    };

    /**
     * Retourne la partie nom du comopsant depuis l'identifiant GeoTools
     * de type "composant.NNN".
     * Utilisé lors de l'exploitation du résultat de la recherche des ouvrages
     * du réseau se trouvant sur un point. En effet, on fait une seule recherhe
     * avec plusieurs UID de FTI, et il faut dans le résultat identifier
     * le comosant auquel appartient la feature trouvée.
     *
     * @param {*} id
     * @returns
     */
    const getFtiNameFromId = (id) => {
      return id.split('.')[0];
    };

    /**
     * Recherche des ouvrages ponctuels du réseau situés
     * sur un point particuleir.
     *
     * @param {*} scope
     * @param {*} ftiUids : uid des composants onctuels du réseau
     * @param {*} pointCoordinates : cordonnées de la localisation
     *                    des ponctuels à chercher.
     * @param {*} polygon : informations liées à la portion en train d'être
     *                      mise en correspondance
     * @returns : promesse de réalisation de la fonction
     */
    const getExtremityNode = (scope, ftiUids, pointCoordinates, polygon) => {
      const def = $q.defer();
      ogcFactory.getfeatures('GetFeature', 'WFS', '1.0.0', ftiUids, 'json',
        getProj(scope), getBoxFilter(pointCoordinates)).then((res) => {
        if (res.data.totalFeatures !== 0) {
          const ftiName = getFtiNameFromId(res.data.features[0].id);
          const ftiUid = FeatureTypeFactory.getFeatureUidByName(ftiName);
          addPolygonOfExtremityNode(scope, polygon, res.data.features[0],
            res, ftiUid).then(() => {
            def.resolve();
          });
        }
        else {
          // -- Pas d'extrémité trouvée sur ce point, mais on doit continuer,
          // -- soit pout l'autre extrémité, soit pour dessiner et
          // -- prendre en compte ce qui a été trouvé.
          def.resolve();
        }
      });
      return def.promise;
    };

    /**
     * Recherche des ouvrages à l'extrémité de la canalisation que
     * l'operateur vient de "cliquer".
     * Cette recherche se fait à travers la rcherche de l'ouvrage
     * de l'extrémité aval puis de l'extrémité amont.
     *
     * @param {*} scope : fonction déportée depuis la directive ...
     * @param {*} polygonCenter : informations liées au trait vert du HTML
     *                            de la mise en correpondance
     * @returns promesse de réalisation de la fonction
     */
    const getExtremityNodes = (scope, polygonCenter) => {
      const def = $q.defer();
      const index = scope.networks.findIndex(
        (x) => x.name === scope.conf.config.network);
      const networkNodesFTI
        = scope.networks[index].nodesTypes.map(x => x.ftiUID);
      const ftiUids = networkNodesFTI.join(',');
      const GEOJSON_FORMAT = new ol.format.GeoJSON();
      const olGeometry = GEOJSON_FORMAT.readGeometry(
        polygonCenter.feature.geometry
      );
      let polygon = getPolygon(scope, 'polygonAmont');
      getExtremityNode(scope, ftiUids, olGeometry.getFirstCoordinate(),
        polygon).then(() => {
        polygon = getPolygon(scope, 'polygonAval');
        getExtremityNode(scope, ftiUids, olGeometry.getLastCoordinate(),
          polygon).then(() => {
          def.resolve();
        });
      });

      return def.promise;
    };

    /**
     * Fonction factorisée, il semble qu'elle permette le dessin
     *  des features des regards et tronçons de la correspondance de l'ITV.
     *
     * @param {*} polygon
     * @param {*} infoCorresp
     * @param {*} name
     * @returns
     */
    const selCenterSetPolygonfeatures = (scope, polygon, infoCorresp, name) => {
      const def = $q.defer();
      const promise = def.promise;
      const ftiUids = polygon.partHeader.featUids;
      let genCount = 0;
      for (let iFeat = 0; iFeat < ftiUids.length; iFeat++) {
        const feat = ftiUids[iFeat];
        itvUtilsFactory.partHeaderValidated(polygon);
        itvUtilsFactory.drawFeature(feat,
          polygon.partHeader.values[iFeat], getProj(scope), 'center').then(
          () => {
            itvUtilsFactory.getitvoperationalLayers().forEach(
              (featureLayer) => {
                if (featureLayer.getProperties().name == 'center') {
                  scope.map.getLayers().insertAt(5, featureLayer);
                }
              });
            getExtremityNodes(scope, polygon, name).then().finally(
              () => {
                if (++genCount === ftiUids.length) {
                  if (infoCorresp.lastDPath != undefined
                    && infoCorresp.lastDPath.fullData) {
                    const lastDPath = infoCorresp.lastDPath;
                    const features = lastDPath.fullData.path.features;
                    const indexFeature = features.features.map((x) => {
                      if (x.id == feat.featId) {
                        return x.id;
                      }
                    }).indexOf(feat.featId);
                    polygon.features[iFeat]
                        = lastDPath.nFeatureCollectionData.features[indexFeature];
                  }
                  def.resolve();
                }
              }
            );
          });
      }
      return promise;
    };

    /**
     * Je ne sais pas ce que fait la fonction.
     */
    const valSelCenterWithFeatUids = (scope, polygon, infoCorresp) => {
      // -- Obtention de l'index du code de l'ouvrage dont
      // -- on valide la mise en correspondance.
      const codeofparthead = polygon.partHeader.code;
      const indexofheaders = polygon.part.objValeur.headers
        .map((x) => {
          return x.code;
        }).indexOf(codeofparthead);
      // -- Obtention du numéro d ordre de la portion inspectée
      // -- dans le fichier ITV
      const indexInfiles = scope.information.impor.itvResponse.objValeur.parts
        .map((x) => {
          return x.partIndex;
        }).indexOf(polygon.part.objValeur.indTroncon);
      const objValeur = scope.information.impor.itvResponse.objValeur;
      const partinitbobj
        = objValeur.parts[indexInfiles].partHeaders.map((x) => {
          return x.code;
        }).indexOf(codeofparthead);
      if (partinitbobj !== -1) {
        objValeur.parts[indexInfiles].partHeaders[partinitbobj]
          = polygon.part.objValeur.headers[indexofheaders];
      }
      else {
        objValeur.parts[indexInfiles].partHeaders.push(
          polygon.part.objValeur.headers[indexofheaders]);
      }
      const indexInfilescurrent
        = infoCorresp.parts.currentPolygon.part.objValeur.headers
          .map((x) => {
            return x.code;
          }).indexOf(polygon.partHeader.code);
      infoCorresp.parts.currentPolygon.part.objValeur.headers[
        indexInfilescurrent] = angular.copy(polygon.partHeader);
      const Key = 'part' + polygon.part.objValeur.indTroncon;

      if (Object.keys(infoCorresp.parts.polygons[Key]).length == 3) {
        // -- Valide si regard départ + cana + regard arrivé
        // -- mis en correspondance.
        objValeur.parts[indexInfiles].matchedAutomatically = true;
        objValeur.parts[indexInfiles].validated = true;
      }
    };

    const selectPolygonText = (polygon, dfPolygons, infoCorresp) => {
      const indexdrawCenter = itvUtilsFactory.getNameIndex(dfPolygons[0], 'polygonCenter');
      if (angular.isDefined(
        dfPolygons[0][indexdrawCenter].__data__.features)) {
        dfPolygons[0][indexdrawCenter].__data__
          = angular.copy(polygon);
        infoCorresp.parts.currentPolygon = angular.copy(
          polygon
        );
      }
      itvUtilsFactory.selectText(polygon);
    };

    /**
     * Méthode illisible
     * @param polygon
     * @param correspDfPolygons
     */
    const valSelCenter = (scope, polygon, name) => {
      const def = $q.defer();
      const infoCorresp = scope.information.correspondance;
      if (angular.isDefined(polygon.partHeader.featUids) &&
        polygon.partHeader.featUids.length > 0) {
        itvUtilsFactory.partHeaderValidated(polygon);
        itvUtilsFactory.selectText(polygon);
        itvUtilsFactory.clearoriginsourcecanal();
        itvUtilsFactory.removeLayerByName(scope, 'center');
        selCenterSetPolygonfeatures(scope, polygon, infoCorresp, name).then(
          () => {
            valSelCenterWithFeatUids(scope, polygon, infoCorresp, name);
            def.resolve();
          }
        );
      }
      else {
        getExtremityNodes(scope, polygon, name).then(
          () => {
            selectPolygonText(polygon, scope.correspPolygons, infoCorresp);
            def.resolve();
          }
        );
      }

      return def.promise;
    };

    /**
     * Récupére l'identiant GeoTools pointée par le header.
     * Dans le cas ArcGIS l'identifiant GeoTools est le champ OBJECTID.
     * C'est ce champ qui est utilisé par le XgGraph du parcours réseau.
     *
     * @param {*} header
     * @returns
     */
    const getGtIdFromHeader = (header) => {
      if (header.partHeader) {
        header = header.partHeader;
      }
      if (header.featId4Net) {
        return { id: header.featId4Net, featUid: header.featUid };
      }
      else {
        return { id: header.value, featUid: header.featUid };
      }
    };

    /**
     * Positionnement des valeurs de travail correspondant
     * au cas de sélection sur la carte soit du noeud amont
     * soit du noeud aval.
     *
     * @param {*} iDrawAmont
     * @param {*} iDrawAval
     * @param {*} polygon
     * @param {*} amontAval
     * @returns
     */
    const vscGetParams = (scope) => {
      const params = {};
      // -- Identifiant des neouds départ et arrivé
      // -- pour recherche des tronçons intermédiaires
      // -- mais uniquement quand le choix des 2 regards a été validé.
      // -- On peut ne pas avoir de partHeaderAmont pour
      // -- une inspection de branchement
      if (scope.corresp.partHeaderAmont.partHeader
        && scope.corresp.partHeaderAmont.partHeader.validated) {
        const idDesc = getGtIdFromHeader(scope.corresp.partHeaderAmont);
        params.idStart = idDesc.id;
        params.featUidStart = idDesc.featUid;
      }
      // -- On peut ne pas avoir de partHeaderAval pour
      // -- une inspection de branchement
      if (scope.corresp.partHeaderAval.partHeader
        && scope.corresp.partHeaderAval.partHeader.validated) {
        const idDesc = getGtIdFromHeader(scope.corresp.partHeaderAval);
        params.idEnd = idDesc.id;
        params.featUidEnd = idDesc.featUid;
      }
      // -- Si les identifiants amont et aval calculés (selon la méthode
      // -- du code d'origine) sont les mêmes, alors on voit à changer
      // -- l'identifiant de celui qui n'est pas dans le "polygon" et qui /
      // -- doit correspondra à la sélection qui vient d'être faite.
      // -- Toutefois, il se peut aussi que l'on n'ait pas encore
      // -- sélectionné les 2 noeuds, dans ce cas on laisse comme c'est.
      if (params.idStart && params.idStart === params.idEnd) {
        const idDesc = getGtIdFromHeader(scope.corresp.partHeaderAval);
        params.idStart = idDesc.id;
        params.featUidStart = idDesc.featUid;
      }
      return params;
    };

    /**
     * Actualisaition de "scope.information.correspondance.parts"
     * pour ne pas perdre la correspondance lors du changement
     * de portion dans la liste des portions.
     *
     * @param {*} header : header actualisé
     * @param {*} headerIndex : index du header actualisé (AAA,AAB,AAD, ou AAF)
     * @param {*} partIndex : index de la portion concernée
     * @param {*} scope : scope de la soumission
     */
    const setCorrespPart = (header, headerIndex, partIndex, scope) => {
      const parts = scope.information.correspondance.parts;
      const indexInWorkList = parts.part.findIndex(
        part => part.objValeur.indTroncon === partIndex
      );
      const part = scope.information.correspondance.parts.part[indexInWorkList];
      part.objValeur.headers[headerIndex] = header;
    };

    const setCorresPartOfPipePart = (scope, p, indexInFiles) => {
      const headers = itvUtilsFactory.getHeadersFrom(p);
      // -- Actualiser les cana modifiées dans la portion de la liste
      // -- des portions modifiées, soit scope.information.correspondance.parts
      let headerIndex = headers.findIndex(pHeader =>
        pHeader ? pHeader.code === 'AAA' : true
      );
      setCorrespPart(headers[headerIndex], headerIndex, indexInFiles, scope);

      // -- Actualiser les regards modifiées dans la portion de la liste
      // -- des portions modifiées, soit scope.information.correspondance.parts
      headerIndex = headers.findIndex(pHeader =>
        pHeader ? pHeader.code === 'AAD' : true
      );
      setCorrespPart(headers[headerIndex], headerIndex, indexInFiles, scope);
      headerIndex = headers.findIndex(pHeader =>
        pHeader ? pHeader.code === 'AAB' : true
      );
      setCorrespPart(headers[headerIndex], headerIndex, indexInFiles, scope);
      setCorrespPart(headers[headerIndex], headerIndex, indexInFiles, scope);
      headerIndex = headers.findIndex(pHeader =>
        pHeader ? pHeader.code === 'AAF' : true
      );
      setCorrespPart(headers[headerIndex], headerIndex, indexInFiles, scope);
    };

    const manageAadAabCases = (scope, p, codeofparthead, header) => {
      const headers = itvUtilsFactory.getHeadersFrom(p);
      const indexofheaders
        = headers.findIndex(header => header.code === codeofparthead);
      headers[indexofheaders] = angular.copy(header);
      const indexAAD = headers.findIndex(header => header.code === 'AAD');
      const indexAAB = headers.findIndex(header => header.code === 'AAB');
      if (headers[indexAAD].validated) {
        manageAadAab(scope, p, indexAAD, 'AAB');
      }
      else if (headers[indexAAB].validated) {
        manageAadAab(scope, p, indexAAB, 'AAD');
      }
    };

    /**
     * Appelé aprés avoir trouvé le cheminentre 2 regards pour définir
     * le ou les tronçons de la portion en affectant le HEADER dont
     * le code est AAA
     *
     * @param {*} scope : scope de la soumission
     * @param {*} p : portion
     * @param {*} edges : tronçons du réseau entre les regards
     * @param {*} def : deferred pour faie resolve quand traitement fini
     * @returns
     */
    const addmultipolygons = (scope, p, edges, def) => {
      if (!scope.corresp.center) {
        scope.corresp.center = {};
      }
      const center = scope.corresp.center;
      center.features = [];
      const partHeader = scope.corresp.partHeaderCenter.partHeader;
      /*
      partHeader.featUids = [];
      partHeader.values = [];
      */
      itvUtilsFactory.clearoriginsourcecanal();
      itvUtilsFactory.removeLayerByName(scope, 'center');
      angular.forEach(edges, (elem, i) => {
        if (partHeader.values.indexOf(elem.featId) === -1) {
          center.features.push(JSON.parse(edges[i].feature));
          itvUtilsFactory.setHeaderId(elem.ftiUID, center.features[i],
            partHeader, i);
          partHeader.featUids.push(elem.ftiUID);

          itvUtilsFactory.partHeaderValidated(partHeader);
          itvUtilsFactory.drawFeature(elem.ftiUID, elem.featId,
            mapProjCode, 'center').then(() => {
            itvUtilsFactory.getitvoperationalLayers().forEach(
              (featureLayer) => {
                const flName = featureLayer.getProperties().name;
                if (flName === 'center' || flName === 'canal') {
                  scope.map.getLayers().insertAt(5, featureLayer);
                }
              });
          });
        }
      });
      itvUtilsFactory.partHeaderValidated(p);
      itvUtilsFactory.selectText(p);

      let header = p.partHeader;
      let codeofparthead;
      if (header) {
        codeofparthead = header.code;
        manageAadAabCases(scope, p, codeofparthead, header);
      }
      else {
        header = p;
        codeofparthead = p.code;
        p = {
          partHeader: header
        };
      }

      // -- Mettre à jour le header dans bien des endroits !!!!
      const parts = scope.information.correspondance.parts;
      const headers = itvUtilsFactory.getHeadersFrom(parts.currentPolygon.part);
      const headerIndex = headers.findIndex(pHeader =>
        pHeader ? pHeader.code === header.code : true
      );
      headers[headerIndex] = angular.copy(header);

      const objValeur = scope.information.impor.itvResponse.objValeur;
      const currentPart = itvUtilsFactory.getCurrentPart(scope);
      const indexInFiles = objValeur.parts.findIndex(
        part => part.partIndex === currentPart.partIndex
      );

      setCorresPartOfPipePart(scope, p, indexInFiles);

      let partinitbobj = -1;
      if (codeofparthead) {
        partinitbobj = objValeur.parts[indexInFiles].partHeaders.findIndex(
          header => header.code === codeofparthead
        );
      }
      if (partinitbobj !== -1) {
        objValeur.parts[indexInFiles].partHeaders[partinitbobj]
          = angular.copy(header);
      }
      else {
        objValeur.parts[indexInFiles].partHeaders.push(
          angular.copy(header));
      }

      const ind = parts.partIndex.indexOf(currentPart.partIndex);
      const partObjValeur = parts.part[ind].objValeur;
      parts.part[ind] = angular.copy(currentPart);
      parts.part[ind].objValeur = partObjValeur;
      const Key = 'part' + currentPart.partIndex;
      parts.polygons[Key].polygonCenter = angular.copy(p);
      parts.polygons[Key].polygonCenter.name = 'polygonCenter';
      parts.polygons[Key].polygonCenter.features = center.features;
      if (center.features.length) {
        parts.polygons[Key].polygonCenter.id = center.features[0].id;
      }

      if (
        Object.keys(scope.information.correspondance.parts.polygons[Key])
          .length === 3
      ) {
        objValeur.parts[indexInFiles].matchedAutomatically = true;
        objValeur.parts[indexInFiles].validated = true;
        currentPart.validated = true;
        if (parts.currentPolygon.part.objValeur) {
          parts.currentPolygon.part.objValeur.validated = true;
        }
        itvUtilsFactory.saveItvPart(objValeur.ItvId, currentPart.partIndex,
          parts.currentPolygon.part.objValeur, scope.conf.config.network,
          scope.conf.config.datastore).then(() => {
          def.resolve();
        });
      }
      else {
        def.resolve();
      }
      return p;
    };

    const soumissionProgressionGotten = (scope, res, res2, polygon,
      name, def, threadId, p) => {
      if (res2.data.etat == 'en cours' && res2.data.progression !== 100) {
        res.data = res2.data;
        $timeout(() => {waitEndOfThread (scope, threadId, res, p, polygon, def, name);}, 2000);
      }
      else if (res2.data.etat == 'erreur') {
        itvUtilsFactory.soumissionProgressionError(res2.data);
        def.reject();
      }
      else {
        // -- Récupération des tronçons
        // -- parmi la liste des ouvrages retournés.
        const amont = scope.corresp.partHeaderAmont.partHeader;
        const aval = scope.corresp.partHeaderAval.partHeader;
        const center = scope.corresp.partHeaderCenter;
        const edges = res2.data.objValeur.map((x) => {
          if (x.featId !== amont.value &&
            x.ftiUID !== amont.featUid &&
            x.featId !== aval.value &&
            x.ftiUID !== aval.featUid
          ) {
            return x;
          }
        }).filter((value) => {
          return value !== undefined;
        });
        try {
          itvUtilsFactory.clearHeaderIds(center);
          addmultipolygons(scope, center, edges, def);
          gaDomUtils.hideGlobalLoader();
          require('toastr').success(
            $filter('translate')('itv.soumission.chargementfini')
          );
        }
        catch (e) {
          console.log(e.stack);
          require('toastr').error(
            $filter('translate')('itv.correspondance.nocorrespondancedone')
          );
          gaDomUtils.hideGlobalLoader();
          def.reject();
        }
      }
    };

    /**
     * La fonction utilise la méthode itvSoumFactory.getProgression pour récupérer
     * la progression du thread avec le threadId donné.
     * Une fois la progression récupérée, elle appelle la fonction soumissionProgressionGotten
     * avec la progression récupérée et d'autres paramètres.
     * En résumé, cette fonction récupére la progression du traitement
     * de l'enregistrement de la correspondance de la portion courante et
     * appelle ensuite "soumissionProgressionGotten" avec la progression récupérée
     * pour agir en fonction de l'avancement du traitement.
     *
     * @param {*} scope  Ce paramètre fait référence à la portée dans laquelle
     *                la fonction est exécutée. Il s'agit de la portée du contrôleur du widget.
     * @param {*} threadId  Ce paramètre est l'identifiant unique du thread
     *                 pour lequel la progression est récupérée.
     * @param {*} res  Ce paramètre est la réponse de l'appel au service "getPathBetweenNodes".
     * @param {*} p  Description des noeuds
     * @param {*} polygon  Description des ouvrages schématiques
     * @param {*} def  Defer pour terminer la promesse dans "soumissionProgressionGotten"
     * @param {*} name  Nom du "poygon" courant
     */
    const waitEndOfThread = (scope, threadId, res, p, polygon, def, name) => {
      itvSoumFactory.getProgression(threadId).then((res2) => {
        soumissionProgressionGotten (scope, res, res2, polygon,
          name, def, threadId, p);
      });
    };


    /**
     * Recherche des tronçons situés entre le noeud amont et
     * le noeud aval désignés.
     *
     * @param {*} correspDfPolygons
     * @param {*} polygon
     * @param {*} amontAval
     */
    const valSelCorresp = (scope, polygon, amontAval, name) => {
      const def = $q.defer();

      const p = vscGetParams(scope);

      if (p.idStart && p.idEnd && p.idStart !== p.idEnd) {
        // -- Si les noeuds sont validés et différents on fait le parcours réseau.
        gaDomUtils.showGlobalLoader();
        // -- Recherche des canalisations entre les 2 noeuds sélectionnés.
        itvSoumFactory.getPathBetweenNodes(p.idStart, p.idEnd, p.featUidStart,
          p.featUidEnd, getProj(scope), scope.conf.config.network).then((res) => {
          if (res.data.etat === 'en cours') {
            // -- La recherche a commencé mais n'est pas terminée.
            if (res.data.etat == 'en cours') {
              waitEndOfThread(scope, res.data.strValeur, res, p, polygon, def, name);
            }
            else if (res.data.etat == 'erreur') {
              require('toastr').error(
                $filter('translate')('itv.soumission.existerreur')
              );
              gaDomUtils.hideGlobalLoader();
              def.reject();
            }
          }
        });
      }
      else {
        def.resolve();
      }
      return def.promise;
    };

    /**
     * Cherche la portion de scope.information.correspondance ayant l'index
     * de polygon.part.
     *
     * @param {*} parts
     * @param {*} polygon
     * @returns
     */
    const getPartFromCorrespParts = (scope, parts, polygon) => {
      const def = $q.defer();
      if (parts.part) {
        def.resolve(parts.part);
      }
      else {
        const indTroncon = itvUtilsFactory.getIndTronconFrom(polygon.part);
        itvSoumFactory.getitvPart(
          scope.information.impor.itvResponse.objValeur.ItvId, indTroncon)
          .then((res) => {
            def.resolve(res.data);
          });
      }
      return def.promise;
    };

    /**
     *
     * @param {*} scope : scope de la directive de travail
     * @param {*} p     : "polygon"
     * @param {*} index : index du code AAD ou du code AAB
     * @param {*} code  : code "AAD" ou "AAB"
     */
    const manageAadAab = (scope, p, index, code) => {
      itvUtilsFactory.validSelectionManageHeaderCode(scope, p, index, code,
        'fromValidSelection');
    };

    const getCurrentPartIndex = (scope) => {
      const parts = scope.information.impor.itvResponse.objValeur.parts;
      return parts.findIndex(part => part.$hideRows === true);
    };

    const setTrueToValidated = (scope, polygon, name) => {
      const otherParts
        = scope.information.impor.itvResponse.objValeur.parts;
      const ind = getCurrentPartIndex(scope);
      itvUtilsFactory.partHeaderValidated(polygon.partHeader, otherParts, ind);
      let header;
      if (name.includes('Aval')) {
        header = scope.corresp.partHeaderAval;
      }
      else if (name.includes('Amont')) {
        header = scope.corresp.partHeaderAval;
      }
      else {
        header = scope.corresp.partHeaderCenter;
      }
      itvUtilsFactory.partHeaderValidated(header, otherParts, ind);
    };

    /**
     * Cette fonction est juste une factorisation de "validerSelection".
     * Appeler suit clic sur coche du popup menu pour valider le tronçon ou
     * le regard choisi.
     *
     * @edg A quoi sert-elle ?
     * @edg Peut-on scinder cette fonction en plusieurs méthodes moins obscures?
     * @edg Toutes les variables ont des noms obscures.
     * Peut-on les décrire avec un commentaire de ligne?
     */
    const validerSelection02 = (scope, polygonToValidate) => {
      const def = $q.defer();
      const corresp = scope.information.correspondance;
      //              let polygon = angular.copy(corresp.parts.currentPolygon);
      let polygon;
      if (!polygonToValidate) {
        polygon = corresp.parts.currentPolygon;
      }
      else {
        polygon = polygonToValidate;
      }
      try {
        const name = polygon.name;
        if (polygon.partHeader) {
          polygon.partHeader.featUid = polygon.id;
          itvUtilsFactory.setHeaderId(polygon.id, polygon.feature,
            polygon.partHeader, undefined, true);
          // -- Validation du noeud ou du tronçon
          setTrueToValidated(scope, polygon, name);

          polygon.partHeader.featUids = [polygon.id];
          polygon.partHeader.values = [polygon.partHeader.value];

          const codeofparthead = polygon.partHeader.code;
          const headers = itvUtilsFactory.getHeadersFrom(polygon.part);
          const indexofheaders = headers.findIndex(
            header => header ? header.code === codeofparthead : false
          );
          headers[indexofheaders] = angular.copy(polygon.partHeader);
          const codes = itvUtilsFactory.getCodesOfPart(polygon.part);
          const indexAAD = codes.indexOf('AAD');
          const indexAAB = codes.indexOf('AAB');
          if (codeofparthead == 'AAD') {
            manageAadAab(scope, polygon, indexAAD, 'AAB');
          }
          else if (codeofparthead == 'AAB') {
            manageAadAab(scope, polygon, indexAAB, 'AAD');
          }
          else {
            // -- Cas de vérification systématique dans les autres cas.
            // -- Le noeud AAD peut être bon dans le fichier lu,
            // -- dans quel cas one ne passe jamais par les lignes précédentes
            // -- et on ne pourrait pas valider la portion.
            if (indexAAD != -1 && headers[indexAAD].validated) {
              manageAadAab(scope, polygon, indexAAD, 'AAB');
            }
          }
          if (name.indexOf('Center') !== -1) {
            itvUtilsFactory.clearAllSources();
          }
          itvUtilsFactory.addpolygons(scope, polygon).then((polygon) => {
            corresp.parts.currentPolygon = angular.copy(polygon);
            let promise;
            if (name.indexOf('Amont') !== -1) {
              scope.corresp.partHeaderAmont.partHeader
                = angular.copy(polygon.partHeader);
              promise = valSelCorresp(scope, polygon, 'Amont', name);
            }
            else if (name.indexOf('Aval') !== -1) {
              scope.corresp.partHeaderAval.partHeader
                = angular.copy(polygon.partHeader);
              promise = valSelCorresp(scope, polygon, 'Aval', name);
            }
            else if (name.indexOf('Center') !== -1) {
              scope.corresp.partHeaderCenter.partHeader
                = angular.copy(polygon.partHeader);
              promise = valSelCenter(scope, polygon, name);
            }
            else {
              // -- Alone (inspetion de regard)
              scope.corresp.partHeaderCenter.partHeader
                = angular.copy(polygon.partHeader);
            }

            if (!promise) {
              $timeout(() => {
                synchronizePartFlags(scope, polygon.part);
              });
            }
            else {
              const objValeur
                  = scope.information.impor.itvResponse.objValeur;
              promise.then(() => {
              //-- scope.corresp.partHeaderAmont.part
              //-- == scope.corresp.partHeaderAval  --> true
              //-- scope.corresp.partHeaderAmont.part
              //-- == scope.corresp.partHeaderCenter.part --> true
                const correspPart = scope.corresp.partHeaderCenter;
                getPartFromCorrespParts(scope, correspPart, polygon).then(
                  (part) => {
                    itvUtilsFactory.clearitvoperationalLayers();
                    itvDrawPartsFactory.drawParts(scope, part).then(() => {
                      synchronizePartFlags(scope, part);
                      itvSoumFactory.saveItvPart(objValeur, part,
                        scope.dontSaveItvPart, scope.conf.config.network,
                        scope.conf.config.datastore).then(() => {
                        def.resolve();
                      },
                      () => {
                        def.reject();
                      });
                    },
                    () => {
                      def.reject();
                    });
                  });
              },
              () => {
                def.reject();
              });
            }
          },
          () => {
            def.reject();
          });
        }
      }
      catch (e) {
        console.log(e.stack);
        require('toastr').info(
          $filter('translate')(
            'itv.correspondance.nocorrespondancedone'
          )
        );
        def.reject();
      }
      finally {
        if (scope.pointer &&
                  scope.pointer instanceof ol.interaction.Draw) {
          scope.pointer.setActive(false);
        }
        scope.map.removeInteraction(scope.pointer);
        scope.selected = false;
        SelectManager.clear();
        scope.tovalidate = false;
      }
      return def.promise;
    };

    /**
         *
         * @param {*} p
         * @param {*} amav : contient 'amont' ou 'aval'
         */
    const constructFeatureAmontAval = (scope, p, amav, aName) => {
      const def = $q.defer();
      let pathNodePt, layerIndex;
      const corresp = scope.information.correspondance;
      if (amav === 'amont') {
        pathNodePt = 'startPathNode';
        layerIndex = 10;
      }
      else {
        pathNodePt = 'endPathNode';
        layerIndex = 11;
      }
      const ftiUid = corresp.lastDPath.fullData[pathNodePt].ftiUID;
      p.partHeader.featUid = ftiUid;
      p.partHeader.value = corresp.lastDPath.fullData[pathNodePt].featId;
      const pathFeature
            = JSON.parse(corresp.lastDPath.fullData[pathNodePt].feature);
      itvUtilsFactory.setHeaderId(ftiUid, pathFeature, p.partHeader);
      itvUtilsFactory.drawFeature(p.partHeader.featUid,
        itvUtilsFactory.getIdValue(p.partHeader), getProj(scope), amav)
        .then(() => {
          itvUtilsFactory.removeLayerByName(scope, amav);
          itvUtilsFactory.getitvoperationalLayers().forEach((featureLayer) => {
            if (featureLayer.getProperties().name == amav) {
              scope.map.getLayers().forEach((Layer) => {
                if (amav === Layer.getProperties().name) {
                  itvUtilsFactory.removeLayer(scope, Layer);
                }
              });
              scope.map.getLayers().insertAt(layerIndex, featureLayer);
            }
          });
        });
      itvUtilsFactory.partHeaderValidated(p);

      const indexFeature = corresp.lastDPath.nFeatureCollectionData.features.map((x) => {
        if (x.id == p.partHeader.value) {
          return x.id;
        }
      }).indexOf(p.partHeader.value);
      p.feature
            = corresp.lastDPath.nFeatureCollectionData.features[indexFeature];

      const codeofparthead = p.partHeader.code;
      const indexofheaders = p.part.objValeur.headers.map(function (x) {
        return x.code;
      }).indexOf(codeofparthead);
      p.part.objValeur.headers[indexofheaders] = angular.copy(
        p.partHeader
      );

      const indexAAD = p.part.objValeur.headers.map(function (x) {
        return x.code;
      }).indexOf('AAD');
      const indexAAB = p.part.objValeur.headers.map(function (x) {
        return x.code;
      }).indexOf('AAB');
      if (codeofparthead == 'AAD') {
        manageAadAab(scope, p, indexAAD, 'AAB');
      }
      else if (codeofparthead == 'AAB') {
        manageAadAab(scope, p, indexAAB, 'AAD');
      }

      const indexInfiles = scope.information.impor.itvResponse.objValeur.parts
        .map((x) => {
          return x.partIndex;
        }).indexOf(p.part.objValeur.indTroncon);
      const partinitbobj = scope.information.impor.itvResponse.objValeur.parts[
        indexInfiles
      ].partHeaders
        .map((x) => {
          return x.code;
        }).indexOf(codeofparthead);
      scope.information.impor.itvResponse.objValeur.parts[
        indexInfiles
      ].partHeaders[partinitbobj] = angular.copy(p.partHeader);

      const ind = corresp.parts.partIndex.indexOf(
        p.part.objValeur.indTroncon
      );
      corresp.parts.part[ind] = angular.copy(p.part);
      const Key = 'part' + p.part.objValeur.indTroncon;
      corresp.parts.polygons[Key][aName] = angular.copy(p);

      if (Object.keys(corresp.parts.polygons[Key]).length == 3) {
        scope.information.impor.itvResponse.objValeur.parts[
          indexInfiles
        ].matchedAutomatically = true;
      }
      const curInFileInd
            = corresp.parts.currentPolygon.part.objValeur.headers.map((x) => {
              return x.code;
            }).indexOf(p.partHeader.code);
      corresp.parts.currentPolygon.part.objValeur.headers[curInFileInd]
            = angular.copy(p.partHeader);
      def.resolve();
      return def.promise;
    };

    /**
     * Fonction issu d'une factorisation,
     * difficile de dire précisément son rôle.
     *
     * @param {*} p
     * @param {*} corresp
     * @param {*} impor
     * @param {*} name
     */
    const constructFeatureAloneStep2 = (p, corresp, impor, name) => {
      // -- Recherche de l'index du premier tronçon.
      let idVal = p.partHeader.value;
      if (idVal === '' && p.partHeader.featId4Net) {
        idVal = p.partHeader.featId4Net;
      }
      const data = corresp.lastDPath.nFeatureCollectionData;
      const indexFeature = data.features.map((x) => {
        if (x.id == idVal) {
          return x.id;
        }
      }).indexOf(p.partHeader.value);
      // -- Premier tronçon
      p.feature = data.features[indexFeature];
      const codeofparthead = p.partHeader.code;
      const indexofheaders = p.part.objValeur.headers.map((x) => {
        return x.code;
      }).indexOf(codeofparthead);
      p.part.objValeur.headers[indexofheaders] = angular.copy(
        p.partHeader
      );

      const indexInfiles = impor.itvResponse.objValeur.parts.map(
        (x) => {
          return x.partIndex;
        }).indexOf(p.part.objValeur.indTroncon);
      const headers
        = impor.itvResponse.objValeur.parts[indexInfiles].partHeaders;
      const partinitbobj
        = headers.findIndex((x) => x.code === codeofparthead);
      headers[partinitbobj] = angular.copy(p.partHeader);

      const ind = corresp.parts.partIndex.indexOf(
        p.part.objValeur.indTroncon
      );
      corresp.parts.part[ind] = angular.copy(p.part);
      const key = 'part' + p.part.objValeur.indTroncon;
      corresp.parts.polygons[key][name] = angular.copy(p);

      if (Object.keys(corresp.parts.polygons).length == 1) {
        impor.itvResponse.objValeur.parts[
          indexInfiles
        ].matchedAutomatically = true;
      }

      const indexInfilescurrent = corresp.parts.currentPolygon.part.objValeur.headers
        .map((x) => {
          return x.code;
        }).indexOf(p.partHeader.code);
      corresp.parts.currentPolygon.part.objValeur.headers[
        indexInfilescurrent
      ] = angular.copy(p.partHeader);
    };

    /**
     * Fonction issu d'une factorisation,
     * difficile de dire précisément son rôle.
     *
     * @param {*} p
     * @param {*} corresp
     * @param {*} impor
     * @param {*} name
     * @returns
     */
    const constructFeatureAlone = (scope, p, corresp, impor, name) => {
      const def = $q.defer();
      p.partHeader.featUid = corresp.lastDPath.fullData.endPathNode.ftiUID;
      p.partHeader.value = corresp.lastDPath.fullData.endPathNode.featId;
      itvUtilsFactory.partHeaderValidated(p);
      itvUtilsFactory.drawFeature(p.partHeader.featUid,
        itvUtilsFactory.getIdValue(p.partHeader), getProj(scope),
        'alone').then(() => {
        itvUtilsFactory.removeLayers(scope, 12);
        constructFeatureAloneStep2(p, corresp, impor, name);
        def.resolve();
      });
      return def.promise;
    };

    const getIndexOfElem = (corresp, elem) => {
      return corresp.lastDPath.fullData.path.features.features
        .map(x => {
          if (x.id == elem.featId) {
            return x.id;
          }
        }).indexOf(elem.featId);
    };

    /**
     * Dessine les entités mises en correspondance sur la carte.
     *
     * @param {object} scope - Scope de la directive ITV.
     * @param {object} corresp - Descripttion des correspondances.
     * @param {array} idsfullregards - Liste des UID des FTIS des noeuds (regards).
     * @param {object} p - ENtêtes de la portion.
     * @return {void}
     */
    const drawFeatures = (scope, corresp, idsfullregards, p) => {
      const dfPromises = [];
      angular.forEach(corresp.lastDPath.fullData.path.elements,
        (elem, i) => {
          if (!idsfullregards.includes(elem.ftiUID)) {
            const indexFeature = getIndexOfElem(corresp, elem);
            p.partHeader.featUids.push(elem.ftiUID);
            p.features[i] =
                  corresp.lastDPath.nFeatureCollectionData.features[indexFeature];
            const ftiUid = elem.featUid ? elem.featUid : elem.ftiUID;
            itvUtilsFactory.setHeaderId(ftiUid, p.features[i], p.partHeader, i);
            itvUtilsFactory.partHeaderValidated(p);
            const promise = itvUtilsFactory.drawFeature(ftiUid, elem.featId,
              getProj(scope), 'center').then(() => {
              itvUtilsFactory.getitvoperationalLayers().forEach(
                (featureLayer) => {
                  const layerName = featureLayer.getProperties().name;
                  if (layerName === 'center' || layerName === 'canal') {
                    scope.map.getLayers().insertAt(5, featureLayer);
                  }
                });
            });
            dfPromises.push(promise);
          }
        }
      );
      return dfPromises;
    };


    /**
     * Fonction issu d'une factorisation,
     * difficile de dire précisément son rôle.
     *
     * @param {*} p
     * @param {*} corresp
     * @param {*} impor
     * @param {*} name
     * @returns
     */
    const constructAllElements = (scope, p, corresp, impor, name) => {
      const def = $q.defer();
      if (p.partHeader.value !== '') {
        p.partHeader.featUid = '';
        p.partHeader.value = '';
        delete p.feature;
      }
      p.features = [];
      const idsfullregards = [
        corresp.lastDPath.fullData.endPathNode.ftiUID,
        corresp.lastDPath.fullData.startPathNode.ftiUID
      ];
      itvUtilsFactory.clearHeaderIds(p);
      itvUtilsFactory.clearoriginsourcecanal();

      itvUtilsFactory.removeLayerByName(scope, 'center');

      const dfPromises = drawFeatures(scope, corresp, idsfullregards, p);

      const codeofparthead = p.partHeader.code;
      const headerIndex = p.part.objValeur.headers.findIndex(
        (x) => x.code === codeofparthead);
      if (headerIndex !== -1) {
        p.part.objValeur.headers[headerIndex] = angular.copy(p.partHeader);
      }
      else {
        p.part.objValeur.headers.push(angular.copy(p.partHeader));
      }
      const indexInfiles = impor.itvResponse.objValeur.parts.map((x) => {
        return x.partIndex;
      }).indexOf(p.part.objValeur.indTroncon);
      const partinitbobj = impor.itvResponse.objValeur.parts[
        indexInfiles
      ].partHeaders
        .map(function (x) {
          return x.code;
        }).indexOf(codeofparthead);
      if (partinitbobj !== -1) {
        impor.itvResponse.objValeur.parts[indexInfiles].partHeaders[partinitbobj]
          = angular.copy(p.partHeader);
      }
      else {
        impor.itvResponse.objValeur.parts[indexInfiles]
          .partHeaders.push(angular.copy(p.partHeader));
      }
      const indexInfilescurrent
            = corresp.parts.currentPolygon.part.objValeur.headers
              .map(function (x) {
                return x.code;
              }).indexOf(p.partHeader.code);
      corresp.parts.currentPolygon.part.objValeur.headers[
        indexInfilescurrent
      ] = angular.copy(p.partHeader);

      const ind = corresp.parts.partIndex.indexOf(
        p.part.objValeur.indTroncon);
      if (ind !== -1) {
        corresp.parts.part[ind] = angular.copy(p.part);
      }
      else {
        corresp.parts.part.push(angular.copy(p.part));
      }
      const Key = 'part' + p.part.objValeur.indTroncon;
      corresp.parts.polygons[Key][name] = angular.copy(p);

      if (Object.keys(corresp.parts.polygons[Key]).length == 3) {
        impor.itvResponse.objValeur.parts[indexInfiles].matchedAutomatically = true;
        impor.itvResponse.objValeur.parts[indexInfiles].validated
              = true;
      }
      if (dfPromises.length !== 0) {
        $q.all(dfPromises).then(()=> {
          def.resolve();
        });
      }
      else {
        def.resolve();
      }
      return def.promise;
    };

    /**
         * Validation de la correspondance établie suite à l'utilisation
         * du parcours réseau pour sélectionner les ouvrages inspectés.
         */
    const validerSelParcoursReseau = (scope) => {
      const corresp = scope.information.correspondance;
      const impor = scope.information.impor;
      const polygons = scope.correspPolygons;
      let polygonDoneCount = 0;
      itvUtilsFactory.clearAllSources();
      for (const p of polygons) {
        const name = p.name;
        let promise;
        p.partHeader.validated = true;
        if (name.indexOf('Amont') !== -1) {
          promise = constructFeatureAmontAval(scope, p, 'amont', name);
        }
        else if (name.indexOf('Aval') !== -1) {
          promise = constructFeatureAmontAval(scope, p, 'aval', name);
        }
        else if (name.indexOf('Alone') !== -1) {
          promise = constructFeatureAlone(scope, p, corresp, impor, name);
        }
        else {
          promise = constructAllElements(scope, p, corresp, impor, name);
        }

        promise.then(() => {
          if (name == corresp.parts.currentPolygon.name) {
            corresp.parts.currentPolygon = angular.copy(p);
          }
          if (name.indexOf('Amont') !== -1) {
            itvUtilsFactory.selectText(p, '.textAmont');
          }
          else if (name.indexOf('Aval') !== -1) {
            itvUtilsFactory.selectText(p, '.textAval');
          }
          else {
            itvUtilsFactory.selectText(p);
          }
        }
        ).finally(() => {
          if (++polygonDoneCount === polygons.length) {
            corresp.parts.currentPolygon.part.objValeur.validated = true;
            const polygon = angular.copy(corresp.parts.currentPolygon);
            $timeout(() => {
              synchronizePartFlags(scope, polygon.part);
              itvSoumFactory.saveItvPart(impor.itvResponse.objValeur, polygon.part,
                scope.dontSaveItvPart, scope.conf.config.network,
                scope.conf.config.datastore).then(() => {
                require('toastr').success(
                  $filter('translate')('itv.correspondance.parcoursuccess')
                );
                scope.usenetwork = false;
                scope.tovalidate = false;
                scope.lastDPath.fullData = null;
                scope.lastDPath.nFeatureCollectionData.features.splice(0,
                  scope.lastDPath.nFeatureCollectionData.features.length);
              });
            });
          }
        });
      }
    };

    /**
     * Clic sur coche de validation du popover d'un des polygones.
     * Le regard est validé si premier rien de plus si 2 eme parcours réseau
     * entre les 2 regards pour trouver tronçons.
     * Le tronçon est validé, recherche et validation automatique
     * des regards aux extrémités.
     *
     * @param {*} scope : scope de osumission
     * @param {*} partovalidate : portion à valider
     * @param {*} casValiderSel02 : appelé depuis "validerSelection02"
     * @returns
     */
    const validerSelection = (scope, partovalidate, casValiderSel02) => {
      if (partovalidate && !casValiderSel02) {
        // -- Clic sur la coche orange.
        validerSelectionPartDefined(scope, partovalidate).then(() => {
          synchronizePartFlags(scope, partovalidate);
        });
      }
      else {
        if (scope.selected && !scope.usenetwork) {
          return validerSelection02(scope,
            partovalidate);
        }
        else {
          validerSelParcoursReseau(scope);
        }
      }
    };

    return {
      validerSelection02,
      synchronizePartFlags,
      validerSelParcoursReseau,
      validerSelection
    };
  };
  itvSoumValidSelection.$inject = ['$q', 'itvSoumFactory', 'SelectManager',
    '$timeout', '$filter', 'itvDrawPartsFactory', 'itvUtilsFactory',
    'gaDomUtils', 'ogcFactory', 'FeatureTypeFactory'];
  return itvSoumValidSelection;
});
