'use strict';
define(() => {

  /**
   * Lecteur vidéo d'une portion d'ITV
   * Fonctions de liaison entre le lecteur et la carte
   */
  const itvvisualisationvideo = (itvVisuFactory, itvSoumFactory2, itvStylesFactory, mapJsUtils,
      gaJsUtils, QueryFactory, FeatureTypeFactory, SelectManager) => {
    return {
      templateUrl:
          'js/XG/widgets/utilities/itv/views/visualisation/itvVisualisationVideo.html',
      restrict: 'EA',
      scope: {
        map: '=',       // carte openlayers de l'application
        filepath: '=',  // chemin de la vidéo depuis le dossier temporaire tomcat
        part: '=',      // portion inspectée
        conf: '='       // configuration du widget
      },

      link: (scope) => {

        /*****************************************************************************

         LECTEUR VIDEO

         Code from tutorial "How to build a Custom HTML5 Video Player with JavaScript"
         @see https://github.com/Freshman-tech/custom-html5-video

         ****************************************************************************/

            // Select elements here
        const video = document.getElementById('video');
        const videoControls = document.getElementById('video-controls');
        const playButton = document.getElementById('play');
        const playbackIcons = document.querySelectorAll('.playback-icons use');
        const timeElapsed = document.getElementById('time-elapsed');
        const duration = document.getElementById('duration');
        const progressBar = document.getElementById('progress-bar');
        const seek = document.getElementById('seek');
        const seekTooltip = document.getElementById('seek-tooltip');
        const volumeButton = document.getElementById('volume-button');
        const volumeIcons = document.querySelectorAll('.volume-button use');
        const volumeMute = document.querySelector('use[href="#volume-mute"]');
        const volumeLow = document.querySelector('use[href="#volume-low"]');
        const volumeHigh = document.querySelector('use[href="#volume-high"]');
        const volume = document.getElementById('volume');
        const playbackAnimation = document.getElementById('playback-animation');
        const fullscreenButton = document.getElementById('fullscreen-button');
        const videoContainer = document.getElementById('video-container');
        const fullscreenIcons = fullscreenButton.querySelectorAll('use');

        const videoWorks = !!document.createElement('video').canPlayType;
        if (videoWorks) {
          video.controls = false;
          videoControls.classList.remove('hidden');
        }

        // Add functions here

        /**
         * togglePlay toggles the playback state of the video
         * If the video playback is paused or ended, the video is played
         * otherwise, the video is paused
         */
        const togglePlay = () => {
          if (video.paused || video.ended) {
            if (scope.isVideoPause) {
              remainAnimation();
            } else {
              startAnimation();
            }
            video.play();
          } else {
            video.pause();
            pauseAnimation();
          }
        };

        /**
         * updatePlayButton updates the playback icon and tooltip
         * depending on the playback state
         */
        const updatePlayButton = () => {
          playbackIcons.forEach((icon) => icon.classList.toggle('hidden'));

          if (video.paused) {
            playButton.setAttribute('data-title', 'Play (k)');
          } else {
            playButton.setAttribute('data-title', 'Pause (k)');
          }
        };

        /**
         * formatTime takes a time length in seconds and returns the time in
         * minutes and seconds
         * @param {number} timeInSeconds
         * @return {{seconds: string, minutes: string}}
         */
        const formatTime = (timeInSeconds) => {
          const date = new Date(timeInSeconds * 1000);
          if (date instanceof Date && moment(date).isValid()) {
            const result = date.toISOString().substr(11, 8);
            return {
              minutes: result.substr(3, 2),
              seconds: result.substr(6, 2),
            };
          }
        };

        /**
         * initializeVideo sets the video duration, and maximum value of the progressBar
         */
        const initializeVideo = () => {
          const videoDuration = Math.round(video.duration);
          seek.setAttribute('max', videoDuration);
          progressBar.setAttribute('max', videoDuration);
          const time = formatTime(videoDuration);
          duration.innerText = `${time.minutes}:${time.seconds}`;
          duration.setAttribute('datetime', `${time.minutes}m ${time.seconds}s`);
        };

        /**
         * updateTimeElapsed indicates how far through the video
         * the current playback is by updating the timeElapsed element
         */
        const updateTimeElapsed = () => {
          const time = formatTime(Math.round(video.currentTime));
          timeElapsed.innerText = `${time.minutes}:${time.seconds}`;
          timeElapsed.setAttribute('datetime', `${time.minutes}m ${time.seconds}s`);
        };

        /**
         * updateProgress indicates how far through the video
         * the current playback is by updating the progress bar
         */
        const updateProgress = () => {
          seek.value = Math.floor(video.currentTime);
          progressBar.value = Math.floor(video.currentTime);
        };

        /**
         * updateSeekTooltip uses the position of the mouse on the progress bar to
         * roughly work out what point in the video the user will skip to if
         * the progress bar is clicked at that point
         * @param {MouseEvent} event Evènement déclenché au clic sur la barre de progression de la vidéo
         */
        const updateSeekTooltip = (event) => {
          const skipTo = Math.round(
              (event.offsetX / event.target.clientWidth) *
              parseInt(event.target.getAttribute('max'), 10)
          );
          seek.setAttribute('data-seek', skipTo);
          const t = formatTime(skipTo);
          if (t && t.minutes && t.seconds) {
            seekTooltip.textContent = `${t.minutes}:${t.seconds}`;
            const rect = video.getBoundingClientRect();
            seekTooltip.style.left = `${event.pageX - rect.left}px`;
          }
        };

        /**
         * skipAhead jumps to a different point in the video when the progress bar is clicked
         * @param {MouseEvent|object} event Evènement déclenché au clic sur la barre de progression de la vidéo
         * @param {boolean} moveMarker est true quand on cherche à déplacer le marqueur sur la carte alors que la vidéo est en pause
         */
        const skipAhead = (event, moveMarker = true) => {
          const skipTo = event.target.dataset.seek
              ? event.target.dataset.seek
              : event.target.value;
          video.currentTime = skipTo;
          progressBar.value = skipTo;
          seek.value = skipTo;

          if (!scope.animating && (moveMarker || video.paused || scope.isVideoPause)) {
            // déplace le marqueur sur la carte alors que la vidéo est en pause
            const elapsedDistance = mapJsUtils.getMultiLineStringLength(scope.partLine, 0) * skipTo
                / video.duration;
            scope.currentCoords = mapJsUtils.getMultiLineStringCoordinatesAt(scope.partLine, 0,
                elapsedDistance);
            scope.geoMarker.getGeometry().setCoordinates(scope.currentCoords);
            scope.geoMarker.setStyle(itvStylesFactory.VIDEO_MARKER_STYLE);
          }
        };

        /**
         * updateVolume updates the video's volume
         * and disables the muted state if active
         */
        const updateVolume = () => {
          if (video.muted) {
            video.muted = false;
          }

          video.volume = volume.value;
        };

        /**
         * updateVolumeIcon updates the volume icon so that it correctly reflects
         * the volume of the video
         */
        const updateVolumeIcon = () => {
          volumeIcons.forEach((icon) => {
            icon.classList.add('hidden');
          });

          volumeButton.setAttribute('data-title', 'Mute (m)');

          if (video.muted || video.volume === 0) {
            volumeMute.classList.remove('hidden');
            volumeButton.setAttribute('data-title', 'Unmute (m)');
          } else if (video.volume > 0 && video.volume <= 0.5) {
            volumeLow.classList.remove('hidden');
          } else {
            volumeHigh.classList.remove('hidden');
          }
        };

        /**
         * toggleMute mutes or unmutes the video when executed.
         * When the video is unmuted, the volume is returned to the value
         * it was set to before the video was muted
         */
        const toggleMute = () => {
          video.muted = !video.muted;

          if (video.muted) {
            volume.setAttribute('data-volume', volume.value);
            volume.value = 0;
          } else {
            volume.value = volume.dataset.volume;
          }
        };

        /**
         * animatePlayback displays an animation when the video is played or paused
         */
        const animatePlayback = () => {
          playbackAnimation.animate(
              [
                {
                  opacity: 1,
                  transform: 'scale(1)',
                },
                {
                  opacity: 0,
                  transform: 'scale(1.3)',
                },
              ],
              {
                duration: 500,
              }
          );
        };

        /**
         * toggleFullScreen toggles the full screen state of the video.<br>
         * If the browser is currently in fullscreen mode, then it should exit and vice versa.
         */
        const toggleFullScreen = () => {
          if (document.fullscreenElement) {
            document.exitFullscreen();
          } else if (document.webkitFullscreenElement) {
            // Need this to support Safari
            document.webkitExitFullscreen();
          } else if (videoContainer.webkitRequestFullscreen) {
            // Need this to support Safari
            videoContainer.webkitRequestFullscreen();
          } else {
            videoContainer.requestFullscreen();
          }
        };

        /**
         * updateFullscreenButton changes the icon of the full screen button
         * and tooltip to reflect the current full screen state of the video
         */
        const updateFullscreenButton = () => {
          fullscreenIcons.forEach((icon) => icon.classList.toggle('hidden'));

          if (document.fullscreenElement) {
            fullscreenButton.setAttribute('data-title', 'Exit full screen (f)');
          } else {
            fullscreenButton.setAttribute('data-title', 'Full screen (f)');
          }
        };

        /**
         * hideControls hides the video controls when not in use
         * if the video is paused, the controls must remain visible
         */
        const hideControls = () => {
          if (video.paused) {
            return;
          }
          videoControls.classList.add('hide');
        };

        /**
         * showControls displays the video controls
         */
        const showControls = () => {
          videoControls.classList.remove('hide');
        };

        /**
         * keyboardShortcuts executes the relevant functions for
         * each supported shortcut key
         * @param {KeyboardEvent} event
         */
        function keyboardShortcuts(event) {
          const {key} = event;
          switch (key) {
            case 'k':
              togglePlay();
              animatePlayback();
              if (video.paused) {
                showControls();
              } else {
                setTimeout(() => {
                  hideControls();
                }, 2000);
              }
              break;
            case 'm':
              toggleMute();
              break;
            case 'f':
              toggleFullScreen();
              break;
          }
        }

        // Add eventlisteners here
        playButton.addEventListener('click', togglePlay);
        video.addEventListener('play', updatePlayButton);
        video.addEventListener('pause', updatePlayButton);
        video.addEventListener('loadedmetadata', initializeVideo);
        video.addEventListener('timeupdate', updateTimeElapsed);
        video.addEventListener('timeupdate', updateProgress);
        video.addEventListener('volumechange', updateVolumeIcon);
        video.addEventListener('click', togglePlay);
        video.addEventListener('click', animatePlayback);
        video.addEventListener('mouseenter', showControls);
        video.addEventListener('mouseleave', hideControls);
        videoControls.addEventListener('mouseenter', showControls);
        videoControls.addEventListener('mouseleave', hideControls);
        seek.addEventListener('mousemove', updateSeekTooltip);
        seek.addEventListener('input', skipAhead);
        volume.addEventListener('input', updateVolume);
        volumeButton.addEventListener('click', toggleMute);
        fullscreenButton.addEventListener('click', toggleFullScreen);
        videoContainer.addEventListener('fullscreenchange', updateFullscreenButton);
        document.addEventListener('keyup', keyboardShortcuts);

        /*****************

         FIN LECTEUR VIDEO

         *****************/

        const VIDEO_MARKER_ID = 'itv-video-marker';
        const PART_LAYER_ID = 'itv-video-part-layer';
        const DETAILS_LAYER_ID = 'itv-video-details-layer';
        const ANIMATION_LAYER_ID = 'itv-video-layer';
        const PART_LINE_ID = 'itv-video-line';
        const GEOJSON_FORMAT = new ol.format.GeoJSON();

        /**
         * Dessine les objets de la portion inspectée dans la couche de dessin temporaire (conduite + extrémités)
         * @param {object[]} partHeaders headers de la portion inspectée
         * @param {object[]} features objets renvoyés par l'appel API {@link itvVisuFactory.getPartGeometries}
         * @param {ol.source.Vector} drawSource référence de la couche de dessin des objets de la portion d'ITV
         */
        const drawPartFeatures = (partHeaders, features, drawSource) => {
          for (const header of partHeaders) {
            let featureStyle;
            let featureName = '';
            let featureId = header.value;
            switch (header._code) {
              case 'AAA':
                featureStyle = itvStylesFactory.PIPE_ROUTE_STYLE;
                // on affecte un ID prédéfini pour faire fonctionner
                // le chgt de curseur au survol et le déplacement du marqueur au clic
                featureId = PART_LINE_ID;
                break;
              case '@01':
                featureStyle = itvStylesFactory.BRANCHEMENT_STYLE;
                featureId = PART_LINE_ID;
                break;
              case 'AAD':
              case 'AAT':
                featureStyle = itvStylesFactory.NODE_AMONT;
                break;
              case 'AAF':
              case 'AAU':
                featureStyle = itvStylesFactory.NODE_AVAL;
                break;
              case 'CAAA':
                featureStyle = itvStylesFactory.NODE_ALONE;
                break;
            }
            if (featureStyle) {
              const geojsonFeature = features.find(feat => feat.id === header.value);
              if (geojsonFeature) {
                itvSoumFactory2.addGeometriesToSource([geojsonFeature], drawSource, featureStyle,
                    [featureName], [featureId]);
              }
            }
          }
        };

        /**
         * Zoom sur les objets de la portion d'ITV (linéaire + extrémités)
         * @param {ol.source.Vector} drawSource référence de la couche de dessin temporaire
         */
        const zoomOnPartFeatures = (drawSource) => {
          const extent = drawSource.getExtent();
          scope.map.getView().fit(extent, scope.map.getSize());
        };

        /**
         * Initialise le parcours du marqueur de position de la vidéo sur la carte
         */
        const setupAnimation = () => {
          const routeCoords = scope.partLine.getCoordinates();
          // créé le marqueur de la vidéo sur l'extrémité de départ de la canalisation inspectée (tronçon ou branchement)
          scope.geoMarker = new ol.Feature({
            type: 'geoMarker',
            geometry: new ol.geom.Point(routeCoords[0][0])
          });
          scope.geoMarker.setId(VIDEO_MARKER_ID);
          // le marqueur est un rond noir entouré de blanc
          scope.geoMarker.setStyle(itvStylesFactory.VIDEO_MARKER_STYLE);

          // boolean égal à true durant le déplacement du marqueur
          scope.animating = false;

          // Couche ol contenant le marqueur uniquement.
          // Le titre de la couche est important pour la récupération de la couche
          scope.vectorLayer = new ol.layer.Vector({
            source: new ol.source.Vector({
              features: [scope.geoMarker]
            })
          });
          scope.vectorLayer.set('id', ANIMATION_LAYER_ID);

          // place la couche du marqueur au-dessus des couches existantes
          scope.vectorLayer.setZIndex(gaJsUtils.getZIndexMax(scope.map) + 1);

          // boolean qui permet de créer l'animation
          scope.map.loadTilesWhileAnimating = true;

          // ajoute la couche du marqueur à la carte
          scope.map.addLayer(scope.vectorLayer);
        };

        /**
         * Déplace le pointeur de la barre de progression de la vidéo en fonction du point cliqué par l'utilisateur sur la carte
         */
        const skipVideoOnFeatureClick = () => {
          // calcule la distance entre le départ de la canalisation et le point cliqué
          const elapsedDistance = mapJsUtils.getDistanceFromLineStart(scope.coordsByClick,
              scope.partLine);

          if (elapsedDistance === 0) {
            const event = {target: {value: 0, dataset: {}}};
            skipAhead(event);

          } else if (elapsedDistance > 0) {

            // calcule le temps de la vidéo au niveau du point cliqué
            const moveToTime = video.duration * elapsedDistance
                / mapJsUtils.getMultiLineStringLength(scope.partLine, 0);
            const event = {target: {value: moveToTime, dataset: {}}};

            // déplace la lecture de la vidéo à la seconde correspondant à la localisation du point cliqué
            skipAhead(event);
          }
          scope.coordsByClick = null;
        }

        /**
         * Fonction exécutée par le listener du clic sur la carte pour déplacer le marqueur de la vidéo sur la portion d'ITV
         * @param {ol.MapBrowserEvent} event Evènement déclenché au clic sur la portion d'ITV correspondant à la vidéo.
         */
        const setupVideoNavigationByMarkerDisplacement = (event) => {
          // récupère les features cliquées à la position du curseur
          scope.map.forEachFeatureAtPixel(event.pixel, (feature, layer) => {
            if (layer.get('id') === PART_LAYER_ID && feature.getId(PART_LINE_ID)) {
              const featGeometry = feature.getGeometry();
              if (featGeometry.getType() === 'MultiLineString' || featGeometry.getType()
                  === 'LineString') {

                // si l'objet est la portion d'ITV inspectée, on assigne les coordonnées du point à la variable coordsByClick
                // on s'assure que les coordonnées soint celles d'un point de la portion avec getClosestPoint
                scope.coordsByClick = featGeometry.getClosestPoint(event.coordinate);

                if (!scope.animating) {
                  for (const feature of scope.vectorLayer.getSource().getFeatures()) {
                    if (feature.getId() !== VIDEO_MARKER_ID) {
                      scope.vectorLayer.getSource().removeFeature(feature);
                    }
                  }
                  // garde actif le déplacement du curseur et de la vidéo quand celle-ci est en pause
                  scope.geoMarker.getGeometry().setCoordinates(scope.coordsByClick);
                  scope.geoMarker.setStyle(itvStylesFactory.VIDEO_MARKER_STYLE);
                  skipVideoOnFeatureClick();
                }
              }
            }
          });
        };

        /**
         * Déplace le marqueur de la position de la vidéo sur la carte
         * @param {ol.render.Event} event évènement déclenché après la composition des couches.
         * Lorsque l'évènement est dispatché par la map, l'évènement n'a pas de propriété context définie.
         * Lorsque l'évènement est renvoyé par une couche, l'objet event aura une propriété context définie.
         * Seules les couches WebGL envoient actuellement cet évènement
         */
        const moveFeature = (event) => {
          const vectorContext = event.vectorContext;

          if (scope.animating) {
            const multiLineLength = mapJsUtils.getMultiLineStringLength(scope.partLine, 0);
            const elapsedDistance = (video.currentTime * multiLineLength) / video.duration;

            /*console.log('Move feature to : \n Time : ' + video.currentTime.toFixed(2)
                + ' s \n Distance : ' + elapsedDistance.toFixed(2) + ' m');*/

            if (elapsedDistance >= multiLineLength) {
              stopAnimation(true);
              return;
            }

            if (Array.isArray(scope.coordsByClick)) {
              scope.currentCoords = scope.coordsByClick.slice();
            } else {
              scope.currentCoords = mapJsUtils.getMultiLineStringCoordinatesAt(scope.partLine, 0,
                  elapsedDistance);
            }

            const currentPoint = new ol.geom.Point(scope.currentCoords);
            const feature = new ol.Feature(currentPoint);
            vectorContext.drawFeature(feature, itvStylesFactory.VIDEO_MARKER_STYLE);

            // scope.coordsByClick existe uniquement lorsque l'utilisateur a cliqué sur la ligne de la portion ITV
            if (Array.isArray(scope.coordsByClick)) {
              skipVideoOnFeatureClick()
            }

          }
          // tell OpenLayers to continue the postcompose animation
          scope.map.render();
        };

        /**
         * Démarre sur la carte le parcours du marqueur de position de la vidéo
         */
        const startAnimation = () => {
          if (scope.animating) {
            // stoppe l'animation
            stopAnimation(false);
          } else {
            scope.animating = true;

            // masque le marqueur de la vidéo
            scope.geoMarker.setStyle(new ol.style.Style({}));

            // lance l'animation
            scope.map.on('postcompose', moveFeature);
            scope.map.render();
          }
        };

        /**
         * Au clic sur le bouton pause,
         * suspend l'animation et
         * affiche le marqueur de la vidéo sur le dernier point du segment déjà parcouru
         */
        const pauseAnimation = () => {
          scope.animating = false;
          scope.isVideoPause = true;
          scope.geoMarker.getGeometry().setCoordinates(scope.currentCoords);
          scope.geoMarker.setStyle(itvStylesFactory.VIDEO_MARKER_STYLE);
        };

        /**
         * Au clic sur le bouton play après avoir mis la vidéo en pause,
         * masque le marqueur de la vidéo et reprend l'animation
         */
        const remainAnimation = () => {
          scope.geoMarker.setStyle(new ol.style.Style({}));
          scope.isVideoPause = false;
          scope.animating = true;
        };

        /**
         * Stoppe le parcours du curseur de la vidéo sur la carte
         * @param {boolean} ended est true si le curseur est arrivé au bout de la portion inspectée
         */
        const stopAnimation = (ended) => {
          scope.animating = false;

          const routeCoords = scope.partLine.getCoordinates();

          // si l'animation est annulée, le marqueur est renvoyé au début
          const coord = ended ? routeCoords[routeCoords.length - 1] : routeCoords[0];
          scope.geoMarker.getGeometry().setCoordinates(coord);

          //supprime le listener
          scope.map.un('postcompose', moveFeature);
        };

        /**
         * Au clic sur le bouton close ("croix") du lecteur vidéo:<ul><li>
         *   Stoppe l'animation</li><li>
         *   Vide la couche de l'animation</li><li>
         *   Supprime la couche de l'animation</li><li>
         *   Enlève le listener de l'animation</li><li>
         *   Supprime la couche des détails d'observation</li><li>
         *   Enlève le listener du clic sur la portion inspectée</li><li>
         *   Enlève le listener au survol de la portion inspectée</li><li>
         *   Vide la couche de dessin temporaire</li></ul>
         */
        const onExitViewer = () => {
          const mapLayers = scope.map.getLayers().getArray();

          // recherche la couche ol dans laquelle se trouve le marqueur de la vidéo
          const animationLayer = mapLayers.find(layer => layer.get('id') === ANIMATION_LAYER_ID);
          if (animationLayer) {

            // stoppe l'animation
            scope.animating = false;

            // vide la couche (supprime le marker de l'emplacement de la vidéo)
            animationLayer.getSource().clear();

            // supprime la couche
            scope.map.removeLayer(animationLayer);

            // enlève le listener de l'animation
            scope.map.un('postcompose', moveFeature);
          }

          // recherche la couche ol dans laquelle se trouve les détails d'observation du collecteur
          const detailLayer = mapLayers.find(layer => layer.get('id') === DETAILS_LAYER_ID);
          if (detailLayer) {
            // vide la couche des détails
            detailLayer.getSource().clear();

            // supprime la couche des détails
            scope.map.removeLayer(detailLayer);
          }

          // enlève le listener du clic sur la portion inspectée
          scope.map.un('click', setupVideoNavigationByMarkerDisplacement);

          // enlève le listener au survol de la portion inspectée
          scope.map.un('pointermove', showPointerOnPartHover);

          // vide la couche de dessin temporaire (supprime la mise en évidence des objets de la portion)
          clearPartSource();
        };

        /**
         * Fonction exécutée après appui de la touche "Escape" du clavier pour lancer la suppression de l'animation
         * @param {KeyboardEvent} event Evènement déclenché à l'appui d'une touche de clavier lorsque la popup du lecteur vidéo est active
         */
        const onEscape = (event) => {
          let isEscape;
          if ("key" in event) {
            isEscape = (event.key === "Escape" || event.key === "Esc");
          }
          if (isEscape) {
            onExitViewer();
          }
        };

        /**
         * Ajoute un listener sur le bouton close ("croix") pour supprimer l'animation proprement
         */
        const addExitEventListener = () => {
          const videoPopup = document.querySelector('.popupContainer.itv-visu-video');
          if (videoPopup) {

            // ajoute un listener sur la pression de la touche "Escape"
            document.addEventListener('keydown', onEscape, {
              // Cela déclenchera l’événement une fois et le désactivera par la suite
              once: true
            });

            // ajoute un listener sur le bouton close (.closebtn, "croix")
            const btnClose = videoPopup.querySelector('.closebtn');
            if (btnClose) {
              btnClose.addEventListener('click', onExitViewer);
            }
          }
        };

        /**
         * Affiche le curseur pointer "main" au survol de la canalisation inspectée correspondant à la vidéo
         * @param {ol.MapBrowserEvent} event Evènement déclenché au survol de la souris sur la carte
         */
        const showPointerOnPartHover = (event) => {
          const hit = scope.map.forEachFeatureAtPixel(event.pixel, (feature, layer) => {
            return feature.getId() === PART_LINE_ID;
          });
          if (hit) {
            scope.map.getTargetElement().style.cursor = 'pointer';
          } else {
            scope.map.getTargetElement().style.cursor = '';
          }
        };

        /**
         * Transforme une feature geojson en ol.Feature et affecte un style suivant la propriété type de chaque feature correspondant au type d'observation
         * @param {object} feature objet geojson du tableau de features présent dans le retour de {@link QueryFactory.data}
         * @return {ol.Feature} feature openlayers d'une observation (i.e anomalie) de portion ITV
         */
        const transformDetail = (feature) => {
          let style = itvStylesFactory.DETAILS_STYLE.default;
          if (feature.properties && feature.properties.hasOwnProperty('code_principal')
              && feature.properties.code_principal.length > 1) {
            switch (feature.properties.code_principal.trim().substring(0, 2)) {
              case 'BC':
              case 'DC':
                style = itvStylesFactory.DETAILS_STYLE.BC;
                break;
              case 'BA':
              case 'DA':
                style = itvStylesFactory.DETAILS_STYLE.BA;
                break;
              case 'BB':
              case 'DB':
                style = itvStylesFactory.DETAILS_STYLE.BB;
                break;
            }
          }
          const olGeometry = GEOJSON_FORMAT.readGeometry(feature.geometry);
          const olDetailFeature = new ol.Feature({name: '', geometry: olGeometry});
          olDetailFeature.setStyle(style);
          olDetailFeature.setId(feature.id);
          return olDetailFeature;
        };

        /**
         * Affiche les détails/observations (i.e anomalies) de la portion d'ITV inspectée:<ul><li>
         * Détermine l'uid du fti du composant des détails d'observation</li><li>
         * Elabore la clause where pour filtrer les observations de la portion courante</li><li>
         * Exécute l'appel API afin de récupérer les objets geojson des observations</li><li>
         * Exécute la conversion des features en ol.Feature et insère ces features dans une nouvelle couche openlayers ajoutée à la carte</li></ul>
         * Les termes "détail", "observation" et "anomalie" représentent le même objet et sont synonymes.
         * @param {object} part portion d'ITV inspectée correspondant à la vidéo courante
         */
        const drawPartDetailFeatures = (part) => {
          if (part.hasOwnProperty('partId') && Array.isArray(part.details) && part.details.length
              > 0) {
            const itvparts = part.partId;
            const details = part.details.map(detail => detail.id).join(',');
            const ftiName = details.split(',')[0].split('.')[0];

            // construit le critère where pour la requête api de sélection des observations (en fonction de l'identifiant de la portion)
            let where;
            if (scope.conf.arcgiscomponent
                && scope.conf.arcgiscomponent.EVT_Inspection_Part_Detail
                && scope.conf.arcgiscomponent.EVT_Inspection_Part_Detail.fti.type === 'esri') {
              where = 'id_part = ' + itvparts.split('.')[1];
            } else {
              where = 'id_part = \'' + itvparts + '\'';
            }

            const ftiuid = FeatureTypeFactory.getFeatureUidByName(ftiName);
            if (ftiuid) {

              // récupère les objets des observations directement
              QueryFactory.data(ftiuid, where, scope.map.getView().getProjection().getCode()).then(
                  res => {
                    if (res.data) {

                      // construit un objet ayant pour clé l'id du détail et pour valeur le timecode de l'observation/détail
                      extractDetailsTime(res.data.features);

                      // il y a beaucoup d'incohérences entre la position géographique calculée du détail et le timecode du détail
                      // Il y a aussi des détails sans timecode.
                      // Ainsi on publie les informations des observations en console pour permettre à l'admin/PO de déterminer la source des éventuels défauts des observations
                      logFeatures(res.data.features);

                      // transforme chaque feature en ol.Feature et affecte un style suivant le type d'observation
                      const features = res.data.features.map(feat => transformDetail(feat));

                      // créé une layer openlayers pour contenir les détails d'observation de la portion d'ITV
                      const source = new ol.source.Vector();
                      source.addFeatures(features);
                      scope.detailsLayer = new ol.layer.Vector({
                        source: source
                      });
                      scope.detailsLayer.set('id', DETAILS_LAYER_ID);

                      // place la couche du marqueur au-dessus des couches existantes
                      scope.detailsLayer.setZIndex(gaJsUtils.getZIndexMax(scope.map) + 1);
                      scope.map.addLayer(scope.detailsLayer);
                    }
                  }
              )
            }
          }
        };

        /**
         * Publie en console les informations des observations pour permettre à l'admin/PO de déterminer la source des éventuels défauts des observations:<ul><li>
         * Il y a des détails sans timecode.</li><li>
         * Il y a des détails sans géométrie</li><li>
         * Il y a une incohérence entre la position géographique calculée du détail et le timecode du détail</li></ul>
         * @param details tableau des détails de la portion inspectée (hors regards début/fin) retourné par {@link QueryFactory.data}
         */
        const logFeatures = (details) => {
          const logFeatures = details.map(feat => {
            return {
              id: feat.id,
              code: feat.properties.code_principal,
              timecode: feat.properties.ref_video,
              hasGeometry: feat.geometry !== undefined
            }
          });
          for (const feature of logFeatures) {
            let color;
            switch (feature.code.trim().substring(0, 2)) {
              case 'BC':
              case 'DC':
                color = '#5A89AD';
                break;
              case 'BA':
              case 'DA':
                color = '#EAF903';
                break;
              case 'BB':
              case 'DB':
                color = '#D87047';
                break;
              default:
                color = '#62E515';
            }
            console.log('%c ' + JSON.stringify(feature), 'color : ' + color);
          }
        };

        /**
         * Construit un objet où la clé est l'id de l'observation et la valeur est le timecode de l'observation dans la vidéo (en secondes)
         * @param {object[]} details observations de la portion d'ITV
         */
        const extractDetailsTime = (details) => {
          if (Array.isArray(details) && details.length > 0) {
            scope.detailsTimecodes = {};
            for (const detail of details) {
              const properties = detail.properties;
              if (properties.hasOwnProperty('ref_video') && typeof properties.ref_video === 'string'
                  && properties.ref_video.length > 0) {
                const detailTimeParts = properties.ref_video.split(':');
                if (detailTimeParts.length === 3) {
                  const hours = Number.isNaN(detailTimeParts[0]) ? 0 : Number(detailTimeParts[0]);
                  const mins = Number.isNaN(detailTimeParts[1]) ? 0 : Number(detailTimeParts[1]);
                  const secs = Number.isNaN(detailTimeParts[2]) ? 0 : Number(detailTimeParts[2]);
                  scope.detailsTimecodes[detail.id] = hours * 3600 + mins * 60 + secs;
                }
              }
            }
          }
        };

        /**
         * Met en surbrillance les observations/détails lors du passage de la vidéo
         */
        const highlightDetails = () => {
          for (const [detailId, detailTimecode] of Object.entries(scope.detailsTimecodes)) {

            // le détail (i.e l'observation) est mis en évidence durant 3 secondes
            const startHighlightTime = detailTimecode - 1;
            const endHighlightTime = detailTimecode + 1;

            if (video.currentTime >= startHighlightTime && video.currentTime <= endHighlightTime) {

              // la vidéo est actuellement au niveau de l'observation
              // ajoute la feature ol au jeu d'objets sélectionnés
              const olDetailFeature = scope.detailsLayer.getSource().getFeatureById(detailId);
              const geojsonDetailFeature = GEOJSON_FORMAT.writeFeatureObject(olDetailFeature);
              geojsonDetailFeature.id = detailId;
              SelectManager.addFeaturesFromGeojson({features: [geojsonDetailFeature]});

            } else if (SelectManager.hasFeature(detailId)) {
              // si la vidéo n'est pas au niveau de l'observation et si le jeu de sélection contient la feature
              SelectManager.removefeatures(detailId);
            }
          }
        };
        // Exécute highlightDetails à chaque seconde de la vidéo
        video.addEventListener('timeupdate', highlightDetails);

        /**
         * Créé une layer openlayers pour contenir le tracé de la portion d'ITV et de ses extrémités utilisés pour la vidéo
         * @return {ol.source.Vector} source de la couche créée
         */
        const createPartLayer = () => {
          const partLayer = new ol.layer.Vector();
          partLayer.set('id', PART_LAYER_ID);
          const partSource = new ol.source.Vector();
          partLayer.setSource(partSource);
          partLayer.setZIndex(gaJsUtils.getZIndexMax(scope.map) + 1);
          scope.map.addLayer(partLayer);
          return partSource;
        };

        /**
         * Vide la layer openlayers contenant les tracés de la portion d'ITV et de ses extrémités utilisés pour la vidéo
         */
        const clearPartSource = () => {
          scope.map.getLayers().forEach(layer => {
            if (layer.get('id') === PART_LAYER_ID) {
              layer.getSource().clear();
            }
          });
        };

        /*****************
         * INITIALISATION
         ****************/

        const srid = scope.map.getView().getProjection().getCode();
        itvVisuFactory.getPartGeometries(scope.part, srid).then(
            res => {
              if (Array.isArray(res)) {
                const partFeatures = res.map(result => {
                  if (Array.isArray(result.data.features) && result.data.features.length > 0) {
                    return result.data.features[0];
                  }
                });
                const manholeOrConnexionPipeOrPipeHeaders = ['AAA', '@01', 'CAAA'];
                const mainHeader = scope.part.headers.find(
                    header => manholeOrConnexionPipeOrPipeHeaders.includes(header._code));
                if (mainHeader) {
                  const mainFeatId = mainHeader.value;
                  // mainFeature est l'objet geojson constituant la portion inspectée (tronçon, branchement ou regard isolé) sans les objets des extrémités.
                  const mainFeature = partFeatures.find(feat => feat.id === mainFeatId);

                  // Lorsque la portion inspectée est un regard isolée,
                  // on ne créé pas le marqueur de la position de la vidéo ni les objets temporaires sur la carte
                  if (mainFeature && mainHeader._code !== 'CAAA') {

                    // transforme la géométrie du linéaire-support en MultiLineString
                    let mainCoordinates = mainFeature.geometry.coordinates;
                    if (mainFeature.geometry.type === 'LineString') {
                      mainCoordinates = [mainFeature.geometry.coordinates];
                    }
                    scope.partLine = new ol.geom.MultiLineString(mainCoordinates);

                    // créé la layer de dessin
                    const drawSource = createPartLayer();

                    // met en surbrillance les objets de la portion inspectée (canalisation en noir et extrémités vert/rouge)
                    drawPartFeatures(scope.part.headers, partFeatures, drawSource);

                    // affiche les détails d'observation de la portion inspectée
                    drawPartDetailFeatures(scope.part);

                    // zoom sur les objets de la portion inspectée
                    zoomOnPartFeatures(drawSource);

                    // créé le marker de la position de la vidéo et la layer pour contenir celui-ci
                    setupAnimation();

                    // créé l'observateur d'évènement qui va supprimer l'animation à la fermeture de la popup
                    addExitEventListener();

                    // déplace le marker de la vidéo au clic sur la canalisation inspectée
                    scope.map.on('click', setupVideoNavigationByMarkerDisplacement);

                    // affiche le cursoir "pointer" au survol de la canalisation inspectée
                    scope.map.on('pointermove', showPointerOnPartHover);
                  }
                }
              }
            }
        );
      },
    };
  };

  itvvisualisationvideo.$inject = ['itvVisuFactory', 'itvSoumFactory2', 'itvStylesFactory',
    'mapJsUtils', 'gaJsUtils', 'QueryFactory', 'FeatureTypeFactory', 'SelectManager'];
  return itvvisualisationvideo;
});
