/**
 * Gestion compléte de la fonction "rechercheAmontAval":
 *
 * -- Description de la fonction pour la configuration.
 *
 * -- Traitement d'appel du service de recherche des éléments
 *
 * -- Traitement de dessin des éléments sur la carte
 *    (dans la couche highlightLayer)
 *
 * -- Les paramètres en sortie sont :
 *    -- une variable qui contiendra le résultat.
 *       Elle est construite avec 2 properties :
 *       -- riskVertexIds :
 *                dont la valeur est une chaine de caractères
 *                avec les ids des nœuds séparés par une virgule
 *       -- riskEdgeIds :
 *                dont la valeur est une chaine de caractères avec les ids
 *                des linéaires séparés par une virgule
 *
 * Auteur : @EDG
 * Date: 27/08/2021
 */
'use strict';
define(function () {
  const networkfunctions = function () {
    this.$get = function (NetworkFactory, $q, gclayers, $filter,
      gaDomUtils) {

      const init = () => {
        // -- Le get si-dessous enregistre la liste des réseaux dans la factory.
        NetworkFactory.get(false);
      };

      init();


      /**
       * Crée une feature OpenLayers et l'ajoute à la couche des objets
       * mis en évidence.
       *
       * @param {*} geom : Géométrie pour le feature à créer
       * @param faetureName nom à inclure dans une propriété 'name' à créer dans la feature ol.
       * @param {*} geomType : 'line ou 'poin' pour déterminer
       *                       le style à appliquer
       */
      const showFeature = (geom, faetureName, geomType) => {
        let style;
        //-- Création de la feature open layers du tronçon, du point de départ
        //-- ou du point d'arrivée du parcours.
        let feature = new ol.Feature({
          geometry: geom
        });
        if (geomType === 'originalLine') {
          //-- Style de la ligne: trait orange épais
          style = new ol.style.Style({
            stroke: new ol.style.Stroke({
              color: 'rgba(26, 140, 255,0.85)',
              width: 8,
            })
          });
        } else if (geomType === 'line') {
          //-- Style de la ligne: trait orange épais
          style = new ol.style.Style({
            stroke: new ol.style.Stroke({
              color: 'rgba(255,182,43,0.85)',
              width: 8,
            })
          });
        }
        else {
          //-- Styles des ponctuels : rond orange entouré de rouge
          style = new ol.style.Style({
            image: new ol.style.Circle({
              radius: 7,
              fill: new ol.style.Fill({color: 'rgba(255,182,43,0.85'}),
              stroke: new ol.style.Stroke({
                color: [255,0,0], width: 2
              })
            })
          });
        }
        feature.setStyle(style);
        feature.set('name', faetureName);

        // KIS-2960: ajoute l'objet à la couche de surbrillance sans supprimer le style d'origine
        gclayers.addhighLightFeature(feature, false);
      };


      /**
       * Transforme la géométrie reçu dans le JSON en géométrie OpenLayers.
       * On s'azttend à avoir une multiligne ou un point.
       *
       * @param {*} geom : Géométrie contenu dans le JSON pour un objet.
       * @returns : La géométrie OpenLayers pour dessin sur la carte.
       */
      const getOlGeomFrom = (geom) => {
        let coords;
        if (geom.startsWith('MULTILINESTRING')) {
          //-- Cas des tronçons: lecture des coordonnées depuis le JSON
          //-- vers un tableau de tableau de coordonnées.
          coords = geom.replace('MULTILINESTRING ((', '').replace('))', '');
          coords = coords.split(',');
          for (let iCoord = 0; iCoord < coords.length; iCoord++) {
            coords[iCoord] = coords[iCoord].trim().split(' ');
            for (let ind = 0; ind < coords[iCoord].length; ind++) {
              coords[iCoord][ind] = parseFloat(coords[iCoord][ind]);
            }
          }
          //-- Création de la multi ligne OpenLayers.
          return new ol.geom.MultiLineString([coords]);
        }
        else if (geom.startsWith('POINT')) {
          //-- Cas des noeuds : récupération des coordonnées du point.
          coords = geom.replace('POINT (', '').replace(')', '');
          coords = coords.split(' ');
          for (let ind = 0; ind < coords.length; ind++) {
            coords[ind] = parseFloat(coords[ind]);
          }
          //-- Création du point OpenLayers.
          return new ol.geom.Point(coords);
        }
      };


      /**
       * Récupére ml'identifiant de l'arc ou du noeud depuis l'élément retourné
       * par le service "rechercheAmontAval".
       *
       * @param {*} elt : arc ou noeud tel que retourné par le service
       *                  "rechercheAmontAval".
       * @returns identifiant esri selon configuration , sinon
       *          partie numérique de l'identifiant geotools.
       */
      const getIdOfElement = (elt) => {
        if (elt.esriIdFieldValue) {
          return elt.esriIdFieldValue;
        }
        else {
          let ind = elt.id.indexOf('.');
          if (ind !== -1) {
            return elt.id.substring(ind + 1);
          }
          else {
            return elt.id;
          }
        }
      };


      /**
       * Dessin pour mise en évidence sur la carte des éléments
       * parcourus par la recherche amon aval.
       * Stockage des identifiants des éléments pour récupération ultérieures
       * (rapport dans lequel ils seront utilisés).
       *
       * @param {*} elements : objets ayant 2 propriété edges et nodes
       * @param {*} res : zone memoire contenant les variables du formulaire
       * @param {*} originalPipes : tronçons de l'intervention (geojson)
       */
      const highlightNetworkElements = (elements, res, originalPipes) => {
        const { edges, nodes } = elements;
        gclayers.clearhighLightFeatures();
        if (!res.miscResult) {
          res.miscResult = {};
        }
        res.miscResult.riskVertexIds = '';
        res.miscResult.riskEdgeIds = '';

        //-- Dessin des tronçons.
        for (let iLine = 0; iLine < edges.length; iLine++) {
          showFeature(getOlGeomFrom(edges[iLine].geometry), edges[iLine].id, 'line');
          if (res.miscResult.riskEdgeIds !== '') {
            res.miscResult.riskEdgeIds +=  ',';
          }
          res.miscResult.riskEdgeIds += getIdOfElement(edges[iLine]);
        }
        //-- Dessin des tronçons de l'intervention.
        for (let iPipe = 0; iPipe < originalPipes.length; iPipe++) {
          const pipe = originalPipes[iPipe];
          showFeature(new ol.geom.MultiLineString(pipe.geometry.coordinates), pipe.id, 'originalLine' );
        }

        //-- Dessin des noeuds.
        for (let iPt = 0; iPt < nodes.length; iPt++) {
          showFeature(getOlGeomFrom(nodes[iPt].geometry), nodes[iPt].id);
          if (res.miscResult.riskVertexIds !== '') {
            res.miscResult.riskVertexIds +=  ',';
          }
          res.miscResult.riskVertexIds += getIdOfElement(nodes[iPt]);
        }
      };


      /**
       * Retourne les points d'entrée de ce provider:
       * -1- getParametersDesc: Description des paramétres
       *               pour configuration de la "function".
       * -2- searchUpDownStream: fonction appelé depuis "gc_formfunction"
       *               quand cette foncyionnalité est sollicitée
       *               depuis un formulaire.
       */
      return {
        /**
         * la description retournée est exploitée dans functionsBrowser.html.
         *
         * @returns : description des paramétres de la fonction
         */
        getParametersDesc: () => {

          return [
            [0, 'select',
              NetworkFactory.resources.networks.map(
                (network) => {
                  return { label: network.name, id: network.uid };
                }
              )
            ],
            [1, 'fti.attribute'],
            [2, 'fti.attribute'],
            [3, 'variable'],
          ];
        },

        /**
         * Recherche de parcours réseau depuis chaque extrémité
         * du sous réseau décrit par la liste de linéaires (paramétre pipes).
         * Les recherches de parcours se font depuis les extrémités connectées
         * à un autre tronçon qu'un de  ceux du sous réseau (paramétre pipes).
         * Une fois le résultat obtenu, les éléments sont mis en évidence
         * sur la carte dans la couche "highlightLayers".
         *
         * @param {*} formRes : variables du formulaire (ressources du
         *                      formulaire)
         * @param {*} network : réseau à utiliser pour le parcours
         * @param {*} distAmont : distance minimale à parcourir à partir
         *                        des noeuds qui sont des noeuds amont.
         * @param {*} distAval : distance minimale à parcourir à partir
         *                        des noeuds qui sont des noeuds avals.
         * @param {*} pipes : liste des tronçons du sous réseau
         *                    (tronçon associés à l'intervention).
         * @param {*} inSrId : Référence spatiale de la carte.
         *
         * @returns : promesse qui est résolue aprés le dessin
         *            des éléments trouvés.
         */
        searchUpDownStream: (formRes, network, distAmontProp, distAvalProp,
          pipes, inSrId) => {
          const defer = $q.defer();
          let distAm = formRes.current.properties[distAmontProp];
          let distAv = formRes.current.properties[distAvalProp];
          if (!distAm) {
            distAm = 20;
          }
          if (!distAv) {
            distAv = 20;
          }
          if (!formRes[pipes] || !formRes[pipes].features
            || formRes[pipes].features.length === 0) {
            require('toastr').warning($filter('translate')('model.featuretypes.actions.no_line_for_rechercheAmontAval'));
            defer.reject();
          }
          else {
            gaDomUtils.showGlobalLoader('searchUpDownStream');
            NetworkFactory.rechercheAmontAval(network.label, formRes[pipes],
              distAm, distAv, inSrId).then(
              (res) => {
                if (res.data.etat === 'erreur') {
                  let mess = res.data.errorList.map(
                    (err) => { return err.message_kis; }
                  );
                  require('toastr').error(mess.join());
                  defer.reject();
                }
                else {
                  const elements = JSON.parse(res.data.objValeur);
                  if (elements.edges.length !== 0) {
                    highlightNetworkElements(elements, formRes,
                      formRes[pipes].features);
                  }
                  else {
                    swal(
                      {
                        title: $filter('translate')('functions_browser.functions.deactivateFilter.swalTitle'),
                        type: 'error',
                        text: $filter('translate')('functions_browser.functions.deactivateFilter.resultIsEmpty')
                      }
                    );
                  }
                  defer.resolve();
                }
                gaDomUtils.hideGlobalLoader('searchUpDownStream');
              },
              () => {
                gaDomUtils.hideGlobalLoader('searchUpDownStream');
                defer.reject();
              }
            );
          }
          return defer.promise;
        }
      };
    };
    this.$get.$inject = ['NetworkFactory', '$q', 'gclayers',
      '$filter', 'gaDomUtils'];
  };
  return networkfunctions;
});
