'use strict';
define(function() {
  var pv2LayerServices = function(
    $location,
    $filter,
    BaseMapFactory,
    PortalsFactory,
    DataStoreFactory
  ) {
    var scope;

    // Transform an ol.Color to an hexadecimal string
    var toHexa = function(olColor) {
      var hex = '#';
      for (var i = 0; i < 3; i++) {
        var part = olColor[i].toString(16);
        if (part.length === 1 && parseInt(part) < 10) {
          hex += '0';
        }
        hex += part;
      }
      return hex;
    };

    // Transform a ol.style.Style to a print literal object
    var transformToPrintLiteral = function(feature, style) {
      /**
       * ol.style.Style properties:
       *
       *  fill: ol.style.Fill :
       *    fill: String
       *  image: ol.style.Image:
       *    anchor: array[2]
       *    rotation
       *    size: array[2]
       *    src: String
       *  stroke: ol.style.Stroke:
       *    color: String
       *    lineCap
       *    lineDash
       *    lineJoin
       *    miterLimit
       *    width: Number
       *  text
       *  zIndex
       */

      /**
       * Print server properties:
       *
       * fillColor
       * fillOpacity
       * strokeColor
       * strokeOpacity
       * strokeWidth
       * strokeLinecap
       * strokeLinejoin
       * strokeDashstyle
       * pointRadius
       * label
       * fontFamily
       * fontSize
       * fontWeight
       * fontColor
       * labelAlign
       * labelOutlineColor
       * labelOutlineWidth
       * graphicHeight
       * graphicOpacity
       * graphicWidth
       * graphicXOffset
       * graphicYOffset
       * zIndex
       */

      var literal = {
        zIndex: style.getZIndex(),
      };
      var fill = style.getFill();
      var stroke = style.getStroke();
      var textStyle = style.getText();
      var imageStyle = style.getImage();
      var graphicName = style.graphicName;




      if (imageStyle) {
        var size = imageStyle.getSize();
        var anchor = imageStyle.getAnchor();
        var scale = imageStyle.getScale();
        if (feature.getProperties().styleGraphic) {
          literal.rotation = feature.getProperties().styleGraphic.Rotation;
        } else {
          literal.rotation = imageStyle.getRotation();
        }
        if (size) {
          literal.graphicWidth = size[0];
          literal.graphicHeight = size[1];
        }
        if (anchor) {
          //-- original --  literal.graphicXOffset = -anchor[0] * scale;
          //-- original --  literal.graphicYOffset = -anchor[1] * scale;
          literal.graphicXOffset = -anchor[0];
          literal.graphicYOffset = -anchor[1];
        }
        literal.graphicScale = scale;
        if (imageStyle instanceof ol.style.Icon) {
          literal.externalGraphic = imageStyle.getSrc();
          literal.fillOpacity = 1;
        } else {
          // ol.style.Circle
          fill = imageStyle.getFill();
          stroke = imageStyle.getStroke();
          literal.pointRadius = imageStyle.getRadius();
        }
      }

      if (graphicName) {
        literal.graphicName = graphicName;
      }
      if (fill instanceof ol.style.FillPattern) {
        var hachure = feature.getProperties().styleGraphic.hashtemplate;
        literal.fillPatternColor = feature.getProperties().styleGraphic.Fill.CssParameter[0].__text;
        literal.fillColor = feature.getProperties().styleGraphic.Fill.CssParameter[0].__text;
        literal.strokeColor = feature.getProperties().styleGraphic.Stroke.CssParameter[0].__text;
        literal.strokeWidth = 1;
        literal.fillOpacity = 1;
        literal.stylePattern = {};
        literal.stylePattern.hashColor = feature.getProperties().styleGraphic.Fill.CssParameter[1].__text;
        literal.stylePattern.strokeColor = feature.getProperties().styleGraphic.Stroke.CssParameter[0].__text;
        /* var fillHash= fill.patterns[hachure]["fill"];
              if (fillHash ){
                if (fillHash == true){
                    fill.patterns[hachure]["fill"]= 1;
                }else if(fill == false){
                    fill.patterns[hachure]["fill"]= 0;
                }
              }*/
        literal.stylePattern.fillPattern = fill.patterns[hachure];
      }
      //if (fill && fill.getColor() instanceof String) {
      if (fill && !(fill instanceof ol.style.FillPattern)) {
        let color = ol.color.asArray(fill.getColor());
        literal.fillColor = toHexa(color);
        literal.fillOpacity = color[3];
      } else if (!literal.fillOpacity) {
        literal.fillOpacity = 0; // No fill
      }

      if (stroke) {
        let color = ol.color.asArray(stroke.getColor());
        literal.strokeWidth = stroke.getWidth();
        literal.strokeColor = toHexa(color);
        literal.strokeOpacity = color[3];
        literal.strokeLinecap = stroke.getLineCap() || 'round';
        literal.strokeLinejoin = stroke.getLineJoin() || 'round';

        let lineDash = stroke.getLineDash();
        if (lineDash) {
          if (lineDash[0]/literal.strokeWidth==0.25 && lineDash[1]/literal.strokeWidth==4)
            //-- Open layers does not have dot line style, but MapFishPrint does.
            //-- So if the dash line style matches the dot line style setting
            //-- actually this is a dot line style.
            literal.strokeDashstyle = 'dot';
          else
            literal.strokeDashstyle = 'dash';
          literal.strokeDash = lineDash;
        }
        // TO FIX: Not managed by the print server
        // literal.strokeMiterlimit = stroke.getMiterLimit();
      } else {
        literal.strokeOpacity = 0; // No Stroke
      }

      if (textStyle) {
        var fillColor = ol.color.asArray(textStyle.getFill().getColor());
        var textStroke = textStyle.getStroke();
        var strokeColor;
        if (textStroke != undefined)
          strokeColor = ol.color.asArray(textStyle.getStroke().getColor());
        var rotation;
        if (
          feature.getProperties().styleGraphic &&
          feature.getProperties().styleGraphic.Rotation
        ) {
          rotation = feature.getProperties().styleGraphic.Rotation;
        } else {
          rotation = (textStyle.getRotation() * 180) / Math.PI;
        }

        var fontValues;
        if (textStyle.getFont()) {
          fontValues = textStyle.getFont().split(' ');
        }
        var align, baseline, mfplacement;
        if (fontValues.length != 3) {
          if (!fontValues[0].match(/normal|bold|italic/gi))
            fontValues.splice(0, 0, 'normal');
        }
        literal.fontColor = toHexa(fillColor);
        // Fonts managed by print server: COURIER, HELVETICA, TIMES_ROMAN
        literal.fontFamily = fontValues[2].toUpperCase();
        literal.fontSize = parseInt(fontValues[1]);
        literal.fontWeight = fontValues[0];
        literal.label = textStyle.getText();
        align = textStyle.getTextAlign();
        baseline = textStyle.getTextBaseline();
        if (align != undefined) mfplacement = align.substr(0, 1);
        else mfplacement = 'c';
        if (baseline != undefined) mfplacement += baseline.substr(0, 1);
        else mfplacement += 'm';
        if (!mfplacement.match(new RegExp('[rcl][tmb]'))) mfplacement = 'cm';
        literal.labelAlign = mfplacement;
        if (strokeColor) {
          literal.labelOutlineColor = toHexa(strokeColor);
          literal.labelOutlineWidth = textStyle.getStroke().getWidth();
        }
        literal.labelOutlineOpacity = 1.0;
        //literal.labelRotation = textStyle.getRotation() * 180 / Math.PI;
        literal.labelRotation = rotation;
        if (textStyle.getOffsetX() != 0)
          literal.labelXOffset = '' + textStyle.getOffsetX();
        if (textStyle.getOffsetY() != 0)
          literal.labelYOffset = '' + textStyle.getOffsetY();
      }

      return literal;
    };

    // Create a ol.geom.Polygon from an ol.geom.Circle, comes from OL2
    // https://github.com/openlayers/openlayers/blob/master/lib/OpenLayers/Geometry/Polygon.js#L240
    var circleToPolygon = function(circle, sides, rotation) {
      var origin = circle.getCenter();
      var radius = circle.getRadius();
      sides = sides || 40;
      var angle = Math.PI * (1 / sides - 1 / 2);
      if (rotation) {
        angle += (rotation / 180) * Math.PI;
      }
      var points = [];
      for (var i = 0; i < sides; ++i) {
        var rotatedAngle = angle + (i * 2 * Math.PI) / sides;
        var x = origin[0] + radius * Math.cos(rotatedAngle);
        var y = origin[1] + radius * Math.sin(rotatedAngle);
        points.push([x, y]);
      }
      points.push(points[0]); // Close the polygon
      return new ol.geom.Polygon([points]);
    };

    function getLocation(href) {
      var match = href.match(
        /^(https?\:)\/\/(([^:\/?#]*)(?:\:([0-9]+))?)([\/]{0,1}[^?#]*)(\?[^#]*|)(#.*|)$/
      );
      return (
        match && {
          protocol: match[1],
          host: match[2],
          hostname: match[3],
          port: match[4],
          pathname: match[5],
          search: match[6].replace('?', '').split('&'),
          hash: match[7],
        }
      );
    }

    function getDefaultWmtsInfo(layer, params) {
      var ret = {};
      //-- La description de notre "matrice de tuile"
      //-- n'a pas été trouvée.
      //-- On récupére les coordonnées origine de la source et
      //-- on prend 256 comme taille par défaut
      //-- des tuiles en hauteur et largeur.
      var source = layer.getSource();
      var projectedTo = ol.extent.getTopLeft(
        source.getProjection().getExtent()
      );
      if (!params.projImpEqualsProjDisplay)
        projectedTo = proj4(
          scope.olProjection,
          scope.secondProjection,
          projectedTo
        );
      ret.tileDim = [];
      ret.tileDim.push(256);
      ret.tileDim.push(256);
      ret.origin = projectedTo;

      return ret;
    }

    /**
     *   Si le système de projection du WMTS n'est pas celui
     * d'impression, l'origine doit être projetée et ses soordonées
     * exprimées en système de projection de l'impression.
     *
     * @param {[[type]]} origin [[Description]]
     * @param {[[type]]} layer [[Description]]
     * @return {[[type]]} [[Description]]
     */
    function checkWmtsOrigin(origin, layer) {
      var iBkg, bkgs, wmtsProj;

      bkgs = BaseMapFactory.resources.basemaps;
      for (iBkg = 0; iBkg < bkgs.length; iBkg++)
        if (bkgs[iBkg].name == layer.name) {
          wmtsProj = bkgs[iBkg].projectiondesc;
          break;
        }
      if (scope.projImpEqualsProjDisplay(wmtsProj)) return origin;

      return proj4(wmtsProj, scope.secondProjection, origin);
    }

    /**
     *    Récupère les informations tileOrigin, TileWidth, et
     * TileHeight des capabilities du WMTS.
     * Sans ces valeurs l'impression ne marchera pas avec le WMTS.
     *
     * @param {[[type]]} layer [[Description]]
     * @param {[[type]]} proj1 [[Description]]
     * @param {[[type]]} printProj [[Description]]
     * @param {[[type]]} printScale [[Description]]
     * @return {[[type]]} [[Description]]
     */
    function getWmtsInfo(layer, params) {
      var content,
        iLyr,
        iScale,
        tm,
        maxScale = 0;
      var ret = {},
        theScale,
        printScale;

      ret.tileDim = [];

      content = layer.getProperties().capabilities.Contents;

      //-- Recherche de la partie des capabilities
      //-- qui concerne notre "matrice de tuile".
      var matrixSet = layer.getSource().getMatrixSet();
      for (iLyr = 0; iLyr < content.TileMatrixSet.length; iLyr++) {
        if (content.TileMatrixSet[iLyr].Identifier == matrixSet) break;
      }

      if (iLyr == content.Layer.length)
        return getDefaultWmtsInfo(layer, params);

      //-- Recherche de la desctiption du tuilage
      //-- pour le niveau d'échelle qui nous concerne.
      printScale = parseFloat(params.printScale.value);
      tm = content.TileMatrixSet[iLyr].TileMatrix;
      for (iScale = 0; iScale < tm.length; iScale++) {
        if (
          printScale > tm[iScale].ScaleDenominator &&
          tm[iScale].ScaleDenominator > maxScale
        ) {
          maxScale = tm[iScale].ScaleDenominator;
          theScale = tm[iScale];
        }
      }

      ret.tileDim.push(theScale.TileWidth);
      ret.tileDim.push(theScale.TileHeight);
      ret.origin = checkWmtsOrigin(theScale.TopLeftCorner, layer);

      return ret;
    }

    // Encode ol.Layer to a basic js object
    var encodeLayer = function(layer, proj, scope, osmUrl, extentp) {
      var encLayer, encLegend;
      var resolution = scope.map.getView().getResolution();

      if (!(layer instanceof ol.layer.Group) && layer.getSource != undefined) {
        var src = layer.getSource();
        var layerConfig = /*gaLayers.getLayer(layer.bodId) ||*/ {};
        var minResolution = layerConfig.minResolution || 0;
        var maxResolution = layerConfig.maxResolution || Infinity;

        if (resolution <= maxResolution && resolution >= minResolution) {
          if (src instanceof ol.source.WMTS) {
            var params = {};
            params.projImpEqualsProjDisplay = scope.projImpEqualsProjDisplay();
            params.printScale = scope.scale;
            encLayer
              = getEncodedLayer(scope, 'WMTS', layer, layerConfig, params);
            if (encLayer != undefined)
              encLayer.customParams.CRS = scope.printConfig.srid;
          }
          else if (src instanceof ol.source.ImageWMS ||
            src instanceof ol.source.TileWMS ) {
            encLayer = getEncodedLayer(scope, 'WMS', layer, layerConfig, {
              osmUrl: osmUrl,
            });
            if (encLayer != undefined)
              encLayer.customParams.CRS = scope.printConfig.srid;
          }
          else if (src instanceof ol.source.OSM) {
            encLayer = getEncodedLayer(scope, 'OSM', layer, layerConfig, {
              osmUrl: osmUrl,
            });
            if (encLayer != undefined) {
              if (encLayer.customParams == undefined) {
                encLayer.customParams = {};
              }
              encLayer.customParams.CRS = scope.printConfig.srid;
            }
          }
          else if ( src instanceof ol.source.Vector ||
            src instanceof ol.source.ImageVector ) {
            if (src instanceof ol.source.ImageVector) {
              src = src.getSource();
            }

            let features = [];
            let srcFeatures = src.getFeatures();
            for (let ind=0;ind<srcFeatures.length;ind++) {
              if (ol.extent.intersects(extentp,srcFeatures[ind].getGeometry().getExtent()))
                features.push(srcFeatures[ind]);
            }

            if (features.length > 0) {
              encLayer = getEncodedLayer(scope, 'Vector', layer, features);
            }
          }
        }
      }

      if (scope.options.legend /*&& layerConfig.hasLegend*/) {
        if (
          layer.name.toUpperCase() != 'OSM' &&
          layer.type != 'OSM' &&
          layer.type != 'GEOWEBCACHE' &&
          layer.type != 'WMSBACK'
        ) {
          var l = '';
          l =
            $location.protocol() +
            '://' +
            $location.host() +
            ':' +
            $location.port() +
            '/services/' +
            PortalsFactory.getPortalId() +
            '/geoserver/' +
            '/wms?service=WMS&REQUEST=GetLegendGraphic' +
            '&VERSION=1.0.0&FORMAT=image/png&WIDTH=20&' +
            'HEIGHT=20&LAYER=' +
            layer.name +
            '&token=' +
            localStorage.getItem('auth_token');
          if (!angular.isUndefined(layer.style)) {
            l += '&STYLE=' + layer.style;
          }
          encLegend = {
            classes: [
              {
                icons: [l],
                name: '',
                iconBeforeName: true,
              },
            ],
            name:
              $filter('translate')('features.' + layer.name + '.alias').indexOf(
                'features.'
              ) === -1
                ? $filter('translate')('features.' + layer.name + '.alias')
                : layer.label,
          };
        }

      }

      return { layer: encLayer, legend: encLegend };
    };


    const getEpsgNum = () => {
      if (scope.printConfig && scope.printConfig.srid) {
        return parseInt(
          scope.printConfig.srid.toLowerCase().replace('epsg:', '')
        );
      }
      else {
        return scope.map.getView().getProjection().getCode();
      }
    };


    /**
     * Préparation des couches arcgis pour le service d'impression.
     * Le terme utilisé par ESRI pour les layers que l'on passe
     * à son service d'impression est "dynamicLayers",
     * d'où le nom de cette fonction.
     *
     * @param {*} arcgisLayers : liste des layers ArcGIS à préparer
     * @returns : paramétre liste de couche ArcGIS pour le service d'impression.
     */
    const getArcGisLayerDynLayers = (arcgisLayers) => {
      const dynamicLayers = [];

      for (let iLyr = 0; iLyr < arcgisLayers.length; iLyr++) {
        const layerProps = arcgisLayers[iLyr].getProperties();
        const fti = layerProps.fti;
        const dynamicLayerEntry = {
          id: fti.ogcId,
          source: {
            type: 'mapLayer',
            mapLayerId: fti.ogcId,
          },
        };

        //-- Filtre sur couche
        const filter = layerProps.cql_filter;
        if (filter && filter.length != 0 && filter != '1=1') {
          dynamicLayerEntry.definitionExpression = filter;
        }

        //-- Opacité de la couche
        const opacity = layerProps.opacity;
        if (opacity != null) {
          dynamicLayerEntry.drawingInfo = {
            transparency: opacity == 1 ? 0 : 100 - opacity * 100,
          };
        }

        const layerStyle = layerProps.style;
        if (layerStyle && layerStyle !== '') {
          if (!dynamicLayerEntry.drawingInfo) {
            dynamicLayerEntry.drawingInfo = {};
          }
          dynamicLayerEntry.drawingInfo.renderer = layerStyle;
        }

        const layerLabel =layerProps.label;
        if (layerLabel && layerLabel !== '') {
          if (!dynamicLayerEntry.drawingInfo) {
            dynamicLayerEntry.drawingInfo = {};
          }
          dynamicLayerEntry.drawingInfo.labelingInfo = layerLabel;
        }

        dynamicLayers.push(dynamicLayerEntry);
      }
      return dynamicLayers;
    };


    /**
     * Construire l'URL de dessin de la carte depuis l'url stokée
     * dans la propriété wms du FTI et en prenant la clef stockée
     * dans le datastore du FTI.
     *
     * @param {*} fti : composant pour lequel construire l'URL
     *  de fabricaiton de la carte
     * @returns URL de fabrication de la carte
     */
    const getArcGISMapUrl = (fti) => {
      const datastore = DataStoreFactory.getDataStoreByName(fti.storeName);
      const iToken0 = fti.wms.indexOf('/esriproxy/') + 11;
      const iToken1 = fti.wms.indexOf('/', iToken0);
      return fti.wms.substr(0, iToken0) + datastore.proxykey
        + fti.wms.substr(iToken1) + '/export?';
    };


    // Encoders by type of layer
    var encoders = {
      //-----------------------------------------
      //--------------  LAYERS  -----------------
      //-----------------------------------------
      layers: {
        Layer: function(layer) {
          var enc = {
            layer: layer.bodId,
            opacity: layer.getOpacity(),
          };
          return enc;
        },
        Group: function(layer, proj) {
          var encs = [];
          var subLayers = layer.getLayers();
          subLayers.forEach(function(subLayer) {
            if (subLayer.visible) {
              const enc = encoders.layers['Layer'].call(this, layer);
              var layerEnc = encodeLayer(subLayer, proj);
              if (layerEnc && layerEnc.layer) {
                $.extend(enc, layerEnc);
                encs.push(enc.layer);
              }
            }
          });
          return encs;
        },
        Vector: function(layer, features) {
          var enc = encoders.layers['Layer'].call(this, layer);
          var format = new ol.format.GeoJSON();
          var encStyles = {};
          var encFeatures = [];
          var styleId = 0;
          var hasLayerStyleFunction = !!(
            layer.getStyleFunction && layer.getStyleFunction()
          );

          //var opacity;
          angular.forEach(features, function(feature) {
            var encStyle = {
              id: styleId,
            };
            var styles = hasLayerStyleFunction
              ? layer.getStyleFunction()(feature)
              : ol.feature.defaultStyleFunction(feature);

            var geometry = feature.getGeometry();

            // Transform an ol.geom.Circle to a ol.geom.Polygon
            if (geometry.getType() === 'Circle') {
              var polygon = circleToPolygon(geometry);
              feature = new ol.Feature(polygon);
            }

            var encJSON = format.writeFeature(feature);
            if (encJSON.trim != undefined) encJSON = JSON.parse(encJSON);
            if (!encJSON.properties) {
              encJSON.properties = {};

              // Fix https://github.com/geoadmin/mf-geoadmin3/issues/1213
            } else if (encJSON.properties.Style) {
              delete encJSON.properties.Style;
            }

            encJSON.properties._gx_style = styleId;
            encFeatures.push(encJSON);

            if (styles && styles.length > 0) {
              $.extend(encStyle, transformToPrintLiteral(feature, styles[0]));
            }

            encStyles[styleId] = encStyle;
            styleId++;
          });

          angular.extend(enc, {
            type: 'Vector',
            styles: encStyles,
            styleProperty: '_gx_style',
            geoJson: {
              type: 'FeatureCollection',
              features: encFeatures,
            },
            name: layer.bodId,
            opacity: layer.opacity != null ? layer.opacity : 1.0,
          });
          return enc;
        },

        //---   WMS  ----
        WMS: function(layer, config, otherParams) {
          var layers;
          var enc = encoders.layers['Layer'].call(this, layer);
          var params = layer.getSource().getParams();

          var fti = layer.get('fti');

          try {
            layers = params.LAYERS.split(',') || [];
          } catch (e) {
            e.stack;
            layers = params.layers.split(',') || [];
          }

          var styles =
            params.STYLES !== undefined
              ? params.STYLES.split(',')
              : new Array(layers.length).join(',').split(',');
          var cqlfilter =
            params.CQL_FILTER !== undefined ? params.CQL_FILTER : '';
          const urlw = $location.protocol() + '://' + $location.host()
            + ':' + $location.port() + '/services/'
            + PortalsFactory.getPortalId()
            + '/geoserver/wms?token=' + localStorage.auth_token;
          if (layer.type === 'WMS' && fti.type === 'esri') {
            const epsgNum = getEpsgNum();
            if (
              scope.layout &&
              scope.layout.map &&
              scope.layout.map.width &&
              scope.layout.map.height
            ) {
              angular.extend(enc, {
                type: 'WMS',
                baseURL: getArcGISMapUrl(fti),  // fti.wms + '/export?',
                layers: layers,
                styles: styles,
                format: 'PNG32',
                customParams: {
                  EXCEPTIONS: 'XML',
                  TRANSPARENT: 'true',
                  CRS: scope.printConfig.srid, //'EPSG:3857',
                  TIME: params.TIME,
                  BBOXSR: '' + epsgNum,
                  IMAGESR: '' + epsgNum,
                  F: 'image'
                },
                singleTile: true,
              });
            } else {
              angular.extend(enc, {
                type: 'WMS',
                baseURL: fti.wms + '/export?',
                layers: layers,
                styles: styles,
                format: 'PNG32',
                customParams: {
                  EXCEPTIONS: 'XML',
                  TRANSPARENT: 'true',
                  CRS: scope.printConfig.srid, //'EPSG:3857',
                  TIME: params.TIME,
                  BBOXSR: '' + epsgNum,
                  IMAGESR: '' + epsgNum,
                  DPI: '90',
                  F: 'image',
                  token: localStorage.getItem('auth_token'),
                },
                singleTile: true,
              });
            }
            return enc;
          } else if (layer.type == 'GEOWEBCACHE') {
            console.log(layer.getSource().getUrls());

            angular.extend(enc, {
              type: 'tms',
              baseURL: layer
                .getSource()
                .getUrls()[0]
                .replace('wms', 'tms'),
              layer: layers[0],
              //styles: styles,
              format: 'png',
              resolutions: [
                156543.0339,
                78271.51695,
                39135.758475,
                19567.8792375,
                9783.93961875,
                4891.969809375,
                2445.9849046875,
                1222.99245234375,
                611.496226171875,
                305.7481130859375,
                152.87405654296876,
                76.43702827148438,
                38.21851413574219,
                19.109257067871095,
                9.554628533935547,
                4.777314266967774,
                2.388657133483887,
                1.1943285667419434,
                0.5971642833709717,
              ],
              maxExtent: [
                -20037508.3392,
                -20037508.3392,
                20037508.3392,
                20037508.3392,
              ],
              tileSize: [256, 256],

              singleTile: false,
            });
            /*angular.extend(enc, {
                         type: 'WMS',
                         baseURL: layer.getSource().getUrls()[0],
                         layers: layers,
                         //styles: styles,
                         format: 'image/' +'png',
                         customParams: {
                         'EXCEPTIONS': 'XML',
                         'TRANSPARENT': 'true',
                         'CRS': 'EPSG:3857',
                         'TIME': params.TIME
                         },
                         singleTile: false
                         });*/
            return enc;
          } else if (layer.type == 'OSM') {
            if (otherParams.osmUrl) {
              angular.extend(enc, {
                type: 'OSM',
                baseURL: otherParams.osmUrl,
                singleTile: false,
                maxExtent: [
                  -20037508.3392,
                  -20037508.3392,
                  20037508.3392,
                  20037508.3392,
                ],
                tileSize: [256, 256],
                extension: 'png',
                resolutions: [
                  156543.0339,
                  78271.51695,
                  39135.758475,
                  19567.8792375,
                  9783.93961875,
                  4891.969809375,
                  2445.9849046875,
                  1222.99245234375,
                  611.496226171875,
                  305.7481130859375,
                  152.87405654296876,
                  76.43702827148438,
                  38.21851413574219,
                  19.109257067871095,
                  9.554628533935547,
                  4.777314266967774,
                  2.388657133483887,
                  1.1943285667419434,
                  0.5971642833709717,
                ],
              });
              return enc;
            }
          } else if (layer.type == 'WMSBACK') {
            angular.extend(enc, {
              type: 'WMS',
              baseURL: layer.getSource().getUrls()[0],
              layers: layer
                .getSource()
                .getParams()
                .LAYERS.split(','),
              version: layer.getSource().getParams().VERSION || '1.1.1',
              styles: layer.getSource().getParams().STYLES,
              format: 'image/jpeg', // + (config.format || 'png'),
              customParams: {
                EXCEPTIONS: 'XML',
                TRANSPARENT: 'true',
                CRS: scope.map.getView().getProjection().getCode(),
                TIME: params.TIME,
                CQL_FILTER: cqlfilter,
              },
              singleTile: false,
              tiled: true,
            });
            if (!cqlfilter) {
              delete enc.customParams.CQL_FILTER;
            }
            return enc;
          } else {
            var ind;
            for (ind = 0; ind < layers.length; ind++) {
              layers[ind] = PortalsFactory.getPortalId() + ':' + layers[ind];
            }
            console.log(layer.actualEventTarget_);
            angular.extend(enc, {
              type: 'WMS',
              baseURL: config.wmsUrl || urlw,
              layers: layers,
              styles: styles,
              format: 'image/' + (config.format || 'png'),
              customParams: {
                EXCEPTIONS: 'XML',
                TRANSPARENT: 'true',
                CRS: scope.printConfig.srid, //'EPSG:3857',
                TIME: params.TIME,
                CQL_FILTER: cqlfilter,
                token: localStorage.getItem('auth_token'),
              },
              singleTile: config.singleTile || false,
            });
            if (!cqlfilter) {
              delete enc.customParams.CQL_FILTER;
            }
            return enc;
          }
        },

        //-- Opens Street Map
        OSM: function(layer, config, otherParams) {
          if (otherParams.osmUrl) {
            var enc = encoders.layers['Layer'].call(this, layer);

            angular.extend(enc, {
              type: 'OSM',
              baseURL: otherParams.osmUrl,
              singleTile: false,
              maxExtent: [
                -20037508.3392,
                -20037508.3392,
                20037508.3392,
                20037508.3392,
              ],
              tileSize: [256, 256],
              extension: 'png',
              resolutions: [
                156543.0339,
                78271.51695,
                39135.758475,
                19567.8792375,
                9783.93961875,
                4891.969809375,
                2445.9849046875,
                1222.99245234375,
                611.496226171875,
                305.7481130859375,
                152.87405654296876,
                76.43702827148438,
                38.21851413574219,
                19.109257067871095,
                9.554628533935547,
                4.777314266967774,
                2.388657133483887,
                1.1943285667419434,
                0.5971642833709717,
              ],
            });
            return enc;
          }
        },
        WMTS: function(layer, config, params) {
          var enc = encoders.layers['Layer'].call(this, layer);
          var source = layer.getSource();
          var tileGrid = source.getTileGrid();
          var cle;

          try {
            var indkey = source
              .getUrls()[0]
              .toLowerCase()
              .indexOf('key');
            if (indkey !== -1) {
              var hh = getLocation(source.getUrls()[0]);
              var search = hh.search;
              for (var j = 0; j < search.length; j++) {
                if (search[j].toLowerCase().indexOf('key') !== -1) {
                  cle = search[j].split('=')[1];
                  break;
                }
              }
            }
          } catch (e) {
            e.stack;
            cle = undefined;
          }

          var wmtsInfo = getWmtsInfo(layer, params);

          angular.extend(enc, {
            type: 'WMTS',
            baseURL: source.getUrls()[0],
            opacity: layer.getOpacity(),
            layer: source.getLayer(),
            maxExtent: source.getProjection().getExtent(),
            tileOrigin: wmtsInfo.origin,
            tileSize: wmtsInfo.tileDim,
            resolutions: tileGrid.getResolutions(),
            zoomOffset: tileGrid.getMinZoom(),
            version: source.getVersion(),
            requestEncoding: 'KVP',
            extension: source.getFormat().split('/')[1],
            //matrixIds : matrixIds,
            format: source.getFormat(),
            style: 'normal',
            /*dimensions: [ 'TIME' ],*/
            //params : cle ? { 'KEY' : cle , 'TIME': source.getDimensions().Time } : {},
            customParams: cle ? { key: cle } : {},
            matrixSet: source.getMatrixSet(),
          });

          return enc;
        },
        ArcGIS: (arcgisLayers) => {
          const oneFti = arcgisLayers[0].getProperties().fti;
          const epsgNum = getEpsgNum().replace('EPSG:','');
          const urlAg = $location.protocol() + '://' + $location.host()
            + ':' + $location.port() + oneFti.wms.replace('..', '')
            + '/export';
          const encLayer = {
            type: 'ArcGIS',
            dynamicLayers: getArcGisLayerDynLayers(arcgisLayers),
            baseURL: urlAg,
            format: 'PNG32',
            customParams: {
              F: 'image',
              FORMAT: 'PNG32',
              TRANSPARENT: 'true',
              dynamicLayers: getArcGisLayerDynLayers(arcgisLayers),
              tiled: false,
              bboxSR: '' + epsgNum,
              //-- Token tenir compte de la restriction géographique
              //-- le cas échéant.
              kis_token: localStorage.getItem('auth_token'),
              kis_portal_id: localStorage.getItem('portal')
            },
            singleTile: true,
          };

          return encLayer;
        }
      },

      //-----------------------------------------
      //-------------  LEGENDS  -----------------
      //-----------------------------------------
      legends: {}
    };

    function getEncodedLayer(pScope, layerType, p1, p2, otherParams) {
      scope = pScope;
      return encoders.layers[layerType].call(this, p1, p2, otherParams);
    }
    /*
        function getEncodedLegend(pScope,legType,p1,p2) {
            scope = pScope;
            encoders.legends[legType].call(p1,p2);
        }
*/
    return {
      getEncodedLayer: getEncodedLayer,
      encodeLayer: encodeLayer,
    };
  };
  pv2LayerServices.$inject = [
    '$location',
    '$filter',
    'BaseMapFactory',
    'PortalsFactory',
    'DataStoreFactory'
  ];
  return pv2LayerServices;
});
