'use strict';
define(function() {
  let MeasureFactory = function($filter) {
    let MeasureFactory = {};

    // style d'un rectangle de tooltip
    const POINTER_STYLE = new ol.style.Style({
      fill: new ol.style.Fill({
        color: 'rgba(255, 204, 51, 0.7)'
      }),
      zIndex: 10006,
    });

    const MEASURE_PATH_STYLE = new ol.style.Style({
      fill: new ol.style.Fill({
        color: 'rgba(255, 255, 255, 0.2)',
      }),
      stroke: new ol.style.Stroke({
        color: 'rgba(32,32,32, 0.5)',
        lineDash: [5, 5],
        width: 1
      })
    });

    const getTootltipStyle = (feature) => {
      const style = new ol.style.Style({
        fill: new ol.style.Fill({
          color: 'rgba(255, 204, 51, 0.7)'
        }),
        stroke: new ol.style.Stroke({
          width: 1,
          color: 'rgba(255, 255, 255, 0.7)'
        }),
        text: new ol.style.Text({
          text: feature.get('name'),
          font: '12px Arial', // '12 px opensans, sans-serif' ne fonctionne pas
          offsetY: 1,
          fill: new ol.style.Fill({
            color: 'rgba(0, 0, 0, 0.7)'
          }),
          stroke: new ol.style.Stroke({
            color: '#ffffff',
            width: 1
          })
        })
      });

      return [style];
    };

    /**
     * Créée pour l'impression, cette méthode <ul><li>
     *   ajoute la couche des cotations dans une couche d'impression dédiée</li><li>
     *   transforme chaque tooltip de cotation en feature (overlay -> feature)</li><li>
     *   adapte la taille des tooltip à l'échelle d'impression</li><li>
     *   insère les tooltip dans une couche d'impression dédiée</li></ul>
     *
     * @param {ol.Map} map carte openlayers de l'application MAP
     * @param {ol.Collection} layers collection des couches qui seront utilisées pour l'impression
     * @param {number} printScale échelle d'impression
     */
    const addMeasureToolLayers = (map, layers, printScale) => {
      const measureLayer = map.getLayers().getArray().find(layer => layer.name === 'Mesure');
      if (measureLayer && measureLayer.getVisible()) {
        addMeasureLayer(map, layers);
        addMeasureTooltipsAsLayer(map, layers, printScale);
      }
    };

    /**
     * Dans openlayers, les tooltips de mesures ne sont pas des features mais des overlays.<br>
     * Méthode qui convertit les overlays des tooltips de mesure (outil "Cotation") en feature insérés dans une couche dédiée.<br>
     * Deux couches sont créées, une première couche pour les rectangles des tooltips et une seconde couche pour les pointeurs des tooltips.
     * Les rectangles et les triangles sont dissociés pour pouvoir gérer une priorité différente entre les deux:
     * le pointeur doit être placé devant le rectangle associé (afin de masquer la bordure blanche du rectangle au niveau du pointeur)
     * @param {ol.Map} map carte openlayers de l'application MAP
     * @param {ol.Collection} layers collection de couches openlayers qui doit recevoir la couche de mesure
     * @param {number} printScale échelle d'impression
     */
    const addMeasureTooltipsAsLayer = (map, layers, printScale) => {

      const overlays = map.getOverlays();

      if (overlays.getLength() > 0) {

        const radiusInPixel = 4       // Border-radius du tooltip
        const triangleSize = 5.5;     // taille du pointeur du tooltip (triangle vers le bas)
        const dpi = 90;               // Résolution d'impression

        const mpu = ol.proj.METERS_PER_UNIT['m'];           // unité de mesure la carte
        const resolution = map.getView().getResolution();   // résolution de la vue de la carte

        let scaleRatio;               // facteur de mise à l'échelle (rapport entre l'échelle courante et l'échelle d'impression)
        if (printScale) {
          const currentScale = Math.round((100 * resolution * dpi) / (mpu * 2.54));
          scaleRatio = printScale / currentScale;
        }

        // couche qui va contenir le tooltip
        const tooltipLayer = new ol.layer.Vector({
          name: 'measure-tooltip',
          source: new ol.source.Vector(),
          style: getTootltipStyle
        });
        tooltipLayer.name = 'measure-tooltip';

        // couche qui va contenir le pointeur du tooltip (triangle vers le bas)
        const triangleLayer = new ol.layer.Vector({
          name: 'measure-tooltip-pointer',
          source: new ol.source.Vector(),
          style: POINTER_STYLE
        });
        triangleLayer.name = 'measure-tooltip-pointer';

        overlays.forEach((overlay) => {
          const element = overlay.getElement();
          const position = overlay.getPosition();

          // Vérifie si l'élément est un tooltip et possède les classes spécifiées
          const isMeasureTooltip = position !== undefined && element.classList.contains('tooltip')
              && element.classList.contains('tooltip-static');

          if (isMeasureTooltip) {

            // Récupére la position écran du tooltip (en pixels)
            const rect = element.getBoundingClientRect();
            const pixelTop = rect.top + window.pageYOffset;
            const pixelLeft = rect.left + window.pageXOffset;
            const pixelBottom = rect.bottom + window.pageYOffset;
            const pixelRight = rect.right + window.pageXOffset;
            const halfPixWidth = ((pixelRight - pixelLeft) / 2);

            const tlArcCenter = [pixelLeft + radiusInPixel, pixelTop + radiusInPixel];  // top-left
            const trArcCenter = [pixelRight - radiusInPixel, pixelTop + radiusInPixel]; // top-right
            const brCenter = [pixelRight - radiusInPixel, pixelBottom - radiusInPixel]; // btm-right
            const blCenter = [pixelLeft + radiusInPixel, pixelBottom - radiusInPixel];  // btm-left

            // Créé la liste des positions en remplaçant chaque angle de rectangle par un arc de cercle
            const pixelPositions = [
              [pixelLeft, pixelTop + radiusInPixel],
              ...generateArcPoints(tlArcCenter, radiusInPixel, Math.PI, 3*Math.PI/2),
              [pixelLeft + radiusInPixel, pixelTop],
              [pixelRight - radiusInPixel, pixelTop],
              ...generateArcPoints(trArcCenter, radiusInPixel, 3*Math.PI/2, 2*Math.PI),
              [pixelRight, pixelTop + radiusInPixel],
              [pixelRight, pixelBottom - radiusInPixel],
              ...generateArcPoints(brCenter, radiusInPixel, 0, Math.PI/2),
              [pixelRight - radiusInPixel, pixelBottom],
              [pixelLeft + radiusInPixel, pixelBottom],
              ...generateArcPoints(blCenter, radiusInPixel, Math.PI/2, Math.PI),
              [pixelLeft, pixelBottom - radiusInPixel],
              [pixelLeft, pixelTop + radiusInPixel]
            ];

            // Converti les positions en pixels en coordonnées géographiques projetées
            const coordinates = pixelPositions.map(pos => map.getCoordinateFromPixel(pos));

            // Créé une géométrie polygonale avec les angles arrondis
            const tooltipGeometry = new ol.geom.Polygon([coordinates]);

            // créé le tooltip et insère le feature dans la couche de tooltips
            const tooltipFeature = new ol.Feature({
              geometry: tooltipGeometry,
              text: element.innerText
            });
            tooltipFeature.set('name', element.innerText);

            if (scaleRatio) {
              const anchor = map.getCoordinateFromPixel([pixelLeft + halfPixWidth, pixelBottom]);
              tooltipFeature.getGeometry().scale(scaleRatio, scaleRatio, anchor);
            }
            tooltipLayer.getSource().addFeature(tooltipFeature);

            // Dessine le pointeur du tooltip (triangle pointe en bas)
            // Superpose le triangle à la bordure pour la masquer => on ajoute 1 pixel à coordonnée Y de la pointe basse
            const triangleCoords = [];
            const firstVertex = map.getCoordinateFromPixel([pixelLeft + halfPixWidth - triangleSize, pixelBottom - 1]);
            triangleCoords.push(firstVertex);
            const secondVertex = map.getCoordinateFromPixel([pixelLeft + halfPixWidth, pixelBottom + triangleSize + 1]);
            triangleCoords.push(secondVertex);
            const thirdVertex = map.getCoordinateFromPixel([pixelLeft + halfPixWidth + triangleSize, pixelBottom - 1]);
            triangleCoords.push(thirdVertex);
            triangleCoords.push(firstVertex);

            // créé le pointeur et insère le feature dans la couche des triangles
            const triangleFeature = new ol.Feature({
              geometry: new ol.geom.Polygon([triangleCoords]),
            });

            // modifie la taille du pointeur en fonction du ratio d'échelle
            if (scaleRatio) {
              triangleFeature.getGeometry().scale(scaleRatio, scaleRatio, secondVertex);
            }

            triangleLayer.getSource().addFeature(triangleFeature);
          }
        });
        if (triangleLayer.getSource().getFeatures().length > 0
            && tooltipLayer.getSource().getFeatures().length > 0) {
          layers.push(tooltipLayer);
          layers.push(triangleLayer);
        }
      }
    };

    /**
     * Ajoute la couche technique "Mesure" à un tableau de couches
     * La couche "Mesure" doit être présente dans la map
     * @param {ol.Map} map carte openlayers de l'application MAP
     * @param {ol.Collection} layers collection de couches openlayers qui doit recevoir la couche de mesure
     */
    const addMeasureLayer = (map, layers) => {
      const measureLayer = map.getLayers().getArray().find(layer => layer.name === 'Mesure');
      if (measureLayer && measureLayer.getVisible()) {
        measureLayer.setStyle(MEASURE_PATH_STYLE);
        layers.push(measureLayer);
      }
    };

    /**
     * Avec Openlayers 4.6, on ne peut pas créer un polygone avec des sommets arrondis.<br>
     * Pour simuler un sommet arrondi on va créer des points sur un arc de cercle.
     * Plus on créé de points, plus l'arrondi sera précis.
     * @param {number[]} center centre de l'arc de cercle à tracer
     * @param {number} radius rayon de l'arc en pixels
     * @param startAngle angle de départ en radians
     * @param endAngle angle de fin en radians
     * @param numPoints nombre de points à créer
     * @return {number[]} points de l'arc de cercle
     */
    const generateArcPoints = (center, radius, startAngle, endAngle, numPoints) => {
      numPoints = numPoints || 20;

      const angleDelta = (endAngle - startAngle) / numPoints;
      const arcPoints = [];

      for (let i = 0; i <= numPoints; i++) {
        const angle = startAngle + i * angleDelta;
        const x = center[0] + radius * Math.cos(angle);
        const y = center[1] + radius * Math.sin(angle);
        arcPoints.push([x, y]);
      }
      return arcPoints;
    };

    /**
     *
     * @param element
     * @return {boolean|boolean}
     */
    const isMeasureOverlay = (element) => {
      const isMeasureTooltip = element.classList.contains('tooltip') && element.classList.contains('tooltip-static');
      const isMeasureHelpMsg = element.innerText ===
          $filter('translate')('toolbarmeasuredirective.helpMessage')
          || element.innerText === $filter('translate')('toolbarmeasuredirective.startdrawingLine')
          || element.innerText === $filter('translate')('toolbarmeasuredirective.startdrawingPolygon');
      return isMeasureTooltip || isMeasureHelpMsg;
    };

    return {
      MeasureFactory: MeasureFactory,
      addMeasureToolLayers: addMeasureToolLayers,
      isMeasureOverlay: isMeasureOverlay
    };
  };
  MeasureFactory.$inject = ['$filter'];
  return MeasureFactory;
});