'use strict';
define(function () {

      let mapcatalogwidget = function (
          MapCatalogFactory,
          gcPopup,
          ngDialog,
          $rootScope,
          $compile,
          $timeout,
          gaJsUtils
      ) {
        return {
          templateUrl:
              'js/XG/widgets/mapapp/mapcatalog/views/mapcatalogwidget.html',
          restrict: 'EA',

          link: function (scope) {
            scope.catalogFiles = [];
            scope.viewElement = null;

            /**
             * Récupération de la liste des cartes du widget. S'il n'y a aucune carte le tableau récupéré est vide.
             * Création du tableau de cartes de la directive (scope.catalogFiles).
             * Les éléments de ce tableau contiennent les propriétés suivantes: fileName, fileSize, title, category<br>
             * A partir de ce tableau, les cartes sont réparties dans des catégories.
             * Un composant dropzone par catégorie est initialisé.
             */
            const initWidget = () => {
              MapCatalogFactory.getFilesAndConfig().then(
                  (getFilesRes) => {
                    // cartes du widget non classées
                    scope.catalogFiles = getFilesRes.data;

                    // cartes classées par catégories
                    scope.categories = MapCatalogFactory.buildCategories(scope.catalogFiles);

                    // initialise les composants dropzones (1 dropzone / catégorie)
                    scope.dropzoneComponents = {};
                    for (const category of scope.categories) {
                      scope.dropzoneComponents[category.name] = {};
                    }
                  },
                  (error) => {
                    const message = error.data.message ? error.data.message : '';
                    console.log(message);
                  }
              );
            }
            initWidget();

            /**
             * Au clic sur la miniature d'une image
             */
            scope.thumbnailClick = (selectedFile) => {
              MapCatalogFactory.thumbnailClick(selectedFile);
            };

            // largeur du panel de droite du widget
            scope.detView = {
              hasLeftContainer: $rootScope.asideOpened,
            };

            $rootScope.$watch('asideOpened', function (newvalue) {
              scope.detView.hasLeftContainer = newvalue;
            });

            scope.toExpand = false;
            /**
             * Au clic sur le bouton "toggle expand":<br>
             * Exécute l'affichage de la partie droite du widget ou bien sa suppression
             */
            scope.toggleExpand = () => {
              if (scope.toExpand) {
                expandWidget()
              } else {
                collapseWidget();
              }
            };

            /**
             * <ul><li>
             * Créé la div de la partie droite du widget en mode étendu</li><li>
             * Compile cette div pour afficher les variables contenues</li><li>
             * Ajoute la div au body de la page</li></ul>
             */
            const expandWidget = () => {
              const templateUrl = 'js/XG/widgets/mapapp/mapcatalog/views/largeMapCatalog.html';
              const htmlelement = '<div  id="large-catalog-container" ng-include=""  src="\''
                  + templateUrl + '\'"> </div>';

              // enveloppe un élément brut du DOM dans un élement Jquery
              const element = angular.element(htmlelement);

              // compile l'élément Jquery en template HTML AngularJS
              scope.viewElement = $compile(element)(scope);

              $(document.body).append(scope.viewElement);
              $timeout(() => {
                initDropzones();
              }, 500);
            };

            /**
             * Supprime du DOM la partie droite du widget en mode étendu
             * en supprimant la div portant l'id "large-catalog-container"
             */
            const collapseWidget = () => {
              scope.viewElement = null;
              const largeCatalog = document.getElementById('large-catalog-container');
              if (largeCatalog) {
                largeCatalog.remove();
              }
            };

            /**
             * Evènement reçu depuis la directive enfant "mapcatalog" à la saisie de texte dans la zone de filtre de la liste de cartes.<br>
             * Exécute le filtrage des miniatures dans les dropzones
             * "filterValue" est la valeur actuellement saisie dans l'input
             */
            scope.$on('mapCatalogFilterChange', (event, filterValue) => {
              if (scope.mode === 'config') {
                onFilterChange(filterValue);
              } else {
                onStaticFilterChange(filterValue);
              }
            });

            /**
             * Uniquement en mode config.
             * Filtre les miniatures dans les dropzones de catégorie suivant la string "filterValue"
             * @param filterValue string présente dans la zone de filtre de la liste de cartes
             */
            const onFilterChange = (filterValue) => {
              initDropzones(filterValue);
            };

            /**
             * Hors mode config.
             * Filtre les images des catégorie suivant la string "filterValue"
             * @param filterValue string présente dans la zone de filtre de la liste de cartes
             */
            const onStaticFilterChange = (filterValue) => {
              if (!scope.filteredImages) {
                scope.filteredImages = {};
              }
              // récupère toutes les catégories
              const imageCategories = document.querySelectorAll('.readmode-img-row');
              for (const imageCategory of imageCategories) {
                // copie toutes les images de la catégorie dans un objet qui sauvegarde l'état initial
                if (!scope.filteredImages[imageCategory.id]) {
                  scope.filteredImages[imageCategory.id] = imageCategory.querySelectorAll(
                      '.readmode-img-container');
                }
                // vide la catégorie
                while (imageCategory.lastChild) {
                  imageCategory.removeChild(imageCategory.lastChild);
                }
                // affiche les images de la catégorie dont le nom/titre contient la valeur saisie
                if (scope.filteredImages[imageCategory.id].length > 0) {
                  for (const imageContainer of scope.filteredImages[imageCategory.id]) {

                    // récupère le nom de fichier dans l'id de l'image
                    const image = imageContainer.querySelector('img');
                    let filenameComparator = image.id;

                    // regarde si le fichier possède un titre personnalisé
                    // si oui, on utilise le titre comme argument de matchFilter
                    const currentCategoryName = imageCategory.id.substring(5);
                    const category = scope.categories.find(cat => cat.name === currentCategoryName);
                    if (category) {
                      const item = category.items.find(mapItem => mapItem.fileName === image.id);
                      if (item.hasCustomTitle) {
                        filenameComparator = item.title;
                      }
                    }

                    if (matchFilter(filenameComparator, filterValue)) {
                      imageCategory.appendChild(imageContainer);
                    }
                  }
                }
              }
            };

            /**
             * Lance l'insertion des miniatures dans la dropzone de chaque catégorie.<br>
             * Attache un eventHandler permettant de supprimer la carte dans la liste de cartes
             * lorsque l'on clique sur le lien "Supprimer" d'une miniature de dropzone.
             * @param filterValue string présente dans la zone de filtre de la liste de cartes
             */
            const initDropzones = (filterValue = null) => {
              // remplace par null quand filterValue est vide

              for (const category of scope.categories) {
                const dropzoneComponent = scope.dropzoneComponents[category.name].dropzoneComponent;
                if (dropzoneComponent) {

                  if (filterValue !== null) {
                    clearDropzoneFiles(dropzoneComponent, filterValue);
                    scope.isFiltering = true;
                  } else {
                    scope.isFiltering = false;
                  }

                  for (const catalogFile of category.items) {
                    if (!scope.isFiltering || (scope.isFiltering
                        && !MapCatalogFactory.getThumbByDropzoneAndFilename(dropzoneComponent,
                            catalogFile))) {
                      handleAddFileByDropzone(dropzoneComponent, catalogFile, filterValue);
                    }
                  }
                  // attend l'affichage des miniatures de dropzone pour leur attacher des eventHandler
                  $timeout(() => {
                    for (const catalogFile of category.items) {
                      // A l'initialisation de la dropzone,
                      // attache la miniature existante à la liste de cartes du panel gauche (nécessaire pour suppression)
                      addThumbnailDeleteEventListener(catalogFile, category.name);
                      // ajoute l'évènement au clic sur la miniature (nécessaire pour le téléchargement)
                      addThumbnailDownloadEventListener(catalogFile);
                    }
                  }, 500);

                  dropzoneComponent.on('addedfile', (file) => {
                    // récupère le nom de la catégorie dans l'id de l'élément HTML parent de la dropzone
                    const categoryName = dropzoneComponent.element.parentElement.id.substring(3);
                    const currentCategory = scope.categories.find(cat => cat.name === categoryName);
                    // vérifie si le fichier existe déjà dans la catégorie
                    if (!scope.isFiltering && currentCategory && Array.isArray(currentCategory.items)
                        && currentCategory.items.length > 0) {
                      remainProcessOrRisePopupIfFilePresent(dropzoneComponent, file, currentCategory);
                    } else {
                      // si un fichier portant le même nom n'existe pas on poursuit le traitement du dépôt
                      $timeout(() => {
                        dropzoneComponent.processQueue();
                      }, 100);
                    }
                  });

                  // évènements attachés sur chaque dropzone existante
                  dropzoneComponent.on('success', (file) => {
                    // attache le futur fichier déposé à la catégorie et à la liste de cartes du panel gauche
                    attachThumbnailToCategory(file.name, file.size, category.name);
                    // ajoute l'évènement au clic sur la miniature (nécessaire pour le téléchargement)
                    addThumbnailDownloadEventListener({fileName: file.name});
                  });
                }
              }
            };

            /**
             * Au clic sur le bouton "Confirmer" de la popup qui apparaît lorsque l'on dépose un fichier
             * dont le nom existe déjà dans la dropzone.
             */
            scope.confirmEraseFile = () => {
              scope.uploadConfirmed = true;
            };

            /**
             * A la fermeture de la popup qui apparaît lorsque l'on dépose un fichier
             * dont le nom existe déjà dans la dropzone.
             * Ecrase ou préserve le fichier déjà présent dans la dropzone.
             */
            const confirmDialogClosed = () => {
              if (scope.draft.categoryNameToEdit) {
                $timeout(() => {
                  const dropzoneComponent = scope.dropzoneComponents[scope.draft.categoryNameToEdit].dropzoneComponent;
                  if (dropzoneComponent) {
                    // remplace le fichier existant
                    if (scope.uploadConfirmed && scope.fileToConfirm) {
                      const oldThumbnail = MapCatalogFactory.getDzThumbByFilename(
                          scope.fileToConfirm.name);
                      if (oldThumbnail) {
                        oldThumbnail.remove();
                        const currentCategory = scope.categories.find(
                            cat => cat.name === scope.draft.categoryNameToEdit);
                        if (currentCategory) {
                          const oldMapItemIndex = currentCategory.items.findIndex(
                              mapItem => mapItem.fileName === scope.fileToConfirm.name);
                          if (oldMapItemIndex > -1) {
                            currentCategory.items.splice(oldMapItemIndex, 1);
                          }
                        }
                      }
                      dropzoneComponent.processQueue();
                    }
                    // annule l'upload du fichier déposé, préserve le fichier existant
                    else if (scope.fileToConfirm != null) {
                      try {
                        // supprime la miniature du fichier déposé
                        scope.fileToConfirm.previewElement.remove();
                        // termine le processus d'upload sans passer par remove-action comme le ferait 'canceled'
                        dropzoneComponent.emit('queuecomplete');
                        delete scope.fileToConfirm;
                      } catch (error) {
                        console.log(error);
                      }
                    }
                  }
                  scope.uploadConfirmed = false;
                  delete scope.draft.categoryNameToEdit;
                }, 100);
              }
            };

            /**
             * Au dépôt d'un fichier dans une dropzone, vérifie si un fichier portant le même nom existe déjà.<ul><li>
             * Si oui, ouvre une popup pour que l'utilisateur confirme l'écrasement du fichier.</li><li>
             * Si non, autorise la fin de l'import du fichier déposé</li></ul>
             * @param dropzone composant de la dropzone
             * @param file fichier déposé dans la dropzone et en cours d'import
             * @param category catégorie/groupe dans laquelle est située la drozpone
             */
            const remainProcessOrRisePopupIfFilePresent = (dropzone, file, category) => {
              const droppedFilename = file.fileName ? file.fileName : file.name;
              const hasSameFile = category.items.some(mapItem => mapItem.fileName === droppedFilename);
              if (hasSameFile) {
                // si un fichier existe déjà une popup propose l'écrasement du fichier
                scope.fileToConfirm = file;
                scope.draft.categoryNameToEdit = category.name;
                ngDialog.open({
                  template:
                      'js/XG/widgets/mapapp/mapcatalog/views/confirmUploadDialog.html',
                  className: 'ngdialog-theme-plain',
                  closeByDocument: false,
                  scope: scope,
                  preCloseCallback: () => {
                    if (!scope.hasConfirmed) {
                      scope.hasConfirmed = true;
                      confirmDialogClosed();
                    }
                  }
                });
              } else {
                // si un fichier portant le même nom n'existe pas on poursuit le traitement du dépôt
                $timeout(() => {
                  dropzone.processQueue();
                }, 100);
              }
            };

            /**
             * Supprime du DOM les miniatures d'une dropzone
             * Utilisée au filtrage des fichiers lors de la saisie de texte dans la zone de filtre
             * @param dropzoneComponent composant d'une dropzone
             * @param filterStr string saisie dans la zone de filtre
             */
            const clearDropzoneFiles = (dropzoneComponent, filterStr) => {
              if (filterStr !== null) {
                if (dropzoneComponent && dropzoneComponent.element) {
                  while (dropzoneComponent.element.lastChild) {
                    dropzoneComponent.element.removeChild(dropzoneComponent.element.lastChild);
                  }
                }
              }
            };

            /**
             * Insère la miniature d'un fichier dans la dropzone
             * @param dropzoneComponent composant de la dropzone
             * @param catalogFile fichier du catalogue de carte {fileName, fileSize...}
             * @param filterStr string saisie dans la zone de filtre
             */
            const handleAddFileByDropzone = (dropzoneComponent, catalogFile, filterStr) => {
              let filenameComparator = catalogFile.fileName;

              // regarde si le fichier possède un titre personnalisé
              // si oui, on utilise le titre comme argument de matchFilter
              if (catalogFile.hasCustomTitle && catalogFile.title) {
                filenameComparator = catalogFile.title;
              }

              if (catalogFile && catalogFile.fileName && catalogFile.fileName.length > 0) {

                if (matchFilter(filenameComparator, filterStr)) {
                  let catalogFileSize = catalogFile.fileSize;

                  const dzFile = {
                    name: catalogFile.fileName,
                    size: catalogFileSize,
                    clientRemoving: false,
                  };

                  dropzoneComponent.emit('addedfile', dzFile);

                  const logoURL = MapCatalogFactory.getExtensionLogoURL(catalogFile.fileName);

                  dropzoneComponent.emit('thumbnail', dzFile, logoURL);

                  dropzoneComponent.emit('complete', dzFile);
                }
              }
            };

            /**
             * Vérifie l'égalité de deux chaînes: si la valeur de filterStr est contenue dans le nom de fichier catalogFileName
             * @param catalogFileName nom du fichier de carte, comparant de l'évaluation
             * @param filterStr chaîne saisie dans la zone de filtre, comparé de l'évaluation
             * @return {boolean} true si la chaîne saisie est contenue dans le nom de fichier de carte
             */
            const matchFilter = (catalogFileName, filterStr) => {
              if (filterStr == null || filterStr.length === 0) {
                return true;
              } else {
                const lFilterStr = filterStr.toLowerCase();
                const lCatalogFileName = catalogFileName.toLowerCase();
                return lCatalogFileName.indexOf(lFilterStr) !== -1;
              }
            };

            /**
             * Ouvre la popup de confirmation de suppression d'une catégorie au clic sur le bouton "Supprimer" (poubelle).<br>
             * Initialise la variable draft.toDeleteCategoryName correspondant au nom de la catégorie
             * qui sera recherchée dans la liste de catégories pour suppression si l'utilisateur confirme
             * @param categoryName nom de la catégorie à supprimer
             */
            scope.openDeleteCategory = (categoryName) => {
              scope.draft.toDeleteCategoryName = categoryName;
              ngDialog.open({
                template:
                    'js/XG/widgets/mapapp/mapcatalog/views/modal/deleteCategoryModal.html',
                className: 'ngdialog-theme-plain width600 miniclose',
                closeByDocument: false,
                scope: scope
              });
            };

            /**
             * Dans la popup de confirmation de la suppression d'une catégorie,<ul><li>
             * soit la catégorie et les cartes contenues sont supprimées (clic sur "Oui")</li><li>
             * soit la catégorie est supprimée et les cartes sont déplacées dans la catégorie "default" (clic sur "Non")</li></ul>
             * @param shouldDeleteMaps est true si l'utilisateur a cliqué sur le bouton "Oui" souhaitant supprimer également les cartes contenues dans la catégorie
             */
            scope.deleteMapCatalogCategory = (shouldDeleteMaps) => {
              if (scope.draft && scope.draft.toDeleteCategoryName) {
                const toDeleteCategoryIndex = scope.categories.findIndex(
                    cat => cat.name === scope.draft.toDeleteCategoryName);
                if (toDeleteCategoryIndex > -1) {
                  // au clic sur "Oui" pour supprimer les cartes lors de la suppression de la catégorie
                  if (shouldDeleteMaps) {
                    const toDeleteMaps = scope.categories[toDeleteCategoryIndex].items.map(
                        mapItem => mapItem.fileName);
                    MapCatalogFactory.deleteMaps(toDeleteMaps).then(
                        res => {
                          if (Array.isArray(res.data)) {
                            scope.catalogFiles = res.data;
                          }
                          // reconstruit les catégories après suppression
                          refreshRightPanel();
                        }
                    )
                  }
                    // au clic sur "Non" pour renvoyer dans "default" les cartes de la catégorie à supprimer
                    else {
                      // Exclusion des références de fichiers dont la catégorie a été supprimée
                      const categoryNames = scope.categories.map(cat => cat.name);
                      scope.catalogFiles = scope.catalogFiles.filter(file => categoryNames.includes(file.category));

                      for (const catalogFile of scope.catalogFiles) {
                        if (catalogFile.category === scope.draft.toDeleteCategoryName) {
                          catalogFile.category = 'default';
                        }
                      }
                      refreshRightPanel();
                      // sauvegarde le titre et la catégorie dans le json
                      saveMapCatalogConfig();
                  }
                }
              }
            };

            /**
             * Re-initialise les catégories,
             * re-initialise les dropzones,
             * referme et re-ouvre le panel de droite du widget
             */
            const refreshRightPanel = () => {
              scope.categories = MapCatalogFactory.buildCategories(scope.catalogFiles);
              scope.dropzoneComponents = {};
              for (const category of scope.categories) {
                scope.dropzoneComponents[category.name] = {};
              }
              collapseWidget();
              expandWidget();
            };

            /**
             * Ouvre une popup à partir de laquelle l'utilisateur peut saisir le nom d'une nouvelle catégorie vierge à afficher
             */
            scope.openAddCategory = () => {
              ngDialog.open({
                template:
                    'js/XG/widgets/mapapp/mapcatalog/views/modal/newCategoryModal.html',
                className: 'ngdialog-theme-plain width600 miniclose',
                closeByDocument: false,
                scope: scope
              });
            };

            // objet conteneur des inputs des noms de catégorie
            scope.draft = {
              categoryNameToCreate: '',
              editingCategory: false,
            }

            /**
             * Au clic sur le bouton "Enregistrer" de la popup de création d'une nouvelle catégorie,
             * si un titre de catégorie est saisi,<ul><li>
             * Créé une dropzone pour la nouvelle catégorie vierge</li><li>
             * Ajoute une catégorie vierge dans la liste de catégorie</li><li>
             * Après le temps que s'affiche la dropzone créée,
             * ajoute un callback sur la dropzone (après succès d'un dépôt de fichier)
             * pour ajouter le titre et la catégorie de l'image dans le json du repo</li></ul>
             */
            scope.saveAddCategory = () => {
              if (scope.draft.categoryNameToCreate && scope.draft.categoryNameToCreate.length > 0) {
                scope.dropzoneComponents[scope.draft.categoryNameToCreate] = {};
                const newCategory = {
                  name: scope.draft.categoryNameToCreate,
                  items: []
                };
                // KIS-3610: je dois pouvoir ajouter plusieurs catégories (vides) à la suite
                const emptycategories = (scope.categories || []).filter(cat => Array.isArray(cat.items) && cat.items.length === 0);
                emptycategories.push(newCategory);

                scope.categories = MapCatalogFactory.buildCategories(scope.catalogFiles, emptycategories);
                collapseWidget();
                expandWidget();
                delete scope.draft.categoryNameToCreate;
                scope.draft.editingCategory = false;
              }
            };

            /**
             * Affiche un input en lieu et place du titre de catégorie pour modifier celui-ci
             * @param categoryName nom de la catégorie à modifier
             */
            scope.openRenameCategory = (categoryName) => {
              // initialise des variables temporaires pour l'édition qui seront supprimées à la clôture
              scope.draft.editingCategory = true;
              scope.draft.categoryNameToEdit = categoryName;
              scope.draft.newCategoryName = categoryName;
            };

            /**
             * Au clic sur le bouton "Valider" (check vert),<ul><li>
             * enregistre le nouveau titre saisi de la catégorie</li><li>
             * Met à jour la propriété 'category' de toutes les cartes contenues dans la catégorie dans le fichier json du repo</li></ul>
             */
            scope.saveRenameCategory = () => {
              // met à jour le nouveau nom de catégorie dans les propriétés des cartes de la catégorie modifiée
              scope.catalogFiles.map(file => {
                if (file.category === scope.draft.categoryNameToEdit) {
                  file.category = scope.draft.newCategoryName;
                }
                return file;
              });
              // met à jour les objets json des cartes de la catégorie dans le fichier "mapcatalog.json" du repo
              saveMapCatalogConfig();
              // remet à jour les dropzones
              refreshRightPanel();

              // supprime les variables initialisées au début de l'édition du titre de catégorie
              delete scope.draft.newCategoryName;
              delete scope.draft.categoryNameToEdit;
              scope.draft.editingCategory = false;
            };

            /**
             * Au clic sur le bouton "Annuler" (croix rouge),<ul><li>
             * supprime la variable temporaire 'draft.categoryNameToEdit' qui stockait la saisie d'un nouveau titre de catégorie</li><li>
             * passe à false la variable qui désactivait le bouton des autres catégories</li></ul>
             */
            scope.cancelRenameCategory = () => {
              // supprime les variables temporaires de l'édition
              delete scope.draft.newCategoryName;
              delete scope.draft.categoryNameToEdit;
              scope.draft.editingCategory = false;
            };

            /**
             * Construit une carte
             * Ajoute cette carte dans le tableau de cartes du scope et dans la catégorie cible
             * Exécute la sauvegarde du tableau de cartes (#saveConfig)
             * @param fileName nom du fichier pour créer l'objet carte
             * @param fileSize taille du fichier pour créer l'objet carte
             * @param categoryName nom de la catégorie pour créer l'objet carte
             */
            const attachThumbnailToCategory = (fileName, fileSize, categoryName) => {
              if (fileName) {
                const item = {
                  title: fileName,
                  fileName: fileName,
                  fileSize: fileSize,
                  category: categoryName !== 'default' ? categoryName : null,
                };
                const currentCategory = scope.categories.find(cat => cat.name === categoryName);
                if (currentCategory) {
                  item.thumbnail = MapCatalogFactory.getExtensionLogoURL(fileName);
                  scope.catalogFiles.push(item);
                  currentCategory.items.push(item);

                  // tri des cartes dans la catégorie
                  gaJsUtils.sortByKey(currentCategory.items, 'title');

                  // ajoute l'eventListener sur le bouton "Supprimer" de la miniature
                  // qui permet de supprimer le titre de la carte de la liste dans le panel gauche
                  addThumbnailDeleteEventListener(item, categoryName);
                }
                saveMapCatalogConfig();
              }
            };

            /**
             * Sauvegarde le tableau de cartes de la directive.<br>
             * Remplace le json du repo par ce tableau de cartes.
             * Effectue des traitements différents en fonction de la méthode appelante
             */
            const saveMapCatalogConfig = () => {
              MapCatalogFactory.saveConfig(scope.catalogFiles).then(
                  res => {
                    if (!res.data) {
                      require('toastr').error(
                          $filter('translate')('mapcatalogwidget.saveConfigError')
                      );

                      // traitement après changement du nom de la catégorie
                    }
                  },
                  err => {
                    const message = err.data && err.data.message ? err.data.message : '';
                    console.error(message);
                    require('toastr').error(
                        $filter('translate')('mapcatalogwidget.saveConfigError')
                    );
                  }
              );
            };

            /**
             * Ajoute un eventListener au lien "Supprimer" d'une miniature de dropzone.
             * Permet de déclencher la suppression du titre de la carte dans le panel gauche du widget.
             * @param catalogFile nom de fichier de la carte correspondant à la miniature
             * @param categoryName nom de la catégorie à laquelle appartient la carte
             */
            const addThumbnailDeleteEventListener = (catalogFile, categoryName) => {
              const dzThumbnail = MapCatalogFactory.getDzThumbByFilename(catalogFile.fileName);
              if (dzThumbnail) {
                const thumbnailDeleteElement = dzThumbnail.querySelector('.dz-remove');

                if (thumbnailDeleteElement) {
                  // au clic sur le lien "Supprimer" d'une miniature de dropzone
                  thumbnailDeleteElement.addEventListener('click', () => {
                    const mapToDelete = {
                      fileName: catalogFile.fileName,
                      fileSize: catalogFile.fileSize,
                      title: catalogFile.title,
                      category: categoryName
                    }
                    // envoie à la directive enfant "mapCatalog" une demande de suppression dans la liste des cartes
                    // "mapCatalog" supprime également le fichier et la miniature
                    scope.$broadcast('deleteMapCatalogItem', mapToDelete);
                  }, {once: true});
                }
              }
            };

            /**
             * Ajoute un évènement au clic sur l'image de la miniature afin de télécharger le fichier de carte
             * @param catalogFile nom de fichier de la carte correspondant à la miniature
             */
            const addThumbnailDownloadEventListener = (catalogFile) => {
              const htmlElements = [];
              const dzThumbnail = MapCatalogFactory.getDzThumbByFilename(catalogFile.fileName);
              if (dzThumbnail) {
                const dzThumbnailSpan = dzThumbnail.querySelector('.dz-filename');
                if (dzThumbnailSpan) {
                  const selectedItem = {
                    fileName: dzThumbnailSpan.innerText
                  };
                  const dzImage = dzThumbnail.querySelector('.dz-image');
                  const dzDetails = dzThumbnail.querySelector('.dz-details');
                  htmlElements.push(dzImage);
                  htmlElements.push(dzDetails);
                  for (const htmlElement of htmlElements) {
                    htmlElement.addEventListener('click', () => {
                      MapCatalogFactory.thumbnailClick(selectedItem);
                    });
                  }
                }
              }
            };

            /**
             * Permet d'afficher dans une dropzone le message "(Veuillez charger les documents.)"
             * uniquement si la catégorie est vide.<br>
             * Prévient l'affichage du message quand au moins 1 miniature est présente dans la dropzone.
             * @param category catégorie de cartes
             * @return {boolean} true si la catégorie est vide
             */
            scope.isCategoryEmpty = (category) => {
              return (!category || !category.items || category.items.length === 0);
            };

            scope.$on('newCategorie', () => {
              scope.toExpand = false;
              scope.toggleExpand();
            });

            scope.$on('closeTools_mapcatalogwidget', () => {
              scope.toExpand = false;
              scope.toggleExpand();
            });

            scope.$on('openCloseTools_mapcatalogwidget', (event, arg) => {
              if (scope.toExpand) {
                scope.toExpand = false;
                scope.toggleExpand();
              }
            });

            scope.$on('destroy_mapModelMgtWidget', () => {
              scope.toExpand = false;
              scope.toggleExpand();
              $timeout(() => scope.$destroy(), 0);
            });
          },
        };
      };

      mapcatalogwidget.$inject = [
        'MapCatalogFactory',
        'gcPopup',
        'ngDialog',
        '$rootScope',
        '$compile',
        '$timeout',
        'gaJsUtils'
      ];
      return mapcatalogwidget;
    }
);
