'use strict';
define(function () {
  /**
   * Catalogue du widget "Catalogue de cartes"
   */
  let mapcatalog = (ngDialog, MapCatalogFactory, $filter, $timeout) => {
    return {
      restrict: 'E',
      scope: {
        mode: '=',
        categories: '='
      },
      templateUrl:
          'js/XG/widgets/mapapp/mapcatalog/views/mapCatalog.html',
      link: function (scope) {

        /**
         *  A l'initialisation de la directive.<ul><li>
         *  Construit les lignes du catalogue à partir des cartes présentes dans le json</li><li>
         *  construit les catégories à partir de la propriété 'category' de chaque carte issue du json</li><li>
         *  Répartie les cartes dans les catégories</li></ul>
         */
        const buildCatalog = () => {
          scope.hiddenPanels = {};
          scope.allPanelHidden = false;
          // init filterLayers dans un objet sinon filterLayers n'est pas mis à jour depuis le html
          scope.search = {
            filterLayers: ''
          };
        };

        /****************************
         * INITIALISATION DIRECTIVE *
         ***************************/
        buildCatalog();

        /**
         * Masquer les cartes à l'intérieur des catégories
         */
        scope.hideAllPanels = () => {
          for (const categorie of scope.categories) {
            scope.hiddenPanels[categorie.name] = true;
          }
          scope.allPanelHidden = true;
        };

        /**
         * Afficher les cartes à l'intérieur des catégories
         */
        scope.showAllPanels = () => {
          scope.hiddenPanels = {};
          scope.allPanelHidden = false;
        };

        /**
         * Supprime le filtrage quand on appuie sur la touche "escape"
         * Uniquement lors du 1er caractère écrit:<ul><li>
         * Sauvegarde l'état de d'affichage des groupes</li><li>
         * Dégroupe tous les groupes de couches</li><ul>
         * @param event keydown event
         */
        scope.onFilterKeydown = (event) => {
          // reset si escape
          scope.search.filterLayers = event.keyCode === 27 ? '' : scope.search.filterLayers;

          if (!scope.hiddenPanelsPrevState && scope.search.filterLayers.length > 0) {
            if (Object.keys(scope.hiddenPanels).length > 0) {
              scope.hiddenPanelsPrevState = angular.copy(scope.hiddenPanels);
            }
            // modifie la variable allPanelHidden
            scope.showAllPanels();
          }
        };

        /**
         * Après avoir effacé le texte de saisie du filtre,
         * restaure l'état initial d'affichage des groupes de couches
         */
        scope.onFilterKeyup = () => {
          if (scope.search.filterLayers.length === 0) {

            if (scope.filterState) {
              scope.filterState = false;
            }

            // restaure un état initial existant
            if (scope.hiddenPanelsPrevState) {
              scope.hiddenPanels = scope.hiddenPanelsPrevState;
              // restaure allPanelHidden
              if (Object.keys(scope.hiddenPanels).length === Object.keys(scope.categories).length) {
                scope.allPanelHidden = Object.values(scope.hiddenPanels).every(active => active);
              }
              scope.hiddenPanelsPrevState = null;
            }
          }
          scope.$emit('mapCatalogFilterChange', scope.search.filterLayers);
        };

        /**
         * Au clic sur le bouton suppimer (croix) de la zone de saisie du filtre des couches
         */
        scope.resetFilter = () => {
          scope.search.filterLayers = ''
          scope.onFilterKeyup();
        };

        /**
         * Au clic sur le bouton "visualiser" (oeil) d'une carte.<br>
         * Récupère la carte pour l'affecter à la variable que l'on va utiliser dans la popover
         * @param item carte dont on veut afficher la miniature
         * @param categoryName catégorie/groupe de la carte
         */
        scope.showThumbnail = (item, categoryName) => {
          scope.mapToShow = item;
          $timeout(() => {
            const popover = document.getElementById('popover-mapcatalog');
            if (popover) {
              const bsButton = popover.previousElementSibling;
              const categoryIndex = scope.categories.findIndex(cat => cat.name === categoryName);
              if (categoryIndex > -1) {
                const itemIndex = scope.categories[categoryIndex].items.findIndex(
                    catItem => catItem.fileName === scope.mapToShow.fileName);
                if (itemIndex > -1) {
                  adjustPopoverLeftPosition(popover);
                  const isFirstCategory = categoryIndex === 0;
                  const isFirstCategoryHidden = Object.keys(scope.hiddenPanels).includes(
                      scope.categories[0].name);
                  const hasMoreThanOneCategory = scope.categories.length > 1;
                  const isSecondCategory = categoryIndex === 1;
                  const isTwoFirstCategoriesHidden = Object.keys(scope.hiddenPanels).includes(
                      scope.categories[0].name) && hasMoreThanOneCategory && Object.keys(
                      scope.hiddenPanels).includes(scope.categories[1].name);
                  const isThirdCategory = categoryIndex === 2;
                  if (isFirstCategory && itemIndex < 4 || (isFirstCategoryHidden
                      && isSecondCategory && itemIndex < 3)
                      || (isTwoFirstCategoriesHidden && isThirdCategory && itemIndex < 2)) {
                    displayPopoverAboveButton(popover, bsButton);
                  }
                }
              }
            }
          }, 100);
        };

        /**
         * Positionne la popover d'un bouton 'bs-popover' au-dessous de celui-ci.<br>
         * Modifie la propriété CSS 'top' dans le style inline de la popover
         * @param popover popover en tant que élement HTML
         * @param bootstrapButton bouton bs-popover clické en tant que élement HTML
         */
        const displayPopoverAboveButton = (popover, bootstrapButton) => {
          const customCssText = [];
          const inlineStyleArray = popover.style.cssText.split(';');
          if (inlineStyleArray.length > 0 && inlineStyleArray[0].includes('top:')) {
            inlineStyleArray.shift();
          }
          const customTopValue = (bootstrapButton.offsetTop + 44) + 'px';
          customCssText.push('top: ' + customTopValue);
          const inlineCssArray = inlineStyleArray.filter(cssProp => cssProp.length > 0);
          customCssText.push(...inlineCssArray);
          popover.style.cssText = customCssText.join('; ') + ';';
        };

        /**
         * Aligne la popover à gauche pour éviter qu'elle ne passe sous le bandeau des catégories de l'application MAP.<br>
         * Modifie la propriété CSS 'left' dans le style inline de la popover
         * @param popover popover en tant que élement HTML
         */
        const adjustPopoverLeftPosition = (popover) => {
          const inlineStyleArray = popover.style.cssText.split(';');
          const leftPropertyIndex = inlineStyleArray.findIndex(
              inlineCssProp => inlineCssProp.includes('left:'));
          const customCssText = [];
          for (let i = 0; i < inlineStyleArray.length; i++) {
            if (inlineStyleArray[i].length > 0) {
              if (i !== leftPropertyIndex) {
                customCssText.push(inlineStyleArray[i]);
              } else {
                customCssText.push('left: 0px');
              }
            }
          }
          popover.style.cssText = customCssText.join('; ') + ';';
        };

        /**
         * Au clic sur le bouton "éditer" (crayon) d'une carte.
         * Passe à true la variable editing pour afficher l'input de saisie.
         * Passe à true editSessionOpened pour désactiver le bouton "éditer" des autres cartes.
         * Créé une variable pour stocker la saisie de l'input.
         * @param categoryName nom de la catégorie d'appartenance de la carte
         * @param itemIndex rang de la carte dans la catégorie
         */
        scope.openEdit = (categoryName, itemIndex) => {
          const currentCategory = scope.categories.find(cat => cat.name === categoryName);
          if (currentCategory) {
            const item = currentCategory.items[itemIndex];
            item.editing = true;
            scope.editSessionOpened = true;
            scope.draft = {
              itemTitle: angular.copy(item.title)
            }
          }
        };

        /**
         * Au clic sur la croix lors de l'édition du titre d'une carte
         * pour annuler la modification.
         * Supprime la variable qui stocke la saisie
         * Passe à false la variable qui désactive les boutons "edit" des autres cartes
         */
        scope.cancelEdit = (item) => {
          delete scope.draft.itemTitle;
          delete item.editing;
          scope.editSessionOpened = false;
        };

        /**
         * Au clic sur le bouton vert "check" lors de la saisie d'un nouveau titre pour une carte.
         * Remplace le titre de la carte par la chaine de caractères saisie dans l'input.
         * Le titre saisi ne doit pas être égal à un titre existant d'une autre carte.
         * Sinon on affiche un message d'erreur.
         * @param categoryName nom de la catégorie d'appartenance de la carte
         * @param itemIndex rang de la carte dans la catégorie
         */
        scope.saveEdit = (categoryName, itemIndex) => {
          if (scope.draft && scope.draft.itemTitle && scope.draft.itemTitle.length > 0) {

            // regarde si une carte porte déjà le titre saisi
            for (const category of scope.categories) {
              for (let i = 0; i < category.items.length; i++) {
                const item = category.items[i];
                const isSameCategory = categoryName === item.category;
                const isSameIndex = itemIndex === i;
                if (item.title === scope.draft.itemTitle && (!isSameCategory || (isSameCategory
                    && !isSameIndex))) {
                  // lève une erreur si une carte différente porte le même nom
                  // distingue l'item courant pour ne pas lever d'erreur quand on valide sans avoir modifié le titre
                  require('toastr').error(
                      $filter('translate')('mapcatalogwidget.sameTitleError')
                  );
                }
              }
            }
            // met à jour le titre de la carte éditée
            const currentCategory = scope.categories.find(cat => cat.name === categoryName);
            if (currentCategory && currentCategory.items[itemIndex].hasOwnProperty('editing')) {
              currentCategory.items[itemIndex].hasCustomTitle = currentCategory.items[itemIndex].title
                  !== scope.draft.itemTitle && currentCategory.items[itemIndex].title !== 'null';
              currentCategory.items[itemIndex].title = angular.copy(scope.draft.itemTitle);
              delete currentCategory.items[itemIndex].editing;
              delete scope.draft.itemTitle;
              scope.editSessionOpened = false;
            }
            saveConfig();
          }
        };

        /**
         * Au clic sur le bouton supprimer (poubelle) d'une carte.
         * Récupère la carte à supprimer et ouvre une popup de confirmation.
         * @param item carte à supprimer
         * @param categoryName nom de la catégorie d'appartennance de l'item
         */
        scope.openDelete = (item, categoryName) => {
          if (!item.category && categoryName) {
            item.category = categoryName;
          }
          scope.mapToDelete = item;
          ngDialog.open({
            template:
                'js/XG/widgets/mapapp/mapcatalog/views/modal/deleteModal.html',
            className: 'ngdialog-theme-plain width600 miniclose',
            closeByDocument: false,
            scope: scope
          });
        };

        /**
         * Supprime la carte après suppression.
         * La carte est supprimée dans le json, supprimée de la catégorie affichée.
         * Si la cagtégorie affichée est vide alors la catégorie est également masquée.
         */
        scope.deleteItem = () => {
          if (scope.mapToDelete) {
            const categoryName = scope.mapToDelete.category ? scope.mapToDelete.category
                : 'default';

            // recherche la catégorie de la carte
            const currentCategoryIndex = scope.categories.findIndex(
                cat => cat.name === categoryName);
            if (currentCategoryIndex > -1) {

              // recherche la carte dans la catégorie
              const currentCategory = scope.categories[currentCategoryIndex];
              const currentItemIndex = currentCategory.items.findIndex(
                  item => item.fileName === scope.mapToDelete.fileName);
              if (currentItemIndex > -1) {

                // supprime l'affichage de la miniature
                const dzThumbnail = MapCatalogFactory.getDzThumbByFilename(
                    scope.mapToDelete.fileName);
                if (dzThumbnail) {
                  dzThumbnail.remove();
                }

                // supprime la carte dans la catégorie
                currentCategory.items.splice(currentItemIndex, 1);
                if (currentCategory.items.length === 0 && currentCategory.name !== 'default') {

                  // masque l'éventuelle catégorie vide
                  scope.categories.splice(currentCategoryIndex, 1);
                }
              }
            }
            MapCatalogFactory.removeUploaded(scope.mapToDelete.fileName).catch(
                (err) => {
                  const message = err.data && err.data.message ? err.data.message : '';
                  console.error(message);
                  require('toastr').error(
                      $filter('translate')('mapcatalogwidget.saveConfigError')
                  );
                }
            )

          }
        };

        /**
         * Au clic sur le bouton "Supprimer" d'une image de dropzone,
         * lance la suppression de la carte dans la liste
         */
        scope.$on('deleteMapCatalogItem', (event, itemToDelete) => {
          scope.mapToDelete = itemToDelete;
          scope.deleteItem();
        });

        /**
         * Au clic sur le bouton "télécharger" (flèche vers le bas) d'une carte.
         * @param item carte à télécharger
         */
        scope.downloadFile = (item) => {
          MapCatalogFactory.thumbnailClick(item);
        };

        /**
         * Sauvegarde les détails des cartes dans le json du repo
         * Avant sauvegarde, supprime la propriété thumbnail qui est construite dans le front à l'initialisation
         * Sauvegarde la catégorie de la carte quand la catégorie est différente de "default"
         */
        const saveConfig = () => {
          const catalogFiles = [];
          for (const category of scope.categories) {
            const categoryName = category.name;
            for (const mapItem of category.items) {
              if (mapItem.placement) {
                delete mapItem.placement;
              }
              if (categoryName !== 'default') {
                mapItem.category = categoryName;
              } else if (mapItem.category) {
                delete mapItem.category;
              }
              if (mapItem.thumbnail) {
                delete mapItem.thumbnail;
              }
              catalogFiles.push(mapItem);
            }
          }
          MapCatalogFactory.saveConfig(catalogFiles).then(
              res => {
                if (res.data) {
                  scope.catalogFiles = catalogFiles;
                } else {
                  require('toastr').error(
                      $filter('translate')('mapcatalogwidget.saveConfigError')
                  );
                }
              },
              error => {
                const message = error.data && error.data.message ? error.data.message : '';
                console.log(message);
              }
          );
        };
      }
    };
  };

  mapcatalog.$inject = ['ngDialog', 'MapCatalogFactory', '$filter', '$timeout'];
  return mapcatalog;
});