/*global define, $ */
define(['ol3js', 'angular', 'toastr'], function (ol, angular, toastr) {
  'use strict';

  // 1px = 0.28 mm according to the OGC spec for DPI
  var PX_PER_M = (1000 * (window.devicePixelRatio || 1)) / 0.28;

  /*
   * The Annotations Widget allows to draw OpenLayers features on the map.
   * Features are drawn on the drawLayer's source.
   *
   * TODO as of october 2017:
   * - Make better edition dialogs (needs someone who knows a tad about design)
   * - Allow setting symbol X & Y anchor ('auto' or 'N pixels') & label Y offset ('0-50 pixels')
   * - When we edit a feature and save it, it should stay in place instead of going to the foreground
   * - I failed to do this: have a selection style (ie, highlight selected features)
   */
  var mapannotationswidget = function (
    $rootScope,
    $filter,
    $timeout,
    $q,
    gclayers,
    gcInteractions,
    ParametersFactory,
    extendedNgDialog,
    $aside,
    $compile,
    maFeature,
    maAnnotation,
    SridFactory,
    $http,
    ConfigFactory,
    maStyle,
    GeometryFactory,
    maUtils,
    gaJsUtils
  ) {
    var removeDrawListener,
      deferredDraw = $q.defer();
    var draw = {
      source: null,
      layer: null,
      actions: [],
      name: '',
      enabled: false,
      select: false,
      features: null,
      available: deferredDraw.promise,
      getSnap: function () {
        if (!draw.snap) {
          draw.snap = new ol.interaction.Snap({
            source: draw.source,
          });
        }

        return draw.snap;
      },
    };

    /*
     * Get the draw layer as soon as it is available
     */
    if (gclayers.getAnnotationLayer().getSource) {
      deferredDraw.resolve();
    }
    else {
      removeDrawListener = $rootScope.$on('gcTechnicalLayerChange', () => {
        if (gclayers.getAnnotationLayer().getSource) {
          removeDrawListener();
          deferredDraw.resolve();
        }
      });
    }

    /* ------------------------------------------------------
     * Classes used by annotations
     */

    /*
     * Manage the features of an annotation
     */
    function AnnotationFeatures(featureTemplates, map) {
      var self = this; // We extend Array

      // Remove a feature
      self.remove = function (feat) {
        const index = self.indexOf(feat);
        if (index !== -1) {
          draw.source.getFeatures().some(function (olFeat) {
            if (olFeat === feat.vector) {
              draw.source.removeFeature(feat.vector);
              return true;
            }
          });
          self.splice(index, 1);
        }
        return index;
      };

      self.replace = (a, b) => {
        const index = self.remove(a);
        if (index !== -1) {
          self.splice(index, 0, b);
        }
        return index;
      };

      self.vectors = () => {
        return self.map((feat) => {
          return feat.vector;
        });
      };

      self.find = function (vector) {
        var index = -1;
        self.some(function (feat, i) {
          if (
            feat.vector === vector ||
            feat.vectorText === vector ||
            (feat.vectorText != undefined &&
              feat.vectorText.getId() === vector.getId())
          ) {
            index = i;
            return true;
          }
        });
        return self[index];
      };

      // Instantiate the OpenLayers features
      featureTemplates.forEach((template) => {
        self.push(
          new maFeature.Feature(
            template,
            map,
            draw,
            maStyle.getDefaultStyle,
            maStyle.getAnnotationStyle
          )
        );
        const ind = self.length - 1;
        const visible = self[ind].properties.visible;
        if (visible != undefined) self[ind].visible = visible;
      });
    }
    AnnotationFeatures.prototype = new Array();

    /*
     * Manage a selection of features
     */
    function AnnotationSelection(features) {
      var self = this; // We extend Array
      self.allChecked = false;

      self.add = function (feat) {
        if (!feat) {
          return;
        }
        feat.properties.selected = true;
        feat.vector.set('selected', true);
        if (self.indexOf(feat) === -1) {
          self.push(feat);
        }
        if (draw.features.getArray().indexOf(feat.vector) === -1) {
          draw.features.push(feat.vector);
        }
      };

      self.remove = function (feat) {
        var index = self.indexOf(feat);
        if (index !== -1) {
          feat.properties.selected = false;
          feat.vector.set('selected', false);
          self.splice(index, 1);
        }
        draw.features.remove(feat.vector);
      };

      self.toggle = function (feat) {
        if (feat) {
          // If we toggle a single feature
          if (features.indexOf(feat) === -1) {
            return; // Feature doesn't exist
          }
          if (!feat.properties.selected) {
            self.add(feat);
          }
          else {
            self.remove(feat);
          }
        }
        else {
          // If we toggle all features
          self.allChecked = !self.allChecked;
          if (self.allChecked) {
            features.forEach(self.add);
          }
          else {
            self.uncheck();
          }
        }
      };

      self.showHide = function () {
        var visible = !self.isVisible(); // toggle
        self.forEach(function (feat) {
          feat.visible = visible;
          feat.setVisibility();
        });
      };

      self.isVisible = function () {
        var visibleCount = 0;
        self.forEach(function (feat) {
          visibleCount += feat.visible * 2 - 1; // 1 (true) or -1 (false)
        });
        return visibleCount >= 0;
      };

      self.uncheck = function () {
        while (self.length > 0) {
          self.remove(self[0]);
        }
      };

      self.erase = function () {
        self.forEach(function (feat) {
          feat.visible = false;
          feat.setVisibility();
          features.remove(feat);
        });
        self.length = 0;
      };
    }
    AnnotationSelection.prototype = new Array();

    /*
     * Controller for the widget
     * Manages the list of annotations (hence the name 'list')
     */
    function MapAnnotationsWidgetCtrl($scope) {
      var list = this;
      $scope.list = list;
      var map = $scope.map;
      $scope.current = null;
      $scope.annotations = [];
      $scope.draw = draw;
      $scope.Infinity = Infinity;

      /* ------------------------------------------------------
       * 'Annotation' class defined below
       */

      function getLayerAndSource() {
        draw.layer = gclayers.getAnnotationLayer();
        if (draw.layer.getSource) draw.source = draw.layer.getSource();
        else {
          //                    gclayers.createAnnotationLayer();
          $timeout(getLayerAndSource, 500);
        }
      }
      getLayerAndSource();

      $scope.drawSourceClearVectorText = function (removed, scale, zoomDiff) {
        var feats, ind, props;

        if (removed && draw.source) {
          removed = false;
          feats = draw.source.getFeatures();
          for (ind = 0; ind < feats.length; ind++) {
            props = feats[ind].getProperties();
            if (
              props.featureType == 'Polygon' &&
              props.styleGraphic &&
              props.styleGraphic.text == 'Label'
            ) {
              maUtils.removeFeatureFromSource(draw, feats[ind]);
              removed = true;
              break;
            }
          }
          $timeout(function () {
            $scope.drawSourceClearVectorText(removed, scale, zoomDiff);
          }, 100);
        }
        else if (scale != 'noscale') {
          $timeout(function () {
            $scope.annotations.forEach(function (annot) {
              if ($scope.current == annot || annot.showByScale(scale))
                annot.setScaling2(zoomDiff, $scope.selectedFeature, $scope);
            });
          }, 150);
        }
      };

      /* Modifier l'opacity Polygon */
      $scope.$watch('activeGraphicTool.graphic.opacity', function () {
        if ($scope.activeGraphicTool != null) {
          let feat = $scope.current.features.find($scope.selectedFeature);
          feat.properties.styleGraphic.opacity =
            $scope.activeGraphicTool.graphic.opacity;
          feat.setVisibility();
        }
      });

      /* ------------------------------------------------------
       * 'list' object
       */

      /*
       * Tell whether the current user can modify the given annotation
       */
      list.isEditable = function (annotation) {
        // Editable if the user authored the annotation
        if ($rootScope.xgos.user.login === annotation.login) {
          return true;
        }
        // Editable if the user is Root
        return $rootScope.xgos.user.roles.some(function (role) {
          return role
            ? role.name === 'rootUser' || role.name === 'rootuser' || role.name === 'ANCadmin'
            : false;
        });
      };

      /*
       * Open the given annotation
       */
      list.open = function open(annotation) {
        if (!list.isEditable(annotation)) {
          return;
        }
        draw.stop();
        $scope.annotations.forEach(function (annot) {
          annot.hide();
        });
        annotation.show();
        $scope.zoomToAnnotatiopn(annotation);
        annotation.open = true;
        $scope.current = annotation;
      };

      /*
       * Navigate back to the annotation or to the list
       */
      list.back = function back() {
        $scope.resetAll();
        if ($scope.current && draw.enabled) {
          draw.stop();
          draw.enabled = false;
          try {
            draw.source.clear();
          }
          catch (e) {}
        }

        $scope.deactivateLastAside();
        if ($scope.current != null) {
          $scope.current.writeData();
          if (
            !angular.equals($scope.current.source(), $scope.current.untouched)
          ) {
            // Unsaved changes!!
            /*if (!confirm($filter('translate')('annotations.confirm_unsaved'))) {
                        return; // The user canceled
                    }*/
            $scope.current.reset($scope.current.untouched);
          }
          $scope.drawSourceClearVectorText(true, 'noscale', 0);
          $scope.current.hide();
        }

        $scope.current = null;
        $scope.onNewScaleForRes();
        $timeout(() => {
          const container = document.getElementsByClassName('editedFeature');
          if (container != null) $('.editedFeature').remove();
        });
      };

      function annotHasChanged() {
        const d1 = $scope.current.features;
        const d2 = $scope.current.untouched.data.features;

        if (d1.length != d2.length) return true;

        for (let ind = 0; ind < d1.length; ind++) {
          if (!maUtils.sameObject(d1[ind].source(), d2[ind])) return true;
        }

        return false;
      }
      /*
       * annuler les modification apporté et revenir à la dernière version enregistrée
       */
      list.backLastVersion = function backLastVersion() {
        if (!annotHasChanged())
          toastr.error($filter('translate')('mapannotations.no_modif_to_save'));
        else {
          swal(
            {
              title: 'Etes vous sur de vouloir annuler les modifications',
              type: 'warning',
              showCancelButton: true,
              confirmButtonColor: '#DD6B55',
              confirmButtonText: $filter('translate')('common.yes'),
              cancelButtonText: $filter('translate')('common.no'),
              closeOnConfirm: true,
            },
            function (isConfirm) {
              if (isConfirm) {
                $scope.resetAll();
                $scope.deactivateLastAside();
                if ($scope.current != null) {
                  $scope.current.writeData();
                  $scope.current.reset($scope.current.untouched);
                  $scope.current.show(true);
                }
              }
            }
          );
        }
      };
      /*
       * Create a new annotation
       */
      list.create = function create() {
        list.annotationToCopy = null;
        $scope.toCopyAnnotations = [];
        $scope.annotations.forEach(function (annotation) {
          $scope.toCopyAnnotations.push({
            id: annotation.id,
            name: annotation.name,
          });
        });

        var created = new maAnnotation.Annotation(
          { data: { features: [] } },
          $scope,
          AnnotationFeatures,
          AnnotationSelection,
          draw
        );
        list.open(created);
      };

      list.populateNewFromeExistingOne = function populateNewFromeExistingOne() {
        console.log(list.annotationToCopy);
        if (list.annotationToCopy)
          ParametersFactory.getbyid(list.annotationToCopy).then(function (res) {
            if (res.data && res.data.data && res.data.data.features) {
              $scope.current.features = new AnnotationFeatures(
                res.data.data.features,
                map
              );
              $scope.current.selection = new AnnotationSelection(
                $scope.current.features
              );
            }
          });

        //    ParametersFactory.add(template.data, 'Annotation', self.name).then(function(res) {
      };

      /*
       * Remove the given annotation from the list
       */
      list.remove = function remove(annotation) {
        var index = $scope.annotations.indexOf(annotation);
        if (index === -1 || !list.isEditable(annotation)) {
          return; // Error, annotation not found
        }
        /*if (!confirm($filter('translate')('annotations.confirm_delete'))) {
                    return; // The user canceled the removal
                }*/
        if ($scope.current === annotation) {
          $scope.current = null;
        }
        ParametersFactory.remove(annotation.id).then(function () {
          annotation.hide();
          $scope.annotations.splice(index, 1);
        });
      };

      /**
       *    Attend que ol.style.FillPattern soit défini
       * avant de dessiner l'annotation. En effet,
       * au démarrage de KIS, OL peut ne pas être initialisé.
       *
       * @param {[[maAnnotation]]} annotation [[Description]]
       * @param {[[Number]]} scale [[Description]]
       */
      function showAnnotationByScale(annotation, scale) {
        if (ol.style.FillPattern != undefined) annotation.showByScale(scale);
        else
          $timeout(function () {
            showAnnotationByScale(annotation, scale);
          }, 1000);
      }
      /*
       * Fill in the list
       */
      list.fetch = function fetch(goBack) {
        const username = $rootScope.xgos.user.login;
        const isroot = $rootScope.xgos.isroot;
        const srid = $scope.map.getView().getProjection().getCode();
        const bypass = false;
        ParametersFactory.getannotationsbyuser(username, srid, bypass, isroot).then(
          (res) => {
            const currName = $scope.current && $scope.current.name;
            $scope.current = null;
            if (Array.isArray(res.data)) {

              // Delete existing annotations (even unsaved ones)
              for (const annotation of $scope.annotations){
                annotation.hide();
              }

              $scope.annotations = [];

              // Add the new annotations and update the current if any
              let iAnn = 0;
              for (const template of res.data){
                template.data.visible = template.data.svisible;
                const annotation = new maAnnotation.Annotation(
                  template,
                  $scope,
                  AnnotationFeatures,
                  AnnotationSelection,
                  draw,
                  iAnn++
                );
                annotation.svisible = annotation.visible;
                annotation.shared = username !== annotation.untouched.login;
                $scope.annotations.push(annotation);
                if (currName === template.name) {
                  list.open(annotation);
                }
                if (!currName && annotation.svisible) {
                  $timeout(() => {
                    annotation.isHidden = true;
                    //annotation.showHide(annotation.visible);
                    $scope.onNewScaleForRes();
                  }, 750);
                }

                $timeout(() => {
                  if (goBack) {
                    list.back();
                  }
                  annotation.setScaling2(0, $scope);
                });
              }
            }

            // Open the annotation specified by 'attrs' if any
            if ($scope.attrName) {
              for (const annotation of $scope.annotations){
                if (annotation.name === $scope.attrName) {
                  list.open(annotation);
                }
              }
            }
          },
          (err) => {
            // si l'utilisateur $rootScope.xgos.user.login n'existe pas
            toastr.error(err.data.message + '\n' + err.data.detail);
          });
      };
      // The widget does nothing while the draw layer is unavailable
      draw.available.then(function () {
        if ($scope.initialDrawingDone) return;
        $scope.initialDrawingDone = true;
        list.fetch();
        // Store the draw layer
        draw.layer = gclayers.getAnnotationLayer();
        draw.source = draw.layer.getSource();

        // Configure selection (click & collection)
        draw.click = new ol.interaction.Select({
          layer: draw.layer,
          // Alternative: ol.events.condition.altKeyOnly
          // The Shift key is already used by the Modify interaction
          // If you change the condition, update the locale-xx.json files
          toggleCondition: ol.events.condition.always,
        });
        draw.features = draw.click.getFeatures();

        // Configure selection (dragbox)
        draw.box = new ol.interaction.DragBox({
          condition: ol.events.condition.platformModifierKeyOnly, // Ctrl (Command on Mac)
          style: new ol.style.Style({
            stroke: new ol.style.Stroke({ color: [0, 0, 255, 1] }),
          }),
        });
        draw.box.on('boxstart', function () {
          draw.features.clear();
        });
        draw.box.on('boxend', function () {
          var extent = draw.box.getGeometry().getExtent();
          draw.source.forEachFeatureIntersectingExtent(
            extent,
            function (feature) {
              if (feature.getProperties().visible) draw.features.push(feature);
            }
          );
        });

        // Synchronize the OpenLayers selection and our UI selection
        draw.features.on('add', function (event) {
          var feat = $scope.current.features.find(event.element);
          if (!feat.properties.selected) {
            $scope.current.selection.add(feat);
            try {
              $scope.$apply();
            }
            catch (err) {}
          }
        });
        draw.features.on('remove', function (event) {
          if ($scope.current) {
            var feat = $scope.current.features.find(event.element);
            if (feat != undefined && feat.properties.selected) {
              $scope.current.selection.remove(feat);
              try {
                $scope.$apply();
              }
              catch (err) {}
            }
          }
        });
      });

      /*
       * Manage draw interactions
       */
      draw.stop = function () {
        draw.actions.forEach(function (interaction) {
          map.removeInteraction(interaction);
        });
        draw.actions = [];
        var oldName = draw.name;
        draw.name = '';
        draw.select = false;
        return oldName;
      };
      draw.start = function (name, interactions, selecting) {
        draw.name = name;
        draw.select = selecting;
        draw.actions = interactions;
        draw.actions.forEach(function (interaction) {
          if (interaction != undefined) {
            if (
              !(interaction instanceof ol.interaction.Snap) &&
              !(interaction instanceof ol.interaction.Select)
            ) {
              //-- Le setProperties sur le snap a des conséquence malfaisantes.
              //-- Aucune interaction ne fonction si cette propriété
              //-- est positionnée sur l'interaction de type snap.'
              interaction.setProperties({
                gctype: 'kis',
                widget: 'annotations',
                interaction: selecting ? 'select' : 'draw',
              });
            }
            map.addInteraction(interaction);
          }
        });
        gcInteractions.setCurrentToolBar($scope.toolBarWidget);
      };

      //            list.fetch();
    }
    MapAnnotationsWidgetCtrl.$inject = ['$scope', 'maUtils'];

    return {
      templateUrl:
        'js/XG/widgets/mapapp/mapannotations/views/mapannotationswidget.html',
      restrict: 'AE',
      controller: MapAnnotationsWidgetCtrl,
      controllerAs: 'list',
      // Stuff which was not required in the controller
      link: function (scope, elt, attrs, ctrler) {
        let tol = 30;
        const wgs84Sphere = new ol.Sphere(6378137);
        scope.contextMenu = gaJsUtils.getMenuOptions(scope);
        scope.activeAnnotationTool = null;
        scope.new_annot = {};

        /**
         *    Définit la hauteur initiale de la zone du nom de l'annotation
         *  afin que celui-ci soit entiérement visible.
         */
        function updateTextAreaHeight() {
          for (let ind = 0; ind < scope.annotations.length; ind++) {
            let elt = document.getElementById('textAnnotName_' + ind);
            if (elt == null) {
              //-- Widget pas encore construit, on attend un peu plus.
              $timeout(updateTextAreaHeight, 250);
              return;
            }
            elt.style.height = elt.scrollHeight + 'px';
          }
        }

        scope.openMapAnnotationWidget = function () {
          const userlogin = $rootScope.xgos.user.login;
          scope.annotations.forEach((annotation) => {
            annotation.shared = userlogin !== annotation.untouched.login;
          });
          if (!scope.isActive) {
            scope.annotationDlg = extendedNgDialog.open({
              template:
                'js/XG/widgets/mapapp/mapannotations/views/mapannotationswidget2.html',
              className: 'ngdialog-theme-plain width400 nopadding miniclose',
              closeByDocument: false,
              minimizeMaximize: true,
              scope: scope,
              title: $filter('translate')('annotations.widgetname'),
              draggable: true,
              closeCallback: function (data) {
                //-- Quand le widget des annotations est fermé
                //-- soit par le bouton X de la fenêtre,
                //-- soit par clic sur l'icône de la barre d'outil,
                //-- -1- Le widget n'est plus actif
                //-- -2- On réinitialise le dessin des annotations
                //--     à un état de dessin en mode normal.
                scope.isActive = false;
                scope.list.back();
              },
            });
            scope.isActive = true;
            $timeout(updateTextAreaHeight, 250);
          }
          else if (scope.annotationDlg) scope.annotationDlg.close();
        };

        scope.zoomToAnnotatiopn = function (annotation) {
          let ext = annotation.getExtent();
          if (ext != undefined)
            scope.map.getView().fit(ext, {
              duration: 700,
              padding: [50, 50, 50, 350],
            });
        };

        function getHtmlCnt4Popups(text_content, ngDisabled, uniqid) {
          var saveTxt = $filter('translate')('common.save');
          var closeTxt = $filter('translate')('common.close');
          var tpl =
            '<div id="' +
            uniqid +
            '" ' +
            '  class="ol-popup annotationPopup2"' +
            '  style="width: 250px;">' +
            '  <div class="popup-content2">' +
            text_content +
            '  </div>' +
            '  <br>' +
            '  <input type="button" value="' +
            saveTxt +
            '"' +
            '   ng-click="popupSave()"';
          if (ngDisabled != undefined) tpl += ngDisabled;
          tpl +=
            '   class="btn btn-success">&nbsp;' +
            '  <input type="button" value="' +
            closeTxt +
            '"' +
            '    ng-click="popupClose()" class="btn btn-danger">' +
            '</div>';

          return tpl;
        }

        function removeMapPopupOverlay() {
          scope.map.removeOverlay(scope.popupOverlay);
          scope.popupOverlay = undefined;
        }

        /**
         * open popup
         * @param coords
         * @param text_content
         */
        var setPopups = function (
          coords,
          text_content,
          saveActions,
          saveParams,
          ngDisabled,
          closeActions
        ) {
          if (scope.popupOverlay != undefined) return;
          var uniqid = 'geocoderpopup_' + Date.now();

          scope.popupClose = function () {
            removeMapPopupOverlay();
            if (closeActions != undefined) closeActions(saveParams);
            scope.textIsHtml = undefined;
            scope.isSurface = undefined;
            scope.isPerimetre = undefined;
            scope.isLength = undefined;
            stopFeatureRemoval(false);
          };

          scope.popupSave = function () {
            draw.features.clear();
            if (saveActions != undefined) saveActions(saveParams);
            scope.popupClose();
            scope.textIsHtml = undefined;
            scope.isSurface = undefined;
            scope.isPerimetre = undefined;
            scope.isLength = undefined;
          };

          var coordinate = coords;
          var htmlCnt = getHtmlCnt4Popups(text_content, ngDisabled, uniqid);

          angular
            .element(document)
            .find('body')
            .eq(0)
            .append($compile(htmlCnt)(scope));

          var container = document.getElementById(uniqid);

          scope.popupOverlay = new ol.Overlay({
            element: container,
            autoPan: true,
            insertFirst: false,
            autoPanAnimation: {
              duration: 250,
            },
          });
          scope.popupOverlay.set('origin', uniqid);

          scope.map.addOverlay(scope.popupOverlay);
          scope.popupOverlay.setPosition(coordinate);
        };

        function defineAnnotationTools() {
          scope.annotationTools = {
            drawingTools: [
              {
                id: 'drawing-tool-point',
                title: $filter('translate')(
                  'annotations.draw.create_point_tool'
                ),
                type: 'Point',
                class: 'drawing-point-tool-container',
              },
              {
                id: 'drawing-tool-line',
                title: $filter('translate')(
                  'annotations.draw.create_line_tool'
                ),
                type: 'LineString',
                class: 'drawing-common-tool-container',
              },
              {
                id: 'drawing-tool-polygon',
                title: $filter('translate')(
                  'annotations.draw.create_polygon_tool'
                ),
                type: 'Polygon',
                class: 'drawing-common-tool-container',
              },
              {
                id: 'drawing-tool-text',
                title: $filter('translate')(
                  'annotations.draw.create_text_tool'
                ),
                type: 'Text',
                class: 'drawing-common-tool-container',
              },
            ],
            editTools: [
              {
                id: 'drawing-tool-edit',
                title: $filter('translate')('annotations.draw.draw_tool_edit'),
              },
              {
                id: 'drawing-tool-display',
                title: $filter('translate')(
                  'annotations.draw.draw_tool_display'
                ),
              },
              {
                id: 'drawing-tool-tooltip',
                title: $filter('translate')(
                  'annotations.draw.draw_tool_tooltip'
                ),
              },
              {
                id: 'drawing-tool-remove',
                title: $filter('translate')(
                  'annotations.draw.draw_tool_remove'
                ),
              },
            ],
          };
        }

        //-- Au moment de l'instanciation, la traduction des messages
        //-- n'est pas disponible, c'est pourquoi
        //-- on remet la définition des outils à plus tard.
        $timeout(defineAnnotationTools, 2000);

        scope.loadedAsides = {};

        if (attrs.annotationname) {
          scope.attrName = attrs.annotationname;
        }

        // angular.element(document.querySelector('.ol-popup')).css('width', 'fit-content');

        /*
         * Change selection-based action
         */
        scope.addSelect = function (type) {
          if (type === draw.stop()) {
            return;
          }
          var interactions = [draw.click, draw.box];

          if (type === 'Modify') {
            interactions.push(
              new ol.interaction.Modify({
                features: draw.features,
                deleteCondition: function (mapBrowserEvent) {
                  return (
                    ol.events.condition.shiftKeyOnly(mapBrowserEvent) &&
                    ol.events.condition.singleClick(mapBrowserEvent)
                  );
                },
              })
            );
            interactioans.push(draw.getSnap());
          }
          if (type === 'Translate') {
            var translate = new ol.interaction.Translate({
              features: draw.features,
            });
            interactions.push(translate);
            translate.on('translateend', function (evt) {
              evt.features.forEach(function (feature) {
                var cid = feature.getProperties().styleTemplate.containerId;
                var container = document.getElementById(cid);
                container.remove();
                var test = new maFeature.Feature(
                  feature,
                  scope.map,
                  draw,
                  maStyle.getDefaultStyle
                );
                test.setPopup(true);
                //Feature.feature.setPopup()
              });
            });
          }

          draw.start(type, interactions, true);
        };

        scope.deactivateLastAside = function () {
          if (scope.activeAnnotationTool) {
            if (scope.activeAnnotationTool.active)
              scope.activeAnnotationTool.aside.hide();
            scope.activeAnnotationTool = null;
          }
        };

        scope.getCurrentDefaultStyle = function (type) {
          console.log(type);
          return angular.isDefined(maUtils.defaultStyles[type])
            ? angular.copy(maUtils.defaultStyles[type])
            : null;
        };
        scope.editToolIsAlreadyActive = false;
        scope.closeAsideEditTool = function () {
          scope.activeAnnotationTool.aside.hide();
          scope.editToolIsAlreadyActive = true;
        };

        function deactivateCurrentTool(isDrawTool, annoTool) {
          stopFeatureRemoval(false);

          if (scope.activeAnnotationTool.closingSidePanel !== true)
            scope.activeAnnotationTool.active = !scope.activeAnnotationTool
              .aside.$isShown;
          if (
            scope.activeAnnotationTool.active &&
            !scope.activeAnnotationTool.aside.$isShown &&
            scope.activeAnnotationTool.closingSidePanel
          ) {
            scope.activeAnnotationTool.active = false;
            scope.activeAnnotationTool.closingSidePanel = false;
          }

          if (!scope.activeAnnotationTool.active) {
            scope.activeAnnotationTool.aside.hide();
          }
          else {
            scope.activeAnnotationTool.aside.show();
            if (isDrawTool) {
              scope.addGraphic(annoTool.type);
              scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle(
                annoTool.type
              );
            }
          }
        }

        /**
         *     Appelé sur clic d'un des outils de création.
         *
         * @param {[[type]]} annoTool [[Description]]
         * @param {[[type]]} isDrawTool [[Description]]
         */
        scope.activateAnnotationTool = function (annoTool, isDrawTool) {
          scope.selectedFeature = undefined;
          scope.lastAt = annoTool;
          if (annoTool.id == 'drawing-tool-edit') {
            scope.activateEditingTool(annoTool);
            return;
          }
          else if (annoTool.id == 'drawing-tool-display') {
            scope.activateEditingGraphicTool(annoTool);
            return;
          }
          else if (annoTool.id == 'drawing-tool-remove') {
            scope.activateRemoveTool(annoTool);
            return;
          }
          else if (annoTool.id == 'drawing-tool-tooltip') {
            scope.activateEditTooltip(annoTool);
            return;
          }
          if (draw) draw.stop();
          scope.onModifyEnd();
          scope.onModifyGraphicEnd();
          scope.removeDisplayeMeasure();
          // si edit tool  ou graphic tool est active le desactiver
          if (scope.isEditActive) scope.isEditActive = false;
          if (scope.isGraphicActive) scope.isGraphicActive = false;
          if (scope.isRemoveActive) scope.isRemoveActive = false;
          if (scope.isEditTooltip) scope.isEditTooltip = false;
          scope.map.un('click', scope.modifyTooltip);
          scope.map.un('click', scope.removeObject);
          var annoToolType = annoTool.id;

          if (scope.activeAnnotationTool) {
            if (scope.activeAnnotationTool.annotationTool == annoToolType) {
              deactivateCurrentTool(isDrawTool, annoTool);
              return;
            }
            else scope.activeAnnotationTool.aside.hide();
          }

          scope.activeAnnotationTool = {
            annotationTool: annoToolType,
            active: true,
          };

          if (angular.isDefined(scope.loadedAsides[annoToolType])) {
            scope.loadedAsides[annoToolType].show();
            scope.activeAnnotationTool.aside = scope.loadedAsides[annoToolType];

            if (isDrawTool) {
              scope.addGraphic(annoTool.type);
              scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle(
                annoTool.type
              );
            }
          }
          else {
            // Pre-fetch an external template populated with a custom scope
            var drawingToolAside = $aside({
              annoClass: annoTool.class,
              scope: scope,
              backdrop: false,
              keyboard: false,
              footer: false,
              contentTemplate:
                'js/XG/widgets/mapapp/mapannotations/views/palettes/aside-' +
                annoToolType +
                '.html',
              show: true,
            });

            scope.activeAnnotationTool.aside = drawingToolAside;

            scope.loadedAsides[annoToolType] = drawingToolAside;

            if (isDrawTool) {
              scope.addGraphic(annoTool.type);
              scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle(
                annoTool.type
              );
              console.log(scope.activeAnnotationTool.graphic);
            }
          }
        };

        scope.isEditTooltip = false;

        function getOlFeatureFomCoord(coord, textType) {
          var features, ind, coords;
          features = draw.source.getFeatures();
          for (ind = 0; ind < features.length; ind++) {
            if (features[ind].getGeometry().getType() == 'Point') {
              coords = features[ind].getGeometry().getCoordinates();
              if (
                coords[0] == coord[0] &&
                coords[1] == coord[1] &&
                features[ind].getProperties().featureType == textType
              )
                return features[ind];
            }
          }
        }

        /**
         *    Saisie d'un nouveau texte
         * pour l'annotation de type "bulle d'aide".
         *
         * @param {[[maFeature]]} feat [[bulle d'aide à modifier]]
         */
        function modifyHtmlTextAnnotation(feat) {
          var tv;
          //-- Obtention de la valeur actuelle du texte affiché.
          if (
            feat.properties.styleTemplate &&
            feat.properties.styleTemplate.text
          )
            tv = feat.properties.styleTemplate.text;
          else if (
            feat.properties.styleGraphic &&
            feat.properties.styleGraphic.textValue
          )
            tv = feat.properties.styleGraphic.textValue;
          else tv = '';

          //-- Saisie du nouveau texte
          askTextInfo(
            {},
            tv,
            function () {
              //-- En cas de validation de la saise,
              //-- le nouveau texte est pris en compte ici.
              if (feat.properties.styleTemplate)
                feat.properties.styleTemplate.text = scope.inputText;
              else if (
                feat.properties.styleGraphic &&
                feat.properties.styleGraphic.textValue
              ) {
                feat.properties.styleGraphic.textValue = scope.inputText;
              }
              feat.setVisibility();
            },
            feat
          );
        }

        /**
         *     Saisie d'un texte affiché sur la carte pour modification.
         *
         * @param {[[maFeature]]} feat [[Etiquette à modifier]]
         */
        function modifyTextAnnotation(feat) {
          var tv;
          //-- Récupération de la valeur actuelle du texte à modifier.
          if (feat.properties.styleGraphic)
            tv = feat.properties.styleGraphic.textValue;
          else tv = feat.properties.styleTemplate.text.text;

          //-- Saisie du nouveau texte à mettre sur la carte.
          askTextInfo(
            {},
            tv,
            function () {
              //-- Fonction appelée en cas de validation
              //-- du nouveau texte.
              //-- Affectation de la nouvelle valeur
              //-- du texte à afficher.
              var olfeat = getOlFeatureFomCoord(
                feat.geometry.coordinates,
                'Label'
              );
              if (feat.properties.styleGraphic) {
                feat.properties.styleGraphic.textValue = scope.inputText;
                olfeat.setStyle(
                  maStyle.getAnnotationStyle(
                    'text',
                    feat.properties.styleGraphic
                  )
                );
              }
              else {
                feat.properties.styleTemplate.text.text = scope.inputText;
                olfeat.setStyle(
                  maStyle.getDefaultStyle('text', feat.properties.styleTemplate)
                );
              }
              //-- Rafraichissement de la carte.
              $timeout(function () {
                draw.source.refresh();
                var c = scope.map.getView().getCenter();
                c[0] += 1;
                scope.map.getView().setCenter(c);
              }, 500);
            },
            feat
          );
        }

        /**
         * [resetAll reset all interaction(draw, modify, graphic...) ]
         * @return {[type]} [description]
         */
        scope.resetAll = function () {
          scope.onModifyEnd();
          scope.onModifyGraphicEnd();
          scope.removeDisplayeMeasure();
          // si edit tool  ou graphic tool est active le desactiver
          if (scope.isEditActive) scope.isEditActive = false;
          if (scope.isGraphicActive) scope.isGraphicActive = false;
          if (scope.isRemoveActive) scope.isRemoveActive = false;
          if (scope.isEditTooltip) scope.isEditTooltip = false;
          scope.map.un('click', scope.modifyTooltip);
          scope.map.un('click', scope.removeObject);
          if (scope.activeAnnotationTool && scope.activeAnnotationTool.active) {
            scope.activeAnnotationTool.active = false;
            if (
              scope.activeAnnotationTool.aside &&
              scope.activeAnnotationTool.aside.$isShown
            )
              scope.activeAnnotationTool.aside.hide();
          }
          if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
            scope.activeGraphicTool.aside.hide();
            scope.loadedEditAside = {};
          }
        };
        scope.isEditTooltip = false;
        /**
         * [activateEditTooltip activer l'outil de modification de text ou de label]
         * @param  {[type]} annoTool [description]
         * @return {[type]}          [description]
         */
        scope.activateEditTooltip = function () {
          removeMapPopupOverlay();
          stopFeatureRemoval(false);
          // si l'outil est active le desactiver sinon l'activer et desactiver les autres outils
          if (scope.isEditTooltip) {
            scope.isEditTooltip = false;
            scope.map.un('click', scope.modifyTooltip);
            // scope.onRemoveEnd();
          }
          else {
            if (draw) draw.stop();
            if (scope.isEditActive) scope.isEditActive = false;
            if (scope.isGraphicActive) scope.isGraphicActive = false;
            if (scope.isRemoveActive) scope.isRemoveActive = false;
            scope.map.un('click', scope.removeObject);
            scope.map.un('click', scope.modifyTooltip);
            scope.isEditTooltip = true;
            if (
              scope.activeAnnotationTool &&
              scope.activeAnnotationTool.active
            ) {
              scope.activeAnnotationTool.active = false;
              if (
                scope.activeAnnotationTool.aside &&
                scope.activeAnnotationTool.aside.$isShown
              )
                scope.activeAnnotationTool.aside.hide();
            }
            if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
              scope.activeGraphicTool.aside.hide();
            }
            // Modification des features
            scope.editTooltip();
          }
        };
        scope.editTooltip = function () {
          //Recuperation du label si selectionne => A completer avec modification text et label
          scope.map.on('click', scope.modifyTooltip);
        };

        scope.modifyTooltipStep2 = function (feat) {
          var featType = feat.properties.featureType;
          switch (featType) {
          case 'Text':
          case 'text':
            modifyHtmlTextAnnotation(feat);
            break;
          case 'Label':
            modifyTextAnnotation(feat);
            break;
          case 'Polygon':
            // ouvrir une popup ou on peut choisir d'afficcher le perimetre et la surface
            //scope.openPopupPolygon(feat); commenté cf. KIS-964
            break;
          case 'LineString':
            //ouvrir popup line pour afficher longueur
            //alert('is Line');
            //scope.openPopupLine(feat); commenté cf. KIS-964
            break;
          }
        };

        /**
         *    Gestion du clic droit pour fin avec choix lors
         * d'une multisélection pour créer un tooltip,
         * texte à bulle de longueur / surface.
         */
        function clickRightBtnForToolTipMod(event) {
          if (event.button == 2) {
            removeMapPopupOverlay();
            var sf = scope.selection.selectedFeatures;
            scope.modifyTooltipStep2(sf[scope.selection.index].feat);
            unsetToolTipToOneOfMany();
          }
        }

        function unsetToolTipToOneOfMany() {
          scope.map.removeEventListener('mousedown',
            clickRightBtnForToolTipMod);
          scope.tooltipToOneOfMany = 'unset';
        }

        /**
         *    Positionnement de l'écoute de l'événement appuie sur bouton
         * de la souris. Et ce, afin de pouvoir gérer le clic droit
         * qui indique que l'on choisit le featurte mis en surbrillance
         * comme étant celui auquel on veut ajouter
         * un texte à bulle (longueur / surface).
         */
        function setToolTipToOneOfMany() {
          if (scope.tooltipToOneOfMany != 'set')
            map.addEventListener('mousedown', clickRightBtnForToolTipMod);
          scope.tooltipToOneOfMany = 'set';
        }

        scope.modifyTooltip = function (event) {
          if (scope.selection && scope.selection.selectedFeatures.length != 0) {
            //scope.chooseFeatToRemove(event);
            return;
          }
          var promise = scope.getSelectedFeatures(event);

          promise.then(function (res) {
            //-- La liste des objets sélectionnés est obtenue.
            scope.selection = {};
            scope.selection.selectedFeatures = res;
            scope.selection.index = 0;
            scope.selection.action = 'modifyTip';

            if (scope.selection.selectedFeatures.length == 1) {
              scope.modifyTooltipStep2(
                scope.selection.selectedFeatures[0].feat
              );
            }
            else if (scope.selection.selectedFeatures.length == 0) {
              manyObjectsToDelete(0, event.coordinate);
            }
            else {
              scope.manyObjectSelected = 'true';
              manyObjectsToDelete(
                scope.selection.selectedFeatures.length,
                event.coordinate,
                'modifyTip'
              );
              //-- Sélection du premier objet de la liste.
              scope.chooseFeatToRemove();
              setToolTipToOneOfMany();
              //scope.map.un('click', scope.modifyTooltip);
              scope.map.on('click', scope.chooseFeatToRemove);
            }
          });
        };

        scope.openPopupPolygon = function (feat) {
          var p = GeometryFactory.getCentroidInPolygon([feat.geometry]);
          p.then(function (res) {
            var center = res.data;
            var newFeat = new ol.Feature({
              geometry: new ol.geom.Point(center.coordinates),
            });
            var labelTypes = {
              Point: 'Point',
              LineString: 'Line',
              Polygon: 'Polygon',
              text: 'Text',
              Symbol: 'Symbol',
              Label: 'Label',
            };
            var type = 'Text';
            var template = maStyle.getDefaultTemplate(type);
            template.zoomLevel = scope.map.getView().getZoom();
            template.scale100 = $rootScope.xgos.scale4Representation;
            newFeat.setStyle(maStyle.getDefaultStyle(type, template));
            newFeat.setProperties({
              name: labelTypes[type] || 'Unknown',
              selected: false,
              styleTemplate: template,
              featureType: type,
            });
            if (scope.isSurface == undefined) {
              scope.isSurface = true;
            }
            if (scope.isPerimetre == undefined) {
              scope.isPerimetre = true;
            }
            if (scope.textIsHtml) {
              scope.textIsHtml = false;
            }
            //var astootltip1 = $filter('translate')('annotations.surface');
            //var astootltip2 = $filter('translate')('annotations.perimetre');
            var astootltip1 = 'Afficher la surface';
            var astootltip2 = 'Afficher le périmètre';
            var content =
              '<div> <div ng-click=\'isSurface=!isSurface\' class=\'checkbox-custom1\' type=\'checkbox\' ng-class=\'{checkboxcustom1checked: isSurface, checkboxcustom1notchecked: !isSurface}\'></div>';
            content +=
              ' <label ng-click=\'checked=!checked\' >' +
              astootltip1 +
              '</label>';
            content +=
              '<div ng-click=\'isPerimetre=!isPerimetre\' class=\'checkbox-custom1\' type=\'checkbox\' ng-class=\'{checkboxcustom1checked: isPerimetre, checkboxcustom1notchecked: !isPerimetre}\'></div>';
            content +=
              ' <label ng-click=\'checked=!checked\' >' +
              astootltip2 +
              '</label>';
            content += '</div>';
            setPopups(
              newFeat.getGeometry().getCoordinates(),
              content,
              addNewfeature,
              { feature: newFeat, text: true, selectedFeat: feat },
              initTextInput
            );
          });
        };

        scope.openPopupLine = function (feat) {
          //var cc = feat.geometry.coordinates[0];
          var midle = feat.geometry.coordinates.length / 2;

          var cc = feat.geometry.coordinates[Math.trunc(midle)];
          var newFeat = new ol.Feature({
            geometry: new ol.geom.Point(cc),
          });
          var labelTypes = {
            Point: 'Point',
            LineString: 'Line',
            Polygon: 'Polygon',
            text: 'Text',
            Symbol: 'Symbol',
            Label: 'Label',
          };
          var type = 'Text';
          var template = maStyle.getDefaultTemplate(type);
          template.zoomLevel = scope.map.getView().getZoom();
          template.scale100 = $rootScope.xgos.scale4Representation;
          newFeat.setStyle(maStyle.getDefaultStyle(type, template));
          newFeat.setProperties({
            name: labelTypes[type] || 'Unknown',
            selected: false,
            styleTemplate: template,
            featureType: type,
          });
          if (scope.textIsHtml) {
            scope.textIsHtml = false;
          }
          if (scope.isLength == undefined) {
            scope.isLength = true;
          }
          //var astootltip2 = $filter('translate')('annotations.perimetre');
          var astootltip2 = 'Afficher longueur';

          var content =
            '<div><div ng-click=\'isLength=!isLength\' class=\'checkbox-custom1\' type=\'checkbox\' ng-class=\'{checkboxcustom1checked: isLength, checkboxcustom1notchecked: !isLength}\'></div>';
          content +=
            ' <label ng-click=\'checked=!checked\' >' + astootltip2 + '</label>';
          content += '</div>';
          setPopups(
            newFeat.getGeometry().getCoordinates(),
            content,
            addNewfeature,
            { feature: newFeat, text: true, selectedFeat: feat },
            initTextInput
          );
        };

        scope.isRemoveActive = false;

        /**
         * [activateRemoveTool fonction de suppression des objets de l'annotation]
         * @param  {[type]} editingToolType [description]
         * @return {[type]}                 [description]
         */
        scope.activateRemoveTool = function () {
          // si l'outil est active le desactiver sinon l'activer et desactiver les autres outils
          if (scope.isRemoveActive) {
            scope.isRemoveActive = false;
            scope.onRemoveEnd();
            stopFeatureRemoval(false);
          }
          else {
            //scope.onModifyGraphicEnd();
            if (draw) draw.stop();
            if (scope.isGraphicActive) scope.isGraphicActive = false;
            if (scope.isEditActive) scope.isEditActive = false;
            if (scope.isEditTooltip) scope.isEditTooltip = false;
            scope.map.un('click', scope.modifyTooltip);
            scope.isRemoveActive = true;
            if (
              scope.activeAnnotationTool &&
              scope.activeAnnotationTool.active
            ) {
              scope.activeAnnotationTool.active = false;
              if (
                scope.activeAnnotationTool.aside &&
                scope.activeAnnotationTool.aside.$isShown
              )
                scope.activeAnnotationTool.aside.hide();
            }
            if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
              scope.activeGraphicTool.aside.hide();
            }
            // Modification des features
            // scope.removeAnnoObject(annoTool);
            scope.map.on('click', scope.removeObject);
          }
        };

        scope.chooseFeatToRemove = function (event) {
          var src, sf, feat;
          if (scope.selection == undefined) return;
          sf = scope.selection.selectedFeatures;
          if (sf == undefined || sf.length == 0) {
            if (scope.selection.action == 'remove') scope.removeObject(event);
          }
          else {
            //-- Mettre l'objet suivant en surbrillance.
            //-- -- Effacer ce qui est actuellement en surbrillance.
            src = gclayers.getHighLightLayer().getSource();
            src.clear();
            if (event == undefined || event.originalEvent.button == 0) {
              //-- Appelé pour mettre en surbrillance le premier
              //-- objet sélectionné ou suite à clic gauche.
              if (event && ++scope.selection.index >= sf.length)
                scope.selection.index = 0;
              if (sf[scope.selection.index].feat.vectorText)
                feat = sf[scope.selection.index].feat.vectorText.clone();
              else feat = sf[scope.selection.index].feat.vector.clone();
              feat.setStyle(null);
              src.addFeature(feat);
            }
          }
        };

        function stopFeatureRemoval(escape) {
          scope.selection = undefined;
          gclayers.getHighLightLayer().getSource().clear();
          if (!escape) scope.map.un('click', scope.removeObject);

          //-- Suppression des information de longueur et surface.
          var ovls = scope.map.getOverlays();
          ovls.forEach(function (ovl) {
            scope.map.removeOverlay(ovl);
          });
          scope.manyObjectSelected = 'oneChosen';
          unsetToolTipToOneOfMany();
        }

        /**
         *     Rend invisible ou visible le menu contextuel.
         * Le menu contextuel est géré décrit dans left.html.
         * L'action qui le rend visible par click droitn'est
         * pas gérée dans ce code.
         * On a besoin ici de forcer son ivisibilité dans le cas du clic droit
         * qui valide l'objet à supprimer opu à modifier.
         *
         * le paramètre "visible" peut prendre les valeurs "none" ou "block".
         */
        function showHideContextualMenu(visible) {
          var elements = document.getElementsByClassName(
            'angular-bootstrap-contextmenu dropdown clearfix'
          );
          var ind = 0;
          if (visible == 'justOne' && elements.length > 0) {
            ind = 1;
            if (elements[ind] != undefined)
              elements[ind].style.display = 'block';
            visible = 'none';
          }
          for (ind = 0; ind < elements.length; ind++)
            elements[ind].style.display = visible;
        }

        /**
         *    Selon le paramètre on va vider l'élément HTML qui contient
         * la et malheureusement dans certains cas
         * les menus contextuels (duplications).
         */
        function clearContextualMenu(keepOne) {
          var elements = document.getElementsByClassName(
            'angular-bootstrap-contextmenu dropdown clearfix'
          );
          var ind = 1,
            elt,
            len;
          elt = elements[0];
          if (elt != undefined) {
            if (keepOne === false) ind = 0;
            len = elt.childNodes.length;
            for (; ind < len; ind++) {
              elt.removeChild(elt.firstChild);
            }
          }
        }

        scope.removeSelectedObject = function (event) {
          var feat;
          //event.preventDefault();
          //event.stopPropagation();
          if (event != undefined && scope.manyObjectSelected !== 'false') {
            //-- Si appelé par événement alors le menu contextuel
            //-- va s'afficher. Et ceci ne nous convient pas.
            //-- Donc on l'efface.
            $timeout(function () {
              showHideContextualMenu('none');
            }, 150);
            if (scope.selection && scope.manyObjectSelected !== 'true') {
              scope.map.un('click', scope.chooseFeatToRemove);
              scope.selection.selectedFeatures.splice(
                0,
                scope.selection.selectedFeatures.length
              );
            }
          }
          else {
            clearContextualMenu(false);
          }

          if (scope.selection == undefined) {
            if (event.button == 2) {
              //-- Pas de sélection à traiter,
              //-- on doit rétablir le menu contextuel.
              /*
              scope.map.getViewport()
              .addEventListener('contextmenu', addContextMenu, false);
              */
              map.removeEventListener('mousedown', scope.removeSelectedObject);
            }
          }
          else if (event == undefined || event.button == 2) {
            //-- Sur bouton droit choix de l'objet à supprimer:
            //-- donc suppression.
            feat = scope.selection.selectedFeatures[scope.selection.index];
            if (scope.selection.action == 'remove') {
              draw.layer = gclayers.getAnnotationLayer();
              if (draw.source == undefined) {
                draw.source = draw.layer.getSource();
              }
              maUtils.removeFeatureFromSource(draw, feat);
              draw.features.clear();

              scope.current.features.remove(feat.feat);
              //-- Rafraichissement de la carte.
              /*
                        $timeout(function() {
                            draw.layer.setVisible(false);
                            draw.layer.setVisible(true);
                        }, 10);
                        */
            }
            else {
              removeMapPopupOverlay();
              scope.modifyTooltipStep2(feat.feat);
            }

            stopFeatureRemoval(false);
            scope.map.on('click', scope.removeObject);
          }
        };

        /**
         *
         * @param {[[type]]} params [[Description]]
         */
        function manyObjectsToDelete(count, coord, action) {
          var tm = 3500;
          var content = '<label style=\'width:100%;';
          content += 'background-color: #f9f5df;border-radius: 5px;';
          content +=
            'padding-top: 15px;padding-left: 25px; padding-right:25px;\'>';
          if (count == 0) {
            content += 'Pas d\'objet sélectionné</label>';
            tm = 1500;
          }
          else {
            if (action == 'modifyTip')
              content +=
                '' +
                count +
                ' objets sélectionnés: bouton gauche voir le suivant, bouton droit modifier l\'objet</label>';
            else
              content +=
                '' +
                count +
                ' objets sélectionnés: bouton gauche voir le suivant, bouton droit supprimer l\'objet</label>';
          }

          scope.manyObjectsMessage = document.createElement('div');

          scope.manyObjectsMessage.innerHTML = content;
          var manyObjectsMessageOvl = new ol.Overlay({
            element: scope.manyObjectsMessage,
          });
          scope.map.addOverlay(manyObjectsMessageOvl);
          manyObjectsMessageOvl.setPosition([
            coord[0],
            coord[1] + 50 * scope.map.getView().getResolution(),
          ]);
          $timeout(function () {
            scope.map.removeOverlay(manyObjectsMessageOvl);
          }, tm);
        }

        /**
         *     Obtention des objets localisés à l'endroit du clic.
         * Si un seul objet sélectionné: affichage du premier objet
         * de la sélection en surbrillance.
         * Si plusieurs objets sélectionnés mise en place
         * du parcours par clic gauche avec validation par clic droit
         * ou abandon par échappement.
         *
         * @event {    [[Evenement de clic de sélection]]}/**
         *
         */
        scope.removeObject = function (event) {
          if (
            scope.selection != undefined &&
            scope.selection.selectedFeatures != undefined &&
            scope.selection.action == 'remove' &&
            scope.selection.selectedFeatures.length != 0 &&
            scope.selection.selectedFeatures.length != undefined
          )
            return;
          var promise = scope.getSelectedFeatures(event);

          promise.then(function (res) {
            //-- La liste des objets sélectionné est obtenue.
            scope.selection = {};
            scope.selection.selectedFeatures = res;
            scope.selection.index = 0;
            scope.selection.action = 'remove';

            if (scope.selection.selectedFeatures.length == 1) {
              scope.removeSelectedObject();
            }
            else if (scope.selection.selectedFeatures.length == 0) {
              manyObjectsToDelete(0, event.coordinate);
            }
            else {
              scope.manyObjectSelected = 'true';
              manyObjectsToDelete(
                scope.selection.selectedFeatures.length,
                event.coordinate
              );
              //-- Sélection du premier objet de la liste.
              scope.chooseFeatToRemove();
              //-- Suppression théorique de l'écoute de
              //-- l'événement d'affichage du menu contextuel.
              /*
              scope.map
                .getViewport()
                .removeEventListener('contextmenu', addContextMenu);
                */
              map.addEventListener('mousedown', scope.removeSelectedObject);
              scope.map.on('click', scope.chooseFeatToRemove);
            }
          });
        };

        function getGeoJsonObjFromCoord(coords, geomtype) {
          return {
            type: 'FeatureCollection',
            crs: {
              type: 'name',
              properties: {
                name: scope.map.getView().getProjection().getCode(),
              },
            },
            features: [
              {
                type: 'Feature',
                geometry: {
                  type: geomtype,
                  coordinates: coords,
                },
              },
            ],
          };
        }

        function gsfsGetTolerance(feat) {
          tol = scope.map.getView().getResolution() * 5;
          if (feat.properties.featureType == 'Point') {
            if (feat.properties.styleGraphic.ExternalGraphic != undefined)
              tol *= feat.properties.styleGraphic.Size * 20;
            else tol *= feat.properties.styleGraphic.Size / 10;
          }
          return tol;
        }
        function getSelectedFeaturesStep2(
          clickBuffer,
          defer,
          gsfs2IndFeature,
          selectedFeatures
        ) {
          var gsGeom, feat, tol;

          if (gsfs2IndFeature >= scope.current.features.length) {
            //-- La dernière feature de l'annotation a été traitée.
            defer.resolve(selectedFeatures);
          }
          else {
            //-- Récupération de la feature suivante de l'annotation.
            feat = scope.current.features[gsfs2IndFeature];
            //-- Transformer la feature en GeoJson
            if (feat.vectorText)
              gsGeom = getGeoJsonObjFromCoord(
                feat.vectorText.getGeometry().getCoordinates(),
                'Polygon'
              );
            else
              gsGeom = getGeoJsonObjFromCoord(
                feat.geometry.coordinates,
                feat.geometry.type
              );
            if (gsGeom) {
              //-- Obtenir un buffer autour de la feature.
              tol = gsfsGetTolerance(feat);
              GeometryFactory.buffer([gsGeom], 'ROUND', tol).then(function (
                res
              ) {
                //-- Chercher l'intersection entre le buffer
                //-- de l'objet et celui du click de sélection.
                if (!res.data) {
                  //-- Passser à la feature suivante.
                  scope.gsfs2IndFeature++;
                  getSelectedFeaturesStep2(
                    clickBuffer,
                    defer,
                    gsfs2IndFeature + 1,
                    selectedFeatures
                  );
                }
                else {
                  let p = GeometryFactory.intersectTwoGeoms([
                    getGeoJsonObjFromCoord(clickBuffer, 'Polygon'),
                    getGeoJsonObjFromCoord(res.data, 'Polygon'),
                  ]);
                  p.then(function (res) {
                    //-- Si une intersection existe,
                    //-- ajouter la featue dans liste
                    //-- des objets sélectionnés.
                    if (!(res.data == 'null' || res.data == null))
                      selectedFeatures.push({ feat: feat });
                    //-- Passser à la feature suivante.
                    scope.gsfs2IndFeature++;
                    getSelectedFeaturesStep2(
                      clickBuffer,
                      defer,
                      gsfs2IndFeature + 1,
                      selectedFeatures
                    );
                  });
                }
              });
            }
          }
        }

        scope.getSelectedFeatures = function (event) {
          var defer = $q.defer();
          var selectedFeatures = [];
          if (event != undefined) event.preventDefault();
          scope.coordsMap = event.coordinate;

          var featClick = new ol.Feature({
            geometry: new ol.geom.Point(scope.coordsMap),
          });
          var extClick = featClick.getGeometry().getExtent();

          var gsGeom;
          // gsGeom = getGeoJsonObjFromCoord([extClick[0], extClick[1]], 'Point');
          var tol = scope.map.getView().getResolution() * 10;
          gsGeom = getGeoJsonObjFromCoord(
            [
              [extClick[0] - tol, extClick[1] - tol],
              [extClick[0] + tol, extClick[1] - tol],
              [extClick[0] + tol, extClick[1] + tol],
              [extClick[0] - tol, extClick[1] + tol],
              [extClick[0] - tol, extClick[1] - tol],
            ],
            'Polygon'
          );
          GeometryFactory.buffer([gsGeom], 'ROUND', 1).then(function (res) {
            getSelectedFeaturesStep2(res.data, defer, 0, selectedFeatures);
          });

          return defer.promise;
        };

        /**
         * [removeAnnoObject supprimer unobjet de l'annotation]
         * @param  {[type]} annoTool [description]
         * @return {[type]}          [description]
         */
        scope.removeAnnoObject = function () {
          draw.layer = gclayers.getAnnotationLayer();
          if (draw.source == undefined) {
            draw.source = draw.layer.getSource();
          }

          //interaction selection par click
          scope.clickselect = new ol.interaction.Select({
            layer: draw.layer,
            condition: ol.events.condition.click,
            hitTolerance: 20,
            //toggleCondition: ol.events.condition.always
          });
          draw.features = scope.clickselect.getFeatures();
          scope.map.addInteraction(scope.clickselect);
          //ajout des interactions a la map
          draw.start('modify', [scope.clickselect], true);
          scope.editTooltip();
          draw.features.on('add', function (event) {
            var feat = scope.current.features.find(event.element);
            if (feat) {
              draw.source.removeFeature(event.element);
              scope.current.features.remove(feat);
              draw.features.clear();
            }
          });
        };

        // faire un reset à la fin de la modif d'une annotation
        scope.onRemoveEnd = function () {
          draw.features.clear();
          draw.stop();
        };
        scope.isEditActive = false;
        scope.isTranslate = false;
        /**
         * [activateEditingTool fonction de modification de l'annotation]
         * @param  {[type]} editingToolType [description]
         * @return {[type]}                 [description]
         */
        scope.activateEditingTool = function (annoTool) {
          // si l'outil est active le desactiver sinon l'activer et desactiver les autres outils
          if (scope.isEditActive) {
            scope.isEditActive = false;
            scope.onModifyEnd();
          }
          else {
            scope.onModifyGraphicEnd();
            if (scope.isGraphicActive) scope.isGraphicActive = false;
            if (scope.isEditTooltip) scope.isEditTooltip = false;
            scope.map.un('click', scope.modifyTooltip);
            scope.map.un('click', scope.removeObject);
            if (scope.isRemoveActive) scope.isRemoveActive = false;
            scope.isEditActive = true;
            if (
              scope.activeAnnotationTool &&
              scope.activeAnnotationTool.active
            ) {
              scope.activeAnnotationTool.active = false;
              if (
                scope.activeAnnotationTool.aside &&
                scope.activeAnnotationTool.aside.$isShown
              )
                scope.activeAnnotationTool.aside.hide();
            }
            if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
              scope.activeGraphicTool.aside.hide();
            }
            // Modification des features
            scope.updateGeometry('modify', annoTool);
          }
        };

        /**
         *    Appelé aprés modification d'une feature.
         * Cette fonction est appelée aussi quand la feature est
         * le texte à bulle, car on peut le déplacer.
         */
        function updateDimensionTip(event) {
          var maFeat1, maFeat2;
          var feat = event.features.getArray()[0];
          maFeat1 = scope.current.getMaFeatureFromFeat(feat);
          //-- maFeat1 peut être indéfinie si sélection puis zoom
          //-- car à ce moment là, un feature est recréé.
          if (maFeat1 != undefined) {
            maFeat2 = scope.current.getLinkedMaFeatureFromFeat(maFeat1);
            if (maFeat2 != undefined) {
              setFeatureDimensionText(maFeat1, maFeat2.vector);
              maFeat2.cleanVectorText();
              maFeat2.getOlSource().refresh();
              maFeat2.setVisibility();
            }
          }

          if (maUtils.featureIsTipText(scope.selectedFeature)) {
            unselectCurrentFeature();
          }
        }
        var modifyStyle = new ol.style.Style({
          image: new ol.style.Circle({
            radius: 6,
            fill: new ol.style.Fill({
              color: 'rgb(255,255,239,0.5)',
            }),
            stroke: new ol.style.Stroke({
              color: 'rgb(0,143,239,0.7)',
              width: 2,
            }),
          }),
        });
        var modifyStylePolygon = new ol.style.Style({
          fill: new ol.style.Fill({
            color: 'rgb(255,255,239,0.5)',
          }),
          stroke: new ol.style.Stroke({
            color: 'rgb(0,143,239,0.7)',
            width: 3,
          }),
        });

        function unselectCurrentFeature() {
          if (scope.selectedFeature != undefined) {
            if (maUtils.featureIsTipText(scope.selectedFeature)) {
              var zoomDiff = baseZoom - scope.map.getView().getZoom();
              scope.current.setScaling2(
                zoomDiff,
                scope.selectedFeature,
                scope,
                true
              );
            }
            else {
              //-- Rétablissement du style de l'objet précédemment sélectionné.
              scope.selectedFeature.setStyle(scope.styleOfSelectedFeature);
              if (scope.styleOpacityOfSelectedFeature != undefined)
                scope.selectedFeature
                  .getStyle()
                  .getImage()
                  .setOpacity(scope.styleOpacityOfSelectedFeature);
              scope.selectedFeature = undefined;
            }
          }
        }

        function doSelectTheFeature(feat) {
          if (feat != scope.selectedFeature) {
            unselectCurrentFeature();
            //-- Stockage du style de l'objet sélectionné.
            scope.selectedFeature = feat;
            scope.styleOpacityOfSelectedFeature = undefined;
            scope.styleOfSelectedFeature = feat.getStyle();
            if (feat.getGeometry().getType() !== 'Point') {
              feat.setStyle(modifyStylePolygon);
            }
            else if (feat.getStyle().getImage && feat.getStyle().getImage()) {
              scope.styleOpacityOfSelectedFeature = feat
                .getStyle()
                .getImage()
                .getOpacity();
              feat.getStyle().getImage().setOpacity(0.25);
              feat.setStyle([feat.getStyle(), modifyStyle]);
            }
          }
          return true;
        }

        // Mise a jour geometrique d'un objet de l'annotation en fonction de son type
        scope.loadedEditAside = {};
        scope.updateGeometry = function (type) {
          draw.layer = gclayers.getAnnotationLayer();
          if (draw.source == undefined) {
            draw.source = draw.layer.getSource();
          }

          scope.snap = new ol.interaction.Snap({
            source: draw.source,
          });
          //interaction selection par click

          scope.clickselect = new ol.interaction.Select({
            layer: draw.layer,
            condition: ol.events.condition.click,
            hitTolerance: 20,
            filter: function (feat, layer, we) {
              doSelectTheFeature(feat);
              if (maUtils.featureIsTipText(feat, scope)) {
                $timeout(function () {
                  let ptFeat = scope.current.getPointFeatureOfTexteABulle(feat);
                  if (ptFeat) {
                    draw.features.clear();
                    draw.features.push(ptFeat);
                    ptFeat.setStyle(modifyStyle);
                  }
                }, 100);
              }
              return layer != null;
            },
          });

          draw.features = scope.clickselect.getFeatures();

          // interaction modification features

          scope.modifyInteraction = new ol.interaction.Modify({
            features: draw.features,
            deleteCondition: function (mapBrowserEvent) {
              return (
                ol.events.condition.shiftKeyOnly(mapBrowserEvent) &&
                ol.events.condition.singleClick(mapBrowserEvent)
              );
            },
          });
          //ajout des interactions a la map
          draw.start(
            type,
            [scope.modifyInteraction, /*scope.snap ,*/ scope.clickselect],
            true
          );
          scope.modifyInteraction.on('modifyend', updateDimensionTip);

          draw.features.on('add', function (event) {
            var feat = scope.current.features.find(event.element);
            if (feat) {
              if (scope.contextMenu && scope.contextMenu.length >= 3) {
                scope.contextMenu.splice(0, scope.contextMenu.length - 2);
              }
              try {
                scope.$apply();
              }
              catch (err) {}
            }
          });
        };
        // faire un reset à la fin de la modif d'une annotation
        scope.onModifyEnd = function () {
          if (scope.contextMenu && scope.contextMenu.length >= 3) {
            scope.contextMenu.splice(0, scope.contextMenu.length - 2);
          }
          unselectCurrentFeature();
          draw.features.clear();
          draw.stop();
        };

        // Ajout du menu du bouton droit
        scope.map
          .getViewport()
          .addEventListener('contextmenu', addContextMenu, false);
        function addContextMenu(e) {
          e.preventDefault();
          //          if (scope.selection || scope.manyObjectSelected) return;
          clearContextualMenu();
          if (scope.manyObjectSelected === 'oneChosen') {
            scope.manyObjectSelected = 'false';
            return;
          }
          if (
            scope.manyObjectSelected != undefined &&
            scope.manyObjectSelected !== 'false'
          ) {
            $timeout(function () {
              showHideContextualMenu('none');
            }, 100);
            return;
          }
          $timeout(function () {
            showHideContextualMenu('block');
          }, 100);
          scope.coordsMap = scope.map.getEventCoordinate(e);
          console.log(scope.geom, scope.baryCenter);
          // si feature selectionné ajout des actions : déplacer, copier et coller
          if (draw.features && draw.features.getArray().length > 0) {
            scope.isTranslate = true;
            /*** Action deplacer***/
            var deplacer = [
              $filter('translate')('bizedition.menuContext.deplacer'),
              function () {
                if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
                  scope.activeGraphicTool.aside.hide();
                  scope.loadedEditAside = {};
                }
                var ext = draw.features.getArray()[0].getGeometry().getExtent();
                var box = new ol.geom.Polygon.fromExtent(ext);
                scope.featureBox = new ol.Feature({
                  geometry: box,
                  name: 'MyBox',
                });
                draw.source.addFeature(scope.featureBox);
                draw.features.push(scope.featureBox);
                scope.translate = new ol.interaction.Translate({
                  features: draw.features,
                });
                scope.map.addInteraction(scope.translate);
                //-- Curseur par défaut à remettre
                //-- sinon curseur main sur carte.
                scope.map.getViewport().style.cursor = 'move';
                scope.translate.on('translateend', function (event) {
                  scope.map.removeInteraction(scope.translate);
                  if (draw.source == undefined) {
                    draw.source = draw.layer.getSource();
                  }
                  draw.source.removeFeature(scope.featureBox);
                  scope.clickselect.getFeatures().remove(scope.featureBox);
                  draw.features.pop();
                  draw.features = scope.clickselect.getFeatures();

                  if (draw.features.getArray().length != 0) {
                    //-- Rétablissement du style de l'objet précédemment sélectionné.
                    draw.features
                      .getArray()[0]
                      .setStyle(scope.styleOfSelectedFeature);
                  }

                  if (scope.contextMenu && scope.contextMenu.length >= 3) {
                    scope.contextMenu.splice(0, scope.contextMenu.length - 2);
                  }
                  //-- Curseur par défaut à remettre
                  //-- sinon curseur main sur carte.
                  scope.map.getViewport().style.cursor = 'default';
                });
              },
            ];
            /*** Action copier***/
            var copier = [
              $filter('translate')('bizedition.menuContext.copier'),
              function () {
                if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
                  scope.activeGraphicTool.aside.hide();
                  scope.loadedEditAside = undefined;
                }
                var ext = draw.features.getArray()[0].getGeometry().getExtent();
                scope.baryCenter = new ol.extent.getCenter(ext);
                scope.geom = draw.features.getArray()[0].getGeometry().clone();
                scope.styleGraphic = draw.features
                  .getArray()[0]
                  .getProperties()['styleGraphic'];
              },
            ];
            /*** Action coller***/
            var coller = [
              $filter('translate')('bizedition.menuContext.coller'),
              function () {
                if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
                  scope.activeGraphicTool.aside.hide();
                  scope.loadedEditAside = undefined;
                }

                let feat = draw.features.getArray()[0];
                var deltaX = scope.coordsMap[0] - scope.baryCenter[0];
                var deltaY = scope.coordsMap[1] - scope.baryCenter[1];

                let featcopie = $.extend(
                  true,
                  new ol.Feature({}),
                  feat.clone()
                );

                featcopie.setProperties({
                  name: 'Point',
                  selected: false,
                  featureType: 'Point',
                });

                featcopie.getGeometry().translate(deltaX, deltaY);

                let featanno;
                scope.current.features.forEach(function (featano) {
                  if (
                    featano.properties.featureType == 'Text' &&
                    featano.vector.getProperties().styleTemplate.text ==
                      featcopie.getProperties().styleTemplate.text
                  ) {
                    featanno = featano;
                    return;
                  }
                });

                var f = new maFeature.Feature(
                  feat,
                  scope.map,
                  draw,
                  maStyle.getDefaultStyle,
                  maStyle.getAnnotationStyle
                );

                var fa = new maFeature.Feature(
                  featcopie,
                  scope.map,
                  draw,
                  maStyle.getDefaultStyle,
                  maStyle.getAnnotationStyle
                );

                let ftAnnotationz = $.extend(true, f, featanno);
                let lenfeatures = featcopie.getGeometry().getCoordinates()
                  .length;
                let featcopy = $.extend(true, new ol.Feature({}), featcopie);
                let cooordsz =
                  lenfeatures == 2
                    ? $.extend(
                      true,
                      [],
                      fa.vector.getGeometry().getCoordinates()
                    )
                    : $.extend(
                      true,
                      [],
                      fa.vector.getGeometry().getCoordinates()[0][0]
                    );
                featcopy.setGeometry(new ol.geom.Point(cooordsz));
                ftAnnotationz.geometry = fa.geometry;
                ftAnnotationz.vector = featcopy;
                draw.source.addFeature(featcopie);
                scope.current.features.push(ftAnnotationz);
              },
            ];
            // si feature selectionné et bouton droit => ajouter les actions deplacer/copier ou deplacer/coller en fonction
            if (
              scope.contextMenu[0][0] !=
              $filter('translate')('bizedition.menuContext.deplacer')
            ) {
              scope.contextMenu.unshift(deplacer);
            }
            else if (
              scope.geom != undefined &&
              scope.baryCenter != undefined
            ) {
              scope.contextMenu.splice(0, scope.contextMenu.length - 2);
              scope.contextMenu.unshift(deplacer);
            }
          }
        }
        scope.isGraphicActive = false;
        /**
         * [activateEditingTool fonction de modification géometrique des objets d'une annotation]
         * @param  {[type]} editingToolType [description]
         * @return {[type]}                 [description]
         */
        scope.activateEditingGraphicTool = function (annoTool) {
          // si l'outil est active le desactiver sinon l'activer et desactiver les autres outils
          if (scope.isGraphicActive) {
            scope.isGraphicActive = false;
            scope.onModifyGraphicEnd();
          }
          else {
            scope.onModifyEnd();
            scope.isGraphicActive = true;
            if (scope.isEditActive) scope.isEditActive = false;
            if (scope.isEditTooltip) scope.isEditTooltip = false;
            scope.map.un('click', scope.modifyTooltip);
            scope.map.un('click', scope.removeObject);
            if (scope.isRemoveActive) scope.isRemoveActive = false;
            if (
              scope.activeAnnotationTool &&
              scope.activeAnnotationTool.active
            ) {
              scope.activeAnnotationTool.active = false;
              if (
                scope.activeAnnotationTool.aside &&
                scope.activeAnnotationTool.aside.$isShown
              )
                scope.activeAnnotationTool.aside.hide();
            }
            // Modification des styles des objets de l'annotation
            scope.updateGraphic(annoTool);
          }
        };

        // tooltipConf gère les cases à cocher pour l'affichage du périmetre/area
        scope.tooltipConf = {
          isPerimetre: false,
          isSurface: false,
        };

        /**
         * [updateGraphic modifications de styles des objets d'une annotation]
         * @param  {[type]} annoTool [description]
         * @return {[type]}          [description]
         */
        scope.updateGraphic = function (annoTool) {
          draw.layer = gclayers.getAnnotationLayer();
          if (draw.source == undefined) {
            draw.source = draw.layer.getSource();
          }
          //interaction selection par click
          scope.clickselect = new ol.interaction.Select({
            layer: draw.layer,
            condition: ol.events.condition.click,
            hitTolerance: 20,
            //toggleCondition: ol.events.condition.always
          });
          var type = 'Modify';
          draw.features = scope.clickselect.getFeatures();
          //ajout des interactions a la map
          draw.start(type, [scope.clickselect], true);
          //scope.current.selection = new AnnotationSelection(scope.current.features);
          draw.features.on('add', function (event) {
            var feat;
            feat = scope.current.features.find(event.element);
            if (feat) {
              // Recuperer la geometry et ouvrir la popup du style en fonction
              var geomType = feat.geometry.type;
              var annoToolType = '';
              var type = '';
              var featType = feat.properties.featureType;
              scope.isNotToolTipStyle = true;
              switch (featType) {
              case 'Text':
                scope.isNotToolTipStyle = false;
              case 'Label':
                annoToolType = 'drawing-tool-text';
                type = 'Label';
                break;
              case 'Point':
                annoToolType = 'drawing-tool-point';
                type = 'Point';
                break;
              case 'Polygon':
                annoToolType = 'drawing-tool-polygon';
                type = 'Polygon';
                break;
              case 'LineString':
                annoToolType = 'drawing-tool-line';
                type = 'LineString';
                break;
              }
              //A  refaire dans une nouvelle fonction/bouton updateStyle
              scope.styleType = type;
              if (
                scope.activeGraphicTool == null ||
                scope.activeGraphicTool == undefined
              ) {
                scope.activeGraphicTool = {};
              }
              if (!scope.loadedEditAside) {
                scope.loadedEditAside = {};
              }
              var drawingToolAside;
              //if (scope.loadedEditAside[annoToolType]==undefined)
              drawingToolAside = $aside({
                annoClass: annoTool.class,
                scope: scope,
                backdrop: false,
                keyboard: false,
                contentTemplate:
                  'js/XG/widgets/mapapp/mapannotations/views/palettes/aside-' +
                  'drawing-tool-display' +
                  '.html',
                show: true,
                footer: false,
              });
              if (feat.properties.styleGraphic) {
                if (feat.properties.styleGraphic.opacity == undefined)
                  feat.properties.styleGraphic.opacity = 100.0;
                scope.activeGraphicTool.graphic = feat.properties.styleGraphic;
              }
              else {
                scope.activeGraphicTool.graphic = scope.getCurrentDefaultStyle(
                  scope.styleType
                );
              }
              if (feat.properties.showPerimeter) {
                scope.isPerimetre = true;
                scope.tooltipConf.isPerimetre = true;
              }
              else {
                scope.isPerimetre = false;
                scope.tooltipConf.isPerimetre = false;
              }
              if (feat.properties.showSurface) {
                scope.isSurface = true;
                scope.tooltipConf.isSurface = true;
              }
              else {
                scope.isSurface = false;
                scope.tooltipConf.isSurface = false;
              }
              //scope.activeGraphicTool.graphic = scope.getCurrentDefaultStyle(scope.styleType);
              scope.activeGraphicTool.aside = drawingToolAside;
              //if (scope.loadedEditAside[annoToolType]==undefined)
              scope.loadedEditAside[annoToolType] = drawingToolAside;
              scope.selectedFeature = event.element;

              // scope.$apply();
              $timeout(function () {
                scope.ensureNoAsideCloseBtn();
              }, 250);
            }
          });
        };
        scope.appliquerStyle = function (type, graphic) {
          var objType = type;
          if (type == 'Label') objType = 'text';
          var feat;
          feat = scope.current.features.find(scope.selectedFeature);
          var style = maStyle.getAnnotationStyle(
            objType,
            graphic,
            undefined,
            undefined,
            false
          );
          var ratio = feat.getRatio(style);
          scope.selectedFeature.setStyle(
            maStyle.getAnnotationStyle(objType, graphic, ratio)
          );
          feat.properties.styleGraphic = graphic;
          feat.setVisibility(undefined, ratio);
          if (feat.vectorText != undefined) {
            //-- Dans le cas d'un texte à bulle, on recrée le feature
            //-- openlayers de la bulle, il faut donc changer
            //-- le contenu de la liste des features sans quoi le "mafeature"
            //-- stocké référera un desssin de bulle qui n'existe plus.
            scope.selectedFeature = feat.vectorText;
            draw.features.clear();
            draw.features.push(scope.selectedFeature);
          }
        };
        scope.onModifyGraphicEnd = function () {
          if (scope.activeGraphicTool && scope.activeGraphicTool.aside) {
            scope.activeGraphicTool.aside.hide();
            scope.loadedEditAside = {};
          }
          scope.ensureNoAside();
          draw.features.clear();
          draw.stop();
        };
        scope.switchPointSymbolMode = function (tool) {
          /*  var styleMode = scope.activeAnnotationTool.graphic.styleMode;
                    if ('SYMBOL' == styleMode && angular.isDefined(scope.activeAnnotationTool.graphic.ExternalGraphic))
                        scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle('Point');
                    else if ('SVG' == styleMode && !angular.isDefined(scope.activeAnnotationTool.graphic.ExternalGraphic))
                        scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle('PointSVG');*/
          var styleMode = tool.graphic.styleMode;
          if (
            'SYMBOL' == styleMode &&
            angular.isDefined(tool.graphic.ExternalGraphic)
          )
            tool.graphic = scope.getCurrentDefaultStyle('Point');
          else if (
            'SVG' == styleMode &&
            !angular.isDefined(tool.graphic.ExternalGraphic)
          )
            tool.graphic = scope.getCurrentDefaultStyle('PointSVG');
        };

        scope.switchPolygonSymbolMode = function (tool) {
          /* var styleMode = scope.activeAnnotationTool.graphic.styleMode;
                    if ('SYMBOL' == styleMode && angular.isDefined(scope.activeAnnotationTool.graphic.ExternalGraphic))
                        scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle('Polygon');
                    else if ('HASH' == styleMode && !angular.isDefined(scope.activeAnnotationTool.graphic.ExternalGraphic))
                        scope.activeAnnotationTool.graphic = scope.getCurrentDefaultStyle('PolygonHASH');*/
          var styleMode = tool.graphic.styleMode;
          if (
            'SYMBOL' == styleMode &&
            angular.isDefined(tool.graphic.ExternalGraphic)
          )
            tool.graphic = scope.getCurrentDefaultStyle('Polygon');
          else if (
            'HASH' == styleMode &&
            !angular.isDefined(tool.graphic.ExternalGraphic)
          )
            tool.graphic = scope.getCurrentDefaultStyle('PolygonHASH');
        };

        function initTextInput() {
          scope.activateAnnotationTool(scope.lastAt, true);
          scope.activateAnnotationTool(scope.lastAt, true);
        }

        /**
         *
         * @param {[[type]]} params [[Description]]
         */
        function askTextInfo(options, text, saveCb, feature) {
          var astootltip = $filter('translate')('annotations.asTooltip');
          scope.inputText = text;
          var content =
            '<textarea ng-model=\'inputText\' style=\'height: 55px;width:100%;\'></textarea>';
          content += '<div>';
          if (options.lblTypeChoice) {
            content +=
              ' <input  ng-model=\'textIsHtml\'  class=\'checkbox-input1\' type=\'checkbox\' ></input>';
            content +=
              ' <label ng-click=\'checked=!checked\' >' + astootltip + '</label>';
          }
          content += '</div>';
          var coords;
          if (feature.getGeometry)
            coords = feature.getGeometry().getCoordinates();
          else coords = feature.geometry.coordinates;
          setPopups(
            coords,
            content,
            saveCb,
            { feature: feature, text: true },
            'ng-disabled=\'inputText==undefined || inputText.length==0\'',
            initTextInput
          );
        }

        /**
         *    Affectation d'un même identifiant de liaison
         * à 2 objets de type maFeature.
         * Cet identifiant permet de mettre à jour le texte d'un texte à bulle
         * associé à une ligne ou un polygone, de telle sorte que la longueur
         * et le périmétre contenu dans le texte soit toujours cohérent
         * avec l'objet et sa nouvelle forme lorsqu'il est modifié.
         */
        function linkMafeatures(feat1, feat2) {
          var feat,
            idMax = 0;
          //-- Obtenir un identifant de liaison disponible.
          for (var iFeat = 0; iFeat < scope.current.features.length; iFeat++) {
            feat = scope.current.features[iFeat];
            if (
              feat.properties.linkId != undefined &&
              feat.properties.linkId > idMax
            ) {
              idMax = feat.properties.linkId;
            }
          }
          //-- Affecter l'identifiant de liaison aux 2 features.
          ++idMax;
          feat1.properties.linkId = feat2.properties.linkId = idMax;
        }

        /**
         *    Calcul du texte contenant les valeurs de longueur et/ou
         * de superficie de l'objet associé.
         *
         * shapeFeat: objet principal dont on calcul
         *            la longueur et/ou la superficie
         * tipfeat: texte à bulle contenant les valeurs de mesure.
         */
        function setFeatureDimensionText(shapeFeat, tipFeat) {
          let line, coordinates, linkedTo;

          if (shapeFeat.geometry.type == 'Polygon') {
            coordinates = shapeFeat.vector.getGeometry().getCoordinates();

            //coordinates = getProjectedCoords(shapeFeat);

            //-- Construction des géométries à partir
            //-- des coordonnées reprojetées afin d'obtenir les bonnes mesures.
            let polygon = new ol.Feature({
              geometry: new ol.geom.Polygon(coordinates),
            });
            line = new ol.Feature({
              geometry: new ol.geom.LineString(coordinates.pop()),
            });
            //-- Récupération du texte de mesure de la longueur
            //-- dans un format HTML.
            let perimeter = scope.formatLength(line.getGeometry());
            tipFeat.getProperties().styleTemplate.text = ' ';
            if (shapeFeat.properties.showSurface) {
              //-- Récupération du texte de mesure de la superficie
              //-- dans un format HTML.
              let area = scope.formatArea(polygon.getGeometry());
              //-- Formatage du texte sans les codes HTML.
              let re = /<sup>2<\/sup>/gi;
              area = area.replace(re, '²');
              re = /<br>/gi;
              area = area.replace(re, '\n');
              tipFeat.getProperties().styleTemplate.text += area;
              if (shapeFeat.properties.showPerimeter)
                tipFeat.getProperties().styleTemplate.text += '\n';
            }
            if (shapeFeat.properties.showPerimeter)
              tipFeat.getProperties().styleTemplate.text += perimeter;
            linkedTo = shapeFeat;
          }
          else if (shapeFeat.geometry.type == 'LineString') {
            coordinates = shapeFeat.vector.getGeometry().getCoordinates();
            //coordinates = getProjectedCoords(shapeFeat);
            line = new ol.Feature({
              geometry: new ol.geom.LineString(coordinates),
            });
            tipFeat.getProperties().styleTemplate.text = scope.formatLength(
              line.getGeometry()
            );
            linkedTo = shapeFeat;
          }

          return linkedTo;
        }

        /**
         *     Ajout d'une nouvelle forme avec son style
         * sur la carte.
         *
         * @param {[[type]]} params [[Description]]
         */
        function addNewfeature(params) {
          var style, linkedTo;
          if (params.text) {
            if (scope.textIsHtml) {
              params.feature.getProperties().featureType = 'text';
              params.feature.getProperties().styleTemplate.text =
                scope.inputText;
            }
            else if (scope.isSurface || scope.isPerimetre || scope.isLength) {
              var styleGraphic;
              var template = maStyle.getDefaultTemplate('Text');
              template.zoomLevel = scope.map.getView().getZoom();
              template.scale100 =
                $rootScope.xgos.scale4Representation;
              /*params.feature.setStyle(
                                  maStyle.getAnnotationStyle("text",
                                                     scope.getCurrentDefaultStyle("Text")));
                                                     */
              params.feature.setStyle(
                maStyle.getDefaultStyle('text', template)
              );
              styleGraphic = maStyle.getDefaultStyle('text', template);
              styleGraphic.text = 'Label';
              //-- Couleur d'écriture du texte dans la bulle
              styleGraphic.fill = { color: 'rgba(20,20,20,1)' };
              styleGraphic.stroke = { color: 'rgba(220,220,220,0.85)' };
              params.feature.setProperties({ styleGraphic: styleGraphic });

              params.feature.setProperties({ featureType: 'text' });
              params.selectedFeat.properties.showSurface = scope.isSurface;
              params.selectedFeat.properties.showPerimeter = scope.isPerimetre;

              linkedTo = setFeatureDimensionText(
                params.selectedFeat,
                params.feature
              );
            }
            else {
              params.feature.setProperties({ featureType: 'Label' });
              scope.activeAnnotationTool.graphic.textValue = scope.inputText;
              style = maStyle.getAnnotationStyle(
                'Text',
                scope.activeAnnotationTool.graphic
              );
              style.setText(scope.inputText);
              params.feature.setStyle(style);
              params.feature.setProperties({
                styleGraphic: angular.copy(scope.activeAnnotationTool.graphic),
              });
            }
          }
          var feat = new maFeature.Feature(
            params.feature,
            scope.map,
            draw,
            maStyle.getDefaultStyle,
            maStyle.getAnnotationStyle
          );
          if (linkedTo != undefined) {
            linkMafeatures(feat, linkedTo);
          }
          scope.current.features.push(feat);
          feat.setVisibility();
          stopFeatureRemoval(false);
        }

        /**
         * format length output
         * @param {ol.geom.LineString} line
         * @return {string}
         */
        scope.formatLength = (line) => {
          const coordinates = line.getCoordinates();
          let length = 0;
          const sourceProj = scope.map.getView().getProjection();
          for (let i = 0, ii = coordinates.length - 1; i < ii; ++i) {
            const c1 = ol.proj.transform(
              coordinates[i],
              sourceProj,
              'EPSG:4326'
            );
            const c2 = ol.proj.transform(
              coordinates[i + 1],
              sourceProj,
              'EPSG:4326'
            );
            length += wgs84Sphere.haversineDistance(c1, c2);
          }
          return Math.round(length * 100) / 100 + ' ' + 'm';
        };

        /**
         * format area output
         * @param {ol.geom.Polygon} polygon
         * @return {string}
         */
        scope.formatArea = (polygon) => {
          const sourceProj = scope.map.getView().getProjection();
          const geom = /** @type {ol.geom.Polygon} */ (polygon
            .clone()
            .transform(sourceProj, 'EPSG:4326'));
          const coordinates = geom.getLinearRing(0).getCoordinates();
          const area = Math.abs(wgs84Sphere.geodesicArea(coordinates));
          return Math.round(area * 100) / 100 + ' ' + 'm<sup>2</sup>';
        };

        /**
         * Creates a new measure tooltip
         */
        scope.createMeasureTooltip = function () {
          if (scope.measureTooltipElement) {
            scope.measureTooltipElement.parentNode.removeChild(
              scope.measureTooltipElement
            );
          }
          scope.measureTooltipElement = document.createElement('div');
          scope.measureTooltipElement.className = 'tooltip tooltip-measure';
          scope.measureTooltip = new ol.Overlay({
            element: scope.measureTooltipElement,
            offset: [0, -15],
            positioning: 'bottom-center',
          });
          console.log('scope.measureTooltip', scope.measureTooltip);
          scope.map.addOverlay(scope.measureTooltip);
          if (angular.isUndefined(scope.overlayMeasure))
            scope.overlayMeasure = [];
          scope.overlayMeasure.push(scope.measureTooltip);
        };
        /**
         * [fct description]
         * @param  {[type]} evt [description]
         * @return {[type]}     [description]
         */
        var displayMeasure = function (evt) {
          if (evt.dragging) {
            return;
          }
          /** @type {string} */
          //var helpMsg = $filter('translate')('toolbarmeasuredirective.helpMessage');
          /** @type {ol.Coordinate|undefined} */
          var tooltipCoord = evt.coordinate;

          if (scope.sketch) {
            var output;
            var geom = scope.sketch.getGeometry();
            if (geom instanceof ol.geom.Polygon) {
              var line = new ol.Feature({
                geometry: new ol.geom.LineString(geom.getCoordinates().pop()),
              });
              var perim = scope.formatLength(line.getGeometry());
              output = perim + ' ,';
              output += scope.formatArea(geom);
              tooltipCoord = geom.getInteriorPoint().getCoordinates();
            }
            else if (geom instanceof ol.geom.LineString) {
              output = scope.formatLength(geom);
              // output= "length"
              tooltipCoord = geom.getLastCoordinate();
            }
            scope.measureTooltipElement.innerHTML = output;
            scope.measureTooltip.setPosition(tooltipCoord);
          }
        };

        scope.removeDisplayeMeasure = function () {
          scope.map.un('pointermove', displayMeasure);
          if (scope.measureTooltipElement != null) {
            scope.measureTooltipElement.remove();
          }
          scope.measureTooltipElement = null;
        };

        /*
         * Add a shape to the current annotation
         */
        scope.addGraphic = function (type) {
          var action = new ol.interaction.Draw({
            source: draw.source,
            type:
              ['Symbol', 'Label', 'Text'].indexOf(type) !== -1 ? 'Point' : type,
          });
          draw.start(type, [action, draw.getSnap()], false);
          action.on('drawstart', function (evt) {
            scope.sketch = evt.feature;
            scope.createMeasureTooltip();
            scope.map.on('pointermove', displayMeasure);
          });
          /*
           * After drawing, set feature properties
           */
          action.on('drawend', function (event) {
            scope.removeDisplayeMeasure();
            if (scope.popupOverlay != undefined) return;
            var labelTypes = {
              Point: 'Point',
              LineString: 'Line',
              Polygon: 'Polygon',
              text: 'Text',
              Symbol: 'Symbol',
              Label: 'Label',
            };
            var template = maStyle.getDefaultTemplate(type);
            template.zoomLevel = scope.map.getView().getZoom();
            if ($rootScope.xgos.scale4Representation){
              template.scale100 = $rootScope.xgos.scale4Representation;
            }
            var styleGraphic;
            if (
              scope.activeAnnotationTool &&
              scope.activeAnnotationTool.graphic
            ) {
              event.feature.setStyle(
                maStyle.getAnnotationStyle(
                  type,
                  scope.activeAnnotationTool.graphic
                )
              );
              styleGraphic = angular.copy(scope.activeAnnotationTool.graphic);
              if (styleGraphic.Size) {
                styleGraphic.image = {
                  radius: parseFloat(styleGraphic.Size) / 2,
                  scale: 1,
                };
              }
            }
            else {
              event.feature.setStyle(maStyle.getDefaultStyle(type, template));
              styleGraphic = maStyle.getDefaultStyle(type, template);
            }
            if (type == 'Text') {
              draw.stop();
            }
            event.feature.setProperties({
              name: labelTypes[type] || 'Unknown',
              selected: false,
              styleTemplate: template,
              featureType: type,
              styleGraphic: styleGraphic,
            });
            if (type === 'Text') {
              event.feature.setProperties({
                uid: Math.round(Math.random() * 10000000000),
              });
              if (scope.isSurface || scope.isPerimetre || scope.isLength) {
                scope.isSurface = undefined;
                scope.isPerimetre = undefined;
                scope.isLength = undefined;
              }
              if (scope.textIsHtml == undefined) scope.textIsHtml = true;
              scope.inputText = '';
              var astootltip = $filter('translate')('annotations.asTooltip');

              var content =
                '<textarea ng-model=\'inputText\' style=\'height: 55px;width:100%;\'></textarea>';
              content += '<div>';
              content +=
                ' <input  ng-model=\'textIsHtml\'  class=\'checkbox-input1\' type=\'checkbox\' ></input>';
              content +=
                ' <label ng-click=\'checked=!checked\' >' +
                astootltip +
                '</label>';
              content += '</div>';
              setPopups(
                event.feature.getGeometry().getCoordinates(),
                content,
                addNewfeature,
                { feature: event.feature, text: true },
                'ng-disabled=\'inputText==undefined || inputText.length==0\'',
                initTextInput
              );
            }
            else {
              addNewfeature({ feature: event.feature });
            }
          });
          $timeout(function () {
            scope.ensureNoAsideCloseBtn();
          }, 250);
        };

        /*
         * Add a shape to the current annotation
         */
        scope.addShape = function (type) {
          if (type === draw.stop()) {
            return;
          }
          var action = new ol.interaction.Draw({
            source: draw.source,
            type:
              ['Symbol', 'Label', 'text'].indexOf(type) !== -1 ? 'Point' : type,
          });
          draw.start(type, [action, draw.getSnap()], false);

          /*
           * After drawing, set feature properties
           */
          action.on('drawend', function (event) {
            // @RB : shouldn't we keep it active instead ?
            draw.stop();

            var labelTypes = {
              Point: 'Point',
              LineString: 'Line',
              Polygon: 'Polygon',
              text: 'Text',
              Symbol: 'Symbol',
              Label: 'Label',
            };
            var template = maStyle.getDefaultTemplate(type);
            template.zoomLevel = scope.map.getView().getZoom();
            template.scale100 = $rootScope.xgos.scale4Representation;
            event.feature.setStyle(maStyle.getDefaultStyle(type, template));

            event.feature.setProperties({
              name: labelTypes[type] || 'Unknown',
              selected: false,
              styleTemplate: template,
              featureType: type,
            });
            if (type === 'text') {
              event.feature.setProperties({
                uid: Math.round(Math.random() * 10000000000),
              });
            }

            var feat = new maFeature.Feature(
              event.feature,
              scope.map,
              draw,
              maStyle.getDefaultStyle
            );
            scope.current.features.push(feat);
            feat.setVisibility();

            $timeout(function () {
              scope.editFeature(feat);
            });
          });
        };

        /*
         * Edit a Feature (pop up)
         */
        var editFeatureDialog;
        scope.editFeature = function (feat) {
          if (editFeatureDialog) {
            editFeatureDialog.close();
          }
          scope.originalFeature = feat;
          scope.editedFeature = angular.copy(feat.source());
          scope.editedStyle = scope.editedFeature.properties.styleTemplate;

          if (scope.editedFeature.properties.featureType === 'Symbol') {
            scope.imageSelected =
              scope.editedFeature.properties.styleTemplate.image.src;
          }

          editFeatureDialog = extendedNgDialog.open({
            template:
              'js/XG/widgets/mapapp/mapannotations/views/mapeditFeature.html',
            className: 'ngdialog-theme-plain width800 nopadding miniclose',
            closeByDocument: false,
            scope: scope,
            title: $filter('translate')('annotations.widgetname'),
            draggable: true,
          });
        };

        /*
         * Apply the modifications to the currently edited feature
         */
        scope.applyFeatureModifications = function () {
          scope.editedFeature.properties.styleTemplate.scale100 =
            $rootScope.xgos.scale4Representation;
          scope.editedFeature.properties.styleTemplate.zoomLevel = scope.map
            .getView()
            .getZoom();
          scope.originalFeature.update(scope.editedFeature);
          editFeatureDialog.close();
          scope.originalFeature = null;
          scope.editedFeature = null;
          scope.editedStyle = null;
          editFeatureDialog = null;
        };

        /**
         * Convert input degrees if value, else convert slider to input value (decimal is PI * 2)
         */
        scope.convertDegrees = function (value, fromSlider) {
          // convertedRotation
          if (angular.isDefined(value) && !fromSlider)
            scope.editedStyle.image.rotation = (6.28318530718 * value) / 360;
          else if (angular.isDefined(value) && fromSlider)
            $timeout(function () {
              scope.editedStyle.image.convertedRotation = Math.round(
                scope.editedStyle.image.rotation * (180 / Math.PI)
              );
            });
        };

        /*
         * Set image for annotation
         */
        scope.setImage = function (data) {
          //scope.editedFeature.properties.styleTemplate.image.src = data.url + '/' + data.img;
          //scope.imageSelected = data.url + '/' + data.img;
          scope.activeAnnotationTool.graphic.image = {
            src: data.url + '/' + data.img,
            scale: 1,
          };
        };

        /**
         *     Créer une nouvelle annotation, et
         * l'ajouter à la liste des annotations'
         */
        scope.saveNewAnnot = function () {
          //-- Créer la nouvelle annotation vide
          var created = new maAnnotation.Annotation(
            { data: { features: [] } },
            scope,
            AnnotationFeatures,
            AnnotationSelection,
            draw
          );
          created.name = scope.new_annot.name;
          //-- Ajouter la nouvelle annotation à
          //-- notre liste d'annotations existantes.
          scope.annotations.push(created);
          //-- Enregistrer la nouvelle annotation vide
          //-- dans les paramètres en base.
          created.save();
          //-- Ouvrir le panneau de dessin de la nouvelle annotation
          //-- afin de commncer à travailler dessus.
          ctrler.open(created);
          scope.draw.enabled = true;
          //-- Réinitialiser le nom de nouvelle annotation.
          scope.new_annot.name = '';
        };

        /**
         *     Vérifier si le nom pour
         * la nouvelle annotation existe déjà.
         */
        scope.annotNameExists = function () {
          var ind;

          if (!scope.annotations) return true;

          for (ind = 0; ind < scope.annotations.length; ind++) {
            if (scope.annotations[ind].name == scope.new_annot.name)
              return true;
          }

          return false;
        };

        scope.$on('aside.show', function (e, aside) {
          if (
            angular.isDefined(aside.$options.annoClass) &&
            aside.$options.annoClass != ''
          ) {
            aside.$element.addClass(aside.$options.annoClass);
          }
          // si l'annotation a une infobulle périmètre/surface, on supprime l'annotation dupliquée
          // Il serait plus judicieux d'éviter de dupliquer l'objet au clic
          // pour ne pas avoir à le supprimer ici
          if (scope.current.features.length > 1) {
            draw.features.clear();
          }
        });

        var baseZoom;
        scope.map.on('rb_movestart', function () {
          baseZoom = scope.map.getView().getZoom();
        });

        var first = true;
        scope.onNewScaleForRes = function () {
          const scale =  gaJsUtils.resolutionToScale(
            scope.map.getView().getResolution());
          var zoomDiff = baseZoom - scope.map.getView().getZoom();
          scope.drawSourceClearVectorText(true, scale, zoomDiff);
        };

        $rootScope.$on('newScaleForResolution', function () {
          if (first) {
            //$timeout (onNewScaleForRes, 2000);
            first = false;
          }
          else scope.onNewScaleForRes();
        });

        // fire movestart avant ol 4.2.0
        // https://stackoverflow.com/a/31897512
        function onpropertychange() {
          scope.map.dispatchEvent('rb_movestart');
          var view = scope.map.getView();
          view.un('propertychange', onpropertychange);
          scope.map.on('moveend', function () {
            view.on('propertychange', onpropertychange);
          });
        }
        scope.map.getView().on('propertychange', onpropertychange);

        function closeWidget(arg) {
          if (!arg.active) {
            if (scope.activeAnnotationTool && scope.activeAnnotationTool.active)
              deactivateCurrentTool(false);
            draw.stop();
          }
        }
        scope.$on(
          'openCloseTools_mapgcannotationswidget',
          function (event, arg) {
            closeWidget(arg);
          }
        );
        scope.$on('closeTools_mapgcannotationswidget', function (event, arg) {
          closeWidget(arg);
        });

        scope.$on('userEnteredEscape', function () {
          stopFeatureRemoval(true);
          showHideContextualMenu('none');
          unselectCurrentFeature();
        });

        scope.ensureNoAside = function () {
          var ind;
          var elts = document.getElementsByClassName(
            'aside ng-scope right am-fade-and-slide-right'
          );

          for (ind = 0; ind < elts.length; ind++) {
            elts[ind].style.display = 'none';
          }
        };
        scope.ensureNoAsideCloseBtn = function () {
          var ind;
          var elts = document.getElementsByClassName('aside-footer');

          for (ind = 0; ind < elts.length; ind++) {
            elts[ind].style.display = 'none';
          }
        };

        /**
         * au clic dans la case perimetre/surface
         * prépare l'affichage de l'infobulle (cf. addNewFeature)
         */
        scope.displayPerimAreaTooltip = () => {
          const feat = scope.selectedFeature;
          if (scope.isSurface || scope.isPerimetre) {
            removeTooltip();
          }
          scope.isPerimetre = scope.tooltipConf.isPerimetre;
          scope.isSurface = scope.tooltipConf.isSurface;
          if (scope.tooltipConf.isPerimetre || scope.tooltipConf.isSurface) {
            let center = [];
            const isLinestring = feat.getGeometry().getType() == 'LineString';
            if (isLinestring) {
              center = feat.getGeometry().getCoordinateAt(0.5);
            }
            else {
              center = feat.getGeometry().getInteriorPoint();
            }
            const type = 'Text';
            const template = maStyle.getDefaultTemplate(type);
            template.zoomLevel = scope.map.getView().getZoom();
            template.scale100 = $rootScope.xgos.scale4Representation;
            const newFeat = new ol.Feature({
              geometry: new ol.geom.Point(
                isLinestring ? center : center.getCoordinates()
              ),
              properties: { selected: false },
              styleTemplate: template,
            });
            feat.properties = { showPerimeter: false, showSurface: false };
            const selectedFeat = scope.current.getMaFeatureFromFeat(feat);
            const params = {
              feature: newFeat,
              text: true,
              selectedFeat: selectedFeat,
            };
            addNewfeature(params);
            // supprime l'objet dupliqué pour avoir l'infobulle au 1er plan
            draw.features.clear();
          }
          else {
            removeTooltip();
          }
        };

        /**
         * supprime l'infobulle perimetre/area lorsque la case est décochée
         *
         */
        function removeTooltip() {
          //scope.current.clearAnnotationPolygon();
          const features = scope.current.features;
          const selectedFeature = scope.current.features.find(
            scope.selectedFeature
          );
          for (let i = 0; i < features.length; i++) {
            if (
              features[i].properties.featureType == 'text' &&
              selectedFeature.properties.linkId ===
                features[i].properties.linkId
            ) {
              selectedFeature.properties.showPerimeter =
                scope.tooltipConf.isPerimetre;
              selectedFeature.properties.showSurface =
                scope.tooltipConf.isSurface;
              draw.source.removeFeature(features[i].vectorText);
              scope.current.features.remove(features[i]);
            }
          }
        }
      }, // link function
    };
  };

  mapannotationswidget.$inject = [
    '$rootScope',
    '$filter',
    '$timeout',
    '$q',
    'gclayers',
    'gcInteractions',
    'ParametersFactory',
    'extendedNgDialog',
    '$aside',
    '$compile',
    'maFeature',
    'maAnnotation',
    'SridFactory',
    '$http',
    'ConfigFactory',
    'maStyle',
    'GeometryFactory',
    'maUtils',
    'gaJsUtils'
  ];
  return mapannotationswidget;
});
