/*global define, $ */
define(['ol3js', 'angular'], function(ol, angular) {
  'use strict';

  /*
   * Provide important data
   */
  var draw;
  function init(drawObj) {
    draw = drawObj;
  }

  /* -------------------------------------------------------------- *\
     * Helper functions
    \* -------------------------------------------------------------- */

  /*
   * Generate an OpenLayers style from a style template
   */
  function getDefaultStyle(type, styleTemplate) {
    var style = {};
    switch (type) {
      case 'Point':
      case 'text':
        style.image = new ol.style.Circle({
          radius: styleTemplate.image.radius,
          fill: new ol.style.Fill(styleTemplate.image.fill),
        });
        break;
      case 'Symbol':
        style.image = new ol.style.Icon({
          anchor: [0.5, 46],
          anchorXUnits: 'fraction',
          anchorYUnits: 'pixels',
          opacity: 1,
          src: styleTemplate.image.src,
          scale: styleTemplate.image.scale,
          rotation: styleTemplate.image.rotation,
        });
        break;
      case 'Label':
        style.text = new ol.style.Text({
          text: styleTemplate.text.text,
          font: styleTemplate.text.font,
          //offsetX: styleTemplate.text.offsetX,
          //offsetY: styleTemplate.text.offsetY,
          rotation: styleTemplate.text.rotation,
          fill: new ol.style.Fill(styleTemplate.text.fill),
          stroke: new ol.style.Stroke(styleTemplate.text.stroke),
        });
        break;
      case 'Polygon':
        style.fill = new ol.style.Fill(styleTemplate.fill);
        style.stroke = new ol.style.Stroke(styleTemplate.stroke);
        break;
      case 'LineString':
        style.stroke = new ol.style.Stroke(styleTemplate.stroke);
        break;
    }
    return new ol.style.Style(style);
  }

  /*
   * Create a feature from a feature template
   */
  function createFeature(source) {
    var type = source.properties.featureType;
    var style = source.properties.styleTemplate;

    // Secure the properties
    if (type === 'text' && !source.properties.uid) {
      source.properties.uid = Math.round(Math.random() * 100000000);
    }
    if (type === 'Polygon' && style.opacity === undefined) {
      style.opacity = 100;
    }

    // Instantiate and style the feature
    var reader = new ol.format.GeoJSON();
    var layerFt = reader.readFeature(source);
    if (type !== 'text') {
      layerFt.setStyle(getDefaultStyle(type, style));
    }
    return layerFt;
  }

  /*
   * Change the opacity of the given style; it is currently the
   * only way of hiding an OpenLayers feature without removing it
   */
  function updateStyleOpacity(style, opacity) {
    function setOpacity(dest, prop) {
      if (!(dest[prop] && dest[prop]() && dest[prop]().getColor())) {
        return; // The property must have a color
      }
      // ol.color.asArray has a cache; we MUST NOT use it directly
      var color = [].concat(ol.color.asArray(dest[prop]().getColor()));
      color[3] = opacity;
      dest[prop]().setColor(color);
    }
    // Update image style if any (type: 'Symbol' or 'Point')
    if (style.getImage()) {
      style.getImage().setOpacity(opacity);
    }
    // Update text style if any (type: 'Label')
    if (style.getText()) {
      var txt = style.getText();
      setOpacity(txt, 'getFill');
      setOpacity(txt, 'getStroke');
    }
    // Update basic styles if any
    setOpacity(style, 'getFill');
    setOpacity(style, 'getStroke');
    return style;
  }

  /* -------------------------------------------------------------- *\
     * Helper classes
    \* -------------------------------------------------------------- */

  function Feature(template, map) {
    var self = this;

    // Display the overlay for 'text' features
    self.setPopup = function() {
      if (self.overlay) {
        map.removeOverlay(self.overlay);
        self.overlay = null;
      }
      if (!self.visible) {
        return;
      }
      // Visible: create the overlay
      var style = self.properties.styleTemplate;
      var css = [
        'background: ' + style.backgroundColor,
        'color: ' + style.textColor,
        'opacity: ' + style.opacity / 100,
      ].join(';');

      var uniqid = 'textpopup_' + Date.now();
      var tpl =
        '<div id="' +
        uniqid +
        '" class="ol-popup annotationPopup" style="' +
        css +
        '">' +
        '<div class="popup-content"></div>' +
        '</div>';
      angular
        .element(document)
        .find('body')
        .eq(0)
        .append(tpl);

      var container = document.getElementById(uniqid);
      $(container)
        .find('.popup-content')
        .html(style.text);

      self.overlay = new ol.Overlay({
        element: container,
        autoPan: true,
        autoPanAnimation: { duration: 250 },
      });
      self.overlay.set('origin', 'annotations');
      self.overlay.set('uid', self.properties.uid);
      map.addOverlay(self.overlay);
      self.overlay.setPosition(self.vector.getGeometry().getCoordinates());
    };

    // Set the visibility of a single feature
    self.setVisibility = function() {
      var opacity = Number(!!self.visible);
      // Retro-compatibility: 'text' annotations had no own style (null)
      var style = self.vector.getStyle() || draw.layer.getStyle();

      if (self.properties.featureType === 'text') {
        self.setPopup();
      }
      if (
        self.properties.featureType === 'Polygon' &&
        self.visible &&
        typeof self.properties.styleTemplate.opacity === 'number'
      ) {
        opacity = self.properties.styleTemplate.opacity / 100;
      }
      self.vector.setStyle(updateStyleOpacity(style, opacity));
    };

    self.showHide = function() {
      self.visible = !self.visible; // toggle
      self.setVisibility();
    };

    self.zoomTo = function() {
      var extent = self.vector.getGeometry().getExtent();
      map.getView().fit(extent, map.getSize());
    };

    self.source = function() {
      return template;
    };

    self.update = function(newTemplate) {
      template = newTemplate;
      // Clean the template
      delete template.properties.selected;
      if (template.visible === undefined) {
        template.visible = true;
      }

      var found = draw.source.getFeatures().some(function(olFeat) {
        if (self.vector === olFeat) {
          draw.source.removeFeature(olFeat);
          return true;
        }
      });
      angular.extend(self, template);
      self.template = template;
      self.vector = createFeature(template);
      if (found) {
        draw.source.addFeature(self.vector);
        self.setVisibility();
      }
    };

    // Instanciation
    if (template instanceof ol.Feature) {
      // The template is the vector feature
      self.vector = template;
      var writer = new ol.format.GeoJSON({});
      template = writer.writeFeatureObject(self.vector);
      template.visible = true;
      angular.extend(self, template);
    } else {
      // The template is a KIS template
      self.update(template);
    }

    // Unselected by default
    self.properties.selected = false;
    self.vector.set('selected', false);
  }

  /*
   * Manage the features of an annotation
   */
  function AnnotationFeatures(featureTemplates, map) {
    var self = this; // We extend Array

    // Remove a feature
    self.remove = function(feat) {
      var index = self.indexOf(feat);
      if (index !== -1) {
        draw.source.getFeatures().some(function(olFeat) {
          if (olFeat === feat.vector) {
            draw.source.removeFeature(feat.vector);
            return true;
          }
        });
        self.splice(index, 1);
      }
      return index;
    };

    self.replace = function(a, b) {
      var index = self.remove(a);
      if (index !== -1) {
        self.splice(index, 0, b);
      }
      return index;
    };

    self.vectors = function() {
      return self.map(function(feat) {
        return feat.vector;
      });
    };

    self.find = function(vector) {
      var index = -1;
      self.some(function(feat, i) {
        if (feat.vector === vector) {
          index = i;
          return true;
        }
      });
      return self[index];
    };

    // Instantiate the OpenLayers features
    featureTemplates.forEach(function(template) {
      self.push(new Feature(template, map));
    });
  }
  AnnotationFeatures.prototype = new Array();

  /*
   * Manage a selection of features
   */
  function AnnotationSelection(features) {
    var self = this; // We extend Array
    self.allChecked = false;

    self.add = function(feat) {
      if (!feat) {
        return;
      }
      feat.properties.selected = true;
      feat.vector.set('selected', true);
      if (self.indexOf(feat) === -1) {
        self.push(feat);
      }
    };

    self.remove = function(feat) {
      var index = self.indexOf(feat);
      if (index !== -1) {
        feat.properties.selected = false;
        feat.vector.set('selected', false);
        self.splice(index, 1);
      }
    };

    self.toggle = function(feat) {
      var selected = !self.allChecked;
      if (feat) {
        // If we toggle a single feature
        if (features.indexOf(feat) === -1) {
          return; // Feature doesn't exist
        }
        selected = !feat.properties.selected;
        if (selected) {
          self.add(feat);
        } else {
          self.remove(feat);
        }
      } else {
        // If we toggle multiple features
        self.allChecked = selected;
        if (selected) {
          features.forEach(self.add);
        } else {
          self.uncheck();
        }
      }
    };

    self.showHide = function() {
      var visible = !self.isVisible(); // toggle
      self.forEach(function(feat) {
        feat.visible = visible;
        feat.setVisibility();
      });
    };

    self.isVisible = function() {
      var visibleCount = 0;
      self.forEach(function(feat) {
        visibleCount += feat.visible ? 1 : -1;
      });
      return visibleCount >= 0;
    };

    self.uncheck = function() {
      while (self.length > 0) {
        self.remove(self[0]);
      }
    };

    self.erase = function() {
      self.forEach(function(feat) {
        feat.visible = false;
        feat.setVisibility();
        features.remove(feat);
      });
      self.length = 0;
    };
  }
  AnnotationSelection.prototype = new Array();

  return {
    init: init,
    Feature: Feature,
    AnnotationFeatures: AnnotationFeatures,
    AnnotationSelection: AnnotationSelection,
    getDefaultStyle: getDefaultStyle,
    updateStyleOpacity: updateStyleOpacity,
  };
});
