define(['ol3js'], function(ol) {
  'use strict';


  // Options for formatting GeoServer legends
  var wmsOptions =
    'legend_options=' +
    [
      'fontName:Open Sans',
      'fontAntiAliasing:true',
      'fontColor:0x000000',
      'fontSize:12',
      'bgColor:0xFFFFFF',
      'forceLabels:on',
    ].join(';');

  /*
   * Make sure URLs are absolute (cross-browser)
   * https://stackoverflow.com/a/24437713/3553657
   */
  function qualifyURL(url) {
    var a = document.createElement('a');
    a.href = url;
    return decodeURIComponent(a.cloneNode(false).href);
  }

  /*
   * Service generating the map legend, and filling scopes with them
   * - add(scope)       register a scope so that it will contain legends
   * - remove(scope)    unregister a scope and delete its legends
   * - refresh()        refresh the legend for all scopes
   * - generate(scope)  fill the scope without registering it
   *
   * Any object can be used as the scope. There are two optional parameters:
   * - legendLayers: array of KIS layers (default: KIS operational layers)
   * - ignoreScale: truthy if layers shouldn't be filtered out by scale (default: false)
   *
   * The scope is filled with a 'themes' object. Keys are the layer themes,
   * values are an array of legend for this theme (one item per layer):
   * - label: layer name
   * - style: style name
   * - image: url for the legend
   */
  function legendService($rootScope, $filter, $timeout, gclayers,
    PrintLegendFactory, $http, $q, ogcFactory, gaJsUtils) {
    var scopes = [];
    const map = $rootScope.xgos.getMapAppMap();



    const getLegendOfCategories = (layer, leg, PrintLegendFactory, map, defer,
      ignoreMissingObjects) => {
      const ftisAndStyles = {};
      ftisAndStyles.ftiAndStyle = [];
      const fti = layer.get('fti');
      if (fti)
        ftisAndStyles.ftiAndStyle.push({
          ftiUid: fti.uid,
          style:  layer.get('style'),
        });

      const currentRes =  map.getView().getResolution();
      const currentScale = ignoreMissingObjects ? -1
        : gaJsUtils.resolutionToScale(currentRes);
      const bbox = ignoreMissingObjects
        ? [-1,-1,-1,-1] : map.getView().calculateExtent();
      const bboxsrs = ignoreMissingObjects
        ? '' : map.getView().getProjection().getCode();

      return PrintLegendFactory.getLegendForLayers(bbox[0], bbox[2], bbox[1],
        bbox[3], bboxsrs, Math.round( currentScale), ftisAndStyles).then(
        (res) => {
          if (res.data && angular.isArray(res.data) && res.data.length > 0) {
            leg.legendDetails = res.data[0].classes;
            var a = document.createElement('a');
            var iClass;
            for (iClass = 0; iClass < leg.legendDetails.length; iClass++) {
              a.href = leg.legendDetails[iClass].icons[0];
              leg.legendDetails[iClass].image = decodeURIComponent(
                a.cloneNode(false).href
              );
            }
            defer.resolve(leg);
          }
        });
    };


    function getArcGISLayerLegend(layer,themeTitle, localThemes) {
      $http.get(layer.legend).then(function (res) {
        if (res.data && res.data.layers && res.data.layers.length > 0) {
          layer.legendData = res.data.layers[0];
          localThemes[themeTitle].push(layer);
        }
      });
    }


    function generate($scope) {
      $scope.errorNoLegend = $filter('translate')('legend.error');


      /**
       * Vérifie que la couche est dessinée en fonction des échelles minimum et
       *  maximum d'affichage.
       *
       * @param {*} layer
       * @returns
       */
      let isVisible4ScaleRange = (layer) => {
        // Resolution limits
        let minRes = layer.getMinResolution() || 0;
        let maxRes = layer.getMaxResolution() || Infinity;
        let curRes = map.getView().getResolution();
        // Scale limits
        let fti = layer.get('fti');
        let minScl, maxScl, curScl;
        if (fti && fti.type == 'esri') {
          minScl = fti.ogcProperties.maxScale;
          maxScl = fti.ogcProperties.minScale != 0
            ? fti.ogcProperties.minScale : Infinity;
        }
        else {
          minScl = layer.minScale || 0;
          maxScl = layer.maxScale || Infinity;
        }
        curScl = gaJsUtils.resolutionToScale(curRes);

        return curRes >= minRes  // - too low resolution
          && curRes <= maxRes // - too high resolution
          && curScl >= minScl // - too low scale
          && curScl <= maxScl; // - too high scale
      };


      let cfosDesc;
      /**
       * On ne met dans légende que les couches dont un objet
       * au moins est dessiné à l'écran. On vérifie ici que des objets
       * de la couche sont dans la zone dessinée.
       *
       * Par souci de performance, on prépare une seule requête qui regroupe
       * l'ensemble des composants pour lesquels on veut savoir
       * si des objets sont sur la carte.
       * Pour ce faire, on sauvegarde le "defer" de chaque layer,
       * pour résoudre la promesse de chaque layer à réception de la réponse
       * du service..
       *
       * @param {*} def
       * @param {*} layer
       */
      let checkFeaturesOnScreen = (def, layer, first, last) => {
        let fti;
        if (first) {
          //-- Premier layer de la liste, on initialise la préparation
          //-- de la requête HTTP.
          cfosDesc = { uids: ''};
        }

        if (layer) {
          //-- Layer à prendre en compte pour la requête HTTP.
          //-- Si le dernier layer n'est pas visible à cause de la visibilité
          //-- par échelle par exemple, alors cette fonction est appelée
          //-- avec layer positionné à undefined, ainsi
          //-- on lance quand même la requête.
          fti = layer.get('fti');
          if (cfosDesc.uids != '') cfosDesc.uids += ',';
          cfosDesc.uids += fti.uid;
          cfosDesc[fti.uid] = { def: def, layer: layer };
        }

        //-- N'appeler la requête qu'une fois le dernier layer atteint et
        //-- à condition qu'il y ait des layers à requêter.
        if (!last || cfosDesc.uids=='') {
          return;
        }

        let bbox = map.getView().calculateExtent();
        var selGeomCql = 'POLYGON((' + bbox[0] + ' ' + bbox[1] + ','
          + bbox[2] + ' ' + bbox[1] + ','
          + bbox[2] + ' ' + bbox[3] + ','
          + bbox[0] + ' ' + bbox[3] + ','
          + bbox[0] + ' ' + bbox[1]
          + '))';

        let spatialClause = 'INTERSECTS(geom, ' + selGeomCql + ')';
        ogcFactory.getfeatures(
          'GetCount', 'WFS', '1.0.0', cfosDesc.uids, 'json',
          map.getView().getProjection().getCode(), spatialClause
        ).then(
          (res) => {
            //-- Résolution de la promesse de chaque layer.
            for (let ftiUid in res.data) {
              cfosDesc[ftiUid].def.resolve({
                layer: cfosDesc[ftiUid].layer,
                visible: res.data[ftiUid] != 0
              });
            }
          },
          () => {
            for (let ftiUid in cfosDesc) {
              if (ftiUid != 'uids') {
                cfosDesc[ftiUid].def.resolve({ visible: false });
              }
            }
          }
        );
      };


      /**
       *
       * @param {*} def : defer pour résoiudre la promesse
       * @param {*} visi : boolean layer visible true , false, sinon
       * @param {*} first : est-ce le premier layer de la liste
       * @param {*} last ; est-ce le dernier layer de la liste
       * @param {*} layer : layer en cours de traitement
       */
      let resolveVisible = (def, visi, first, last, layer) => {
        checkFeaturesOnScreen (def, undefined, first, last);
        def.resolve({ visible: visi, layer: layer });
      };


      /*
       * Tell whether a KIS operational layer should be used
       */
      function isVisible(layer,first,last) {
        let def = $q.defer();

        if (!layer || (layer.type || '').indexOf('WMS') !== 0) {
          resolveVisible(def, false, first, last);
          return def.promise; // Layer type must begin with 'WMS'
        }

        // Layer is hidden if any of these is true
        // - is hidden
        // - has no data
        // - is completely transparent
        if (!layer.getVisible() || !layer.getSource() || !layer.getOpacity()) {
          resolveVisible(def, false, first, last);
          return def.promise;
        }

        // Requested by PALR (La Rochelle), otherwise WebBackGround would be rejected
        if (layer.theme === 'WebBackGround') {
          var url = (layer.getSource().getUrls() || [])[0];
          if (!url || url.indexOf('/geoserver/') === -1) {
            resolveVisible(def, false, first, last);
            return def.promise;
          }
          var params = layer.getSource().getParams();
          layer.legend =
            url +
            [
              '&SERVICE=WMS&REQUEST=GetLegendGraphic',
              '&VERSION=' + params.VERSION,
              '&FORMAT=image/png&WIDTH=20&HEIGHT=20',
              '&LAYER=' + params.LAYERS,
            ].join('');
        }
        let visible = true;
        if (!$scope.ignoreScale) {
          visible = isVisible4ScaleRange(layer);
        }
        if (visible && !$scope.ignoreMissingObjects) {
          checkFeaturesOnScreen(def, layer, first, last);
        }
        else {
          // Return false if hideOn contains true
          resolveVisible(def, visible, first, last, layer);
        }
        return def.promise;
      }


      /*
       * Generate a legend object from a KIS operational layer
       */
      const getLegend = (layer,themeTitle, localThemes, ignoreMissingObjects) => {
        const defer = $q.defer();
        const alias = $filter('translate')('features.' + layer.name + '.alias');
        const leg = {};
        leg.label = (alias.indexOf('features.') === -1) ? alias : layer.label;
        leg.style = layer.style || '';
        leg.name = layer.name;
        if (layer.fti.type == 'esri') {
          getArcGISLayerLegend(layer, themeTitle, localThemes);
          leg.image = qualifyURL(layer.legend);
          defer.resolve(leg);
        }
        else {
          getLegendOfCategories(layer, leg, PrintLegendFactory, map,
            defer, ignoreMissingObjects).then(() => {
            //-- On arrive dans ce "then" uniquement si le style permet
            //-- le dessin des objets dans la zone et à l'échelle de la carte.
            leg.image = qualifyURL(
              layer.legend + (layer.style ? '&STYLE=' + layer.style : '') +
                '&LANGUAGE=' + $rootScope.xgos.portal.lang +
                '&' + wmsOptions
            );
          }
          );
        }

        return defer.promise;
      };


      let handleVisibleLayers = (layers, parentTheme, localThemes,
        localThemesUi, ignoreMissingObjects) => {
        let iLyr;
        for (iLyr = 0; iLyr < layers.length; iLyr++) {
          isVisible(layers[iLyr],iLyr==0,iLyr==layers.length-1).then(
            (res) => {
              if (res.visible) {
                handle(res.layer, res.layer.theme || parentTheme, localThemes,
                  localThemesUi, ignoreMissingObjects);
              }
            }
          );
        }
      };

      /*
       * Unfold OpenLayers group or extract the legend
       */
      function handle(layer, parentTheme, localThemes, localThemesUi,
        ignoreMissingObjects) {
        if (layer instanceof ol.layer.Group) {
          handleVisibleLayers(layer.getLayers().getArray(),parentTheme,
            ignoreMissingObjects);
        }
        else {
          var title = $filter('translate')(
            'theme.' + layer.theme || parentTheme
          );
          if (!localThemes[title]) {
            localThemes[title] = [];
            localThemesUi.push(
              { themeName: title, layers: localThemes[title] }
            );
          }
          if (layer.fti.type == 'esri')
            getLegend(layer, title, localThemes, ignoreMissingObjects);
          else {
            getLegend(layer, undefined, localThemes, ignoreMissingObjects).then(
              (legend) => {
                //-- On arrive dans ce "then" uniquement si le style permet
                //-- le dessin des objets dans la zone et à l'échelle
                //-- de la carte. Et ce, de sorte que si le tableau
                //-- localThemes[title] est vide au final, un ng-show angular
                //-- empêchera l'affichage du thème vide.
                localThemes[title].push(legend);
                localThemes[title].sort((a, b) => {
                  const nameA = a.name.toLowerCase(),
                    nameB = b.name.toLowerCase();
                  if (nameA < nameB)
                    //sort string ascending
                    return -1;
                  if (nameA > nameB) return 1;
                  return 0; //default return value (no sorting)
                });
                console.log(localThemes[title]);
              }
            );

          }
        }
      }

      /*
       * Extract legends from the list of operational layers
       */
      var layers;
      //-- Je ne sais pas pourquoi on fait cela, mais il y a le cas où
      //-- il s'agit d'un fond de plan (groupe de layer geoserver) et où
      //-- il n'y a pas de légende.
      var layer0 = map.getLayers().item(0); // No better way in KIS...  !!!
      if (layer0.theme == 'WebBackGround')
        layers = $scope.legendLayers || gclayers.getOperationalLayer();
      // No better way in KIS...
      else
        // No better way in KIS...
        layers = $scope.legendLayers ||
          [].concat(gclayers.getOperationalLayer(),layer0);
      if (layers instanceof ol.Collection) {
        layers = layers.getArray();
      }

      let localThemes = {};
      let localThemesUi = [];

      handleVisibleLayers(layers, '', localThemes, localThemesUi,
        $scope.ignoreMissingObjects);

      $scope.themes = localThemes;
      $scope.themesUi = localThemesUi;
    }


    function refresh() {
      $timeout(function() {
        scopes.forEach(generate);
      });
    }

    function add($scope) {
      if (scopes.indexOf($scope) !== -1) {
        return;
      }
      scopes.push($scope);
      $scope.$watch('ignoreScale', function(newVal, oldVal) {
        if (newVal != oldVal) generate($scope);
      });
      $scope.$watch('ignoreMissingObjects', function (newVal, oldVal) {
        if (newVal != oldVal) generate($scope);
      });
      generate($scope);
    }

    function remove($scope) {
      var index = scopes.indexOf($scope);
      if (index !== -1) {
        scopes.splice(index, 1);
        delete $scope.themes;
      }
    }

    // Initialize the service
    map.on('moveend', refresh);

    $rootScope.$on('gcOperationalLayerChange', refresh);
    $rootScope.$on('gcBackGroundLayerChange', refresh);

    return {
      add: add,
      remove: remove,
      refresh: refresh,
      generate: generate,
      getArcGISLayerLegend: getArcGISLayerLegend
    };
  }

  legendService.$inject = ['$rootScope','$filter','$timeout','gclayers',
    'PrintLegendFactory','$http', '$q', 'ogcFactory', 'gaJsUtils'
  ];

  return legendService;
});
