'use strict';
define(function () {
  let ReportIndicatorFactory = function (
      $http,
      $q,
      $rootScope,
      PortalsFactory
  ) {
    let ReportIndicatorFactory = {};

    let dataStoreName = '';

    let portalid = angular.module('gcMain').portalid;
    if (portalid === undefined || portalid === '') {
      portalid = PortalsFactory.getPortalId();
    }
    const appName = localStorage.getItem('app');

    function setDataStoreName(value) {
      dataStoreName = value;
      return true;
    }

    /**
     * Récupère la liste des noms de rapports Jasper d'une application dans le repo d'un portail
     * @returns {Promise} contenant une liste de DTO {name,creation}
     */
    function getAllReportNames(appname, isAction = false) {
      if (!appname) {
        //-- Quand cette méthode est appelée depuis l'administartion,
        //-- il n'y a pas d'application choisie pour la page web.
        //-- L'application vient d'un élément de choix dans
        //-- la page d'administration et on ne peut pas compter
        //-- sur le mécanisme d'interception, qui mettra "unknown"
        //-- à la place de "{appname}".
        appname = '{appname}';
      }
      return $http({
        url: '/services/{portalid}/indicator/' + appname + '/getreportnames?' +
            'isAction=' +
            isAction,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });
    }

    /**
     * Récupère les rapports Jasper de type "action" ou bien
     * "tableaux de bord" de toutes les applications d'un portail
     * @param appname nom de l'application dans le cas d'une exécution
     *     depuis le widget de tableaux de bord
     *     (redirection vers getAllReportNames())
     * @param isAction est true pour récupérer les rapport "action"
     *     (false pour les rapports "tableaux de bord")
     * @returns {Promise}
     */
    function getAllReportNamesFromApplication(appname, isAction = false) {
      if (appname) {
        return getAllReportNames(appname, isAction);
      } else {
        return $http({
          url:
              '/services/{portalid}/indicator/' +
              isAction +
              '/getportalreportnamesbytype',
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
            encoding: 'UTF-8',
          },
        });
      }
    }

    /**
     * Request the report generation and the export in pdf and jpeg
     * @param {type} jasperFileName
     * @param {type} jsonParameters
     * @param datastore nom de la datasource
     * @param exportFormat format dans lequel doit être généré le rapport de l'indicateur (pdf/rtf/xls)
     * @param isAction est true si le rapport est situé dans le dossier "/reports_action". Argument optionnel
     * @returns {Promise}
     */
    function generateReport(
        jasperFileName,
        jsonParameters,
        datastore,
        exportFormat,
        isAction = false
    ) {
      if (!datastore) {
        datastore = dataStoreName;
      }
      var data = '' + JSON.stringify(jsonParameters);
      return $http.post(
          '/services/{portalid}/indicator/{appname}/generate' +
          '?jasperFileName=' + jasperFileName +
          '&dataStoreName=' + datastore +
          '&exportFormat=' + exportFormat +
          '&isAction=' + isAction,
          data
      );
    }

    /**
     * Request the report generation and the export in pdf and jpeg
     * @param {type} jasperFileName
     * @param dsName nom de la source de données
     * @param exportFormat format d'export du rapport Jasper ('pdf' par défaut)
     * @param isAction est true si le rapport est de la catégorie "action". Argument optionnel
     * @param {{}} jsonParameters
     * @returns {Promise}
     */
    function generateReportExtended(
        jasperFileName,
        dsName,
        jsonParameters,
        exportFormat,
        isAction = false
    ) {
      //code pour la démo à ne pas merger
      console.log(
          '%c CODE DEMO CD93, NE DOIT PAS ETRE MERGE AILLEURS',
          'background: red; color: white; display: block;'
      );
      if (dataStoreName === 'Reseau_Edit') {
        dsName = 'Report';
      }
      //fin code de démo

      const data = '' + JSON.stringify(jsonParameters);
      return $http.post(
          '/services/{portalid}/indicator/{appname}/generate' +
          '?jasperFileName=' +
          jasperFileName +
          '&dataStoreName=' +
          dsName +
          '&exportFormat=' +
          exportFormat +
          '&isAction=' +
          isAction,
          data
      );

    }

    /**
     * Request the report generation and the export in pdf and jpeg
     * @param {type} jasperFileName nom du fichier jasper
     * @param dsName nom de la datasource
     * @param isAction est true si le rapport est situé dans le dossier "/reports_action"
     * @param {type} jsonParameters paramètres d'exécution du rapport
     * @returns {Promise}
     */
    function generateMultipleReportExtended(
        jasperFileName,
        dsName,
        jsonParameters,
        isAction = false
    ) {
      const data = '' + JSON.stringify(jsonParameters);
      return $http.post(
          '/services/{portalid}/indicator/{appname}/generate/on/component' +
          '?jasperFileName=' +
          jasperFileName +
          '&dataStoreName=' +
          dsName +
          '&isAction=' +
          isAction,
          data
      );

    }

    /**
     * Request the report generation and the export in pdf and jpeg
     * @param {type} jasperFileName
     * @param data
     * @returns {Promise}
     */
    function uploadReport(jasperFileName, data) {
      var promise = $http.post(
          '/services/{portalid}/indicator/{appname}/upload/report' +
          '?jasperFileName=' +
          jasperFileName,
          data,
          {
            transformRequest: angular.identity,
            headers: {'Content-Type': undefined},
          }
      );

      return promise;
    }

    /**
     * Get the generated report in PDF format
     * @param {type} jasperFileName
     * @param exportFormat format dans lequel est exporté le rapport (PDF, XLS ou RTF)
     * @returns {Promise}
     */
    function getPdf(jasperFileName, exportFormat) {
      if (!exportFormat) {
        exportFormat = 'pdf';
      }
      return $http({
        url:
            '/services/{portalid}/indicator/{appname}/getpdf' +
            '?jasperFileName=' +
            jasperFileName +
            '&exportFormat=' +
            exportFormat,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
        responseType: 'blob',
      });
    }

    /**
     * Get the download link of page 'pageIndex' of generated report in jpeg format
     * @param {type} jasperFileName
     * @param {type} pageIndex
     * @returns {Promise}
     */
    function getJpeg(jasperFileName, pageIndex) {
      //            var promise = $http.get('/services/{portalid}/indicator/{appname}/getjpeg'+
      //                    '?jasperFileName='+jasperFileName+"&subDirectoryPath="+subDirectoryPath
      //                    +"&pageIndex="+pageIndex);

      var promise = $http({
        url:
            '/services/{portalid}/indicator/{appname}/getjpeglink' +
            '?jasperFileName=' +
            jasperFileName +
            '&pageIndex=' +
            pageIndex,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });

      return promise;
    }

    /**
     * Get the download link of overview ('thumbnail') of generated report in jpeg format
     * @param {type} jasperFileName
     * @returns {Promise}
     */
    function getThumbnail(jasperFileName) {
      //            var promise = $http.get('/services/{portalid}/indicator/{appname}/getthumbnail'+
      //                    '?jasperFileName='+jasperFileName+"&subDirectoryPath="+subDirectoryPath);

      var promise = $http({
        url:
            '/services/{portalid}/indicator/{appname}/getthumbnaillink' +
            '?jasperFileName=' +
            jasperFileName,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });

      return promise;
    }

    /**
     * Récupère les noms des rapports Jasper de toutes les applications
     * Soit les rapports actions, soit les rapports tableaux de bord
     * @param isAction est true si on souhaite récupérer les rapports "action"
     * @returns {Promise} promise contenant une liste de DTO {name,app,creation}
     */
    function getPortalReportNamesByType(isAction = false) {
      return $http({
        url:
            '/services/{portalid}/indicator/' +
            isAction +
            '/getportalreportnamesbytype',
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });
    }

    /**
     * Récupère les images des rapports jasper au sein des repos de toutes les applications du portail
     * Soit les images des rapports actions, soit celles des rapports tableaux de bord
     * @param isAction est true si on souhaite récupérer les images des rapports actions, false pour les rapports tableaux de bord
     * @returns {Promise} promise contenant une liste de DTO {name,app,creation}
     */
    function getImages(isAction = false) {
      return $http({
        url: '/services/{portalid}/report/jasper/getimages' +
            '?isAction=' + isAction,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });
    }

    /**
     * Vérifie l'existence d'un rapport Jasper dans le dossier "/reports_action"
     * @param filename nom du fichier du rapport jasper
     * @return {Promise} contenant contenant true si le fichier existe dans le dossier "/reports_action"
     */
    const checkActionReportExists = (filename) => {
      return $http({
        url: '/services/{portalid}/report/{appname}/jasper/checkactionreportexists' +
            '?filename=' + filename,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });
    };

    /**
     * Récupère les rôles de l'utilisateur parmi ses rôles et ses groupes
     * @param login login de l'utilisateur xgos.user.login
     * @return {Promise} contenant un tableau des noms de rôle de l'utilisateur connecté
     */
    function getRolesByUser(login) {
      return $http({
        url: '/services/{portalid}/report/{appname}/jasper/getrolesbyuser' +
            '?login=' + login,
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          encoding: 'UTF-8',
        },
      });
    }

    /**
     * Télécharge le fichier du rapport d'indicateur. Méthode utilisée lorsque le format de l'indicateur est rtf ou xls
     * @param filename nom du rapport généré (indicator.fileNameGenerated)
     * @param exportFormat format dans lequel le rapport a été généré (indicator.format)
     */
    const downloadReport = (filename, exportFormat) => {
      const url = '/services/' + portalid + '/indicator/' + appName + '/downloadfile?f=json'
          + '&fileName=' + filename
          + '&exportFormat=' + exportFormat;
      window.open(url);
    };

    /**
     * Filtre les indicateurs en fonction des rôles de celui-ci
     * L'indicateur doit avoir un rôle correspondant à un rôle de l'utilisateur connecté
     * Si l'indicateur n'a pas de rôle alors tout utilisateur peut le voir
     * Exécutée à la récupération et à la sauvegarde de la config
     * @see ReportIndicatorFactory.getRolesByUser
     */
    const filterIndicatorsByUserRole = (indicators) => {
      const defer = $q.defer();
      let indicatorsByUserRole = [];
      if (Array.isArray(indicators) && indicators.length > 0) {
        const login = $rootScope.xgos.user.login;
        getRolesByUser(login).then(
            res => {
              if (res.data) {
                const userRoles = res.data;

                if (userRoles && userRoles.length > 0) {

                  for (const indicator of indicators) {

                    if (indicator.roles && indicator.roles.length > 0) {

                      // l'indicateur doit avoir un rôle en commun avec l'utilisateur pour que celui-ci soit affiché dans la liste du widget
                      for (const indicatorRole of indicator.roles) {
                        if (userRoles.includes(indicatorRole)) {
                          indicatorsByUserRole.push(indicator);
                          break;
                        }
                      }
                    } else {
                      // si l'indicateur n'a pas de rôle alors on l'ajoute à la liste des indicateurs du widget
                      indicatorsByUserRole.push(indicator);
                    }
                  }
                } else {
                  // si l'utilisateur n'a pas de rôle alors il récupère tous les indicateurs (cas impossible)
                  indicatorsByUserRole = indicators;
                }
              }
              defer.resolve(indicatorsByUserRole);
            },
            () => {
              require('toastr').error(
                  $filter('translate')('dashboard.configuration.rolesretrieveerror'));
              defer.resolve(indicators);
            }
        );
      }
      return defer.promise;
    };

    /**
     * Affiche l'erreur serveur dans un message toastr et une erreur en console
     * @param serverError erreur de type string ou bien objet JS contenant une propriété <code>data</code> de type string
     * ou sinon objet JS contenant un objet <code>data</code> contenant une propriété <code>message</code> de type string
     */
    const handleServerError = (serverError) => {
      let error;
      if (typeof serverError === 'string') {
        error = serverError;
      } else if (serverError.data) {
        if (typeof serverError.data === 'string') {
          error = serverError.data;
        } else if (serverError.data.message && typeof serverError.data.message === 'string') {
          error = serverError.data.message;
        }
      }
      if (error && error.length > 0) {
        require('toastr').error(error);
        console.error(error);
      }
    };

    return {
      ReportIndicatorFactory: ReportIndicatorFactory,
      generate: generateReport,
      uploadReport: uploadReport,
      generateReportExtended: generateReportExtended,
      generateMultipleReportExtended: generateMultipleReportExtended,
      getThumbnail: getThumbnail,
      getJpeg: getJpeg,
      getPdf: getPdf,
      setDataStoreName: setDataStoreName,
      //uploadJasperFile:uploadJasperFile,
      getAllReportNames: getAllReportNames,
      getAllReportNamesFromApplication: getAllReportNamesFromApplication,
      getPortalReportNamesByType: getPortalReportNamesByType,
      getImages: getImages,
      checkActionReportExists: checkActionReportExists,
      getRolesByUser: getRolesByUser,
      downloadReport: downloadReport,
      filterIndicatorsByUserRole: filterIndicatorsByUserRole,
      handleServerError: handleServerError
    };
  };

  ReportIndicatorFactory.$inject = [
    '$http',
    '$q',
    '$rootScope',
    'PortalsFactory'
  ];
  return ReportIndicatorFactory;
});
