/**
 *
 */
'use strict';
define(function () {
  /**
   * Report Ctrl
   * @param $scope
   * @param ApplicationFactory
   * @param ReportIndicatorFactory
   * @param ReportFactory
   * @param ngDialog
   * @param $rootScope
   * @param $filter
   * @param $window
   * @param $timeout
   * @constructor
   */
  var ReportCtrl = function (
      $scope,
      ApplicationFactory,
      ReportIndicatorFactory,
      ReportFactory,
      ngDialog,
      $rootScope,
      $filter,
      $window,
      $timeout
  ) {

    /*************** FONCTIONS DES BOUTONS EDITER/TELECHARGER/SUPPRIMER ***************/

    /**
     * Ouvre la boite de dialogue de remplacement du fichier (edit)
     * Fonction exécutée au clic sur le bouton "Editer" d'une ligne du tableau
     * La dialog à afficher dépend de l'onglet actif
     * @see $scope.editListCfg.editFunction
     */
    $scope.replaceFile = () => {
      openEditDialog($scope.editListCfg.activeTab);
    };

    /**
     * Supprime un fichier (image/rapport)
     * Fonction exécutée au clic sur le bouton "Supprimer" de l'en-tête de la page "/reports"
     * @see $scope.editListCfg.removeFunction
     */
    $scope.removeFile = () => {
      const isAction = $scope.editListCfg.activeTab === 0
          || $scope.editListCfg.activeTab === 2;
      const isImage = $scope.editListCfg.activeTab === 2
          || $scope.editListCfg.activeTab === 3;
      const filename = isImage ? $scope.edit_resource.name : $scope.edit_resource.label;
      ReportFactory.removeuploaded($scope.edit_resource.app, isAction, filename, isImage).then(
          () => {
            refreshList();
          },
          (error) => {
            require('toastr').error($filter('translate')('tools.reports.delete_error')+ $scope.edit_resource.name, error);
          });
    };

    /**
     * Récupère la liste des lignes du tableau de l'onglet actif
     * Exécutée après mise à jour d'un fichier
     * Exécutée après ajout d'un fichier
     */
    $scope.refreshReportsList = () => {
      refreshList();
      if (jasperDialog){
        jasperDialog.close();
      }
    };

    /**
     * Récupère les lignes du tableau suivant l'onglet actif
     * @return {[JSON]} liste de DTO des lignes du tableau {name,app,creation}
     * @see $scope.refreshReportsList
     */
    let refreshList = () => {
      $scope.currentResources = [];
      $scope.selected_resource_index = undefined;
      switch ($scope.editListCfg.activeTab) {
        case 0:
          $scope.getActionReports();
          break;
        case 1:
          $scope.getDashboardReports();
          break;
        case 2:
          $scope.getActionImages();
          break;
        case 3:
          $scope.getDashboardImages();
          break;
      }
    };

    /**
     * Lance le téléchargement d'un fichier
     * Fonction exécutée au clic sur le bouton "télécharger" d'une ligne
     * @see $scope.editListCfg.extraActions[0].fn
     */
    $scope.getFile = () => {
      $window.open(
          '/services/' +
          $rootScope.xgos.portal.uid +
          '/report/' +
          $scope.edit_resource.app +
          '/jasper/downloadFile?f=json&name=' +
          $scope.edit_resource.name +
          '&isAction=' +
          ($scope.editListCfg.activeTab === 0 || $scope.editListCfg.activeTab
              === 2)
      );
    };

    /*************** FONCTIONS DES BOUTONS ET ONGLETS DE L'EN-TÊTE ***************/

    /**
     * Au clic sur le bouton global "Ajouter un rapport",
     * récupère la liste des applications,
     * ouvre la dialog contenant une dropzone
     * un select des applications et un select du type de rapport
     * @see $scope.editListCfg.extraGlobalActions[0].fn
     */
    $scope.addReport = () => {
      if (ApplicationFactory.resources.applications.length === 0) {
        // get applications
        ApplicationFactory.get().then(function () {
          $scope.applicationList = ApplicationFactory.resources.applications.map(
              app => app.name);
          openAddDialog(false);
        });
      } else {
        $scope.applicationList = ApplicationFactory.resources.applications.map(
            app => app.name);
        openAddDialog(false);
      }
    };

    /**
     * Au clic sur le bouton global "Ajouter une image",
     * récupère la liste des applications,
     * ouvre la dialog contenant une dropzone
     * un select des applications et un select du type de rapport
     * @see $scope.editListCfg.extraGlobalActions[1].fn
     */
    $scope.addImage = () => {
      if (ApplicationFactory.resources.applications.length === 0) {
        // get applications
        ApplicationFactory.get().then(function () {
          $scope.applicationList = ApplicationFactory.resources.applications.map(
              app => app.name);
          openAddDialog(true);
        });
      } else {
        $scope.applicationList = ApplicationFactory.resources.applications
        .filter(app => app.name && app.name.length > 0)
        .map(app => app.name);
        openAddDialog(true);
      }
    };

    /**
     * Récupère les rapports "actions" de toutes les applications du portail courant
     * Emplacement: repo/portalid/APPS/?/reports_action
     * @see refreshList
     * @see $scope.editListCfg.extraFilterTabs[0]
     */
    $scope.getActionReports = () => {
      $scope.currentResources = [];
      ReportIndicatorFactory.getPortalReportNamesByType(true).then(
          (res) => {
            if (res.data) {
              $scope.editListCfg.cols = ['label', 'app', 'creationdate'];
              $scope.currentResources = res.data;
              $scope.tableParams.reload();
            }
          });
    };

    /**
     * Récupère les rapports "tableaux de bord" de toutes les applications du portail courant
     * Emplacement: repo/portalid/APPS/?/reports
     * @see refreshList
     * @see $scope.editListCfg.extraFilterTabs[1]
     */
    $scope.getDashboardReports = () => {
      $scope.currentResources = [];
      ReportIndicatorFactory.getPortalReportNamesByType(false).then(
          (res) => {
            if (res.data) {
              $scope.editListCfg.cols = ['label', 'app', 'creationdate'];
              $scope.currentResources = res.data;
              $scope.tableParams.reload();
            }
          });
    };

    /**
     * Récupère les images liées aux rapports "actions" de toutes les applications du portail courant
     * Extensions recherchées: jpg, png, jpeg, gif
     * Emplacement: repo/portalid/APPS/?/reports_action
     * @see refreshList
     * @see $scope.editListCfg.extraFilterTabs[2]
     */
    $scope.getActionImages = () => {

      ReportIndicatorFactory.getImages(true).then(
          (res) => {
            if (res.data) {
              $scope.editListCfg.cols = ['name', 'app', 'creationdate'];
              $scope.currentResources = res.data;
              $scope.tableParams.reload();
            }
          });
    };

    /**
     * Récupère les images liées aux rapports "tableaux de bord" de toutes les applications du portail courant
     * Extensions recherchées: jpg, png, jpeg, gif
     * Emplacement: repo/portalid/APPS/?/reports
     * @see refreshList
     * @see $scope.editListCfg.extraFilterTabs[3]
     */
    $scope.getDashboardImages = () => {
      $scope.currentResources = [];
      ReportIndicatorFactory.getImages(false).then(
          (res) => {
            if (res.data) {
              $scope.editListCfg.cols = ['name', 'app', 'creationdate'];
              $scope.currentResources = res.data;
              $scope.tableParams.reload();
            }
          });
    };

    /*************** PERSONNALISATION DE LA PAGE REPORTS (BOUTONS, ONGLETS, COLONNES...) ***************/

    /**
     * Objet de configuration de la page "/reports"
     * @type {{extraGlobalActions: [{fn: *, label: string, class: string}, {fn: *, label: string, class: string}],
     *        addFunction: $scope.builderForm, dataModule: string, editFunction: $scope.builderForm,
     *        resource_type: string,
     *        extraActions: [{disableCondition: string, icon: string, fn: $scope.getFile, label: string}],
     *        addResourceButton: boolean, removeFunction: $scope.removeForm, cols: string[],
     *        activeTab: number,
     *        extraFilterTabs: [{fn: string, label: string}, {fn: string, label: string}, {fn: string, label: string}, {fn: string, label: string}]}}
     */
    $scope.editListCfg = {
      dataModule: 'tools',
      resource_type: 'reports',
      cols: ['label', 'app', 'creationdate'],
      addResourceButton: false,
      editFunction: $scope.replaceFile,
      removeFunction: $scope.removeFile,
      activeTab: 0, // nécessaire pour rafraîchir la liste au chgt d'onglet
      newResourceAppname: '',
      newResourceType: '',
      sort: true,
      extraActions: [
        {
          icon: "<i class='fa fa-download'></i>",
          label: $filter('translate')('tools.reports.download'),
          fn: $scope.getFile,
          disableCondition: 'download',
        },
      ],
      extraFilterTabs: [
        // les fonctions sont définies en string
        // car ng-click ne marche pas sur les onglets
        {
          label: 'tools.reports.action',
          fn: 'getActionReports'
        },
        {
          label: 'tools.reports.dashboard',
          fn: 'getDashboardReports'
        },
        {
          label: 'tools.reports.img_action',
          fn: 'getActionImages'
        },
        {
          label: 'tools.reports.img_dashboard',
          fn: 'getDashboardImages'
        }
      ],
      extraGlobalActions: [
        {
          label: 'tools.reports.add_report',
          fn: $scope.addReport,
        },
        {
          label: 'tools.reports.add_image',
          fn: $scope.addImage,
        }
      ]
    };

    /**
     * Au clic sur le bouton "Enregistrer" de la popup d'ajout d'un(e) rapport/image
     * Déplace les fichiers temporaires dans le sous-dossier "report(_action)" de l'application du portail
     */
    $scope.saveFile = () => {
      if ($scope.newResource && $scope.newResource.appname && $scope.newResource.type){
        const appname = $scope.newResource.appname;
        const type = $scope.newResource.type;
        const appnameExists = appname.length > 0;
        const typeExists = type.length > 0;

        if (appnameExists && typeExists) {
          const filenames = Object.values($scope.tempFiles);
          ReportFactory.saveTempFile(appname, type, filenames).then(
              () => {
                $scope.refreshReportsList();
                $scope.tempFiles = undefined;
              },
              () => {
                require('toastr').error($filter('translate')('tools.reports.save_error'));
              }
          )
        }
      }else{
        require('toastr').error($filter('translate')('tools.reports.field_absent_error'));
      }
    }

    /*************** FONCTIONS EXECUTEES A L'INITIALISATION DE LA PAGE ***************/

    // récupère la liste des rapports action,
    // l'onglet actif au démarrage étant la liste des rapports action
    $scope.getActionReports();

    setTimeout(() => {
    }, 500);

    /**
     * Au chargement de la page et à chaque changement d'onglet
     * Exécute la méthode associée à l'onglet actif: extraFilterTabs[activeTab].fn
     * Contourne l'impossibilité d'exécuter directement tab.fn() au clic sur un titre d'onglet comme c'est le cas pour les boutons
     * @see scope.editListCfg.extraFilterTabs
     */
    $scope.$watch('editListCfg.activeTab', () => {

      if ($scope.isReportPage()) {
        // vide la sélection de lignes du tableau
        delete $scope.selected_resource_index;
        delete $scope.selected_resource;
        delete $scope.edit_resource;

        // execute la fonction extraFilterTabs[index].fn()
        const index = $scope.editListCfg.activeTab;
        const method = $scope.editListCfg.extraFilterTabs[index].fn;
        $scope[method]();
      }
    });

    /*************** GESTION DES DIALOGS (IMAGE/RAPPORT, ADD/EDIT) ***************/

    /**
     * Lance l'ouverture de la boite de dialogue de l'ajout de fichier (rapport/image)
     * Fournit la classe qui sera appliquée à la popup
     * @param isImage est true quand on clique sur le bouton "Ajouter une image"
     * @see $scope.addReport
     * @see $scope.addImage
     */
    function openAddDialog(isImage) {
      const className = 'ngdialog-theme-plain ng-dialog_report miniclose';
      openDialog(className, isImage, true);
    }

    /**
     * Lance l'ouverture de la boite de dialogue du remplacement du fichier
     * Fournit la classe qui sera appliquée à la popup
     * Définit s'il s'agit d'une image ou d'un rapport
     * @param activeTab
     * @see $scope.replaceFile
     */
    function openEditDialog(activeTab) {
      let isImage = (activeTab === 2 || activeTab === 3);
      const className = 'ngdialog-theme-plain ng-dialog_report miniclose';
      openDialog(className, isImage, false);
    }

    let jasperDialog;
    /**
     * Ouvre la boite de dialogue pour éditer/ajouter un(e) rapport/image
     * Laisse un délai avant d'exécuter la création d'un eventListener sur la dropzone
     * pour être sûr que celle-ci soit existe dans le dom
     * @param className nom de la classe CSS à appliquer sur la boite de dialogue.
     * Elle est différente entre ajout et édition
     * @param isImage est true s'il s'agit d'un(e) ajout/modification d'image
     * @param isNew est true s'il s'agit d'une création d'image ou de rapport
     * @see openAddDialog
     * @see openEditDialog
     */
    function openDialog(className, isImage, isNew) {
      $scope.isNewResource = isNew;
      $scope.isImageResource = isImage
      $scope.newResource = {};
      jasperDialog = ngDialog.open({
        template: 'js/XG/modules/tools/views/modals/modal.jasper.report.html',
        className: className,
        closeByDocument: false,
        scope: $scope,
        preCloseCallback: deleteTempFilesList
      });
      // exécute la fonction createDropzoneRemoveEventListener après un délai de 2sec
      $timeout(createDropzoneEventListener, 2000);
    }

    /*************** GESTION DES FICHIERS DONT L'APPLICATION ET LE TYPE SONT DEFINIS APRES UPLOAD ***************/

    /**
     * Après upload du fichier dans le dossier "TEMP" du repo (complete-action)
     * Récupère le nom du fichier déposé dans la dropzone
     * Sauvegarde le nom du fichier déposé dans une liste temporaire
     * qui est nécessaire pour savoir où déplacer les fichiers temporaires
     * suivant les choix des options type de rapport / application
     * qui sont permis d'être effectués après upload
     * @param filename nom du fichier déposé dans la dropzone
     */
    $scope.keepFileNameToSave = (filename) => {
      if (filename) {
        if (!$scope.tempFiles) {
          $scope.tempFiles = {};
        }
        const newIndex = Object.keys($scope.tempFiles).length;
        $scope.tempFiles[newIndex] = filename;
      }
    }

    /***************************************************************************************************
     * Afin de pouvoir supprimer un nom de fichier de la liste des noms de fichiers temporaires,        *
     * $scope.tempFiles, au clic sur le lien "Supprimer" des fichiers déposés dans la dropzone,         *
     * sans accès aux évènements levés par la dropzone et ne pouvant modifier le script de la dropzone, *
     * on utilise un eventListener JavaScript                                                           *
     ***************************************************************************************************/

    /**
     * Création d'un eventListener sur la dropzone après dépôt des fichiers
     * Le timeout permet d'attendre la création des liens "Supprimer" dans le dom
     * pour exécuter la fonction qui créé un eventListener sur les liens "Supprimer"
     * @see openDialog
     */
    function createDropzoneEventListener() {
      // HTMLCollection
      const dropzones = document.getElementsByClassName('dropzone');

      // En plus du délai, on vérifie la présence de la dropzone pour éviter les erreurs
      if (dropzones && dropzones.length > 0) {
        const dropzone = dropzones.item(0);

        // ajoute un auditeur d'évènement qui déclenche au dépôt de fichier dans la dropzone
        // l'exécution de la fonction createDropzoneRemoveEventListener() après un délai de 2 sec
        dropzone.addEventListener('drop', () => {
          $timeout(createDropzoneRemoveEventListener, 2000);
        }, false);
      }
    }

    /**
     * A l'apparition de la dropzone,
     * création d'un eventListener sur le lien "Supprimer" de chaque fichier déposé dans la dropzone:
     * suppression du nom du fichier de la liste des noms de fichiers devant être sauvegardés
     * Permet de tenir à jour la liste des noms de fichiers devant être sauvegardés, $scope.tempFiles
     * @see createDropzoneEventListener
     */
    function createDropzoneRemoveEventListener() {

      // HTMLCollection des boutons "supprimer" (1 / fichier déposé)
      const deleteLink = document.getElementsByClassName('dz-remove');

      // Malgré la présence d'un délai de 2s, on vérifie la présence de boutons "Supprimer" dans le dom
      if (deleteLink && deleteLink.length > 0) {
        for (let i = 0; i < deleteLink.length; i++) {

          // création d'un eventListener à usage unique (EventListener créé à chaque nouveau dépôt)
          deleteLink[i].addEventListener('click', () => {

            // supprime le nom du fichier de la liste temporaire
            delete $scope.tempFiles[i];

            // supprime la liste des fichiers temporaire si elle est vide (action également réalisée à la fermeture de la dialog)
            if (Object.keys($scope.tempFiles).length === 0) {
              $scope.tempFiles = undefined;
            }
          }, {once: true});
        }
      }
    }

    /*************** UTILITAIRES ***************/

    /**
     * Supprime la liste des fichiers temporaires du $scope
     * @see openDialog
     */
    function deleteTempFilesList() {
      if ($scope.tempFiles) {
        $scope.tempFiles = undefined;
      }
    }

    /**
     * Détermine si la page est "/reports" si existent 2 propriétés spécifiques dans l'objet de configuration
     * @returns {boolean} true si l'onglet actif existe et si la liste d'onglets existe et contient au moins 1 élément
     */
    $scope.isReportPage = () => {
      return $scope.editListCfg.extraFilterTabs !== undefined
          && $scope.editListCfg.extraFilterTabs.length > 0
          && $scope.editListCfg.activeTab !== undefined;
    };
  };

  ReportCtrl.$inject = [
    '$scope',
    'ApplicationFactory',
    'ReportIndicatorFactory',
    'ReportFactory',
    'ngDialog',
    '$rootScope',
    '$filter',
    '$window',
    '$timeout'
  ];
  return ReportCtrl;
});
