'use strict';
define(function() {
  /**
   * Gestion des paramètres régionaux
   * Class : RegionalFactory
   */
  const RegionalFactory = function($q, $translate, gaJsUtils) {

    /**
     * Caractères séparateurs de champ disponibles pour la lecture/écriture d'un fichier CSV
     * @type {[{alias: string, id: string}, {alias: string, id: string}, {alias: string, id: string}]}
     */
    const csvColumnSeparators = [
      {
        id: ',',
        alias: 'importdiverswidget.virgule'
      },
      {
        id: ';',
        alias: 'importdiverswidget.pointVirgule'
      },
      {
        id: 'TAB',
        alias: 'importdiverswidget.tabulation'
      }
    ];

    /**
     * Méthode permettant de récupérer les alias traduits des séparateurs de champ d'un fichier CSV
     * @return {string[]} séparateurs de champ disponibles
     */
    const getCsvColumnSeparators = () => {

      // méthode interne pour construire la promesse
      const getCsvColumnSeparator = (sep) => {
        const defer = $q.defer();
        $translate(sep.alias).then(
            res => {
              defer.resolve({
                id: sep.id,
                alias: res
              });
            },
            () => {
              console.error(`Impossible de traduire ${sep.alias}`);
            }
        );
        return defer.promise;
      };

      const defer = $q.defer();
      const promises = [];
      for (const sep of csvColumnSeparators) {
        promises.push(getCsvColumnSeparator(sep));
      }
      $q.all(promises).then(
          res => {
            defer.resolve(res);
          }
      );
      return defer.promise;
    };

    /**
     * Encodages de texte disponibles pour la lecture/écriture de fichiers CSV
     * @type {{alias: string, id: number, name: string}[]}
     */
    const textEncodings = [
      {
        id: 0,
        alias: 'portals.regional.encodings.ascii',
        name: 'US-ASCII'
      },
      {
        id: 1,
        alias: 'portals.regional.encodings.westernEurope',
        name: 'ISO-8859-15'
      },
      {
        id: 2,
        alias: 'portals.regional.encodings.latin1',
        name: 'windows-1252'
      },
      {
        id: 3,
        alias: 'portals.regional.encodings.unicode',
        name: 'UTF-8'
      }
    ];

    /**
     * Méthode permettant de récupérer les encodages de textes disponibles d'un fichier CSV en ayant les alias traduits
     * @return {string[]} nom des encodages de texte disponibles
     */
    const getTextEncodings = () => {

      // méthode interne pour construire la promesse
      const getTextEncoding = (enc) => {
        const defer = $q.defer();
        $translate(enc.alias).then(
            res => {
              defer.resolve({
                id: enc.id,
                alias: res,
                name: enc.name
              });
            },
            () => {
              console.error(`Impossible de traduire ${enc.alias}`);
            }
        );
        return defer.promise;
      };

      const defer = $q.defer();
      const promises = [];
      for (const enc of textEncodings) {
        promises.push(getTextEncoding(enc));
      }
      $q.all(promises).then(
          res => {
            defer.resolve(res);
          }
      );
      return defer.promise;
    };

    /**
     * Formats de dates disponibles pour décrire le format de date utilisé lors d'import de fichier
     * @see importdiverswidget
     * @type {string[]} formats de dates disponibles
     */
    const dateFormats = [
      'dd/MM/yyyy',
      'dd-MM-yyyy',
      'MM/dd/yyyy',
      'MM-dd-yyyy'
    ];

    /**
     * Récupération des paramètres régionaux du portail s'ils existent
     * @param {object} portal portail en cours d'utilisation, propriété de $rootScope.xgos
     * @return {{encoding: number|null, separator: string|null}}
     */
    const getPortalRegionalData = (portal) => {
      let encoding = null;
      let separator = null;
      if (gaJsUtils.notNullAndDefined(portal, 'parameters.regional')) {
        const regionalParameters = portal.parameters.regional;
        if (regionalParameters) {
          if (regionalParameters.hasOwnProperty('textEncoding')) {
            encoding = regionalParameters.textEncoding;
          }
          if (regionalParameters.hasOwnProperty('columnSeparator')) {
            separator = regionalParameters.columnSeparator;
          }
        }
      }
      return {encoding: encoding, separator: separator};
    };

    /**
     * Récupération de l'encodage de texte à utiliser pour générer/lire des fichiers CSV
     * L'encodage est défini dans les paramètres régionaux du portail
     * @param {object} portal portail en cours d'utilisation, propriété de $rootScope.xgos
     * @return {null|number}
     */
    const getPortalTextEncoding = (portal) => {
      if (gaJsUtils.notNullAndDefined(portal, 'parameters.regional')
          && portal.parameters.regional.hasOwnProperty('textEncoding')) {
        return portal.parameters.regional.textEncoding;
      }
      return null;
    };

    /**
     * Encode une chaîne de caractère dans un charset différent
     * @param stringToEncode texte à modifier
     * @param outputEncoding encodage dans lequel traduire le texte
     */
    const encodeString = (stringToEncode, outputEncoding) => {
      if (Number.isInteger(outputEncoding) && textEncodings.some(enc => enc.id === outputEncoding)) {
        outputEncoding = textEncodings.find(enc => enc.id === outputEncoding).name;
      }
      if (typeof outputEncoding !== 'string') {
        return stringToEncode;
      }
      const encoder = new TextEncoder(outputEncoding);
      return encoder.encode(stringToEncode);
    };

    return {
      RegionalFactory: RegionalFactory,
      getCsvColumnSeparators: getCsvColumnSeparators,
      getTextEncodings: getTextEncodings,
      dateFormats: dateFormats,
      getPortalRegionalData: getPortalRegionalData,
      getPortalTextEncoding: getPortalTextEncoding,
      encodeString: encodeString
    }
  };

  RegionalFactory.$inject = ['$q', '$translate', 'gaJsUtils'];
    return RegionalFactory;
  });
