'use strict';
define(function () {
  var homepageNews = function (
      $rootScope,
      $filter,
      extendedNgDialog,
      gaJsUtils,
      gaDomUtils,
      homeFactory,
      authFactory
  ) {
    return {
      templateUrl: 'js/XG/modules/home/views/directives/homepageNews.html',
      restrict: 'EA',
      scope: {
        homepage: '=',    // (OBLIGATOIRE) objet de la page d'accueil (cf. Homepage.java)
        headerHtml: '=?', // string contenant le HTML personnalisé du header de la page d'accueil
        portalId: '=?',   // uid identifiant du portail. Requis pour l'usage des dropzones
      },
      link: function (scope) {

        // objet des dropzones
        scope.dzMemShare = {}
        // Nom de la popover actuellement ouverte (=> 1 à la fois): 'logo', 'image', 'group', 'link'
        scope.activePopover = '';
        // Objet du statut des carousels pour affichage des flèches gauche/droite.<br>
        // Mis à jour par les évènements mouseEnter, mouseMove et mouseLeave sur les carousels
        scope.itemCfg = {
          group: {
            active: false,
            minScrollLeft: false,
            maxScrollLeft: true,
          },
          link: {
            active: false,
            minScrollLeft: true,
            maxScrollLeft: false,
          }
        };

        scope.hasEditOnce = false;

        /**
         * Au clic sur un élément du carousel des groupes,
         * filtre les liens affichés dans le carousel de liens
         */
        scope.$on('onHomepageGroupSelected', (event, selectedGroupId) => {
          filterItems(selectedGroupId);
        });

        /**
         * Après sauvegarde d'un lien,
         * ajoute le lien dans le tableau global de liens de la page d'accueil
         */
        scope.$on('addItemIntoHomepage', (event, data) => {
          const isGroup = data.isGroup;
          const edit_resource = data.edit_resource;
          if (isGroup && scope.homepage.groups && Array.isArray(scope.homepage.groups)) {
            const itemIndex = scope.homepage.groups.findIndex(i => i.id === edit_resource.id);
            if (itemIndex > -1) {
              scope.homepage.groups.splice(itemIndex, 1, edit_resource);
            } else {
              edit_resource.id = getNextId(true);
              edit_resource.index = scope.homepage.groups.length;
              scope.homepage.groups.push(edit_resource);
            }
          } else if (scope.homepage.links && Array.isArray(scope.homepage.links)) {
            const itemIndex = scope.homepage.links.findIndex(i => i.id === edit_resource.id);
            if (itemIndex > -1) {
              scope.homepage.links.splice(itemIndex, 1, edit_resource);
            } else {
              edit_resource.id = getNextId(false);
              const activeGroup = scope.homepage.groups.find(grp => grp.active);
              if (activeGroup) {
                edit_resource.groupid = activeGroup.id;
              }
              edit_resource.index = scope.homepage.links.length;
              scope.homepage.links.push(edit_resource);
            }
            filterItems(edit_resource.groupid);
          }
        });

        /**
         * Après suppression d'un lien,
         * l'enfant demande à homepageNews de supprimer l'item sélectionné: data.edit_resource
         */
        scope.$on('deleteHomepageItem', (event, data) => {
          const isGroup = data.isGroup;
          const edit_resource = data.edit_resource;
          if (isGroup && scope.homepage.groups && Array.isArray(scope.homepage.groups)) {
            const index = scope.homepage.groups.findIndex(grp => grp.id === edit_resource.id);
            if (index > -1) {
              scope.homepage.groups.splice(index, 1);
            }
          } else if (scope.homepage.links && Array.isArray(scope.homepage.links)){
            const index = scope.homepage.links.findIndex(link => link.id === edit_resource.id);
            if (index > -1) {
              scope.homepage.links.splice(index, 1);
            }
            const filteredIndex = scope.filteredLinks.findIndex(link => link.id === edit_resource.id);
            if (filteredIndex > -1) {
              scope.filteredLinks.splice(filteredIndex, 1);
            }
          }
        });

        /**
         * Surveille le basculement vers le mode config
         * pour remplacer le texte on/off du toggle-switch par des icônes
         */
        scope.$watch('mode', (newVal) => {
          if (newVal === 'config') {
            insertToggleIcon();
            // distingue le 1er clic sur le bouton "Configuration"
            if (!scope.hasEditOnce) {
              scope.hasEditOnce = true;
            }
          }
          // si l'utilisateur a l'acces à la config on recharge la page d'accueil
          // pour rafraîchir le filtrage des filtres:
          // cf. spec post-ticket : "Quand je configure un Thème avec une restriction
          // sur un rôle de l’application alors l’utilisateur qui a le rôlekis_homepage doit
          // toujours voir les Thèmes restreints même s’ils n’ont pas le rôle kis_homepage
          // et ce uniquement en mode Configuration"

          // récupère tous les liens en mode config: bypassRights = true
          // Ne s'exécute pas à l'initialisation de la directive.
          // S'exécute à partir du 1er clic sur le bouton violet "Configuration"
          if (scope.homepage && scope.userCanConfigureHomepage && scope.hasEditOnce) {
            gaDomUtils.showGlobalLoader();
            homeFactory.getByNameAndHeader(scope.homepage.name, scope.mode === 'config').then(
                res => {
                  if (res.homepage) {
                    scope.homepage = res.homepage;
                  }
                  if (scope.homepage && Array.isArray(scope.homepage.groups)) {
                    const activeGroup = scope.homepage.groups.find(grp => grp.active);
                    if (activeGroup) {
                      filterItems(activeGroup.id);
                    }
                    if (res.header) {
                      // charge l'éventuel header html personnalisé de la page d'accueil
                      scope.header = res.header;
                    }
                  }
                },
                err => {
                  require('toastr').error(
                      $filter('translate')('portals.homepage.getHomepageError'));
                  if (err.data && err.data.message) {
                    console.error(err.data.message);
                  }
                }
            ).finally(
                () => {
                  gaDomUtils.hideGlobalLoader();
                }
            );
          }
        });

        /**
         * Au clic sur un groupe de lien,
         * filtre le tableau du carousel enfant selon l'élement sélectionné dans le tableau du carousel parent
         * @param {number} groupId nombre entier id identifiant le groupe sur lequel l'utilisateur a cliqué
         */
        const filterItems = (groupId) => {
          // on traite uniquement dans la directive du carousel enfant (liens)
          if (!scope.isPrimary) {
            let filteredLinksTemp = [];
            if (gaJsUtils.notNullAndDefined(groupId) && Number.isInteger(groupId)) {
              // cas où l'on a cliqué sur un groupe
              if (scope.homepage.links && Array.isArray(scope.homepage.links)
                  && scope.homepage.links.length > 0) {
                // Alors que tous les liens sont affichés, au 1er clic sur un groupe de la page,
                // on les stocke dans une autre variable 'backupItems'
                // qui contiendra un tableau de tous les liens quelque soit le groupe sélectionné
                filteredLinksTemp = Object.assign([], scope.homepage.links);
              }
              // filtre les liens suivant l'id du groupe sélectionné (groupid)
              // garde uniquement les liens attachés au groupe sélectionné
              // et les liens n'ayant pas de groupe de rattachement
              filteredLinksTemp = filteredLinksTemp.filter(item => {
                  return (item.groupid === null || item.groupid === '') || (item.hasOwnProperty(
                      'groupid') && item.groupid === groupId);
                });
            } else {
              // groupid est null, tous les groupes sont déselectionnés
              // alors on affiche tous les liens depuis 'backuplinks'
              console.log('[Accueil] Affiche tous les liens');
              if (Array.isArray(scope.homepage.links)) {
                filteredLinksTemp = Object.assign([], scope.homepage.links);
              }
            }
            // on affecte scope.filteredLinks seulement quand la liste est correct car
            // l'allocation déclenche un watcher. Les changements ultérieur ne seront pas pris en compte.
            scope.filteredLinks = filteredLinksTemp;
          }
        };

        /**
         * Insère les icônes fontawesome 'majic'(mode classique) et 'code' (mode HTML)
         * dans les labels du toggle-switch du header de la page d'accueil
         */
        const insertToggleIcon = () => {
          const toggle = document.getElementById('custom-header-toggle');
          if (toggle) {
            const leftSection = toggle.querySelector('.switch-left');
            const rightSection = toggle.querySelector('.switch-right');
            if (leftSection) {
              leftSection.innerHTML = `<i class="fa fa-code" aria-hidden="true"></i>`;
            }
            if (rightSection) {
              rightSection.innerHTML = `<i class="fa fa-magic" aria-hidden="true"></i>`;
            }
          }
        };

        /**
         * Au clic sur le toggle-switch,
         * initialise la variable qui stocke le HTML modifié par l'utilisateur.
         * Supprime le HTML modifié si on bascule en mode classique
         * @see scope.saveHomepage le header est sauvegardé en même temps que la page d'accueil
         */
        scope.toggleCustomHeader = () => {
          const header = document.getElementById('kis-home-header');
          if (scope.homepage.customheader) {
            scope.headerHtml = header.innerHTML;
          } else {
            scope.headerHtml = null;
          }
        };

        /**
         * Au clic sur le bouton violet de configuration,
         * on bascule en mode 'configuration'.<br>
         * Définition les modes de configuration en fonction des rôles de l'utilisateur
         */
        scope.editHomepage = () => {
          if ($rootScope.xgos && $rootScope.xgos.user) {

            const hasActuRole = gaJsUtils.userHasRole($rootScope.xgos.user, 'kis_homepage_actu');
            const hasHomepageRole = gaJsUtils.userHasRole($rootScope.xgos.user, 'kis_homepage');

            if ((hasActuRole && hasHomepageRole) || $rootScope.xgos.user.login === 'root') {
              // l'utilisateur ayant les deux rôles peut modifier la page d'accueil et les actualités
              scope.mode = 'config';
              homeFactory.backupHomepage = angular.copy(scope.homepage);

            } else if (hasHomepageRole && !hasActuRole) {
              // l'utilisateur ayant uniquement le rôle "kis_homepage" peut modifier la page d'accueil
              scope.mode = 'config_homepage';
              homeFactory.backupHomepage = angular.copy(scope.homepage);

            } else if (hasActuRole && !hasHomepageRole) {
              // l'utilisateur ayant uniquement le rôle "kis_homepage_actu" peut modifier les actualités
              scope.mode = 'config_actu';
            }
          }
        };

        /**
         * Au clic sur le bouton "Annuler" de configuration,
         * on quitte le mode 'configuration' sans sauvegarde
         */
        scope.cancelHomepage = () => {
          scope.mode = 'read';
          scope.activePopover = '';
          if (homeFactory.backupHomepage) {
            scope.homepage = angular.copy(homeFactory.backupHomepage);
            homeFactory.backupHomepage = null;
          }
        };

        /**
         * Au clic sur le bouton "Enregistrer" de configuration.
         * Sauvegarde la page d'accueil
         */
        scope.saveHomepage = () => {
          if (scope.homepage.title && scope.homepage.title.hasOwnProperty('isBold')) {
            delete scope.homepage.title.isBold;
          }
          if (scope.homepage.subtitle && scope.homepage.subtitle.hasOwnProperty('isBold')) {
            delete scope.homepage.subtitle.isBold;
          }
          if (scope.homepage.group.title && scope.homepage.group.title.hasOwnProperty('isBold')) {
            delete scope.homepage.group.isBold;
          }
          if (scope.homepage.link.title && scope.homepage.link.title.hasOwnProperty('isBold')) {
            delete scope.homepage.link.isBold;
          }

          // les index ne servent à rien puisqu'on affiche les elements dans l'ordre du tableau
          // on met quand même à jour les index avant la sauvegarde... on sait jamais
          homeFactory.resetIndex(scope.homepage.groups);
          homeFactory.resetIndex(scope.homepage.links);

          homeFactory.save(scope.homepage, scope.headerHtml).then(
              (homepage) => {
                if (homepage) {
                  scope.homepage = homepage;
                  const activeGroup = scope.homepage.groups.find(grp => grp.active);
                  if (activeGroup && activeGroup.id) {
                    filterItems(activeGroup.id);
                  }
                } else {
                  console.error('homepage is undefined');
                }
              }).finally(
              () => {
                scope.mode = 'read';
                scope.activePopover = '';
                if (homeFactory.backupHomepage) {
                  homeFactory.backupHomepage = null;
                }
              }
          );
        };

        /**
         * Au clic sur le bouton "Quitter la configuration" pour un utilisateur ayant le rôle "kis_homepage_actu"
         * Bascule en mode "lecture"
         */
        scope.exitNewsConfig = () => {
          scope.mode = 'read';
        };


        /**
         * Au clic sur le bouton "charger la nouvelle image" de la popover de changement de logo
         * ou d'image d'arrière-plan.
         * on charge l'image et on la selectionne automatiquement
         * @param {string} popoverName should be 'background' ou 'logo'
         */
        scope.addHomepageImage = (popoverName) => {
          if (scope.dzMemShare.dropzoneComponent.files.length > 0 && !scope.uploadPending) {
            // empêche complete-action d'être exécutée 2x par le composant dropzone
            scope.uploadPending = true;

            // dépôt d'une image dans la dropzone
            const filename = scope.dzMemShare.dropzoneComponent.files[0].name;

            // copie l'image du dossier "UPLOAD" dans le dossier "CSS-HTML"
            homeFactory.copyUploadedImgToCssRepo(
                scope.portalid,
                scope.dzMemShare.uploadProcessID,
                filename,
                true
            ).then(
                (img) => {
                  if (img.data && img.data.length > 0) {
                    // dans homepageNews on gère uniquement le logo et les arrière-plans
                    // on met à jour le tableau 'other' en insèrant la nouvelle image dans la mosaïque
                    const extension = filename.split('.').pop();
                    const prefix = 'data:image/' + (extension === 'svg' ? extension + '+xml'
                            : extension)
                        + ';base64,';
                    const image = {
                      base: prefix + img.data,
                      src: filename
                    };
                    scope.images.other.push(image);

                    // après un ajout, on selectionne la nouvelle image automatiquement
                    switch(popoverName) {
                      case 'background':
                        scope.homepage.image = image.src;
                        break;
                      case 'logo':
                        scope.homepage.logo = image.src;
                        break;
                      default:
                        break;
                    }
                  } else {
                    require('toastr').error(
                        $filter('translate')('portals.homepage.addHomepageImageError'));
                  }
                },
                () => {
                  require('toastr').error(
                      $filter('translate')('portals.homepage.addHomepageImageError'));
                }
            ).finally(() => {
              scope.uploadPending = false;
            });
          }
        };

        /**
         * Définit le font d'écran avec l'image cliquée.<br>
         * Si l'image était déjà selectionnée alors on enlève le fond d'écran
         * @param {object} selectedBackground objet de l'image cliqué
         */
        scope.changeBackground = (selectedBackground) => {
          if (scope.homepage.image !== selectedBackground.src) {
            scope.homepage.image = selectedBackground.src;
          } else {
            scope.homepage.image = undefined;
          }
        };

        /**
         * définit le logo avec l'image cliqué
         * si l'image était déjà selectionné alors enlève le logo
         * @param {object} seletedLogo objet de l'image cliqué
         */
        scope.changeLogo = (seletedLogo) => {
          if (scope.homepage.logo !== seletedLogo.src) {
            scope.homepage.logo = seletedLogo.src;
          } else {
            scope.homepage.logo = undefined;
          }
        };

        /**
         * Au clic sur le bouton 'Police en gras' de l'un des champs suivants:
         * titre, sous-titre, titre du bloc gauche, titre du bloc droit.
         * Change la propriété weight de la police et
         * modifie la propriété temporaire "isBold" pour répercuter l'état sur le bouton
         * @param {string} itemType 'title' | 'subtitle' | 'group' | 'link'
         */
        scope.onBoldChange = (itemType) => {
          if (scope.homepage) {
            if (gaJsUtils.notNullAndDefined(scope.homepage[itemType])) {
              let title;
              if (itemType === 'group' || itemType === 'link') {
                title = scope.homepage[itemType].title;
              } else {
                title = scope.homepage[itemType];
              }
              if (title) {
                homeFactory.toggleBoldByWeight(title);
              }
            }
          }
        };

        /**
         * Au clic sur le bouton 'Alignement horizontal' de l'un des champs suivants:
         * titre, sous-titre, titre du bloc gauche, titre du bloc droit
         * @param {string} itemType nom du champ concerné: 'title', 'subtitle', 'group' ou 'link'
         */
        scope.onHalignChange = (itemType) => {
          // le tableau des positions doit exister dans le service
          if (homeFactory.hasOwnProperty('hAlignPositions') && Array.isArray(
              homeFactory.hAlignPositions) && homeFactory.hAlignPositions.length > 0) {

            if (Object.keys(scope.itemCfg).includes(itemType)) {
              // itemType === 'group' || itemType === 'link'
              homeFactory.setHalign(scope.homepage[itemType].title);
            } else {
              // itemType === 'title' || itemType === 'subtitle'
              homeFactory.setHalign(scope.homepage[itemType]);
            }
          }
        };

        /**
         * Au clic sur le bouton "Publier la page d'accueil"
         */
        scope.onPublishChange = () => {
          scope.homepage.published = !scope.homepage.published;
          scope.saveHomepage();
        };

        /**
         * Renvoie l'image en string à partir du nom du fichier de l'image
         * @param {string} filename nom du fichier de l'image
         * @return {string|null} image en string ou null
         */
        scope.getEncodedImageByName = (filename) => {
          if (scope.images && scope.images.other && scope.images.other.length > 0 && filename
              && filename.length > 0) {
            const image = scope.images.other.find(img => img.src === filename);
            if (image && image.hasOwnProperty('base')) {
              return image.base;
            }
          }
        };

        /**
         * A l'entrée du survol d'un carousel,
         * ou lorsque la souris bouge en survolant un carousel
         * on vérifie la position du scroll du carousel
         * pour l'affichage des boutons de navigation - / +
         * @param {string} itemType type de l'item ('group', 'link', ...)
         * @param {boolean} isMove est true quand l'évènement est de type OnMouseMove
         */
        scope.activeCarousel = (itemType, isMove = false) => {
          if (scope.itemCfg.hasOwnProperty(itemType)) {
            if (isMove) {
              scope.itemCfg[itemType].minScrollLeft = isMinScrollLeft(itemType);
              scope.itemCfg[itemType].maxScrollLeft = isMaxScrollLeft(itemType);
            } else {
              for (const anySuffix of Object.keys(scope.itemCfg)) {
                scope.itemCfg[anySuffix].active = false;
              }
              if (scope.itemCfg.hasOwnProperty(itemType)) {
                scope.itemCfg[itemType].minScrollLeft = isMinScrollLeft(itemType);
                scope.itemCfg[itemType].maxScrollLeft = isMaxScrollLeft(itemType);
                scope.itemCfg[itemType].active = true;
              }
            }
          }
        };

        /**
         * A la sortie du survol d'un carousel,
         * on modifie la propriété active de l'objet correspondant au carousel
         * @param {string} itemType type de l'item ('group', 'link', ...)
         */
        scope.deactiveCarousel = (itemType) => {
          if (scope.itemCfg.hasOwnProperty(itemType)) {
            scope.itemCfg[itemType].active = false;
          }
        };

        /**
         * Vérifie si la date de diffusion de l'actualité est antérieure à la date actuelle
         * @param {object} newsEvent actualité de page d'accueil
         * @return {boolean} true si la date de l'actualité est antérieure à la date actuelle
         */
        scope.hasPublishDateBeforeNow = (newsEvent) => {
          return newsEvent.date < Date.now();
        };

        /**
         * Vérifie si un carousel a le défilement horizontal bloqué à gauche.
         * Le bouton flèche gauche est masqué si true.
         * @param {string} itemType 'group' ou 'link' : distingue les 2 carousels
         * @return {boolean} true si le défilement horizontal du carousel est au maximum à gauche
         */
        const isMinScrollLeft = (itemType) => {
          let isScrollMin = false;
          const carouselElement = homeFactory.getCarousel(itemType);
          if (carouselElement) {
            isScrollMin = carouselElement.scrollLeft === 0;
          }
          return isScrollMin;
        };

        /**
         * Vérifie si un carousel a le défilement horizontal bloqué à droite.
         * Le bouton flèche droite est masqué si true.
         * @param {string} itemType 'group' ou 'link' : distingue les 2 carousels
         * @return {boolean} true si le défilement horizontal du carousel est au maximum à droite
         */
        const isMaxScrollLeft = (itemType) => {
          let isScrollMax = false;
          const carouselElement = homeFactory.getCarousel(itemType);
          if (carouselElement) {
            const maxScrollLeft = carouselElement.scrollWidth - carouselElement.clientWidth;
            isScrollMax = carouselElement.scrollLeft >= maxScrollLeft;
          }
          return isScrollMax;
        };

        /**
         * Au clic sur un bouton flèche gauche (-) / flèche droite (+)
         * d'un carousel.
         * Modifie la valeur de la propriété scrollLeft du carousel
         * @param {object} event évènement au clic sur le bouton
         * @param {string} direction 'left' ou 'right'
         * @param {string} itemType 'group' ou 'link': distingue les 2 carousels
         */
        scope.slide = (event, direction, itemType) => {
          event.preventDefault();
          const carouselId = homeFactory.getCarouselId(itemType);
          if (direction === 'left') {
            document.getElementById(carouselId).scrollLeft -= 40;
          } else {
            document.getElementById(carouselId).scrollLeft += 40;
          }
        };

        /**
         * Trie les items des 2 carousels selon la propriété index (asc)
         */
        const sortItemsByIndex = () => {
          if (scope.homepage && scope.homepage.groups) {
            homeFactory.sortByIndex(scope.homepage.groups);
          }
          if (scope.homepage && scope.homepage.links) {
            homeFactory.sortByIndex(scope.homepage.links);
          }
        };

        /**
         * Au clic sur le bouton de déconnexion, exécute une méthode de authFactory
         */
        scope.logout = () => {
          authFactory.logout();
        };

        /**
         * Renvoie l'entier supérieur à la valeur la plus grande
         * de la propriété id d'un item dans le tableau scope.homepage.groups ou links
         * @param {boolean} isGroup est true quand on veut déterminer le prochain id dans les groupes de la page d'accueil,
         * false pour regarder dans les thèmes de la page d'accueil
         * @return {number} entier supérieur au id le plus grand de scope.homepage.groups ou links
         */
        const getNextId = (isGroup) => {
          let nextId = -1;
          const list = isGroup ? scope.homepage.groups : scope.homepage.links;
          if (scope.homepage && Array.isArray(list) && list.length> 0) {
            for (const item of list) {
              if (Number.isInteger(item.id) && nextId < item.id) {
                nextId = item.id;
              }
            }
          }
          nextId = nextId > -1 ? (nextId + 1) : 0;
          return nextId;
        };

        /**
         * Lance l'ouverture de la popup qui propose de supprimer une image de la page d'accueil
         * @param {number} indexToDelete rang de l'image dans le tableau d'images de la directive
         * @param {string} imageCategory est égal à "background" ou "logo"
         * @see homeFactory.ShowDialogToDeleteImage
         */
        scope.ShowDialogToDeleteImage = (indexToDelete, imageCategory) => {
          homeFactory.ShowDialogToDeleteImage(scope.homepage.portalid, scope.images.other, indexToDelete, imageCategory)
        };

        /**
         * met à jour la liste des liens pour qu'ils soient dans l'ordre
         * @param {object[]} links liste des liens du groupe actuel
         */
        scope.copyLinksOrderToConfig = (links) => {
          //on récupère le groupe actif
          const activeGroup = scope.homepage.groups.find(grp => grp.active);
          //on supprime les liens du groupe actif (car ils sont pas dans le bon ordre)
          scope.homepage.links = scope.homepage.links.filter(link => link.groupid !== activeGroup.id);
          // on ajoute les liens du groupe actif (dans l'ordre)
          scope.homepage.links = scope.homepage.links.concat(links);
        };

        /**
         * Ouvre la popup pour configurer les actualités de la page d'accueil
         * La popup contient la directive homepageNewsEventConfig
         */
        scope.openNewsConfig = () => {
          extendedNgDialog.open({
            template: 'js/XG/modules/home/views/modals/modal.home.news.config.html',
            className: 'ngdialog-theme-plain width800 nopadding miniclose kis-home-config-dialog',
            closeByDocument: false,
            scope: scope,
            title: $filter('translate')('portals.homepage.v2.news.config.title'),
            draggable: true,
            scrollable: true,
            minimizeMaximize: true,
            resizable: true,
            minWidth: '570px',
            minHeight: '400px'
          });
        };

        /**
         * Teste si la longueur du message de l'actualité dépasse la largeur de la div conteneur
         * Si oui, affiche le bouton "..." et applique la classe CSS <code>exceed-width</code>
         * @param {number} index rang de l'actualité dans le tableau <code>homepage.news.events</code>
         * @return {boolean} true si la longueur du message de l'actualité dépasse la largeur de la div conteneur
         */
        scope.isEventWidthExceeding = (index) => {
          const newsDiv = document.getElementById('kis-home-news');
          if (newsDiv) {
            const contentBox = newsDiv.querySelector('.content');
            const newsEventRow = contentBox.children[index];
            if (newsEventRow) {
              const textRow = newsEventRow.querySelector('.event-content');
              if (textRow) {
                return textRow.scrollWidth > contentBox.offsetWidth;
              }
            }
          }
          return false;
        };

        /**
         * Au clic sur le bouton '...' d'une actualité,
         * assigne le message de l'actualité à la variable <code>expandedEventText</code>
         * @param {number} index rang de l'actualité dans le tableau <code>homepage.news.events</code>
         */
        scope.setExpandedEventText = (index) => {
          scope.expandedEventText = scope.homepage.news.events[index].text;
        };

        /**********************
         *   INITIALISATION   *
         **********************/

        // enlève la navbar
        let navbar = document.getElementById('xgos_nav');
        if (navbar) {
          navbar.style.display = 'none';
          navbar = null;
        }
        // supprime la marge supérieure du contenu de la page
        // étire le corps de la page jusqu'au bas de l'écran
        let main = document.getElementById('xgos_main');
        if (main) {
          main.style.top = '0px';
          main.children[0].style.height = '100%';
          main = null;
        }

        if (scope.homepage) {
          // trie les groupes et les liens selon l'index enregistré
          sortItemsByIndex();

          // get applications
          homeFactory.getApplicationThenGetConfiguration().then(
              (conf) => {
                // à ce moment là ApplicationFactory.resources est chargé
                if (conf && conf !== '') {
                  scope.config = conf;
                }
              }
          );
          // affectue l'uid du portail à la page d'accueil
          if (scope.portalId) {
            scope.homepage.portalid = scope.portalId;
          }

          // ajoute la propriété 'isBold' aux 4 types de titres existants dans la page d'accueil
          // 'isBold' permet de gérer l'activation du bouton "gras"
          // on stocke le gras dans un nombre "weight" pour garder le système évolutif
          // on utilise actuellement 2 valeurs possibles: 400 ou 700.
          homeFactory.addTitlesIsBoldProperty(scope.homepage);

          if (Array.isArray(scope.homepage.groups) && scope.homepage.groups.length > 0) {
            // sélectionne le 1er groupe et affiche ses liens
            const activeGroup = scope.homepage.groups.find(grp => grp.active);
            for (const group of scope.homepage.groups) {
              // déselectionnne tous les groupes différents de activeGroup
              group.active = activeGroup !== undefined && group.id === activeGroup.id;
              // assigne un alignement vertical si absent
              if (!group.title.valign) {
                group.title.valign = homeFactory.vAlignPositions[1]; // center
              }
            }
            if (activeGroup && activeGroup.id !== undefined) {
              filterItems(activeGroup.id);
            } else {
              scope.homepage.groups[0].active = true;
              filterItems(scope.homepage.groups[0].id);
            }
          }

          if (Array.isArray(scope.homepage.links) && scope.homepage.links.length > 0) {
            for (const link of scope.homepage.links) {
              // assigne un alignement vertical si absent
              if (!link.title.valign) {
                link.title.valign = homeFactory.vAlignPositions[1]; // center
              }
            }
          }
        }

        // vérifie si l'utilisateur à accès à la config de la page d'accueil
        scope.userCanConfigureHomepage = 
          homeFactory.isUserAllowedToConfigureHomepage($rootScope.xgos.user);

        // récupère les noms des images du dossier CSS-HTML du repo
        // les svg pour les groupes sont placés dans images.groups
        // les svg pour les liens sont placés dans images.links
        // les images (autre que svg) pour le logo et arrière-plan sont placés dans images.other
        homeFactory.getImages().then(
            (images) => {
              // les images reçues sont transformées en objet {filename,active,base64} dans le service
              // @see homeFactory.transformImage()
              if (homeFactory.resources.images && homeFactory.resources.images) {
                scope.images = {};
                scope.images.other = images.other ? images.other : [];
                scope.images.links = images.links ? images.links : [];
                scope.images.groups = images.groups ? images.groups : [];
              } else {
                scope.images = {groups: [], links: [], other: []};
              }
            }
        );

        // initialise au minimum l'encart du fil d'actualité: le titre 'Fil d'actualité' par défaut
        if (!scope.homepage.news) {
          scope.homepage.news = {};
        }
        if (!scope.homepage.news.title) {
          scope.homepage.news.title = $filter('translate')('portals.homepage.v2.news.title');
        }
      },
    };
  };

  homepageNews.$inject = [
    '$rootScope',
    '$filter',
    'extendedNgDialog',
    'gaJsUtils',
    'gaDomUtils',
    'homeFactory',
    'authFactory'
  ];
  return homepageNews;
});