'use strict';
define(function () {
  const getFeaturesFromLineAssociationRule = ($rootScope, GeometryFactory, $q, FeatureTypeFactory,
      extendedNgDialog, $filter, gclayers) => {

    /**
     * Construit une map contenant le champ identifiant du composant de l'association patrimoine chargée dans la règle métier.
     * Il est nécessaire de connaître le champ utilisé dans l'association
     * pour construire les objets allégés qui seront affichés dans la table des objets associés
     * @param baseFtiName nom du composant contenant la règle métier. Le composant est celui de l'objet en cours d'édition
     * @param ruleConf configuration de la règle métier getFeaturesFromLineAssociation
     * @return {Map<string, string>} nom du fti en clé et nom du champ identifiant en valeur
     */
    const getAssocIdFieldnames = (ruleConf, baseFtiName) => {
      const associations = ruleConf.parameters.associationsToLoad;
      const assocIdFieldnames = new Map();
      for (const assoc of associations) {

        // déclare et initialise deux constantes atable et btable avec les valeurs respectives des propriétés atable et btable de l'objet assoc
        const {atable, btable, afield, bfield} = assoc;

        if (atable && btable) {
          const ftiNameToAdd = (atable === baseFtiName) ? btable : atable;
          const idFieldName = (atable === baseFtiName) ? bfield : afield;
          assocIdFieldnames.set(ftiNameToAdd, idFieldName);
        }
      }
      return assocIdFieldnames;
    };

    /**
     * Construit la liste des objets associés allégés de l'objet en cours d'édition
     * @param geojsonData objets chevauchant l'objet en cours d'édition
     * @param baseFtiName nom du composant contenant la règle métier. Le composant est celui de l'objet en cours d'édition
     * @param ruleConf configuration de la règle métier getFeaturesFromLineAssociation
     * @return {{features: *[], totalFeatures: number}} objet geojson contenant un tableau "features"
     * des objets associés allégés pour affichage dans la table de la popup des objets associés à l'objet en cours d'édition
     */
    const setAssociatedFeatures = (geojsonData, ruleConf, baseFtiName) => {

      // récupère le nom du champ utilisé comme identifiant du composant associé
      const assocIdFieldnames = getAssocIdFieldnames(ruleConf, baseFtiName);

      const associatedFeatures = {features: [], totalFeatures: 0};

      if (geojsonData && Array.isArray(geojsonData.features)) {
        for (const feature of geojsonData.features) {

          const ftiName = feature.id.split('.')[0];

          // création des objets associés avec une structure allégée et des champs particuliers pour affichage dans la table
          if (ftiName && ftiName.length > 0 && assocIdFieldnames.has(ftiName)) {
            const idFieldName = assocIdFieldnames.get(ftiName);
            const lightFeature = {
              geometry: feature.geometry,
              id: feature.id,
              properties: {
                Identifiant: feature.properties[idFieldName],
                Couche: feature.id.split('.')[0],
                Alias: ''
              },
              type: 'Feature'
            };
            associatedFeatures.features.push(lightFeature);
            associatedFeatures.totalFeatures++;
          }
        }
      }
      return associatedFeatures;
    };

    /**
     * Créé une propriété scope pour la popup contenant la datatable des objets associés à l'objet en cours d'édition.
     * Initialise les propriétés nécessaires pour le fonctionnement de la datatable et du select-drag.
     * Initialse les méthodes de suppression de ligne, la méthode de mise à jour de la table après sélection par cadre
     * et la méthode exécutée au clic sur le bouton "Valider"
     * @param editDescription objet en cours d'édition
     * @param map carte openlayers de l'application KIS-Map
     * @param overlapGeojson objets chevauchant l'objet en cours d'édition
     * @param ruleConf configuration de la règle métier getFeaturesFromLineAssociation
     * @param baseFtiName nom du composant contenant la règle métier. Le composant est celui de l'objet en cours d'édition
     */
    const setPopupScope = (editDescription, map, overlapGeojson, ruleConf, baseFtiName) => {
      const scope = $rootScope.$new();
      scope.map = map;
      scope.manualComponent = {};
      scope.manualSelection = {};
      scope.geomselect = null;
      scope.editDescription = editDescription;
      scope.associatedFeatures = setAssociatedFeatures(overlapGeojson, ruleConf, baseFtiName);

      scope.geojsonData = overlapGeojson;

      const ftiNames = getAssocIdFieldnames(ruleConf, baseFtiName).keys();
      scope.associatedFtis = Array.from(ftiNames).map(name => FeatureTypeFactory.getFeatureByName(name)).filter(fti => fti !== null);
      console.log("ftis ", scope.associatedFtis);

      // méthode pour supprimer une ligne de la table des objets associés à l'objet édité
      scope.selectedObjectRemove = (feature) => {
        scope.associatedFeatures.features = scope.associatedFeatures.features.filter(
            selFeat => selFeat.id !== feature.id);
        scope.associatedFeatures.totalFeatures -= 1;
        gclayers.clearhighLightFeatures();
        scope.$broadcast('reloadDatatable');
      };

      // méthode exécutée au clic sur le bouton "Valider" pour copier les objets associés
      // dans une variable de l'objet editDescription pour une prise en compte à la fin de la session d'édition
      scope.validateAssociatedFeatures = () => {
        if (scope.associatedFeatures.features.length > 0) {
          scope.editDescription.objectCibleToAdd = scope.associatedFeatures;
          scope.editDescription.associationToLoad = ruleConf.parameters.associationsToLoad;
        }
      };

      // méthode exécutée après selection par cadre
      scope.refreshTable = () => {
        if (scope.manualSelection.geojson && Array.isArray(scope.manualSelection.geojson.features)) {
          let isFtiAssociated = true;
          for (const feature of scope.manualSelection.geojson.features) {

            const ftiName = scope.manualComponent.value.name;
            const assocIdFieldnames = getAssocIdFieldnames(ruleConf, baseFtiName);

            // création des objets associés avec une structure allégée et des champs particuliers pour affichage dans la table
            if (ftiName && ftiName.length > 0 && assocIdFieldnames.has(ftiName)) {
              const idFieldName = assocIdFieldnames.get(ftiName);
              const lightFeature = {
                geometry: feature.geometry,
                id: feature.id,
                properties: {
                  Identifiant: feature.properties[idFieldName],
                  Couche: feature.id.split('.')[0],
                  Alias: ''
                },
                type: 'Feature'
              };
              scope.associatedFeatures.features.push(lightFeature);
              scope.associatedFeatures.totalFeatures++;
            } else {
              isFtiAssociated = false;
            }
          }
          if (isFtiAssociated) {
            scope.$broadcast('reloadDatatable');
          } else {
            require('toastr').error($filter('translate')('rulecfg.getFeaturesFromLineAssociation.noAssociatedFti') + baseFtiName);
          }
        }
      };

      // fake fti utile uniquement pour l'initialisation de la datatable
      scope.relationObjFti = {
        'uid': '3a716ffb-77f4-4ffa-909e-5bae9f581f48',
        'name': 'relationObjFti',
        'alias': 'relationObjFti',
        'attributes': [
          {
            'name': 'Identifiant',
            'alias': 'Identifiant',
            'type': 'java.lang.Integer',
            '$$hashKey': 'object:6097'
          },
          {
            'name': 'Couche',
            'alias': 'Couche',
            'type': 'java.lang.String',
            '$$hashKey': 'object:6096'
          }
        ],
        'nameAliasMapping': {
          'Couche': 'Couche',
          'Identifiant': 'Identifiant'
        },
        '$selected': true,
      };
      // bouton d'action dans la datatable
      scope.specialActions = [
        {
          class: 'btn btn-danger btn-xs',
          icone: 'fa fa-times',
          text: $filter('translate')('rulecfg.getFeaturesFromLineAssociation.delete'),
          callFunction: scope.selectedObjectRemove,
        },
      ];
      return scope;
    };

    /**
     * Récupère le tableau des fti existants parmi les associations patrimoine définies concernant le composant de l'objet en cours d'édition
     * @param baseFtiName nom du composant de l'objet en cours d'édition contenant la règle métier.
     * @param ruleConf configuration de la règle métier getFeaturesFromLineAssociation
     * @return {object[]} tableau des fti correspondant à l’ensemble des composants récupérés des associations
     */
    const getAssocFtis = (ruleConf, baseFtiName) => {
      const assocFtis = [];
      const associations = ruleConf.parameters.associationsToLoad;
      for (const assoc of associations) {

        // déclare et initialise deux constantes atable et btable avec les valeurs respectives des propriétés atable et btable de l'objet assoc
        const {atable, btable} = assoc;

        if (atable && btable) {
          const ftiNameToAdd = (atable === baseFtiName) ? btable : atable;
          const uidToAdd = FeatureTypeFactory.getFeatureUidByName(ftiNameToAdd);
          if (uidToAdd) {
            assocFtis.push(uidToAdd);
          }
        }
      }
      return assocFtis;
    };

    /**
     * Méthode principale de la règle métier exécutée par EditRulesFactory
     * @param editDescription objet en cours d'édition
     * @param ruleConf configuration de la règle métier getFeaturesFromLineAssociation
     * @param fti caractéristiques du composant de l'objet en cours d'édition contenant la règle métier
     * @param map carte openlayers de l'application KIS-Map
     * @return {Promise}
     */
    const getFeaturesFromLineAssociation = (editDescription, ruleConf, fti, map) => {
      const defer = $q.defer();
      const baseFtiName = fti.name;

      if (ruleConf.parameters && Array.isArray(ruleConf.parameters.associationsToLoad)) {

        // transformation de la géométrie de l'objet édité en geojson
        const format = new ol.format.GeoJSON();
        const geometry = editDescription.editedfeature.getGeometry();
        const geometryAsGeojson = format.writeGeometryObject(geometry);

        // création de la liste des uid de fti associés au composant de l'objet édité
        const assocFtis = getAssocFtis(ruleConf, fti.name);

        // srid de la carte
        const mapSrid = map.getView().getProjection().getCode();

        // tolérance de 10 cm pour compenser l'approximation de l'accrochage-objet (snapOn)
        GeometryFactory.overlap(geometryAsGeojson, assocFtis, null, mapSrid, 0.1).then(
            res => {

              // le scope de la popup doit contenir:
              // - la datatable des objets associés à l'objet en cours d'édition
              // - les propriétés nécessaires pour le fonctionnement de la datatable et du select-drag
              // - les méthodes de suppression de ligne, la méthode de mise à jour de la table après sélection par cadre
              // - la méthode exécutée au clic sur le bouton "Valider"
              const scope = setPopupScope(editDescription, map, res.data, ruleConf, baseFtiName);

              const baseFti = FeatureTypeFactory.getFeatureByName(baseFtiName);
              const popupTitle = $filter('translate')('rulecfg.getFeaturesFromLineAssociation.popupTitle') + (baseFti ? baseFti.alias : '');

              extendedNgDialog.open({
                template: 'js/XG/modules/edit/views/rules/rulesPopup/getFeaturesFromLineAssociationRule.html',
                className: 'ngdialog-theme-plain miniclose getFeaturesFromLineAssociation-popup',
                closeByDocument: false,
                scope: scope,
                title: popupTitle,
                draggable: true,
                scrollable: true,
                minimizeMaximize: true,
                resizable: true,
                minWidth: '530px',
                minHeight: '350px'
              });
              defer.resolve();
            },
            () => {
              // erreur serveur (la factory affiche déjà une erreur console)
              defer.reject('getFeaturesFromLineAssociation');
            }
        );
      }
      return defer.promise;
    };
    return {
      getFeaturesFromLineAssociation: getFeaturesFromLineAssociation
    };
  };

  getFeaturesFromLineAssociationRule.$inject = ['$rootScope', 'GeometryFactory', '$q',
    'FeatureTypeFactory', 'extendedNgDialog', '$filter', 'gclayers'];
  return getFeaturesFromLineAssociationRule;
});
