/**
 * Implémentation des règles métiers d'édition.
 * @returns {EditRulesProvider_L6.EditRulesProvider}
 */
'use strict';
define(function() {
  let EditRulesProvider = {
    $get: function(
      gclayers,
      gcWFS,
      $q,
      ogcFactory,
      EditTypesFactory,
      GeometryFactory,
      gcPopup,
      $rootScope,
      SridFactory,
      ConfigFactory,
      ChartsFactory,
      $filter,
      What3WordFactory,
      $timeout,
      CopyPasteAttributeFactory,
      gaJsGeneral,
      gaDomUtils,
      EditRuleAllowDelete,
      EditRuleMoveObjectOnEnd,
      EditRuleObjectNotAllowedAtIntersection,
      EditRuleSetDefaultValueByRole,
      EditRuleSnappedOnFeature,
      ngDialog,
      MustNotSelfIntersectRule,
      MustNotSelfOverlapRule,
      GetFeaturesFromLineAssociationRule,
      bizeditProvider
    ) {
      ///////// Fonctions utilitaires ///////////////////////////////////

      let format = new ol.format.GeoJSON();

      /**
       * Vérifie si l'objet partagé est present dans le tableau des objets liés destinés à être sauvegardés.
       * @param {type} shareObjectName: nom de l'objet partagé.
       * @param {type} editdescription: objet contenant les informations sur l'édition en cours (voir bizeditwidget.js).
       * @returns {Boolean} true si l'objet partagé est present dans le tableau de sauvegarde, false sinon.
       */
      function isObjectInRelatedFeatures(shareObjectName, editdescription) {
        var isFeatureExistInRelated = false;
        for (var k = 0; k < editdescription.relatedfeatures.length; k++) {
          if (
            editdescription.relatedfeatures[k].shareObject == shareObjectName
          ) {
            isFeatureExistInRelated = true;
            break;
          }
        }
        return isFeatureExistInRelated;
      }

      /**
       * Renvoie l'objet layer situé dans le tableau layers passé en second argument et
       * dont le nom correspond au premier argument layerName.
       * @param {type} layerName
       * @param {type} layers
       * @returns {unresolved}
       */
      function getLayerByName(layerName, layers) {
        var layer = undefined;
        for (var i = 0; i < layers.length; i++) {
          var l = layers[i];
          if (l.name == undefined) continue;
          if (layerName == l.name) {
            layer = l;
            break;
          }
        }
        return layer;
      }

      /**
       * Retourne l'angle du segment P1P2 par rapport à l'axe des x,
       * dans le sens horaire, en degré.
       * @param {type} p1
       * @param {type} p2
       * @returns {Number}
       */
      function getP1P2Angle(p1, p2) {
        //Angle trigo entre le segment p1p2 et l'axe des x

        var dx = p2[0] - p1[0];
        var dy = p2[1] - p1[1];

        var radian = Math.atan(Math.abs(dy) / Math.abs(dx));

        //L'angle considéré par le style a le sens inverse trigonometrique
        if (dy > 0 && dx > 0) {
          radian = 2 * Math.PI - radian;
        } else if (dy < 0 && dx > 0) {
        } else if (dx < 0 && dy > 0) {
          radian = Math.PI + radian;
        } else if (dy < 0 && dx < 0) {
          radian = Math.PI - radian;
        }

        //Conversion en degre et arrondi à l'entier le plus proche.
        return Math.round(radian * (180 / Math.PI));
      }

      /**
       * Affecte une valeur d'orientation dans le featurePoint à partir de l'objet intersectedFeatureLine et d'un offset eventuel.
       * @param {type} featurePoint
       * @param {type} rotateFieldName
       * @param {type} intersectedFeatureLine
       * @param {type} rotationOffset
       * @returns {undefined}
       */
      function setPointRotationFromLineObject(
        featurePoint,
        rotateFieldName,
        intersectedFeatureLine,
        rotationOffset
      ) {
        var pointCoordinate = featurePoint.getGeometry().getCoordinates();

        var line = intersectedFeatureLine.getGeometry();
        var isSimpleLine = line instanceof ol.geom.LineString;
        var isMultiLine = line instanceof ol.geom.MultiLineString;

        //Index du segment sur lequel couper la line.
        var cuttingIndex = undefined;
        var cuttinglineIndex = undefined;

        //Points du segment courant pour lequel on recherche l'intersection avec le featurePoint
        var p1 = undefined;
        var p2 = undefined;
        if (isSimpleLine) {
          var lineCoordinates = line.getCoordinates();
          cuttingIndex = getIndexOnSimpleLine(lineCoordinates, pointCoordinate);

          if (cuttingIndex == undefined) {
            console.info(
              'setPointRotationFromLineObject() Pas de segment trouvé sous le point'
            );
            return;
          }
          p1 = lineCoordinates[cuttingIndex];
          p2 = lineCoordinates[cuttingIndex + 1];
        } else if (isMultiLine) {
          var simpleLines = line.getCoordinates();
          //Pour chaque simple line de la multiline
          for (var lineIndex = 0; lineIndex < simpleLines.length; lineIndex++) {
            cuttingIndex = getIndexOnSimpleLine(
              simpleLines[lineIndex],
              pointCoordinate
            );
            //Si l'index du segement de la simple line a été trouvé
            if (cuttingIndex != undefined) {
              //mémorisation de l'index de la simple line
              cuttinglineIndex = lineIndex;
              break;
            }
          }
          if (cuttingIndex == undefined) {
            console.info(
              'setPointRotationFromLineObject() Pas de segment trouvé sous le point'
            );
            return;
          }
          var simpleLine = simpleLines[cuttinglineIndex];
          p1 = simpleLine[cuttingIndex];
          p2 = simpleLine[cuttingIndex + 1];
        }

        //Affectation de la valeur de la rotation
        var property = {};
        property[rotateFieldName] = getP1P2Angle(p1, p2);
        if (rotationOffset != undefined) {
          property[rotateFieldName] =
            property[rotateFieldName] + rotationOffset;
        }
        featurePoint.setProperties(property);
      }

      /**
       * Ajoute l'objet dans le tableau des objets à sauver uniqument si il n'est pas déjà present dans ce tableau.
       * @param {type} editdescription
       * @param {type} objectToDelete
       * @returns {undefined}
       */
      function recordObjectIfNotExists(editdescription, objectToDelete) {
        var isAlreadyRecorded = false;
        for (var i = 0; i < editdescription.relatedfeatures.length; i++) {
          var related = editdescription.relatedfeatures[i];
          if (related.editType == objectToDelete.editType) {
            if (related.feature.getId() == objectToDelete.feature.getId()) {
              isAlreadyRecorded = true;
              break;
            }
          }
        }
        if (!isAlreadyRecorded) {
          editdescription.relatedfeatures.push(objectToDelete);
        }
      }

      /**
       * Donne l'index 'cuttingIndex' correspondant au premier point de la simpleLine 'lineCoordinates' du segment sur lequel se trouve le point 'intersectionPointCoordinate'.
       * Retourne undefined si le point n'est pas situé sur la simple line.
       * @param {type} lineCoordinates
       * @param {type} intersectionPointCoordinate
       * @returns {Number}cuttingIndex
       */
      function getIndexOnSimpleLine(
        lineCoordinates,
        intersectionPointCoordinate
      ) {
        var cuttingIndex = undefined;

        //Pour chaque segment de la simple line
        for (var i = 0; i < lineCoordinates.length - 1; i++) {
          //Les deux points du segments
          var p1 = lineCoordinates[i];
          var p2 = lineCoordinates[i + 1];

          //Si le point édité est posé sur le premier point du segment ou sur le second point (x et y sont les mêmes à 1cm près)
          if (
            (Math.round(intersectionPointCoordinate[0] * 100) ==
              Math.round(p1[0] * 100) &&
              Math.round(intersectionPointCoordinate[1] * 100) ==
                Math.round(p1[1] * 100)) ||
            (Math.round(intersectionPointCoordinate[0] * 100) ==
              Math.round(p2[0] * 100) &&
              Math.round(intersectionPointCoordinate[1] * 100) ==
                Math.round(p2[1] * 100))
          ) {
            cuttingIndex = i;
            break;
          }
          //Si le point édité est posé entre les deux points du segment
          else {
            //Coef directeur de la line
            var coef = (p2[1] - p1[1]) / (p2[0] - p1[0]);
            //Coef directeur de la droite intersectionPointCoordinate et une des extrémité (celle qui ne se superpose pas au point édité)
            var coef2 = NaN;
            coef2 =
              (intersectionPointCoordinate[1] - p1[1]) /
              (intersectionPointCoordinate[0] - p1[0]);
            //Si intersectionPoint appartient à la DROITE line correspondant au segment en cours P1P2
            if (
              Math.abs(Math.round(coef * 100)) ==
              Math.abs(Math.round(coef2 * 100))
            ) {
              //Vérification si intersectionPointCoordinate appartient au SEGMENT P1P2
              if (
                intersectionPointCoordinate[0] <= Math.max(p1[0], p2[0]) &&
                intersectionPointCoordinate[0] >= Math.min(p1[0], p2[0]) &&
                intersectionPointCoordinate[1] <= Math.max(p1[1], p2[1]) &&
                intersectionPointCoordinate[1] >= Math.min(p1[1], p2[1])
              ) {
                cuttingIndex = i;
                break;
              }
            }
          }
        }
        return cuttingIndex;
      }

      /**
       * Non utilisée.
       * @param {type} line1
       * @param {type} line2
       * @param {type} intersectingPoint
       * @returns {ol.geom.MultiLineString}
       */
      function mergeTwoLines(line1, line2, intersectingPoint) {
        var deletedCoordinate = intersectingPoint.getCoordinates();
        var firstLine = undefined;
        var secondLine = undefined;

        //Si les coordonnées en x et en y sont égales à 10cm près
        if (
          Math.round(line1.getLastCoordinate()[0] * 10) ==
            Math.round(deletedCoordinate[0] * 10) &&
          Math.round(line1.getLastCoordinate()[1] * 10) ==
            Math.round(deletedCoordinate[1] * 10)
        ) {
          firstLine = line1;
          secondLine = line2;
        } else if (
          Math.round(line2.getLastCoordinate()[0] * 10) ==
            Math.round(deletedCoordinate[0] * 10) &&
          Math.round(line2.getLastCoordinate()[1] * 10) ==
            Math.round(deletedCoordinate[1] * 10)
        ) {
          firstLine = line2;
          secondLine = line1;
        }
        //Concatenation de la first line avec la seconde line
        var isSimpleLine1 = firstLine instanceof ol.geom.LineString;
        var isMultiLine1 = firstLine instanceof ol.geom.MultiLineString;
        var isSimpleLine2 = secondLine instanceof ol.geom.LineString;
        var isMultiLine2 = secondLine instanceof ol.geom.MultiLineString;

        var multiCoordinates = [];

        if (isSimpleLine1) {
          multiCoordinates.push(firstLine.getCoordinates());
        } else if (isMultiLine1) {
          multiCoordinates = multiCoordinates.concat(
            firstLine.getCoordinates()
          );
        }
        if (isSimpleLine2) {
          multiCoordinates.push(secondLine.getCoordinates());
        } else if (isMultiLine2) {
          multiCoordinates = multiCoordinates.concat(
            secondLine.getCoordinates()
          );
        }
        //-- Retourne la ligne fusionnée
        return new ol.geom.MultiLineString(multiCoordinates, 'XY');
      }

      function toRad(Value) {
        /** Converts numeric degrees to radians */
        return (Value * Math.PI) / 180;
      }

      var wgs84Sphere = new ol.Sphere(6378137);
      function getPreciseLineLength(simpleLine, map) {
        var length = 0;
        var coordinates = simpleLine.getCoordinates();
        var sourceProj = map
          .getView()
          .getProjection()
          .getCode();

        for (var i = 0, ii = coordinates.length - 1; i < ii; ++i) {
          var c1 = ol.proj.transform(coordinates[i], sourceProj, 'EPSG:4326');
          var c2 = ol.proj.transform(
            coordinates[i + 1],
            sourceProj,
            'EPSG:4326'
          );
          length += wgs84Sphere.haversineDistance(c1, c2);
        }
        return length;
      }
      function getSimpleLineLength(simpleLine, map, fti) {
        var ftiUid = '';
        if (
          fti &&
          angular.isDefined(fti) &&
          fti.uid &&
          angular.isDefined(fti.uid)
        ) {
          ftiUid = fti.uid;
          var fc = {
            type: 'FeatureCollection',
            crs: {
              type: 'name',
              properties: {
                name: map
                  .getView()
                  .getProjection()
                  .getCode(),
              },
            },
            features: [
              {
                type: 'Feature',
                properties: {},
                geometry: {
                  type: simpleLine.getType(),
                  coordinates: simpleLine.getCoordinates(),
                },
                uid: ftiUid,
              },
            ],
            totalFeatures: 1,
          };
          return ChartsFactory.areaperimeter(fc, false);
        } else {
          console.error(
            'Règle getSimpleLineLength mal ou pas configurée : vérifie le fti de la couche'
          );
        }
      }

      function getSimplePolygonArea(simpleLine, map, featureType) {
        var fc = {
          type: 'FeatureCollection',
          crs: {
            type: 'name',
            properties: {
              name: map
                .getView()
                .getProjection()
                .getCode(),
            },
          },
          features: [
            {
              type: 'Feature',
              properties: {},
              geometry: {
                type: simpleLine.getType(),
                coordinates: simpleLine.getCoordinates(),
              },
              uid: featureType.uid,
            },
          ],
          totalFeatures: 1,
        };

        return ChartsFactory.areaperimeter(fc, false);
      }

      //////// FIN METHODES UTILITAIRES //////////////////////////////////

      ////////// Implémentation des règles métiers ///////////////////////

      /// Régles métiers executées à l'initialisation ////////////////////

      function snapFailed(deferred) {
        require('toastr').error(
          $filter('translate')('rulecfg.snapon.snapIsMandatory')
        );
        deferred.reject({
          error: true,
          errorType: 'snapFailed',
        });
      }

      function snapCrExistMoveExtremityLines(params) {
        var ind;
        for (ind = 0; ind < params.rules.length; ind++)
          if (params.rules[ind].name == 'MoveExtremityLines') return true;
        return false;
      }

      function checkSnapRestrictionsIsOk(
        editdescription,
        params,
        ruleConf,
        features
      ) {
        var iFeat, layerName, snapcfg, val;

        if (ruleConf.parameters.must == 'true') return true;

        for (iFeat = 0; iFeat < features.length; iFeat++) {
          layerName = GlobalServices.getLayerNameFromFullIdValue(
            features[iFeat].id
          );
          snapcfg = ruleConf.parameters.snapOnSet[layerName];
          if (snapcfg && snapcfg.fieldName) {
            if (snapcfg.snapSet != 'attribute') return true;
            val = features[iFeat].properties[snapcfg.fieldName];
            if (snapcfg.mustAttributeVal[val] == 'true') return true;
          }
        }
        return false;
      }

      function snapOnHaveToSnap(snapParams, editdesc, fti) {
        if (snapParams['must'] == 'true') return true;
        if (
          snapParams['must'] == 'attribute' &&
          snapParams.mustAttribute == editdesc.subtypefn
        ) {
          return snapParams.mustAttributeVal[editdesc.subtypevalue] == 'true';
        }
        return false;
      }
      /**
       * Règle d'accroche sur objets appartenant aux couches wfs chargées une à une sur la carte
       *  et dont les noms correspondent à ceux configurés pour cette règle.
       * @param {object} editdescription objet de l'édition métier en cours
       * @param {object} ruleConf configuration de la règle snapOn. Element du tableau fti.rules
       * @param {object} featureType fti sur lequel s'applique la règle métier
       * @param {ol.Map} map carte
       * @return {Promise} vide si la règle a pu s'appliquer sinon contient une erreur serveur
       */
      function snapOn(editdescription, ruleConf, featureType, map) {
        var layersToLoad = [];
        var layers = gclayers.getOperationalLayer();
        var layerNames = ruleConf.parameters['layers'];

        function snapCheckResult(editdescription, params, ruleConf) {
          var deferred = $q.defer();
          var coord;
          if (
            editdescription.originalfeature != undefined &&
            (editdescription.strictsnap == undefined ||
              !editdescription.strictsnap) &&
            snapCrExistMoveExtremityLines(params)
          )
            //-- Dans le cas du déplacement d'un point où on va déplacer
            //-- les lignes aux extrémités on doit vérifier que le point
            //-- original est bien connecté aux ligne
            //-- et non pas le nouveau (la nouvelle position).
            coord = editdescription.originalfeature
              .getGeometry()
              .getCoordinates();
          else {
            if (params.evt.feature == undefined) {
              //-- On doit être ici par erreur !!!
              deferred.resolve();
              return deferred.promise;
            }
            coord = params.evt.feature.getGeometry().getCoordinates();
          }
          while (typeof coord[0] == 'object') coord = coord[0];
          var tol = 0.25;
          var selGeomCql =
            'POLYGON((' +
            (coord[0] - tol) +
            ' ' +
            (coord[1] - tol) +
            ',' +
            (coord[0] + tol) +
            ' ' +
            (coord[1] - tol) +
            ',' +
            (coord[0] + tol) +
            ' ' +
            (coord[1] + tol) +
            ',' +
            (coord[0] - tol) +
            ' ' +
            (coord[1] + tol) +
            ',' +
            (coord[0] - tol) +
            ' ' +
            (coord[1] - tol) +
            '))';

          var spatialClause = 'INTERSECTS(geom, ' + selGeomCql + ')';
          var promise = ogcFactory.getfeatures(
            'GetFeature',
            'WFS',
            '1.0.0',
            params.ftiList,
            'json',
            map
              .getView()
              .getProjection()
              .getCode(),
            spatialClause
          );
          promise.then(
            function(res) {
              if (
                res.data.features.length &&
                checkSnapRestrictionsIsOk(
                  editdescription,
                  params,
                  ruleConf,
                  res.data.features
                )
              )
                deferred.resolve();
              else snapFailed(deferred);
            },
            function(res) {
              snapFailed(deferred);
            }
          );
          return deferred.promise;
        }

        if (snapOnHaveToSnap(ruleConf.parameters, editdescription, featureType))
          ruleConf.checkResult = snapCheckResult;
        else ruleConf.checkResult = undefined;
        ruleConf.checkResultParams = {};
        ruleConf.checkResultParams.ftiList = [];
        let filters = [];
        //Pour chaque nom de layer en config de cette règle
        for (var j = 0; j < layerNames.length; j++) {
          var name = layerNames[j].name;
          //Pour chaque layer opérationelle
          for (var i = 0; i < layers.length; i++) {
            var l = layers[i];
            if (l.name == undefined) {
              continue;
            }
            if (name == l.name) {
              var fti = l.fti;
              layersToLoad.push(fti);
              ruleConf.checkResultParams.ftiList.push(fti.uid);
              //set esri filters
              if(featureType.type == 'esri' && ruleConf.parameters.must=='attribute'){
                let desc = ruleConf.parameters.snapOnSet[name];
                let strFilter = '';
                for (var prop in desc.mustAttributeVal) {
                  if (strFilter != '') strFilter += ' OR ';
                  strFilter += desc.fieldName + '=\'' + prop + '\'';
                }
                filters.push({
                  'ftiUid': fti.uid,
                  'filter': '(' + strFilter + ')'
                });
              }
              break;
            }
          }
        }
        //Chargement des objets des couches sur lesquelles s'accrocher
        let wfsLayer = gcWFS.getOlLayerFromFeaturetypeInfoArray(layersToLoad, map, (filters.length == 0) ? undefined : filters);
        gclayers.addSnapLayer(wfsLayer);
        var srcDrawedDXF = gclayers.getImportLayer().getSource();
        wfsLayer.getSource().addFeatures(srcDrawedDXF.getFeatures());

        var config = map.get(editdescription.theme);
        var snapTolerance;
        //Récupération de la tolérance de snapping (paramètre de configuration du widget)
        config
          ? (snapTolerance = Number(config.snaptolerance))
          : (snapTolerance = Number(map.get('snaptolerance'))); // * resolution;

        if (snapTolerance == undefined || isNaN(snapTolerance)) {
          snapTolerance = 20;
        } // * resolution;

        //Définir la couleur de la couche wfs ol d'accrochage
        //Si l'utilisateur choisi une couleur depuis l'interface admin (thème),
        //on affiche la couleur sélectionnée sinon on met la couleur par défaut (couleur du logo KIS)
        let snapColor;
        if($rootScope && $rootScope.xgos && $rootScope.xgos.portal 
            && $rootScope.xgos.portal.parameters && $rootScope.xgos.portal.parameters.snapColor) {
          snapColor = $rootScope.xgos.portal.parameters.snapColor;
        } else {
          snapColor = '#f31f72';
        }
        // Définir un style spécifique basé sur le type de géométrie
        let getLayerStyle = (type) => {
          if (type === 'POLYGON') {
            return new ol.style.Style({
              fill: new ol.style.Fill({
                color: 'rgba(0, 0, 0, 0)' // Remplissage transparent
              }),
              stroke: new ol.style.Stroke({
                color: snapColor,
                width: 2
              })
            });
          } else {
            return new ol.style.Style({
              fill: new ol.style.Fill({
                color: snapColor
              }),
              stroke: new ol.style.Stroke({
                color: snapColor,
                width: 2
              }),
              image: new ol.style.Circle({
                radius: 5,
                fill: new ol.style.Fill({
                  color: snapColor
                }),
                stroke: new ol.style.Stroke({
                  color: '#000000',
                  width: 1
                })
              })
            });
          }
        };

        // Appliquer un style spécifique à chaque couche dans layersToLoad
        layersToLoad.forEach(layer => {
          let type = layer.typeInfo; // Type de la géométrie
          let layerStyle = getLayerStyle(type); 
          wfsLayer.setStyle(layerStyle);
        });
        
        //Instanciation de l'interaction OpenLayer d'accroche sur objets
        // appartenant aux layers chargées dans l'objet 'source' de la couche WFS crée.
        var snap = new ol.interaction.Snap({
          source: wfsLayer.getSource(),
          edge: true,
          vertex: true,
          pixelTolerance: snapTolerance,
        });

        snap.set('gctype', 'kis');
        snap.set('interaction', 'Snap');
        snap.set('widget', 'Edition');
        snap.setActive(true);

        //Ajout de l'interaction à la carte
        map.addInteraction(snap);
        if (
          map
            .getInteractions()
            .getArray()
            .indexOf(snap) !== -1
        ) {
          //Envoi de l'information de l'interaction ajoutée.
          map.dispatchEvent({ type: 'snapAddedEvent' });
          //Enregistrement de la reference vers l'interaction pour la retirer de la carte au besoin.
          editdescription.interactions.push(snap);
          console.info('Bizeditwidget: intéraction snapOn chargée.');
        }
        var mess = ruleConf.parameters['inputMessage'];
        if (mess != undefined && mess.length !== 0) {
          var t = require('toastr');
          t.info(mess, 'Info', {
            showDuration: 1800,
            timeOut: 2500,
          });
        }
      }

      /**
       * Copie les features éditées et enregistre ces copies dans l'objet editdescription
       * en vue d'être ajouté à la couche d'historisation lorsque la sauvegarde de la session d'édition aura lieu.
       * @param {object} editdescription objet de l'édition métier en cours
       * @param {object} ruleConf configuration de la règle snapOn. Element du tableau fti.rules
       * @param {object} featureType fti sur lequel s'applique la règle métier
       * @param {ol.Map} map carte
       * @returns {Promise} contient un string
       */
      function historicize(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        function isEditType(element, indice, array) {
          return element.name == this;
        }

        //Traitement de l'objet principal
        //Copie du feature
        var mainFeatureCopie = editdescription.editedfeature.clone();
        //Copie de l'id
        mainFeatureCopie.setId(editdescription.editedfeature.getId());
        //Enregistrement de la copie pour historisation
        var newObject = {
          shareObject: '',
          editType: EditTypesFactory.editTypes.tohistorize.name,
          feature: mainFeatureCopie,
          fti: editdescription.fti,
        };
        editdescription.relatedfeatures.push(newObject);

        //Traitement des objets secondaires
        var featuresToHistorize = [];
        for (var i = 0; i < editdescription.relatedfeatures.length; i++) {
          var related = editdescription.relatedfeatures[i];
          //Si le type d'édition de l'objet secondaire est un de ceux précisés dans la config de la règle et
          // que la layer de l'objet secondaire est la même que celle de l'édition principale
          if (
            related.fti.uid === editdescription.fti.uid &&
            ruleConf.editTypes.some(isEditType, related.editType)
          ) {
            var feature = related.feature;
            var featureCopie = feature.clone();
            //Copie de l'id
            featureCopie.setId(feature.getId());
            //Ajout de l'objet en tant qu'objet à historiser.
            //Faire une copie de l'objet car cette copie va recevoir un nouvel id dans la table d'historique. Les relations avec les interventions se feront coté serveur.
            newObject = {
              shareObject: '',
              editType: EditTypesFactory.editTypes.tohistorize.name,
              feature: featureCopie,
              fti: related.fti,
            };
            featuresToHistorize.push(newObject);
          }
        }
        if (featuresToHistorize.length > 0) {
          editdescription.relatedfeatures = editdescription.relatedfeatures.concat(
            featuresToHistorize
          );
        }

        deferred.resolve('historicize terminé !');
        return deferred.promise;
      }

      /// Régles métiers executées sur l'evenement de fin d'édition //////

      /**
       * Crée automatiquement un objet ponctuel en amont et /ou en aval de l'objet lineaire principal en cours d'édition.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setObjectOnExtremity(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle setObjectOnExtremity non adaptée à la couche ' +
              featureType.name
          );
          var deferred = $q.defer();
          deferred.resolve('terminé !');
          return deferred.promise;
        }
        var deferred = $q.defer();
        var layers = gclayers.getOperationalLayer();

        /**
         * Utilitaire: indique si un objet du même nom et de la même layer existe déjà.
         * @param {type} currentExtremity
         * @returns {Boolean}
         */
        function isShareObjectAlreadyInCurrentEdit(currentExtremity) {
          var isShareObjectPresent = false;
          var shareObjectName = currentExtremity.name;
          angular.forEach(editdescription.shareObjects, function(
            shareObj,
            key
          ) {
            if (
              !isShareObjectPresent &&
              shareObj.shareObject == shareObjectName
            ) {
              isShareObjectPresent = true;
            }
          });
          return isShareObjectPresent;
        }

        /**
         * Vérification qu'aucun des objets possible n'existe déjà à cette extrémité
         * @param shareObjectsConfigs
         * @returns {boolean}
         */
        const isOneShareObjectAlreadyInCurrentEdit = (shareObjectsConfigs) => {
          for (const shareObjectConfig of shareObjectsConfigs) {
            if (isShareObjectAlreadyInCurrentEdit(shareObjectConfig)) {
              return true;
            }
          }
          return false;
        };

        /**
         * Utilitaire: crée un objet feature et l'ajoute à l'objet 'editdescription' décrivant la session d'édition
         * @param {type} currentExtremity
         * @param {type} pointCoordinate
         * @returns {undefined}
         */
        function createObject(currentExtremity, pointCoordinate) {
          var layerName = currentExtremity.layer;
          var shareObjectName = currentExtremity.name;
          for (var i = 0; i < layers.length; i++) {
            var l = layers[i];
            if (l.name == undefined) {
              continue;
            }
            if (layerName == l.name) {
              var pointFeature = new ol.Feature({
                geometry: new ol.geom.Point(pointCoordinate),
              });
              gclayers
                .getDrawLayer()
                .getSource()
                .addFeature(pointFeature);
              var newObject = {
                shareObject: shareObjectName,
                editType: EditTypesFactory.editTypes.add.name,
                feature: pointFeature,
                fti: l.fti,
                isNew: true
              };
              editdescription.relatedfeatures.push(newObject);
              editdescription.shareObjects.push(newObject);
              break;
            }
          }
        }

        // Récupération des données de configuration
        let amontShareObjectsConfigs;
        let avalShareObjectsConfigs;
        if (ruleConf.parameters.amont) {
          amontShareObjectsConfigs = ruleConf.parameters.amont.shareObjects;
        }
        if (ruleConf.parameters.aval) {
          avalShareObjectsConfigs = ruleConf.parameters.aval.shareObjects;
        }
        const linearGeometry = editdescription.editedfeature.getGeometry();

        const resetBizEdiWidget = () => {
          bizeditProvider.stopApplyRulesButtonBlinking(true);
          $rootScope.$emit('resetBizEditWidget');
        }

        /**
         * Selection de la configuration de l'objet à créer en amont, et continuation du traitement
         */
        const selectAmontShareObjectConfigAndContinue = () => {
          // Vérification qu'un ponctuel d'un des types configuré n'est pas déjà présent
          if (isOneShareObjectAlreadyInCurrentEdit(amontShareObjectsConfigs)) {
            selectAvalShareObjectConfigAndContinue();
            return;
          }
          // Scope à utiliser pour la popup de sélection du type de feature à créer
          let popupScope = $rootScope.$new(true);
          popupScope.resetBizEdiWidget = resetBizEdiWidget;
          popupScope.shareObjectConfigs = amontShareObjectsConfigs;
          popupScope.stepForPopupTitle = 'Amont';
          // Fonction de sélection du type de feature à créer
          popupScope.pickShareObjectConfig = (shareObjectConfig) => {
            createObject(shareObjectConfig, linearGeometry.getFirstCoordinate());
            // Continuation sur l'aval
            if (avalShareObjectsConfigs
              && avalShareObjectsConfigs.length !== 0) {
              selectAvalShareObjectConfigAndContinue();
            }
            else {
              //-- Seul l'objet en extrémité amont est à créer
              deferred.resolve();
            }
          };
          // Si une seule configuration possible, on la choisi automatiquement
          if (amontShareObjectsConfigs.length === 1) {
            popupScope.pickShareObjectConfig(amontShareObjectsConfigs[0]);
          } else {
            // Sinon, ouverture d'une popup de selection
            ngDialog.open({
              template: 'js/XG/modules/edit/views/rules/rulesPopup/popupSelectShareObjectConfig.html',
              className: 'width400 ngdialog-theme-plain miniclose nopadding',
              closeByDocument: false,
              showClose: false,
              scope: popupScope
            });
          }
        };


        /**
         * Selection de la configuration de l'objet à créer en aval, et continuation du traitement
         */
        const selectAvalShareObjectConfigAndContinue = () => {
          // Vérification qu'un ponctuel d'un des types configuré n'est pas déjà présent
          if (isOneShareObjectAlreadyInCurrentEdit(avalShareObjectsConfigs)) {
            deferred.resolve();
            return;
          }
          // Scope à utiliser pour la popup de sélection du type de feature à créer
          let popupScope = $rootScope.$new(true);
          popupScope.resetBizEdiWidget = resetBizEdiWidget;
          popupScope.shareObjectConfigs = avalShareObjectsConfigs;
          popupScope.stepForPopupTitle = 'Aval';
          // Fonction de sélection du type de feature à créer
          popupScope.pickShareObjectConfig = (shareObjectConfig) => {
            if (!isShareObjectAlreadyInCurrentEdit(shareObjectConfig)) {
              createObject(shareObjectConfig, linearGeometry.getLastCoordinate());
              // Le traitement est terminé
              deferred.resolve();
            }
          };
          // Si une seule configuration possible, on la choisi automatiquement
          if (avalShareObjectsConfigs.length === 1) {
            popupScope.pickShareObjectConfig(avalShareObjectsConfigs[0]);
          } else {
            // Sinon, ouverture d'une popup de selection
            ngDialog.open({
              template: 'js/XG/modules/edit/views/rules/rulesPopup/popupSelectShareObjectConfig.html',
              className: 'width400 ngdialog-theme-plain miniclose nopadding',
              closeByDocument: false,
              showClose: false,
              scope: popupScope
            });
          }
        };


        // Détection d'une ancienne configuration
        if (angular.isUndefined(amontShareObjectsConfigs) && angular.isUndefined(avalShareObjectsConfigs)
         && ((ruleConf.parameters.amont && ruleConf.parameters.amont.isAmont)
                || (ruleConf.parameters.amont && ruleConf.parameters.amont.isAval))) {
          // AMONT
          const amontConfig = ruleConf.parameters ? ruleConf.parameters.amont : undefined;
          if (angular.isDefined(amontConfig) && amontConfig.isAmont && !isShareObjectAlreadyInCurrentEdit(amontConfig)) {
            createObject(amontConfig, linearGeometry.getFirstCoordinate());
          }
          // AVAL
          const avalConfig = ruleConf.parameters ? ruleConf.parameters.aval : undefined;
          if (angular.isDefined(avalConfig) && avalConfig.isAval && !isShareObjectAlreadyInCurrentEdit(avalConfig)) {
            createObject(avalConfig, linearGeometry.getLastCoordinate());
          }
          deferred.resolve();
        }
        // Avec nouvelle configuration
        else {
          // Selection de la configuration (type d'objet / nom de l'objet partagé) du ponctuel créé
          if (amontShareObjectsConfigs && amontShareObjectsConfigs.length > 0) {
            selectAmontShareObjectConfigAndContinue();
          }
          else
          if (avalShareObjectsConfigs && avalShareObjectsConfigs.length !== 0) {
            selectAvalShareObjectConfigAndContinue();
          }
          else {
            //-- Juste pour le cas où, et éviter de bloquer les régles
            deferred.resolve();
          }
        }

        return deferred.promise;
      }


      /**
       * Affectation d'une valeur d'attribut à l'objet principal.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        //Nom de l'attribut de l'objet principal à compléter
        var targetFieldName = ruleConf.parameters['targetField'];

        //Nom de l'objet partagé
        var shareObjectName = ruleConf.parameters['dataSource']['sourceName'];
        //layers auxquelles peut appartenir l'objet partagé.
        var layers = ruleConf.parameters['dataSource']['layers'];

        //Récupération de l'objet partagé, source de données pour compléter la valeur d'attribut de l'objet principal.
        var featureSource = undefined;
        var sourceFieldName = undefined;
        for (var i = 0; i < editdescription.shareObjects.length; i++) {
          if (editdescription.shareObjects[i].shareObject == shareObjectName) {
            featureSource = editdescription.shareObjects[i].feature;
            //Recherche de la layer correspondante à l'objet partagé, pour récupérer le bon nom d'attribut de l'objet partagé.
            for (var j = 0; j < layers.length; j++) {
              var layerName = layers[j].layer;
              if (layerName == editdescription.shareObjects[i].fti.name) {
                sourceFieldName = layers[j].field;
                break;
              }
            }
            break;
          }
        }
        if (featureSource != undefined) {
          //Affectation de la valeur attributaire de l'objet principal
          var newValue = featureSource.getProperties()[sourceFieldName];
          var newProperty = {};
          newProperty[targetFieldName] = newValue;
          editdescription.editedfeature.setProperties(newProperty);
          if (editdescription.relatedfeatures.length > 0) {
            editdescription.relatedfeatures.map(function(feat) {
              if (
                feat.feature.getId() != undefined &&
                feat.feature.getId().split('.')[0] === editdescription.fti.name
              ) {
                feat.feature.setProperties(newProperty);
              }
            });
          }
        } else {
          console.error(
            'Règle setAttribute n\'a pas trouvé l\'objet partagé ' +
              shareObjectName +
              ', impossible d\'affecter la valeur d\'attribut ' +
              targetFieldName +
              ' à l\'objet ' +
              editdescription.fti.name
          );
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Sets a default value to the target attribute of the edited object
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setDefaultValue(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        var targetFieldName = ruleConf.parameters['targetField'];
        var defaultValue = ruleConf.parameters['defaultValue'];
        var properties = editdescription.editedfeature.getProperties();
        if (angular.isUndefined(properties[targetFieldName])) {
          var propertyToSet = {};
          let parsedValue = parseValue(
            editdescription,
            targetFieldName,
            defaultValue
          );
          if (angular.isDefined(parsedValue)) {
            propertyToSet[targetFieldName] = parsedValue;
          }
          editdescription.editedfeature.setProperties(propertyToSet);
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      function parseValue(editdescription, attributeName, valueToParse) {
        let targetAttribute = editdescription.fti.attributes.filter(
          attribute => attribute.name === attributeName
        );
        if (targetAttribute.length > 0) {
          let attributeType = targetAttribute[0].type;
          if (angular.isDefined(attributeType)) {
            if (
              attributeType === 'java.lang.Integer' ||
              attributeType === 'java.lang.Long'
            ) {
              let valueAsInt = parseInt(valueToParse);
              if (!isNaN(valueAsInt)) {
                return valueAsInt;
              } else {
                return undefined;
              }
            }
            if (
              attributeType === 'java.lang.Double' ||
              attributeType === 'java.math.BigDecimal' ||
              attributeType === 'java.lang.Float'
            ) {
              let valueAsFloat = parseFloat(valueToParse);
              if (!isNaN(valueAsFloat)) {
                return valueAsFloat;
              } else {
                return undefined;
              }
            }
          }
        }
        return valueToParse;
      }

      function showErrorMsg() {
        require('toastr').warning(
          $filter('translate')('rulecfg.setattributebyformula.error_msg')
        );
      }

      function doFormulaCalculation(formula, propertyToSet, targetFieldName, featureType, targetFieldVariable) {
        try {
          let calculatedFormula =
            math.eval(formula.join('').toString()).toFixed(2) * 1;
          if (Number.isFinite(calculatedFormula)) {
            if (targetFieldVariable) {
              featureType.rulesVariables = featureType.rulesVariables || {};
              featureType.rulesVariables[targetFieldVariable] = calculatedFormula;
            } else {
              propertyToSet[targetFieldName] = calculatedFormula;
            }
          } else {
            showErrorMsg();
          }
        } catch (e) {
          showErrorMsg();
        }
      }

      function addMessage(
        editdescription,
        targetField,
        new_value,
        old_value,
        ruleId
      ) {
        if (old_value == new_value) {
          return;
        }
        let message = {
          attribute: targetField,
          old_value: old_value,
          new_value: new_value,
          ruleId: ruleId,
        };

        if (editdescription.messages) {
          let attrIsPressent = false;
          editdescription.messages.forEach(function(msg) {
            if (msg.attribute === targetField) {
              msg.new_value = new_value;
              attrIsPressent = true;
            }
          });
          if (!attrIsPressent) {
            editdescription.messages.push(message);
          }
        } else {
          let messages = [];
          messages.push(message);
          editdescription.messages = messages;
        }
      }

      function getAlreadyCaculatedValue(editdescription, item) {
        let alreadyCalc = false;
        let calcValue;
        if (editdescription.messages) {
          editdescription.messages.forEach(function(msg) {
            if (item === msg.attribute) {
              alreadyCalc = true;
              calcValue = msg.new_value;
            }
          });
        }
        if (alreadyCalc) {
          return calcValue;
        } else {
          return false;
        }
      }

      function setCalculatedProperty(
        deferred,
        editdescription,
        properties,
        propertyToSet,
        targetFieldName,
        ruleId
      ) {
        if (
          properties[targetFieldName] &&
          propertyToSet[targetFieldName] &&
          propertyToSet[targetFieldName] !== properties[targetFieldName]
        ) {
          // add message to show in confirmation dialog
          addMessage(
            editdescription,
            targetFieldName,
            propertyToSet[targetFieldName],
            properties[targetFieldName],
            ruleId
          );
        } else {
          editdescription.editedfeature.setProperties(propertyToSet);
        }
        deferred.resolve('terminé !');
      }

      function setAttributeByFormula(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();
        const targetFieldName = ruleConf.parameters['targetField'];
        const targetFieldVariable = ruleConf.parameters['variableName'];
        cleanupMessages(ruleConf.id, editdescription.messages);

        if (targetFieldName || (!targetFieldName && targetFieldVariable)) {
          var formula = [];
          var propertyToSet = {};
          var validFormula = true;
          var properties = editdescription.editedfeature.getProperties();

          ruleConf.parameters.formula.numerotation.parts.forEach(function(
            item
          ) {
            switch (item.key) {
              case 'attribut':
                let alreadyCalc = getAlreadyCaculatedValue(
                  editdescription,
                  item.format
                );
                if (alreadyCalc || alreadyCalc === 0) {
                  formula.push(alreadyCalc);
                } else {
                  if (!angular.isUndefined(properties[item.format])) {
                    formula.push(properties[item.format]);
                  } else {
                    validFormula = false;
                  }
                }
                break;
              case 'operator':
                if (item.format === '√') {
                  formula.push('sqrt');
                } else {
                  formula.push(item.format);
                }
                break;
              case 'variable':
                if (featureType.rulesVariables && featureType.rulesVariables[item.format]) {
                  formula.push(featureType.rulesVariables[item.format]);
                }
                break;
              case 'value':
                formula.push(item.format);
                break;
            }
          });
          if (validFormula) {
            if (!window.math) {
              gaJsGeneral.loadScript('lib/mathjs/mathjs.min.js').then(() => {
                doFormulaCalculation(formula, propertyToSet, targetFieldName, featureType, targetFieldVariable);
                setCalculatedProperty(
                  deferred,
                  editdescription,
                  properties,
                  propertyToSet,
                  targetFieldName,
                  ruleConf.id
                );
              });
            } else {
              doFormulaCalculation(formula, propertyToSet, targetFieldName, featureType, targetFieldVariable);
              setCalculatedProperty(
                deferred,
                editdescription,
                properties,
                propertyToSet,
                targetFieldName,
                ruleConf.id
              );
            }
          } else {
            showErrorMsg();
            deferred.resolve('terminé avec erreur !');
          }
        } else {
          showErrorMsg();
          deferred.resolve('terminé avec erreur !');
        }
        return deferred.promise;
      }

      function setAttributeByConcat(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        cleanupMessages(ruleConf.id, editdescription.messages);
        var deferred = $q.defer();
        var targetFieldName = ruleConf.parameters['targetField'];

        if (targetFieldName) {
          var propertyToSet = {};
          var properties = editdescription.editedfeature.getProperties();

          // collect the valid values in an array of pairs [value, isSeparator]
          var formulaItems = [];
          ruleConf.parameters.formula.numerotation.parts.forEach(function(
            item
          ) {
            if (item.key === 'attribut') {
              let alreadyCalc = getAlreadyCaculatedValue(
                editdescription,
                item.format
              );
              if (properties[item.format]) {
                if (alreadyCalc || alreadyCalc === 0) {
                  formulaItems.push([alreadyCalc, false]);
                } else {
                  formulaItems.push([properties[item.format], false]);
                }
              }
            } else if (item.key === 'fid') {
              let featureId = editdescription.editedfeature
                .getId()
                .substring(
                  editdescription.editedfeature.getId().lastIndexOf('.') + 1
                );
              formulaItems.push([featureId, false]);
            } else if (item.key === 'variable') {
              if (featureType && featureType.rulesVariables && featureType.rulesVariables[item.format]) {
                formulaItems.push([featureType.rulesVariables[item.format], false]);
              }
            } else {
              formulaItems.push([item.format, item.key === 'separator']);
            }
          });
          // remove the leading separators
          while (getSecondElement(formulaItems[0])) {
            formulaItems.shift();
          }
          // remove the trailing separators
          while (getSecondElement(formulaItems[formulaItems.length - 1])) {
            formulaItems.pop();
          }
          // collect the final concatenation items while removing the extra separators
          var formula = [];
          var canAddSep = false;
          formulaItems.forEach(function(item) {
            if (item[1]) {
              if (canAddSep) {
                canAddSep = false;
                formula.push(item[0]);
              }
            } else {
              canAddSep = true;
              formula.push(item[0]);
            }
          });
          // set the concatenation result to the target field
          propertyToSet[targetFieldName] = formula.join('').toString();

          if (
            properties[targetFieldName] &&
            propertyToSet[targetFieldName] &&
            propertyToSet[targetFieldName] !== properties[targetFieldName] &&
            ruleConf.type === 'OnPostValidation'
          ) {
            addMessage(
              editdescription,
              targetFieldName,
              propertyToSet[targetFieldName],
              properties[targetFieldName],
              ruleConf.id
            );
          } else {
            editdescription.editedfeature.setProperties(propertyToSet);
            editdescription.performUpdateOnPostsaving = true;
          }
        }

        deferred.resolve(propertyToSet);
        return deferred.promise;
      }

      function cleanupMessages(ruleId, messages) {
        if (messages && messages.length) {
          messages.forEach(message => {
            if (message.ruleId === ruleId) {
              messages.splice(messages.indexOf(message), 1);
            }
          });
        }
      }

      /**
       * Affectation d'une valeur d'attribut à un objet partagé.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setRemoteAttribute(editdescription, ruleConf, featureType, map) {
        let fieldName;
        var deferred = $q.defer();
        //Récupération de la source de données
        var dataSource = ruleConf['parameters']['dataSource'];
        //Valeur de l'attribut à affecter
        var valueToSet = undefined;
        //Si la source de donnnées est l'objet principal édité
        if (dataSource['type'] == 'mainObject') {
          fieldName = ruleConf['parameters']['dataSource']['field'];
          valueToSet = editdescription.editedfeature.get(fieldName);
        }
        //Si la source de données est un objet partagé
        else if (dataSource['type'] == 'shareObject') {
          var shareObjectName = dataSource['shareObjectName'];
          fieldName = dataSource['field'];
          for (var i = 0; i < editdescription.shareObjects.length; i++) {
            if (
              editdescription.shareObjects[i].shareObject == shareObjectName
            ) {
              valueToSet = editdescription.shareObjects[i].feature.get(
                fieldName
              );
              break;
            }
          }
        }
        //Affectation de la valeur dans le(s) objet(s) cible(s)
        var targetObjects = ruleConf.parameters['targets'];
        //Pour chaque objet cible
        for (var j = 0; j < targetObjects.length; j++) {
          //Nom de l'objet cible
          var shareObjectName = targetObjects[j]['shareObjectName'];
          //Recherche de l'objet cible parmi les objets partagés
          for (var k = 0; k < editdescription.shareObjects.length; k++) {
            if (
              editdescription.shareObjects[k].shareObject == shareObjectName
            ) {
              var targetfeature = editdescription.shareObjects[k].feature;
              var newProperty = {};
              var targetField = targetObjects[j]['field'];
              newProperty[targetField] = valueToSet;
              targetfeature.setProperties(newProperty);

              if (editdescription.relatedfeatures.length > 0) {
                editdescription.relatedfeatures.map(function(feat) {
                  if (
                    feat.feature.getId() != undefined &&
                    feat.feature.getId().split('.')[0] ===
                      editdescription.fti.name
                  ) {
                    feat.feature.setProperties(newProperty);
                  }
                });
              }

              //Si l'objet partagé n'est pas present dans le tableau des features liés à sauvegarder, alors ajout de celui-ci.
              if (
                !isObjectInRelatedFeatures(
                  editdescription.shareObjects[k].shareObject,
                  editdescription
                )
              ) {
                editdescription.relatedfeatures.push(
                  editdescription.shareObjects[k]
                );
              }
              break;
            }
          }
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      function verifZoneSaisie(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        var promises = [deferred.promise];
        //console.log(ruleConf);
        var editedgeom = editdescription.editedfeature.getGeometry();
        var wktStr = new ol.format.WKT().writeGeometry(editedgeom);
        //-- Uid du composant zone interdite.
        var uid = ruleConf.parameters.component.uid;

        //var uid = ruleConf.parameters.component.uid;
        var cql_filter = 'INTERSECTS(geom, ' + wktStr + ')';
        var clauseWhere = ruleConf.parameters.clauseWhere;
        var message = ruleConf.parameters.message;
        var promise;
        if (clauseWhere != '' && clauseWhere != undefined) {
          promise = ogcFactory.getfeatures(
            'GetFeature',
            'WFS',
            '1.0.0',
            uid,
            'json',
            map
              .getView()
              .getProjection()
              .getCode(),
            cql_filter + ' AND ' + clauseWhere
          );
        } else {
          promise = ogcFactory.getfeatures(
            'GetFeature',
            'WFS',
            '1.0.0',
            uid,
            'json',
            map
              .getView()
              .getProjection()
              .getCode(),
            cql_filter
          );
        }
        promise.then(
          function(res) {
            console.log(res.data);
            if (ruleConf.parameters.strict == undefined) {
              ruleConf.parameters.strict = false;
            }
            if (
              res.data.features.length > 0 &&
              ruleConf.parameters.strict == true
            ) {
              //console.log(ruleConf.parameters.component);
              $timeout(function() {
                swal(
                  {
                    title: message,
                    //text: "You will not be able to recover this imaginary file!",
                    type: 'warning',
                    showCancelButton: false,
                    confirmButtonColor: '#DD6B55',
                    confirmButtonText: 'ok',
                    //cancelButtonText: $filter('translate')('common.no'),
                    closeOnConfirm: true,
                  },
                  function(isConfirm) {}
                );
                deferred.reject(message);
                return deferred.promise;
              }, 700);

              //deferred.reject(message);
              //return deferred.promise;
            } else if (
              res.data.features.length > 0 &&
              ruleConf.parameters.strict == false
            ) {
              swal(
                {
                  //title:"<div> message </div> <br/>> <div>Voulez vous continuer?</div>",
                  title: message,
                  text: 'Voulez vous continuer?',
                  //text: "You will not be able to recover this imaginary file!",
                  type: 'warning',
                  showCancelButton: true,
                  confirmButtonColor: '#DD6B55',
                  confirmButtonText: $filter('translate')('common.yes'),
                  cancelButtonText: $filter('translate')('common.no'),
                  closeOnConfirm: true,
                },
                function(isConfirm) {
                  if (isConfirm) {
                    deferred.resolve('terminé !');
                    return deferred.promise;
                  } else {
                    deferred.reject(message);
                    return deferred.promise;
                  }
                }
              );
            } else {
              deferred.resolve('terminé !');
              return deferred.promise;
            }
          },
          function(error) {}
        );
        promises.push(promise);
        //deferred.resolve("terminé !");
        return $q.all(promises);
      }

      /**
       * Récupére les objets intersectant l'amont et / ou l'aval de l'objet lineaire principal en cours d'édition.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {undefined}
       */
      function getFeaturesIntersectingLine(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        //var deferred = $q.defer();
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle getFeaturesIntersectingLine non adaptée à la couche ' +
              featureType.name
          );
          var deferred = $q.defer();
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        //Récupération de la configuration de la règle: config des  objets partagés
        var cfgUpShareObjects = ruleConf.parameters['amont']['shareObjects'];
        var cfgDownShareObjects = ruleConf.parameters['aval']['shareObjects'];

        //Si pas de config
        if (
          (cfgUpShareObjects == undefined &&
            !Array.isArray(cfgUpShareObjects)) ||
          (cfgDownShareObjects == undefined &&
            !Array.isArray(cfgDownShareObjects))
        ) {
          console.error(
            'Règle getFeaturesIntersectingLine mal ou pas configurée pour la couche ' +
              featureType.name
          );
          var deferred = $q.defer();
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        // Recherche et ajoute les features appartenant aux layers dont le nom est contenu dans cfgShareObjects en paramètre,
        // et intersectant les coordonées en paramètre.
        function searchIntersections(cfgShareObjects, pointCoordinate) {
          var promises = [];
          var leftX = pointCoordinate[0] - 0.2;
          var bottomY = pointCoordinate[1] - 0.2;
          var rightX = pointCoordinate[0] + 0.2;
          var topY = pointCoordinate[1] + 0.2;
          var cql_filter =
            'INTERSECTS(geom, POLYGON((' +
            leftX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            bottomY +
            ')))';

          for (var j = 0; j < cfgShareObjects.length; j++) {
            var cfgShareObj = cfgShareObjects[j];
            var l = getLayerByName(cfgShareObj.layerName, layers);
            if (l == undefined) {
              continue;
            }

            //La requete d'intersection de pointCoordinate avec les objets de la couche cfgShareObj.layerName.
            var promise = ogcFactory.getfeatures(
              'GetFeature',
              'WFS',
              '1.0.0',
              l.fti.uid,
              'json',
              map
                .getView()
                .getProjection()
                .getCode(),
              cql_filter
            );
            //Reponse.
            var callBackSuccess = getCallbackSuccess(cfgShareObj, l);
            //promise2 sera resolue lorsque la callback aura finie d'être executée (car elle ne renvoie pas de promesse)
            var promise2 = promise.then(callBackSuccess, function(error) {
              console.error(
                'Erreur à l\'appel de  ogc.data(),avec la couche:' +
                  l.name +
                  'cql_filter:' +
                  cql_filter
              );
            });
            promises.push(promise2);
          }
          return promises;
        }

        /**
         * Appelée lors du résultat de la requete de feature
         * @param {type} cfgShareObj
         * @param {type} layer
         * @returns {Function}
         */
        function getCallbackSuccess(cfgShareObj, layer) {
          return function(result) {
            var intersectedFeatures = format.readFeatures(result.data);
            if (
              intersectedFeatures != undefined &&
              intersectedFeatures.length > 0
            ) {
              addShareObject(intersectedFeatures, cfgShareObj.name, layer);
            }
          };
        }

        function addShareObject(intersectedFeatures, shareObjectName, layer) {
          for (const intersectedFeature of intersectedFeatures) {
            editdescription.shareObjects.push(
              {
                shareObject: shareObjectName,
                editType: EditTypesFactory.editTypes.updateattributes.name,
                feature: intersectedFeature,
                fti: layer.fti,
              }
            );
          }
        }

        var editedgeom = editdescription.editedfeature.getGeometry();
        var upHillpointCoordinate = editedgeom.getFirstCoordinate();
        var downHillpointCoordinate = editedgeom.getLastCoordinate();

        //Récupération des objets intersectés en amont
        var upHillpromises = searchIntersections(
          cfgUpShareObjects,
          upHillpointCoordinate
        );
        //Récupération des objets intersectés en aval
        var downHillpromises = searchIntersections(
          cfgDownShareObjects,
          downHillpointCoordinate
        );
        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all(upHillpromises.concat(downHillpromises));
      }

      /**
       * Récupère les objets portés par les objets linéaires édités (modification ou suppression).
       * Chaque objet relatedFeature dans le tableau des relatedfeatures comportera aussi un tableau shareObjects
       * representant les objets partagés par les règles lors du traitement de l'objet relatedfeature par la présente règle.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function getAllFeaturesIntersectingLine(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle getAllFeaturesIntersectingLine non adaptée à la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        //Récupération de la configuration de la règle: config des  objets partagés
        var cfgUpShareObjects = ruleConf.parameters['shareObjects'];

        //Si pas de config
        if (
          cfgUpShareObjects == undefined &&
          !Array.isArray(cfgUpShareObjects)
        ) {
          console.error(
            'Règle getAllFeaturesIntersectingLine mal ou pas configurée pour la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        // Recherche et ajoute les features appartenant aux layers dont le nom est contenu dans cfgShareObjects en paramètre,
        // et intersectant les coordonées en paramètre.
        function searchIntersections(
          cfgShareObjects,
          featureLine,
          currenteditdescription
        ) {
          var lineGeom = featureLine.getGeometry();
          var geoJson = format.writeGeometryObject(lineGeom);

          //Buffer de la geometrie de l'objet ligne courant.
          var firstPromise = GeometryFactory.buffer(geoJson, 'ROUND', 1);

          //Une fois le buffer de l'objet ligne effectué, on peut rechercher les intersections.
          var mainPromise = firstPromise.then(
            function(res) {
              var polygon = res.data;
              //-- Retourne la promesse du travail d'intersection.
              return requestIntersections(polygon);
            },
            function(error) {
              console.error(
                'Erreur lors de la requete de bufferisation de l\'objet édité line de la couche ' +
                  currenteditdescription.fti.name
              );
              //Fin de la méthode de la recherche d'intesection pour l'objet édité featureLine (résolution de la mainPromise)
              return undefined;
            }
          );

          /**
           * Recherche sur chaque layer d'intersection configurée les
           * features intersectant la ligne édité buferisé 'polygon' passé en paramètre de cette méthode.
           * @param {type} polygon
           * @returns {$q@call;defer.promise}
           */
          function requestIntersections(polygon) {
            var promises = [];

            var lineCoordsString2 = '';

            if (polygon.type == 'Polygon') {
              var polygonGeom = format.readGeometry(polygon);
              angular.forEach(polygonGeom.getLinearRings(), function(
                linearRing,
                key
              ) {
                var coords = linearRing.getCoordinates();
                lineCoordsString2 += '(';
                angular.forEach(coords, function(value, key) {
                  lineCoordsString2 += value[0] + ' ' + value[1] + ',';
                });
                lineCoordsString2 = lineCoordsString2.substring(
                  0,
                  lineCoordsString2.length - 1
                );
                lineCoordsString2 += ')';
              });
            } else {
              var deferred = $q.defer();
              console.error(
                'Règle getAllFeaturesIntersectingLine featureLine after buffer operation is not a Polygon !'
              );
              deferred.resolve('terminé !');
              return deferred.promise;
            }

            var cql_filter =
              'INTERSECTS(geom, Polygon(' + lineCoordsString2 + '))';

            //Pour chaque layer d'intersection configurée
            for (var j = 0; j < cfgShareObjects.length; j++) {
              //configuration de la layer sur laquelle on souhaite rechercher les intersections.
              var cfgShareObj = cfgShareObjects[j];
              var l = getLayerByName(cfgShareObj.layerName, layers);
              if (l == undefined) {
                continue;
              }

              //Préparation du traitement de la Reponse.
              var callBackSuccess = getCallbackSuccess(
                cfgShareObj,
                l,
                currenteditdescription
              );

              //La requete d'intersection de la featureLine avec les objets de la couche cfgShareObj.layerName.
              var promise = ogcFactory.getfeatures(
                'GetFeature',
                'WFS',
                '1.0.0',
                l.fti.uid,
                'json',
                map
                  .getView()
                  .getProjection()
                  .getCode(),
                cql_filter
              );

              //promise2 sera resolue lorsque la callback aura fini d'être executée (car elle ne renvoie pas de promesse)
              var promise2 = promise.then(callBackSuccess, function(error) {
                console.error(
                  'Erreur à l\'appel de ogc.data(), avec la couche:' +
                    l.name +
                    'cql_filter:' +
                    cql_filter
                );
              });
              promises.push(promise2);
            }
            return $q.all(promises);
          }

          return mainPromise;
        }

        /**
         * Renvoie la méthode qui sera appelé lors de la réponse à la recherche de feature intersecté (ogcFactory.getfeatures)
         * @param {type} cfgShareObj
         * @param {type} layer
         * @param {type} currenteditdescription
         * @returns {Function}
         */
        function getCallbackSuccess(
          cfgShareObj,
          layer,
          currenteditdescription
        ) {
          return function(result) {
            var intersectedFeatures = format.readFeatures(result.data);
            if (
              intersectedFeatures != undefined &&
              intersectedFeatures.length > 0
            ) {
              addShareObject(
                intersectedFeatures,
                cfgShareObj.name,
                layer,
                currenteditdescription
              );
            }
          };
        }

        function addShareObject(
          intersectedFeatures,
          shareObjectName,
          layer,
          currenteditdescription
        ) {
          angular.forEach(intersectedFeatures, function(
            aFeatureintersected,
            key
          ) {
            var newObject = {
              shareObject: shareObjectName,
              editType: EditTypesFactory.editTypes.updateattributes.name,
              feature: aFeatureintersected,
              fti: layer.fti,
            };
            if (currenteditdescription.shareObjects == undefined) {
              currenteditdescription.shareObjects = [];
            }
            currenteditdescription.shareObjects.push(newObject);
          });
        }

        //Pour la ligne principale
        //Récupération des objets intersectés par le point.
        var promises = searchIntersections(
          cfgUpShareObjects,
          editdescription.editedfeature,
          editdescription
        );

        //Pour les Les AUTRES lignes éditées (édition multiple de lignesgetAllFeaturesIntersectingLinedelete)
        //Index de l'objet secondaire édité
        var relatedfeaturesIndex = 0;
        //on mémorise le nombre initial d'objets secondaires car on va au cours de l'execution possiblement alimenter ce tableau.
        //Si la règle ajoutait de nouveaux objets et que l'on voulait prendre en compte ces nouveaux objets,
        //on aurait comparé 'relatedfeaturesIndex' à 'editdescription.relatedfeatures.length' dans treatNextLineObject().
        var nbRelatedFeature = editdescription.relatedfeatures.length;

        function treatNextLineObject() {
          if (relatedfeaturesIndex < nbRelatedFeature) {
            var related = editdescription.relatedfeatures[relatedfeaturesIndex];
            if (
              related.editType == editdescription.editType &&
              related.fti.name == editdescription.fti.name
            ) {
              var anOtherpromises = searchIntersections(
                cfgUpShareObjects,
                related.feature,
                related
              );
              anOtherpromises.then(
                function(res) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                },
                function(error) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                }
              );
            } else {
              relatedfeaturesIndex++;
              treatNextLineObject();
            }
          }
          //si fin du parcours des objets
          else {
            deferred.resolve('rule end !');
          }
        }

        treatNextLineObject();

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all([promises, deferred.promise]);
      }

      /**
       * Récupére les objets intersectant l'objet ponctuel principal en cours d'édition.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {undefined}
       */
      function getFeaturesIntersectingPoint(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        //var deferred = $q.defer();
        if (featureType.typeInfo != 'POINT') {
          console.error(
            'Règle getFeaturesIntersectingPoint non adaptée à la couche ' +
              featureType.name
          );
          var deferred = $q.defer();
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        //Récupération de l'objet configuration.
        var cfgUpShareObjects = ruleConf.parameters['shareObjects'];

        //Si pas de config.
        if (
          cfgUpShareObjects == undefined &&
          !Array.isArray(cfgUpShareObjects)
        ) {
          console.error(
            'Règle getFeaturesIntersectingPoint mal ou pas configurée pour la couche ' +
              featureType.name
          );
          var deferred = $q.defer();
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        // Recherche et ajoute les features appartenant aux layers dont le nom est contenu dans cfgShareObjects en paramètre,
        // et intersectant les coordonées en paramètre.
        function searchIntersections(cfgShareObjects, pointCoordinate) {
          var promises = [];

          const leftX = Number.parseFloat(pointCoordinate[0]) - 0.05;
          const bottomY = Number.parseFloat(pointCoordinate[1]) - 0.05;
          const rightX = Number.parseFloat(pointCoordinate[0]) + 0.05;
          const topY = Number.parseFloat(pointCoordinate[1]) + 0.05;
          var cql_filter =
            'INTERSECTS(geom, POLYGON((' +
            leftX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            bottomY +
            ')))';
          //var cql_filter = "INTERSECTS(geom,Point("+pointCoordinate[0]+' '")";

          for (var j = 0; j < cfgShareObjects.length; j++) {
            var cfgShareObj = cfgShareObjects[j];
            var l = getLayerByName(cfgShareObj.layerName, layers);
            if (l == undefined) {
              continue;
            }

            //La requete d'intersection de pointCoordinate avec les objets de la couche cfgShareObj.layerName.
            var promise = ogcFactory.getfeatures(
              'GetFeature',
              'WFS',
              '1.0.0',
              l.fti.uid,
              'json',
              map
                .getView()
                .getProjection()
                .getCode(),
              cql_filter
            );
            //Reponse.
            var callBackSuccess = getCallbackSuccess(cfgShareObj, l);
            //promise2 sera resolue lorsque la callback aura finie d'être executée (car elle ne renvoie pas de promesse)
            var promise2 = promise.then(callBackSuccess, function(error) {
              console.error(
                'Erreur à l\'appel de  ogc.data(),avec la couche:' +
                  l.name +
                  'cql_filter:' +
                  cql_filter
              );
            });
            promises.push(promise2);
          }
          return promises;
        }

        /*
         * Renvoie la méthode qui sera appelée lors de la réponse à la recherche de feature intersecté (ogcFactory.getfeatures)
         * @param {type} cfgShareObj
         * @param {type} layer
         * @returns {Function}
         */
        function getCallbackSuccess(cfgShareObj, layer) {
          return function(result) {
            var intersectedFeatures = format.readFeatures(result.data);
            if (
              intersectedFeatures != undefined &&
              intersectedFeatures.length > 0
            ) {
              addShareObject(intersectedFeatures, cfgShareObj.name, layer);
            }
          };
        }

        function addShareObject(intersectedFeatures, shareObjectName, layer) {
          for (const intersectedFeature of intersectedFeatures) {
            editdescription.shareObjects.push(
              {
                shareObject: shareObjectName,
                editType: EditTypesFactory.editTypes.updateattributes.name,
                feature: intersectedFeature,
                fti: layer.fti,
              }
            );
          }
        }

        //Récupération de l'objet édité
        var editedgeom = editdescription.editedfeature.getGeometry();
        //Récupération des cooordonnées de l'objet édité.
        var coordinate = editedgeom.getCoordinates();
        //Récupération des objets intersectés par le point.
        var promises = searchIntersections(cfgUpShareObjects, coordinate);

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all(promises);
      }

      /**
       * Récupére les objets intersectant l'objet ponctuel principal en cours d'édition.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {undefined}
       */
      const getFeatureIntersectingPolygon = (
        editdescription,
        ruleConf,
        featureType,
        map
      ) => {

        //Récupération de l'objet configuration.
        const cfgUpShareObjects = ruleConf.parameters['shareObjects'];
        if (!cfgUpShareObjects === undefined || !Array.isArray(cfgUpShareObjects)) {
          const deferred = $q.defer();
          console.error('Règle GetFeatureIntersectingPolygon mal ou pas configurée pour la couche ' +featureType.name);
          deferred.resolve();
          return deferred.promise;
        }

        const layers = gclayers.getOperationalLayer();

        // Recherche et ajoute les features appartenant aux layers dont le nom est contenu dans cfgShareObjects en paramètre,
        // et intersectant les coordonées en paramètre.
        function searchIntersections(cfgShareObjects, polygonCoordinate) {
          const intersectingFeaturesPromises = [];

          // Creation du filtre
          let cql_filter =
              'INTERSECTS(geom, POLYGON((';
          for (const coordinate of polygonCoordinate[0]) {
            cql_filter += coordinate[0] + ' ' + coordinate[1] + ',';
          }
          cql_filter = cql_filter.slice(0, cql_filter.length - 1);
          cql_filter += ')))';

          for (const cfgShareObject of cfgShareObjects) {
            const layer = getLayerByName(cfgShareObject.layerName, layers);

            const promise = ogcFactory.getfeatures(
              'GetFeature',
              'WFS',
              '1.0.0',
              layer.fti.uid,
              'json',
              map
                .getView()
                .getProjection()
                .getCode(),
              cql_filter
            );

            promise.then(
              // Succès
              result => {
                const intersectedFeatures = format.readFeatures(result.data);
                if (intersectedFeatures && intersectedFeatures.length > 0) {
                  editdescription.shareObjects.push({
                    shareObject: cfgShareObject.name,
                    editType: EditTypesFactory.editTypes.updateattributes.name,
                    feature: intersectedFeatures[0],
                    fti: layer.fti,
                  });
                }
              },
              // Echec
              () => {
                console.error('Erreur à l\'appel de ogc.data() avec la couche:' +
                      layer.name + 'cql_filter:' + cql_filter);
              }
            );
            intersectingFeaturesPromises.push(promise);
          }
          return intersectingFeaturesPromises;
        }

        //Récupération des objets intersectés par le point.
        const promises = searchIntersections(cfgUpShareObjects,
          editdescription.editedfeature.getGeometry().getCoordinates());

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all(promises);
      };

      /**
       * Scinde la ligne intersectée par l'objet principal lineaire ou ponctuel.
       * Les deux objets lineraires résultant sont issus de la mise à jour de la ligne scindée d'origine pour l'un
       * et de la création d'un deuxième objet lineaire pour l'autre.
       *
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function cutIntersectingLine(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();

        var editedGeom = editdescription.editedfeature.getGeometry();

        /**
         * Récupére le feature partagé à partir des informations de configuration (contenant le nom de l'objet (shareObject) et le nom de la layer)
         * @param {type} cfgShareObjects
         * @returns {EditRulesProvider_L7.EditRulesProvider.$get.cutIntersectingLine.getIntersectedShareObject.EditRulesProviderAnonym$4|undefined}
         */
        function getIntersectedShareObject(cfgShareObjects) {
          for (var j = 0; j < cfgShareObjects.length; j++) {
            for (var k = 0; k < editdescription.shareObjects.length; k++) {
              if (
                editdescription.shareObjects[k].shareObject ==
                  cfgShareObjects[j].name &&
                editdescription.shareObjects[k].fti.name ==
                  cfgShareObjects[j].layerName
              ) {
                return {
                  shareObject: editdescription.shareObjects[k],
                  isHistoricize: cfgShareObjects[j].isHistoricize,
                };
              }
            }
          }
          return undefined;
        }

        /**
         * Vérifie si le feature Amont correspond aussi au feature aval , si c'est le cas cherche dans le shared feature amont le tronçon qui contient le point
         * @param {type} cfgShareObjects
         * @returns {EditRulesProvider_L7.EditRulesProvider.$get.cutIntersectingLine.getIntersectedShareObject.EditRulesProviderAnonym$4|undefined}
         */
        function UpdateIntersectedShareObjectAval(
          intersectedShareObject,
          intersectedShareObject2,
          pointCoordiantes
        ) {
          editdescription.relatedfeatures.map(function(x, i) {
            if (
              x.feature.getGeometry().getType() == 'MultiLineString' &&
              x.feature.getGeometry().getCoordinates() !== undefined &&
              getCuttingIndexOnSimpleLine(
                x.feature.getGeometry().getCoordinates()[0],
                pointCoordiantes
              ) !== undefined
            ) {
              intersectedShareObject2.shareObject.feature = x.feature.clone();
              intersectedShareObject2.shareObject.feature.setId(
                x.feature.getId()
              );
              intersectedShareObject2.shareObject.fti = angular.copy(x.fti);
              intersectedShareObject2.shareObject.editType = angular.copy(
                x.editType
              );
              editdescription.relatedfeatures.splice(i, 1);
            } else if (
              x.feature.getGeometry().getType() == 'LineString' &&
              x.feature.getGeometry().getCoordinates() !== undefined &&
              getCuttingIndexOnSimpleLine(
                x.feature.getGeometry().getCoordinates(),
                pointCoordiantes
              ) !== undefined
            ) {
              intersectedShareObject2.shareObject.feature = x.feature.clone();
              intersectedShareObject2.shareObject.feature.setId(
                x.feature.getId()
              );
              intersectedShareObject2.shareObject.fti = angular.copy(x.fti);
              intersectedShareObject2.shareObject.editType = angular.copy(
                x.editType
              );
              editdescription.relatedfeatures.splice(i, 1);
            }
          });
          return intersectedShareObject2;
        }

        /**
         * Donne l'index 'cuttingIndex' du dernier point de la simpleLine 'lineCoordinates' qui appartiendra à la première ligne issue de la scission.
         * (cuttingIndex+1 est l'index du point qui appartiendra à la deuxième ligne issue de la scission.
         * Si le point intersectionPointCoordinate est situé à une extrémité, la méthode retourne undefined.
         * @param {type} lineCoordinates
         * @param {type} intersectionPointCoordinate
         * @returns {Number}cuttingIndex
         */
        function getCuttingIndexOnSimpleLine(
          lineCoordinates,
          intersectionPointCoordinate
        ) {
          var cuttingIndex = undefined;

          //Pour chaque segment de la simple line
          for (var i = 0; i < lineCoordinates.length - 1; i++) {
            //Les deux points du segments
            var p1 = lineCoordinates[i];
            var p2 = lineCoordinates[i + 1];

            //Si le point édité est posé sur le premier point du segment ou sur le second point (x et y sont les mêmes à 1cm près)
            if (
              (Math.round(intersectionPointCoordinate[0] * 100) ==
                Math.round(p1[0] * 100) &&
                Math.round(intersectionPointCoordinate[1] * 100) ==
                  Math.round(p1[1] * 100)) ||
              (Math.round(intersectionPointCoordinate[0] * 100) ==
                Math.round(p2[0] * 100) &&
                Math.round(intersectionPointCoordinate[1] * 100) ==
                  Math.round(p2[1] * 100))
            ) {
              cuttingIndex = i;
              break;
            } else {
              //Coef directeur de la line
              var coef = (p2[1] - p1[1]) / (p2[0] - p1[0]);
              //Coef directeur de la droite intersectionPointCoordinate P1
              var coef2 =
                (intersectionPointCoordinate[1] - p1[1]) /
                (intersectionPointCoordinate[0] - p1[0]);

              //Si intersectionPoint appartient à la DROITE line correspondant au segment en cours P1P2
              if (
                Math.abs(Math.round(coef * 100)) ==
                Math.abs(Math.round(coef2 * 100))
              ) {
                //Vérification si intersectionPointCoordinate appartient au SEGMENT P1P2
                if (
                  intersectionPointCoordinate[0] <= Math.max(p1[0], p2[0]) &&
                  intersectionPointCoordinate[0] >= Math.min(p1[0], p2[0]) &&
                  intersectionPointCoordinate[1] <= Math.max(p1[1], p2[1]) &&
                  intersectionPointCoordinate[1] >= Math.min(p1[1], p2[1])
                ) {
                  cuttingIndex = i;
                  break;
                }
              }
            }
          }
          return cuttingIndex;
        }

        /**
         * Crée et enregistre les deux objets ol.Feature issus de la scission de la SimpleLine ou Multiligne 'intersectedShareObject.feature' au point de coordonnée 'intersectionPointCoordinate'.
         * @param {type} intersectionPointCoordinate
         * @param {type} intersectedShareObject
         * @returns {undefined}
         */
        function performCutting(
          intersectionPointCoordinate,
          intersectedShareObject,
          isTobeHistoricize
        ) {
          var deferred = $q.defer();
          if (intersectedShareObject != undefined) {
            var intersectedFeature = intersectedShareObject.feature;

            //Copie de l'objet qui va être scindé pour historisation eventuelle
            if (isTobeHistoricize) {
              var mainFeatureCopie = intersectedFeature.clone();
              mainFeatureCopie.setId(intersectedFeature.getId());
            }

            var line = intersectedFeature.getGeometry();

            var isSimpleLine = line instanceof ol.geom.LineString;
            var isMultiLine = line instanceof ol.geom.MultiLineString;

            //Index du segment sur lequel couper la line.
            var cuttingIndex = undefined;
            var cuttinglineIndex = undefined;

            var line1 = undefined;
            var line2 = undefined;

            var line1Length = 0;
            var line2Length = 0;

            const promisesLength = [];

            if (
              (Math.round(intersectionPointCoordinate[0] * 100) ===
                Math.round(line.getFirstCoordinate()[0] * 100) &&
                Math.round(intersectionPointCoordinate[1] * 100) ===
                  Math.round(line.getFirstCoordinate()[1] * 100)) ||
              (Math.round(intersectionPointCoordinate[0] * 100) ===
                Math.round(line.getLastCoordinate()[0] * 100) &&
                Math.round(intersectionPointCoordinate[1] * 100) ===
                  Math.round(line.getLastCoordinate()[1] * 100))
            ) {
              const featId = intersectedFeature instanceof ol.Feature ? intersectedFeature.getId() : intersectedFeature.id;
              console.info('Pas de scission possible de l\'objet intersecté id =', featId,
                  ' de la couche ', intersectedShareObject.fti.name);
              return;
            }

            if (isSimpleLine) {
              var originCoordinates = line.getCoordinates();
              cuttingIndex = getCuttingIndexOnSimpleLine(
                originCoordinates,
                intersectionPointCoordinate
              );

              if (cuttingIndex == undefined) {
                console.info(
                  'Pas de scission possible de l\'objet intersecté id=' +
                    intersectedFeature.id +
                    ' de la couche =' +
                    intersectedShareObject.fti.name
                );
                return;
              }
              //Création des deux objets lines issus de la scission.
              var coordinates1 = originCoordinates.slice(0, cuttingIndex + 1); //slice extrait jusqu'à endIndex exclu.
              coordinates1.push(intersectionPointCoordinate);
              line1 = new ol.geom.LineString(coordinates1, 'XY');

              var prom1 = getSimpleLineLength(
                line1,
                map,
                intersectedShareObject.fti
              );
              prom1.then(function(res) {
                line1Length = Math.round(res.data[0].perimeter * 100) / 100;
              });

              var coordinates2 = [intersectionPointCoordinate];
              coordinates2 = coordinates2.concat(
                originCoordinates.slice(cuttingIndex + 1)
              );
              line2 = new ol.geom.LineString(coordinates2, 'XY');

              var prom2 = getSimpleLineLength(
                line2,
                map,
                intersectedShareObject.fti
              );
              prom2.then(function(res) {
                line2Length = Math.round(res.data[0].perimeter * 100) / 100;
              });

              promisesLength.push(prom1, prom2);
            } else if (isMultiLine) {
              var simpleLines = line.getCoordinates();
              //Pour chaque simple line de la multiline
              for (
                var lineIndex = 0;
                lineIndex < simpleLines.length;
                lineIndex++
              ) {
                cuttingIndex = getCuttingIndexOnSimpleLine(
                  simpleLines[lineIndex],
                  intersectionPointCoordinate
                );
                //Si l'index du segement de la simple line a été trouvé
                if (cuttingIndex != undefined) {
                  //mémorisation de l'index de la simple line
                  cuttinglineIndex = lineIndex;
                  break;
                }
              }
              if (cuttingIndex == undefined) {
                console.info(
                  'Pas de scission possible de l\'objet intersecté id=' +
                    intersectedFeature.id +
                    ' de la couche =' +
                    intersectedShareObject.fti.name
                );
                return;
              }
              //Création des deux objets Multilines issus de la scission.

              //Premiere Multiligne:
              var multiCoordinates1 = [];

              //Récupération des simples lignes précedents celle sur laquelle se trouve la scission.
              for (
                var lineIndex2 = 0;
                lineIndex2 < cuttinglineIndex;
                lineIndex2++
              ) {
                multiCoordinates1.push(simpleLines[lineIndex2]);
              }
              //Ajout de la nouvelle simple line sur laquelle se trouve la scission (première partie de la simple line jusqu'au point d'intersection).
              var coordinates1 = simpleLines[cuttinglineIndex].slice(
                0,
                cuttingIndex + 1
              ); //slice extrait jusqu'à endIndex exclu.
              coordinates1.push(intersectionPointCoordinate);
              multiCoordinates1.push(coordinates1);

              line1 = new ol.geom.MultiLineString(multiCoordinates1, 'XY');

              //Seconde Multiligne:
              var multiCoordinates2 = [];

              //Ajout de la nouvelle simple line sur laquelle se trouve la scission (première partie de la simple line jusqu'au point d'intersection).
              var coordinates2 = [intersectionPointCoordinate];
              var lastPoints = simpleLines[cuttinglineIndex].slice(
                cuttingIndex + 1
              ); //slice extrait jusqu'à endIndex exclu.
              coordinates2 = coordinates2.concat(lastPoints);
              multiCoordinates2.push(coordinates2);

              //Récupération des simples lignes qui suivent celle sur laquelle se trouve la scission.
              for (
                var lineIndex3 = cuttinglineIndex + 1;
                lineIndex3 < simpleLines.length;
                lineIndex3++
              ) {
                multiCoordinates2.push(simpleLines[lineIndex3]);
              }

              line2 = new ol.geom.MultiLineString(multiCoordinates2, 'XY');

              angular.forEach(line1.getLineStrings(), function(value, key) {
                var prom1 = getSimpleLineLength(
                  value,
                  map,
                  intersectedShareObject.fti
                );
                prom1.then(function(res) {
                  line1Length += res.data[0].perimeter;
                });
                promisesLength.push(prom1);
              });

              angular.forEach(line2.getLineStrings(), function(value, key) {
                var prom2 = getSimpleLineLength(
                  value,
                  map,
                  intersectedShareObject.fti
                );
                prom2.then(function(res) {
                  line2Length += res.data[0].perimeter;
                });
                promisesLength.push(prom2);
              });
            }

            var majorPromise = $q.all(promisesLength);
            majorPromise.then(function() {
              line1Length = Math.round(line1Length * 100) / 100;
              line2Length = Math.round(line2Length * 100) / 100;

              //Création d'une deuxième feature ligne : deuxième objet issu de la scission.
              var newFeature = new ol.Feature({
                geometry: line2,
              });
              //Recopie des attributs dans le deuxième objet
              //Itérerer parmi les propriétés et les ajouter au nouveau Feature
              var properties = intersectedFeature.getProperties();
              var geomPropertieName = intersectedFeature.getGeometryName();
              angular.forEach(properties, function(value, key) {
                if (key != geomPropertieName) {
                  var prop = {};
                  prop[key] = value;
                  newFeature.setProperties(prop);
                }
              });
              //Mise à jour de l'attribut longueur si il a été configuré
              if (
                ruleConf.parameters['layerInfos'][
                  intersectedShareObject.fti.name
                ] != undefined &&
                ruleConf.parameters['layerInfos'][
                  intersectedShareObject.fti.name
                ]['lengthFieldName'] != undefined
              ) {
                var lengthFieldName =
                  ruleConf.parameters['layerInfos'][
                    intersectedShareObject.fti.name
                  ]['lengthFieldName'];
                var lengthProperties1 = {};
                lengthProperties1[lengthFieldName] = line1Length;
                intersectedFeature.setProperties(lengthProperties1);

                var lengthProperties2 = {};
                lengthProperties2[lengthFieldName] = line2Length;
                newFeature.setProperties(lengthProperties2);
              }

              //Mise à jour de la geometry de la feature ligne intersectée: premier objet issu de la scission.
              intersectedFeature.setGeometry(line1);

              //Visualisation et enregistrement de ces objets pour la sauvegarde:

              //Objet modifié (1ere objet issu de la scission)
              gclayers
                .getDrawLayer()
                .getSource()
                .addFeature(intersectedFeature);
              var updatedObject = {
                shareObject: '',
                editType:
                  intersectedFeature.getId() == undefined
                    ? EditTypesFactory.editTypes.add.name
                    : EditTypesFactory.editTypes.update.name,
                feature: intersectedFeature,
                fti: intersectedShareObject.fti,
              };
              editdescription.relatedfeatures.push(updatedObject);

              //Objet créé (2e objet issu de la scission)
              gclayers
                .getDrawLayer()
                .getSource()
                .addFeature(newFeature);
              var newObject = {
                shareObject: '',
                editType: EditTypesFactory.editTypes.add.name,
                feature: newFeature,
                fti: intersectedShareObject.fti,
              };
              editdescription.relatedfeatures.push(newObject);

              //Historisation eventuelle de l'objet lineaire modifié : Copie du feature
              if (isTobeHistoricize) {
                //Enregistrement de la copie pour historisation
                var copie = {
                  shareObject: '',
                  editType: EditTypesFactory.editTypes.tohistorize.name,
                  feature: mainFeatureCopie,
                  fti: intersectedShareObject.fti,
                };
                editdescription.relatedfeatures.push(copie);
              }
            });
          }
          return $q.all(majorPromise);
        }

        //Si l'objet principal édité est de type line
        if (featureType.typeInfo == 'LINE') {
          //Amont
          var promincut = new Array();
          var cfgUpShareObjects = ruleConf.parameters['amont']['shareObjects'];
          //Récupération de l'objet partagé intersecté par l'amont de la ligne éditée et mis en mémoire par une règle executée juste avant
          var intersectedShareObject = getIntersectedShareObject(
            cfgUpShareObjects
          );
          if (intersectedShareObject != undefined) {
            var amontCoordinate = editedGeom.getFirstCoordinate();
            var deff2 = $q.defer();
            var prom1 = performCutting(
              amontCoordinate,
              intersectedShareObject.shareObject,
              intersectedShareObject.isHistoricize
            );
            if (prom1 !== undefined) {
              prom1.then(
                function() {
                  //Aval
                  var cfgDownShareObjects =
                    ruleConf.parameters['aval']['shareObjects'];
                  //Récupération de l'objet partagé intersecté par l'aval de la ligne éditée et mis en mémoire par une règle executée juste avant
                  var intersectedShareObject2 = getIntersectedShareObject(
                    cfgDownShareObjects
                  );

                  if (
                    cfgUpShareObjects != undefined &&
                    cfgDownShareObjects != undefined &&
                    intersectedShareObject != undefined &&
                    intersectedShareObject2 != undefined &&
                    intersectedShareObject.shareObject.feature.getId() ==
                      intersectedShareObject2.shareObject.feature.getId()
                  ) {
                    intersectedShareObject2 = UpdateIntersectedShareObjectAval(
                      intersectedShareObject,
                      intersectedShareObject2,
                      editedGeom.getLastCoordinate()
                    );
                  }

                  if (intersectedShareObject2 != undefined) {
                    var avalCoordinate = editedGeom.getLastCoordinate();
                    deff2.promise = performCutting(
                      avalCoordinate,
                      intersectedShareObject2.shareObject,
                      intersectedShareObject2.isHistoricize
                    );
                    deff2.promise.then(
                      function(res) {
                        console.log(res);
                        deff2.resolve('terminé !');
                      },
                      function() {
                        deff2.resolve('terminé !');
                      }
                    );
                  } else {
                    deff2.resolve('terminé !');
                  }
                },
                function() {
                  deff2.resolve('terminé !');
                }
              );
              promincut.push(prom1, deff2.promise, deferred.promise);
              deferred.resolve('terminé !');
              return $q.all(promincut);
            } else {
              var cfgDownShareObjects =
                ruleConf.parameters['aval']['shareObjects'];
              //Récupération de l'objet partagé intersecté par l'aval de la ligne éditée et mis en mémoire par une règle executée juste avant
              var intersectedShareObject2 = getIntersectedShareObject(
                cfgDownShareObjects
              );

              if (
                cfgUpShareObjects != undefined &&
                cfgDownShareObjects != undefined &&
                intersectedShareObject != undefined &&
                intersectedShareObject2 != undefined &&
                intersectedShareObject.shareObject.feature.getId() ==
                  intersectedShareObject2.shareObject.feature.getId()
              ) {
                intersectedShareObject2 = UpdateIntersectedShareObjectAval(
                  intersectedShareObject,
                  intersectedShareObject2,
                  editedGeom.getLastCoordinate()
                );
              }

              if (intersectedShareObject2 != undefined) {
                var avalCoordinate = editedGeom.getLastCoordinate();
                deff2.promise = performCutting(
                  avalCoordinate,
                  intersectedShareObject2.shareObject,
                  intersectedShareObject2.isHistoricize
                );
                if (deff2.promise !== undefined) {
                  deff2.promise.then(
                    function(res) {
                      console.log(res);
                      deff2.resolve('terminé !');
                    },
                    function() {
                      deff2.resolve('terminé !');
                    }
                  );
                  promincut.push(prom2, deferred.promise);
                  deferred.resolve('terminé !');
                  return $q.all(promincut);
                } else {
                  deferred.resolve('terminé !');
                  return deferred.promise;
                }
              } else {
                deff2.resolve('terminé !');
              }
            }
          } else {
            //Aval
            var cfgDownShareObjects =
              ruleConf.parameters['aval']['shareObjects'];
            //Récupération de l'objet partagé intersecté par l'aval de la ligne éditée et mis en mémoire par une règle executée juste avant
            var intersectedShareObject2 = getIntersectedShareObject(
              cfgDownShareObjects
            );

            if (
              cfgUpShareObjects != undefined &&
              cfgDownShareObjects != undefined &&
              intersectedShareObject != undefined &&
              intersectedShareObject2 != undefined &&
              intersectedShareObject.shareObject.feature.getId() ==
                intersectedShareObject2.shareObject.feature.getId()
            ) {
              intersectedShareObject2 = UpdateIntersectedShareObjectAval(
                intersectedShareObject,
                intersectedShareObject2,
                editedGeom.getLastCoordinate()
              );
            }

            if (intersectedShareObject2 != undefined) {
              var avalCoordinate = editedGeom.getLastCoordinate();
              var prom2 = performCutting(
                avalCoordinate,
                intersectedShareObject2.shareObject,
                intersectedShareObject2.isHistoricize
              );
              if (prom2 !== undefined) {
                prom2.then(function() {});
                promincut.push(prom2, deferred.promise);
                deferred.resolve('terminé !');
                return $q.all(promincut);
              } else {
                deferred.resolve('terminé !');
                return deferred.promise;
              }
            }
          }
          promincut.push(prom1, prom2, deferred.promise);
          deferred.resolve('terminé !');
          return $q.all(promincut);
        }
        //Si l'objet principal édité est de type point
        else if (featureType.typeInfo == 'POINT') {
          var deff1 = $q.defer();
          var promincut = new Array();
          var cfgShareObjects = ruleConf.parameters['point']['shareObjects'];
          //Récupération de l'objet partagé intersecté par l'objet ponctuel éditée et mis en mémoire par une règle executée juste avant
          var intersectedShareObject3 = getIntersectedShareObject(
            cfgShareObjects
          );
          if (intersectedShareObject3 != undefined) {
            var coordinate = editedGeom.getCoordinates();
            deff1.promise = performCutting(
              coordinate,
              intersectedShareObject3.shareObject,
              intersectedShareObject3.isHistoricize
            );
            if (deff1.promise !== undefined) {
              deff1.promise.then(function() {
                deff1.resolve('terminé !');
              });
              promincut.push(deff1.promise, deferred.promise);
              //deff1.resolve("terminé !");
              deferred.resolve('terminé !');
              return $q.all(promincut);
            } else {
              deferred.resolve('terminé !');
              return deferred.promise;
            }
          }
        }
        promincut.push(deferred.promise);
        deferred.resolve('terminé !');
        return $q.all(promincut);
      }

      /*function performCutting(intersectionPointCoordinate,intersectedShareObject, isTobeHistoricize){
                    if (intersectedShareObject != undefined){
                        var intersectedFeature = intersectedShareObject.feature;

                        //Copie de l'objet qui va être scindé pour historisation eventuelle
                        if (isTobeHistoricize){
                            var mainFeatureCopie = intersectedFeature.clone();
                            mainFeatureCopie.setId(intersectedFeature.getId());
                        }

                        var line = intersectedFeature.getGeometry();

                        var isSimpleLine = line instanceof ol.geom.LineString;
                        var isMultiLine = line instanceof ol.geom.MultiLineString;

                        //Index du segment sur lequel couper la line.
                        var cuttingIndex = undefined;
                        var cuttinglineIndex = undefined;

                        var line1 = undefined;
                        var line2 = undefined;

                        var line1Length = 0;
                        var line2Length = 0;

                        if ( (Math.round(intersectionPointCoordinate[0]*100) ==  Math.round(line.getFirstCoordinate()[0]*100) && Math.round(intersectionPointCoordinate[1]*100) ==  Math.round(line.getFirstCoordinate()[1]*100) )
                                || ( Math.round(intersectionPointCoordinate[0]*100) ==  Math.round(line.getLastCoordinate()[0]*100) && Math.round(intersectionPointCoordinate[1]*100) ==  Math.round(line.getLastCoordinate()[1]*100 ) ) ){
                                console.info("Pas de scission possible de l'objet intersecté id="+intersectedFeature.id+" de la couche ="+intersectedShareObject.fti.name);
                                return;
                        }


                        if (isSimpleLine){
                            var originCoordinates = line.getCoordinates();
                            cuttingIndex = getCuttingIndexOnSimpleLine(originCoordinates, intersectionPointCoordinate);

                            if (cuttingIndex == undefined){
                                console.info("Pas de scission possible de l'objet intersecté id="+intersectedFeature.id+" de la couche ="+intersectedShareObject.fti.name);
                                return;
                            }
                            //Création des deux objets lines issus de la scission.
                            var coordinates1 = originCoordinates.slice(0,cuttingIndex+1);//slice extrait jusqu'à endIndex exclu.
                            coordinates1.push(intersectionPointCoordinate);
                            line1 = new ol.geom.LineString(coordinates1,'XY');
                            line1Length = getSimpleLineLength(line1,map);
                            line1Length = (Math.round(line1Length*100)) / 100;

                            var coordinates2 = [intersectionPointCoordinate];
                            coordinates2 = coordinates2.concat(originCoordinates.slice(cuttingIndex+1));
                            line2 = new ol.geom.LineString(coordinates2,'XY');
                            line2Length = getSimpleLineLength(line2,map);
                            line2Length = (Math.round(line2Length*100)) / 100;
                        }
                        else if (isMultiLine){

                            var simpleLines = line.getCoordinates();
                            //Pour chaque simple line de la multiline
                            for (var lineIndex=0;lineIndex<simpleLines.length;lineIndex++){
                                cuttingIndex = getCuttingIndexOnSimpleLine(simpleLines[lineIndex], intersectionPointCoordinate);
                                //Si l'index du segement de la simple line a été trouvé
                                if (cuttingIndex != undefined){
                                    //mémorisation de l'index de la simple line
                                     cuttinglineIndex = lineIndex;
                                     break;
                                 }
                            }
                            if (cuttingIndex == undefined){
                                console.info("Pas de scission possible de l'objet intersecté id="+intersectedFeature.id+" de la couche ="+intersectedShareObject.fti.name);
                                return;
                            }
                            //Création des deux objets Multilines issus de la scission.

                            //Premiere Multiligne:
                            var multiCoordinates1 = [];

                            //Récupération des simples lignes précedents celle sur laquelle se trouve la scission.
                            for (var lineIndex2=0; lineIndex2<cuttinglineIndex; lineIndex2++){
                                 multiCoordinates1.push(simpleLines[lineIndex2]);
                            }
                            //Ajout de la nouvelle simple line sur laquelle se trouve la scission (première partie de la simple line jusqu'au point d'intersection).
                            var coordinates1 = simpleLines[cuttinglineIndex].slice(0,cuttingIndex+1);//slice extrait jusqu'à endIndex exclu.
                            coordinates1.push(intersectionPointCoordinate);
                            multiCoordinates1.push(coordinates1);

                            line1 = new ol.geom.MultiLineString(multiCoordinates1,'XY');

                            //Seconde Multiligne:
                            var multiCoordinates2 = [];

                            //Ajout de la nouvelle simple line sur laquelle se trouve la scission (première partie de la simple line jusqu'au point d'intersection).
                            var coordinates2 = [intersectionPointCoordinate];
                            var lastPoints = simpleLines[cuttinglineIndex].slice(cuttingIndex+1);//slice extrait jusqu'à endIndex exclu.
                            coordinates2 = coordinates2.concat(lastPoints);
                            multiCoordinates2.push(coordinates2);

                            //Récupération des simples lignes qui suivent celle sur laquelle se trouve la scission.
                            for (var lineIndex3=cuttinglineIndex+1; lineIndex3<simpleLines.length; lineIndex3++){
                                 multiCoordinates2.push(simpleLines[lineIndex3]);
                            }

                            line2 = new ol.geom.MultiLineString(multiCoordinates2,'XY');

                            angular.forEach(line1.getLineStrings(), function(value, key) {
                                line1Length += getSimpleLineLength(value,map);
                            });
                            line1Length = (Math.round(line1Length*100)) / 100;

                            angular.forEach(line2.getLineStrings(), function(value, key) {
                                line2Length += getSimpleLineLength(value,map);
                            });
                            line2Length = (Math.round(line2Length*100)) / 100;
                        }

                        //Création d'une deuxième feature ligne : deuxième objet issu de la scission.
                        var newFeature = new ol.Feature({
                             geometry : line2
                        });
                        //Recopie des attributs dans le deuxième objet
                        //Itérerer parmi les propriétés et les ajouter au nouveau Feature
                        var properties = intersectedFeature.getProperties();
                        var geomPropertieName = intersectedFeature.getGeometryName();
                        angular.forEach(properties, function(value, key) {
                            if (key != geomPropertieName ){
                                var prop = {};
                                prop[key] = value;
                                newFeature.setProperties(prop);
                            }
                         });
                        //Mise à jour de l'attribut longueur si il a été configuré
                        if ( ruleConf.parameters['layerInfos'][intersectedShareObject.fti.name] != undefined && ruleConf.parameters['layerInfos'][intersectedShareObject.fti.name]['lengthFieldName'] != undefined){
                            var lengthFieldName = ruleConf.parameters['layerInfos'][intersectedShareObject.fti.name]['lengthFieldName'];
                            var lengthProperties1 = {};
                            lengthProperties1[lengthFieldName] = line1Length;
                            intersectedFeature.setProperties(lengthProperties1);

                            var lengthProperties2 = {};
                            lengthProperties2[lengthFieldName] = line2Length;
                            newFeature.setProperties(lengthProperties2);
                        }

                        //Mise à jour de la geometry de la feature ligne intersectée: premier objet issu de la scission.
                        intersectedFeature.setGeometry(line1);

                        //Visualisation et enregistrement de ces objets pour la sauvegarde:

                        //Objet modifié (1ere objet issu de la scission)
                        gclayers.getDrawLayer().getSource().addFeature(intersectedFeature);
                        var typeis = intersectedFeature.getId() == undefined ? EditTypesFactory.editTypes.add.name : EditTypesFactory.editTypes.update.name ;
                        var updatedObject = {shareObject:'',editType:typeis, feature: intersectedFeature, fti:intersectedShareObject.fti};
                        editdescription.relatedfeatures.push(updatedObject);

                        //Objet créé (2e objet issu de la scission)
                        gclayers.getDrawLayer().getSource().addFeature(newFeature);
                        var newObject = {shareObject:'',editType:EditTypesFactory.editTypes.add.name, feature: newFeature, fti:intersectedShareObject.fti};
                        editdescription.relatedfeatures.push(newObject);

                        //Historisation eventuelle de l'objet lineaire modifié : Copie du feature
                        if (isTobeHistoricize){
                            //Enregistrement de la copie pour historisation
                            var copie = {shareObject:'',editType:EditTypesFactory.editTypes.tohistorize.name, feature: mainFeatureCopie, fti:intersectedShareObject.fti};
                            editdescription.relatedfeatures.push(copie);
                        }
                    }
                }


                //Si l'objet principal édité est de type line
                if (featureType.typeInfo == 'LINE'){
                   //Amont
                    var cfgUpShareObjects = ruleConf.parameters['amont']['shareObjects'];
                    //Récupération de l'objet partagé intersecté par l'amont de la ligne éditée et mis en mémoire par une règle executée juste avant
                    var intersectedShareObject = getIntersectedShareObject(cfgUpShareObjects);
                    if (intersectedShareObject != undefined){
                        var amontCoordinate = editedGeom.getFirstCoordinate();
                        performCutting(amontCoordinate, intersectedShareObject.shareObject, intersectedShareObject.isHistoricize);
                    }

                    //Aval
                    var cfgDownShareObjects = ruleConf.parameters['aval']['shareObjects'];
                    //Récupération de l'objet partagé intersecté par l'aval de la ligne éditée et mis en mémoire par une règle executée juste avant
                    var intersectedShareObject2 = getIntersectedShareObject(cfgDownShareObjects);

                    if (    cfgUpShareObjects != undefined
                            && cfgDownShareObjects != undefined
                            && intersectedShareObject != undefined
                            && intersectedShareObject2 != undefined
                            && intersectedShareObject.shareObject.feature.getId() == intersectedShareObject2.shareObject.feature.getId() )
                            intersectedShareObject2 = UpdateIntersectedShareObjectAval(intersectedShareObject , intersectedShareObject2 , editedGeom.getLastCoordinate());

                    if (intersectedShareObject2 != undefined){
                        var avalCoordinate = editedGeom.getLastCoordinate();
                        performCutting(avalCoordinate, intersectedShareObject2.shareObject, intersectedShareObject2.isHistoricize);
                    }

                }
                //Si l'objet principal édité est de type point
                else if (featureType.typeInfo == 'POINT'){
                    var cfgShareObjects = ruleConf.parameters['point']['shareObjects'];
                    //Récupération de l'objet partagé intersecté par l'objet ponctuel éditée et mis en mémoire par une règle executée juste avant
                    var intersectedShareObject3 = getIntersectedShareObject(cfgShareObjects);
                    if (intersectedShareObject3 != undefined){
                        var coordinate = editedGeom.getCoordinates();
                        performCutting(coordinate, intersectedShareObject3.shareObject, intersectedShareObject3.isHistoricize);
                    }
                }

                deferred.resolve("terminé !");
                return deferred.promise;
            };*/

      /**
       * Déplace les objets ponctuels portés par les extrémités de la ligne principale déplacée.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function moveExtremityPoint(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        var editedGeom = editdescription.editedfeature.getGeometry();

        /*
         * Récupére le feature partagé à partir des informations de configuration
         * (contenant le nom de l'objet (shareObject) et le nom de la layer)
         * @param {type} cfgShareObjects
         * @returns {undefined|EditRulesProvider_L7.EditRulesProvider.$get.moveExtremityPoint.getIntersectedShareObject.EditRulesProviderAnonym$8}
         */
        function getIntersectedShareObject(cfgShareObjects) {
          for (var j = 0; j < cfgShareObjects.length; j++) {
            for (var k = 0; k < editdescription.shareObjects.length; k++) {
              if (
                editdescription.shareObjects[k].shareObject ==
                  cfgShareObjects[j].name &&
                editdescription.shareObjects[k].fti.name ==
                  cfgShareObjects[j].layerName
              ) {
                return {
                  shareObject: editdescription.shareObjects[k],
                  isHistoricize: cfgShareObjects[j].isHistoricize,
                };
              }
            }
          }
          return undefined;
        }

        /**
         * Déplace l'objet partagé 'intersectedShareObject' sur les coordonnée de l'extrémité 'lineExtremityCoordinate'
         * @param {type} lineExtremityCoordinate
         * @param {type} intersectedShareObject
         * @param {type} isTobeHistoricize
         * @returns {undefined}
         */
        function performMove(
          lineExtremityCoordinate,
          intersectedShareObject,
          isTobeHistoricize
        ) {
          if (intersectedShareObject != undefined) {
            var intersectedFeaturePoint = intersectedShareObject.feature;

            var intersectedGeom = intersectedFeaturePoint.getGeometry();
            if (!(intersectedGeom instanceof ol.geom.Point)) {
              console.info(
                'moveExtremityPoint: objet intersecté ' +
                  intersectedShareObject.shareObject +
                  ' de la layer ' +
                  intersectedShareObject.fti.name +
                  ' n\'est pas un ponctuel !'
              );
              return;
            }

            //Copie de l'objet qui va être scindé pour historisation eventuelle
            if (isTobeHistoricize) {
              var mainFeatureCopie = intersectedFeaturePoint.clone();
              mainFeatureCopie.setId(intersectedFeaturePoint.getId());
            }

            //Objet ponctuel déplacé vers la nouvelle extrémité de la ligne
            //Mise à jour de la géométrie du point (le déplécement)
            intersectedGeom.setCoordinates(lineExtremityCoordinate);

            //Enregistrement
            //KIS-3631: try-catch pour eviter le crash si l'objet est deja dans la couche
            try {
              gclayers
              .getDrawLayer()
              .getSource()
              .addFeature(intersectedFeaturePoint);
            } catch(e) {
              console.info('Feature déja dans la couche');
            }
            var updatedObject = {
              shareObject: intersectedShareObject.shareObject,
              editType: EditTypesFactory.editTypes.update.name,
              feature: intersectedFeaturePoint,
              fti: intersectedShareObject.fti,
            };
            editdescription.relatedfeatures.push(updatedObject);

            //Historisation eventuelle de l'objet lineaire modifié : Copie du feature
            if (isTobeHistoricize) {
              //Enregistrement de la copie pour historisation
              var copie = {
                shareObject: '',
                editType: EditTypesFactory.editTypes.tohistorize.name,
                feature: mainFeatureCopie,
                fti: intersectedShareObject.fti,
              };
              editdescription.relatedfeatures.push(copie);
            }
          }
        }

        //Si l'objet principal édité est de type line
        if (featureType.typeInfo == 'LINE') {
          //Amont
          var cfgUpShareObjects = ruleConf.parameters['amont']['shareObjects'];
          var intersectedShareObject = getIntersectedShareObject(
            cfgUpShareObjects
          );
          if (intersectedShareObject != undefined) {
            var amontCoordinate = editedGeom.getFirstCoordinate();
            performMove(
              amontCoordinate,
              intersectedShareObject.shareObject,
              intersectedShareObject.isHistoricize
            );
          }

          //Aval
          var cfgDownShareObjects = ruleConf.parameters['aval']['shareObjects'];
          var intersectedShareObject2 = getIntersectedShareObject(
            cfgDownShareObjects
          );
          if (intersectedShareObject2 != undefined) {
            var avalCoordinate = editedGeom.getLastCoordinate();
            performMove(
              avalCoordinate,
              intersectedShareObject2.shareObject,
              intersectedShareObject2.isHistoricize
            );
          }
        }
        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Déplace les objets ponctuels et linéaires portés par les lignes éditées.
       * Utilise les objets récupérés par la règle 'GetAllFeaturesIntersectingInitialLine'.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function moveObjectsOnLine(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle moveObjectsOnLine non adaptée à la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        //Récupération des informations de configuration sur les objets partagés à utiliser dans cette règle (ces objets partagés sont ceux ajoutés par des règles executées juste avant).
        var cfgUpShareObjects = ruleConf.parameters['shareObjects'];

        //Si pas de config
        if (
          cfgUpShareObjects == undefined &&
          !Array.isArray(cfgUpShareObjects)
        ) {
          console.error(
            'Règle moveObjectsOnLine mal ou pas configurée pour la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        //Récupération de l'ensemble des layers
        var layers = gclayers.getOperationalLayer();

        //Récupère les features partagés stockés dans 'currenteditdescription' et dont les noms d'objets partagés sont dans l'objet de configuration 'cfgUpShareObjects'
        function getIntersectedShareObjects(currenteditdescription) {
          var shareObjects = [];

          //Pour chaque layer d'intersection de la configuration
          for (var j = 0; j < cfgUpShareObjects.length; j++) {
            if (currenteditdescription.shareObjects !== undefined) {
              //Recherche dans les features récupérés les objets partagés du même nom.
              for (
                var k = 0;
                k < currenteditdescription.shareObjects.length;
                k++
              ) {
                if (
                  currenteditdescription.shareObjects[k].shareObject ==
                    cfgUpShareObjects[j].name &&
                  currenteditdescription.shareObjects[k].fti.name ==
                    cfgUpShareObjects[j].layerName
                ) {
                  shareObjects.push({
                    shareObject: currenteditdescription.shareObjects[k],
                    isHistoricize: cfgUpShareObjects[j].isHistoricize,
                  });
                }
              }
            }
          }
          return shareObjects;
        }

        /**
         * Utilitaire: permet de savoir si l'objet feature passé en argument
         * est déjà present dans l'objet representant l'action d'édition ('editdescription').
         * @param {type} featureID
         * @returns {Boolean}
         */
        function isObjectRecordedInRelatedfeatures(featureID) {
          var isPresent = false;
          for (var i = 0; i < editdescription.relatedfeatures.length; i++) {
            if (
              featureID == editdescription.relatedfeatures[i].feature.getId()
            ) {
              isPresent = true;
              break;
            }
          }
          return isPresent;
        }

        //Traitement d'un objet porté par la ligne featureLine modifiée.
        function treatNextShareObject(
          shareObject,
          formerFeatureLine,
          newFeatureLine
        ) {
          var shareObjectlayer = getLayerByName(
            shareObject.shareObject.fti.name,
            layers
          );
          if (shareObjectlayer == undefined) {
            var deferred = $q.defer();
            console.error(
              'Règle moveObjectsOnLine treatNextShareObject(): shareObjectlayer undefined !'
            );
            deferred.resolve('terminé !');
            return deferred.promise;
          }

          //Récupération de la géometrie de l'objet feature partagé qui est à déplacer.
          var intersectedGeom = shareObject.shareObject.feature.getGeometry();

          //Préparation des données pour appeler le service java de calcul des nouvelles coordonnnées.
          var geoms = [];
          var firstPromise = undefined;

          var formerLineGeoJson = format.writeGeometryObject(
            formerFeatureLine.getGeometry()
          );
          var newLineGeoJson = format.writeGeometryObject(
            newFeatureLine.getGeometry()
          );
          var geomtoMoveGeoJson = format.writeGeometryObject(intersectedGeom);
          geoms.push(formerLineGeoJson);
          geoms.push(newLineGeoJson);
          geoms.push(geomtoMoveGeoJson);

          //Appelle du service java de calcul des nouvelles coordonnnées
          //Si la géometrie à déplacer est de type point
          if (intersectedGeom instanceof ol.geom.Point) {
            firstPromise = GeometryFactory.movepointwithline(geoms);
          } else if (
            intersectedGeom instanceof ol.geom.LineString ||
            intersectedGeom instanceof ol.geom.MultiLineString
          ) {
            firstPromise = GeometryFactory.moveLineWithLine(geoms);
          }
          //Sinon
          else {
            console.info(
              'treatNextShareObject() dans  movePointWithLine : objet à déplacer de la couche ' +
                shareObjectlayer.fti.name +
                ' non traité car pas de type point'
            );
            //Fin de la méthode de la recherche d'intesection pour l'objet édité featureLine (résolution de la mainPromise)
            var deferred = $q.defer();
            deferred.resolve('terminé !');
            return deferred.promise;
          }

          //Récupération de la réponse du service de calcul des nouvelles coordonnées (lorsque la prommesse est resolue)
          var mainPromise = firstPromise.then(
            function(res) {
              var movedGeom = format.readGeometry(res.data);
              var featureToUpdate = shareObject.shareObject.feature;
              featureToUpdate.setGeometry(movedGeom);

              //Si l'objet n'est pas déjà enregistré, on l'enregistre
              if (
                !isObjectRecordedInRelatedfeatures(
                  shareObject.shareObject.feature.getId()
                )
              ) {
                var newObject = {
                  shareObject: shareObject.shareObject.name,
                  editType: EditTypesFactory.editTypes.update.name,
                  feature: featureToUpdate,
                  fti: shareObjectlayer.fti,
                };
                editdescription.relatedfeatures.push(newObject);
                //Historisation si configurée
                if (shareObject.isHistoricize) {
                  var newObject2 = {
                    shareObject: shareObject.shareObject.name,
                    editType: EditTypesFactory.editTypes.tohistorize.name,
                    feature: featureToUpdate,
                    fti: shareObjectlayer.fti,
                  };
                  editdescription.relatedfeatures.push(newObject2);
                }
              }
            },
            function(error) {
              console.error(
                'Erreur lors de la requete  movePointWithLine de l\'objet à déplacer de la couche ' +
                  shareObjectlayer.fti.name
              );
              //Fin de la méthode de la recherche d'intesection pour l'objet édité featureLine (résolution de la mainPromise)
              var deferred = $q.defer();
              deferred.resolve('terminé !');
              return deferred.promise;
            }
          );

          return mainPromise;
        }

        //Traitement des objets par la ligne modifié
        function treatAnEditedLine(currenteditdescription, featureLine) {
          var promises = [];

          //Récupération des features intersectés par la ligne éditée dont les noms d'objets partagés sont dans cfgUpShareObjects.
          var shareObjects = getIntersectedShareObjects(currenteditdescription);

          //Récupérer la featureLine tel qu'elle est en base (avant sa modification).
          var cql_filter = 'IN (\'' + featureLine.getId() + '\')';
          //La requete d'intersection de la featureLine avec les objets de la couche cfgShareObj.layerName.
          var promise = ogcFactory.getfeatures(
            'GetFeature',
            'WFS',
            '1.0.0',
            currenteditdescription.fti.uid,
            'json',
            map
              .getView()
              .getProjection()
              .getCode(),
            cql_filter
          );
          promises.push(promise);

          //Traitement de chaque objet shareObject porté par la featureLine
          var promise2 = promise.then(
            function(res) {
              var promises = [];
              var formerFeatureLine = format.readFeatures(res.data);
              if (
                formerFeatureLine != undefined &&
                formerFeatureLine.length > 0
              ) {
                //Pour chaque shareObject contenant le feature intersecté par la ligne éditée
                for (var j = 0; j < shareObjects.length; j++) {
                  var shareObject = shareObjects[j];
                  var promise = treatNextShareObject(
                    shareObject,
                    formerFeatureLine[0],
                    featureLine
                  );
                  promises.push(promise);
                }
              }
              return $q.all(promises);
            },
            function(error) {
              console.error(
                'Erreur à l\'appel de ogc.data(), cql_filter:' + cql_filter
              );
              var deferred = $q.defer();
              deferred.resolve('terminé !');
              return deferred.promise;
            }
          );
          promises.push(promise2);

          return $q.all(promises);
        }

        //Pour la ligne principale
        //Suppression des objets portés
        var promises = treatAnEditedLine(
          editdescription,
          editdescription.editedfeature
        );

        //Pour les Les AUTRES lignes éditées (édition multiple de lignes)
        //Index de l'objet secondaire édité
        var relatedfeaturesIndex = 0;
        //on mémorise le nombre initial d'objets secondaires car on va au cours de l'execution possiblement alimenter ce tableau.
        //Si la règle ajoutait de nouveaux objets et que l'on voulait prendre en compte ces nouveaux objets,
        //on aurait comparé 'relatedfeaturesIndex' à 'editdescription.relatedfeatures.length' dans treatNextLineObject().
        var nbRelatedFeature = editdescription.relatedfeatures.length;

        function treatNextLineObject() {
          if (relatedfeaturesIndex < nbRelatedFeature) {
            var related = editdescription.relatedfeatures[relatedfeaturesIndex];
            if (
              related.editType == editdescription.editType &&
              related.fti.name == editdescription.fti.name
            ) {
              var anOtherpromises = treatAnEditedLine(related, related.feature);

              anOtherpromises.then(
                function(res) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                },
                function(error) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                }
              );
            } else {
              relatedfeaturesIndex++;
              treatNextLineObject();
            }
          }
          //si fin du parcours des objets
          else {
            deferred.resolve('rule end !');
          }
        }

        treatNextLineObject();

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all([promises, deferred.promise]);
      }

      /**
       * Supprime les objets partagés portés par les objets linéaire édités.
       * Cette règle utilise le résultat de la règle 'getAllFeaturesIntersectingLine' qui récupère les objets portés par les lignes éditées.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function deleteObjectsOnLine(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle deleteObjectsOnLine non adaptée à la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }
        var cfgUpShareObjects = ruleConf.parameters['shareObjects'];

        if (
          cfgUpShareObjects == undefined &&
          !Array.isArray(cfgUpShareObjects)
        ) {
          console.error(
            'Règle deleteObjectsOnLine mal ou pas configurée pour la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        function isThemeLayer(layer) {
          return layer.fti.theme == editdescription.theme;
        }

        function getThemeLayerUids() {
          var layerUids = '';
          var themeLayers = layers.filter(isThemeLayer);
          angular.forEach(themeLayers, function(l, key) {
            layerUids += l.fti.uid + ',';
          });
          layerUids.substring(0, layerUids.length - 1);
          return layerUids;
        }

        function getLayerUidsFromRule() {
          var layerUids = cfgUpShareObjects
            .map(function(x) {
              var fti = gclayers.getOperationalLayerByName(x.layerName);
              if (fti) {
                return fti.uid;
              }
            })
            .filter(function(x) {
              if (x) {
                return x;
              }
            });
          return layerUids.join(',');
        }

        //Récupère les features partagés stockés dans 'currenteditdescription' et dont les noms d'objets partagés sont dans l'objet de configuration 'cfgUpShareObjects'
        function getIntersectedShareObjects(currenteditdescription) {
          var shareObjects = [];

          if (currenteditdescription.shareObjects != undefined) {
            //Pour chaque layer d'intersection de la configuration
            for (var j = 0; j < cfgUpShareObjects.length; j++) {
              //Recherche dans les features récupérés les objets partagés du même nom.
              for (
                var k = 0;
                k < currenteditdescription.shareObjects.length;
                k++
              ) {
                if (
                  currenteditdescription.shareObjects[k].shareObject ==
                    cfgUpShareObjects[j].name &&
                  currenteditdescription.shareObjects[k].fti.name ==
                    cfgUpShareObjects[j].layerName
                ) {
                  shareObjects.push({
                    shareObject: currenteditdescription.shareObjects[k],
                    isHistoricize: cfgUpShareObjects[j].isHistoricize,
                  });
                }
              }
            }
          }
          return shareObjects;
        }

        function getFeatureCountInEditdescription(featureID) {
          var count = 0;
          angular.forEach(editdescription.shareObjects, function(
            shareObj,
            key
          ) {
            if (shareObj.feature.getId() == featureID) {
              count++;
            }
          });
          angular.forEach(editdescription.relatedfeatures, function(
            related,
            key
          ) {
            angular.forEach(related.shareObjects, function(shareObj, key) {
              if (shareObj.feature.getId() == featureID) {
                count++;
              }
            });
          });
          return count;
        }

        function isObjectRecordedInRelatedfeatures(featureID) {
          var isPresent = false;
          for (var i = 0; i < editdescription.relatedfeatures.length; i++) {
            if (
              featureID == editdescription.relatedfeatures[i].feature.getId()
            ) {
              isPresent = true;
              break;
            }
          }
          return isPresent;
        }

        function treatNextShareObject(shareObject, promises) {
          var shareObjectlayer = getLayerByName(
            shareObject.shareObject.fti.name,
            layers
          );
          if (shareObjectlayer == undefined) {
            return;
          }

          var intersectedGeom = shareObject.shareObject.feature.getGeometry();
          console.log(shareObject, editdescription);

          var geoJson = format.writeGeometryObject(intersectedGeom);

          //Buffer de la geometrie de l'objet ligne courant.
          var firstPromise = GeometryFactory.buffer(geoJson, 'ROUND', 1);
          promises.push(firstPromise);

          //Une fois le buffer de l'objet ligne effectué, on peut rechercher les intersections.
          var mainPromise = firstPromise.then(
            function(res) {
              var polygon = res.data;
              var lineCoordsString2 = '';

              if (polygon.type == 'Polygon') {
                var polygonGeom = format.readGeometry(polygon);
                angular.forEach(polygonGeom.getLinearRings(), function(
                  linearRing,
                  key
                ) {
                  var coords = linearRing.getCoordinates();
                  lineCoordsString2 += '(';
                  angular.forEach(coords, function(value, key) {
                    lineCoordsString2 += value[0] + ' ' + value[1] + ',';
                  });
                  lineCoordsString2 = lineCoordsString2.substring(
                    0,
                    lineCoordsString2.length - 1
                  );
                  lineCoordsString2 += ')';
                });
                var cql_filter =
                  'INTERSECTS(geom, Polygon(' + lineCoordsString2 + '))';

                //UID des layers du theme d'édition sur lequel on désire connaitre les intersections avec le shareObject courant.
                var uidSharedObj = shareObject.shareObject.fti.uid;
                var layersUids = editdescription.fti.uid + ',' + uidSharedObj;
                //var layersUids = "08f6b94b-5031-4c19-a370-161b0d0d4f6a,dd2ab6fb-2347-4d5b-b05e-e7fcbecfa28e";

                //La requete d'intersection de la featureLine avec les objets de la couche cfgShareObj.layerName.
                var promise = ogcFactory.getfeatures(
                  'GetFeature',
                  'WFS',
                  '1.0.0',
                  layersUids,
                  'json',
                  map
                    .getView()
                    .getProjection()
                    .getCode(),
                  cql_filter
                );
                //Préparation du traitement de la Reponse.
                var callBackSuccess = getCallbackSuccess(
                  shareObject,
                  shareObjectlayer
                );
                //promise2 sera resolue lorsque la callback aura fini d'être executée (car elle ne renvoie pas de promesse)
                var promise2 = promise.then(callBackSuccess, function(error) {
                  console.error(
                    'Erreur à l\'appel de ogc.data(), cql_filter:' + cql_filter
                  );
                });
                return promise2;
              } else {
                var deferred = $q.defer();
                console.error(
                  'Règle deleteObjectsOnLine featureLine after buffer operation is not a Polygon !'
                );
                deferred.resolve('terminé !');
                return deferred.promise;
              }
            },
            function(error) {
              console.error(
                'Erreur lors de la requete de bufferisation de l\'objet édité line de la couche ' +
                  currenteditdescription.fti.name
              );
              //Fin de la méthode de la recherche d'intesection pour l'objet édité featureLine (résolution de la mainPromise)
              return undefined;
            }
          );

          promises.push(mainPromise);
        }

        //Supprime les objets partagés de la ligne éditée presents dans 'currenteditdescription' correspondant à ceux configurés dans la règle.
        //Condition pour la suppression: l'objet partagé n'intersecte pas d'autres objets non supprimés.
        function treatAnEditedLine(currenteditdescription) {
          var promises = [];

          //Récupération des features intersectés par la ligne éditée dont les noms d'objets partagés sont dans cfgUpShareObjects.
          var shareObjects = getIntersectedShareObjects(currenteditdescription);

          //Pour chaque shareObject contenant le feature intersecté par la ligne éditée
          for (var j = 0; j < shareObjects.length; j++) {
            var shareObject = shareObjects[j];
            treatNextShareObject(shareObject, promises, currenteditdescription);
          }

          return $q.all(promises);
        }

        function getCallbackSuccess(shareObj, shareObjectlayer) {
          return function(result) {
            var intersectedFeatures = format.readFeatures(result.data);

            //Si la geométrie de l'objet n'intersecte que deux objets (l'objet lui-même et celui supprimé, alors on peut supprimer cet objet partagé)
            if (
              intersectedFeatures != undefined &&
              intersectedFeatures.length == 2
            ) {
              var newObject = {
                shareObject: shareObj.shareObject.name,
                editType: EditTypesFactory.editTypes.delete.name,
                feature: shareObj.shareObject.feature,
                fti: shareObjectlayer.fti,
              };
              editdescription.relatedfeatures.push(newObject);

              //Historisation si configurée
              if (shareObj.isHistoricize) {
                var newObject = {
                  shareObject: shareObj.shareObject.name,
                  editType: EditTypesFactory.editTypes.tohistorize.name,
                  feature: shareObj.shareObject.feature,
                  fti: shareObjectlayer.fti,
                };
                editdescription.relatedfeatures.push(newObject);
              }
              //Mise en évidence du feature intersecté à supprimer
              gclayers
                .getselectSource()
                .addFeature(shareObj.shareObject.feature);
            }
            //Si le nombre d'intersections trouvées au niveau serveur est égale au nombre de fois que l'objet est present dans les shareobjets
            // (nombre de fois où l'objet est present sur l'ensemble des lignes à supprimer).
            //(+1 car le serveur considère l'intersection de la geometrie de l'objet avec lui-même)
            else if (
              intersectedFeatures.length ==
              1 +
                getFeatureCountInEditdescription(
                  shareObj.shareObject.feature.getId()
                )
            ) {
              if (
                !isObjectRecordedInRelatedfeatures(
                  shareObj.shareObject.feature.getId()
                )
              ) {
                var newObject = {
                  shareObject: shareObj.shareObject.name,
                  editType: EditTypesFactory.editTypes.delete.name,
                  feature: shareObj.shareObject.feature,
                  fti: shareObjectlayer.fti,
                };
                editdescription.relatedfeatures.push(newObject);

                //Mise en évidence du feature intersecté à supprimer
                gclayers
                  .getselectSource()
                  .addFeature(shareObj.shareObject.feature);

                //Historisation si configurée
                if (shareObj.isHistoricize) {
                  var newObject = {
                    shareObject: shareObj.shareObject.name,
                    editType: EditTypesFactory.editTypes.tohistorize.name,
                    feature: shareObj.shareObject.feature,
                    fti: shareObjectlayer.fti,
                  };
                  editdescription.relatedfeatures.push(newObject);
                }
              }
            }
            /* else {
                                //Si le feature à supprimer n'est pas déjà enregistré, ( cas possible si le feature est à l'intersection d'au moins deux linéaires supprimés)


                                    var newObject = {shareObject:shareObj.shareObject.name,editType:EditTypesFactory.editTypes.delete.name, feature: shareObj.shareObject.feature, fti:shareObjectlayer.fti};
                                    editdescription.relatedfeatures.push(newObject);

                                    //Mise en évidence du feature intersecté à supprimer
                                     gclayers.getselectSource().addFeature(shareObj.shareObject.feature);

                                    //Historisation si configurée
                                    if (shareObj.isHistoricize){
                                        var newObject2 = {shareObject:shareObj.shareObject.name,editType:EditTypesFactory.editTypes.tohistorize.name, feature: shareObj.shareObject.feature, fti:shareObjectlayer.fti};
                                        editdescription.relatedfeatures.push(newObject2);
                                    }
                                }
                            }*/
          };
        }

        //Pour la ligne principale
        //Suppression des objets portés
        var promises = treatAnEditedLine(editdescription);

        //Pour les Les AUTRES lignes éditées (édition multiple de lignes)
        //Index de l'objet secondaire édité
        var relatedfeaturesIndex = 0;
        //on mémorise le nombre initial d'objets secondaires car on va au cours de l'execution possiblement alimenter ce tableau.
        //Si la règle ajoutait de nouveaux objets et que l'on voulait prendre en compte ces nouveaux objets,
        //on aurait comparé 'relatedfeaturesIndex' à 'editdescription.relatedfeatures.length' dans treatNextLineObject().
        var nbRelatedFeature = editdescription.relatedfeatures.length;

        function treatNextLineObject() {
          if (relatedfeaturesIndex < nbRelatedFeature) {
            var related = editdescription.relatedfeatures[relatedfeaturesIndex];
            if (
              related.editType == editdescription.editType &&
              related.fti.name == editdescription.fti.name
            ) {
              var anOtherpromises = treatAnEditedLine(related);

              anOtherpromises.then(() => {
                relatedfeaturesIndex++;
                treatNextLineObject();
              },
              () => {
                relatedfeaturesIndex++;
                treatNextLineObject();
              }
              );
            } else {
              relatedfeaturesIndex++;
              treatNextLineObject();
            }
          }
          //si fin du parcours des objets
          else {
            deferred.resolve('rule end !');
          }
        }

        treatNextLineObject();

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all([promises, deferred.promise]);
      }

      /**
       * Pour chaque objet ponctuel supprimé (Considère l'édition multiple),
       * recherche les objets intersectés des couches précisées
       * en configuration de cette règle
       * et effectue une fusion des objets linéaires intersectés trouvés
       * si les conditions le permettent.
       *
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function mergeIntersectingLines(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();
        if (featureType.typeInfo != 'POINT') {
          console.error(
            'Règle mergeIntersectingLines non adaptée à la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        /**
         * Cherche les objets de la layer décrite par l'objet cfglayerObject qui intersecte l'objet supprimé deletedFeaturePoint;
         * Puis effectue une fusion si les conditions sur les objets intersectés trouvés le permettent.
         * @param {type} cfglayerObject
         * @param {type} deletedFeaturePoint
         * @returns {Array}
         */
        function deleteAndMergeIntersectingLines(
          cfglayerObject,
          deletedFeaturePoint
        ) {
          var editedgeom = deletedFeaturePoint.getGeometry();
          var pointCoordinate = editedgeom.getCoordinates();

          var leftX = pointCoordinate[0] - 0.05;
          var bottomY = pointCoordinate[1] - 0.05;
          var rightX = pointCoordinate[0] + 0.05;
          var topY = pointCoordinate[1] + 0.05;
          var cql_filter =
            'INTERSECTS(geom, POLYGON((' +
            leftX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            bottomY +
            ',' +
            rightX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            topY +
            ',' +
            leftX +
            ' ' +
            bottomY +
            ')))';
          //var cql_filter = "INTERSECTS(geom,Point("+pointCoordinate[0]+' '")";

          var l = getLayerByName(cfglayerObject.layerName, layers);
          if (l == undefined) {
            console.error(
              'Erreur à l\'appel de  ogc.data(),avec la couche:' +
                cfglayerObject.layerName +
                'cql_filter:' +
                cql_filter
            );
            var deferred = $q.defer();
            deferred.resolve('terminé !');
            return deferred.promise;
          }

          //La requete d'intersection de pointCoordinate avec les objets de la couche cfgShareObj.layerName.
          var promise = ogcFactory.getfeatures(
            'GetFeature',
            'WFS',
            '1.0.0',
            l.fti.uid,
            'json',
            map.getView().getProjection().getCode(),
            cql_filter
          );
          //Reponse.
          var callBackSuccess = getCallbackSuccess(
            cfglayerObject,
            l,
            deletedFeaturePoint
          );
          //promise2 sera resolue lorsque la callback aura finie d'être executée
          var promise2 = promise.then(callBackSuccess, function (error) {
            console.error(
              'Erreur à l\'appel de  ogc.data(),avec la couche:' +
                l.name +
                'cql_filter:' +
                cql_filter
            );
          });

          return promise2;
        }


        /**
         * Affichage d'un toastr pour les messages ayant comme
         * chemin d'accés JSON "rulecfg.mergeintersectinglines".
         *
         * @param {*} messageName : Clef identifiant le message
         */
        const mergeToastr = (messageName) => {
          require('toastr').error(
            $filter('translate')('rulecfg.mergeintersectinglines.' + messageName));
        };


        /**
         * Vérifier que les canalisations à fusionner ont le même matériau
         * et le même diamètre.
         *
         * @param {*} prop1 : Propriétés de la canalisation 1
         * @param {*} prop2 : Propriétés de la canalisation 2
         * @param {*} cfglayerObject : Configuration où sont stockés
         *                  le nom des attributs matériau et diamètre
         * @returns VRAI si la fusion est possible, FAUX sinon
         */
        const checkMaterialOfPipes = (prop1, prop2, cfglayerObject) => {
          let ok = true;
          if (prop1[cfglayerObject.materialFieldName] !==
            prop2[cfglayerObject.materialFieldName]) {
            mergeToastr('differentMaterials');
            ok = false;
          }
          if (prop1[cfglayerObject.diameterFieldName] !==
            prop2[cfglayerObject.diameterFieldName]) {
            mergeToastr('differentDiameters');
            ok = false;
          }
          if (ok) {
            return true;
          }
          else {
            mergeToastr('notCompatible');
            return false;
          }
        };


        function getCallbackSuccess(
          cfglayerObject,
          intersectionlayerLine,
          deletedFeaturePoint
        ) {
          return function(result) {
            var deferred = $q.defer();
            var intersectedFeatures = format.readFeatures(result.data);
            //Si  intersection de deux cana uniquement,alors possibilité de fusion
            if (intersectedFeatures != undefined) {
              if (intersectedFeatures.length === 2) {
                var feature1 = intersectedFeatures[0];
                var feature2 = intersectedFeatures[1];
                var prop1 = feature1.getProperties();
                var prop2 = feature2.getProperties();

                //Vérification des types des cana pour accepter la fusion.
                if (checkMaterialOfPipes(prop1,prop2,cfglayerObject)) {
                  //Renvoi une promesse qui sera résolue lorsque
                  //les opérations de fusion seront terminées pour ces lignes.
                  return doMerge(deletedFeaturePoint,intersectedFeatures,
                    cfglayerObject,intersectionlayerLine);
                }
              }
              else {
                if (intersectedFeatures.length < 2) {
                  //-- Isolé ou connecté à un seul tronçon, on n'interdit pas
                  //-- la suppression(en tout cas pas dans cette régle).
                  deferred.resolve('terminé !');
                  return deferred.promise;
                }
              }
            }

            //-- -- ANNULATION DE LA SUPPRESSION

            if (intersectedFeatures != undefined
              && intersectedFeatures.length > 2) {
            //-- Message du cas où on essaye de supprimer un point connecté
            //-- à plus de 2 tronçons
              mergeToastr('moreThan2Lines');
            }

            if (editdescription.editedfeature.getId() ==
              deletedFeaturePoint.getId()
            ) {
              if (editdescription.relatedfeatures.length > 0) {
                // take the first relatedFeature and put it in editedFeature
                editdescription.editedfeature =
                  editdescription.relatedfeatures[0];
                editdescription.relatedfeatures.splice(0, 1);
              } else {
                editdescription.editedfeature = null;
              }
            } else {
              for(let index in editdescription.relatedfeatures){
                if (editdescription.relatedfeatures[index].getId() ==
                  deletedFeaturePoint.getId()) {
                  editdescription.relatedfeatures.splice(index,1);
                }
              }
            }


            /* else {
                            alert ('le noeud est raccordé à plus de deux tronçons')
                        }  */
            //CallBack fait aussi un return dans la condition interne, donc pas de else, sinon, si la deuxième condition n'est pas vérifié, la promesse résolue ne serait pas envoyée.
            //Si pas de fusion, cette callback renvoi une promesse resolue.
            deferred.resolve('terminé !');
            return deferred.promise;
          };
        }

        /**
         * tempAllMergedObjects: Object contenant les tableaux par layers des feature line fusionnées pour acceder plus facilement à ces objetss lors de vérifications.
         * @type Array
         */
        function getMergedObjectsArrayByLayerName(editdescription, layerName) {
          if(!editdescription.tempAllMergedObjects) {
            editdescription.tempAllMergedObjects = {};
          }
          if (editdescription.tempAllMergedObjects[layerName] == undefined) {
            editdescription.tempAllMergedObjects[layerName] = [];
          }
          return editdescription.tempAllMergedObjects[layerName];
        }

        //overlapPoints est un tableau d'objets representant chacun une coordonnée commune entre les deux lines.
        //Chaque objet de overlapPoints contient les valeurs d'index et subindex de la coordonnée commune.
        function getOverlapPoints(currentLine, newMergeLine) {
          var overlapPoints = [];

          var coordinates1 = currentLine.getCoordinates();
          var coordinates2 = newMergeLine.getCoordinates();

          //Pour chaque point de la première line
          for (
            var lineStringIndex1 = 0;
            lineStringIndex1 < coordinates1.length;
            lineStringIndex1++
          ) {
            var subcoordinates1 = coordinates1[lineStringIndex1];

            //Pour chaque point de la deuxième line, vérification si currentpoint1 et currentpoint2 ont les memes valeurs
            for (
              var lineStringIndex2 = 0;
              lineStringIndex2 < coordinates2.length;
              lineStringIndex2++
            ) {
              var subcoordinates2 = coordinates2[lineStringIndex2];

              if (
                Math.round(subcoordinates1[0] * 10) ==
                      Math.round(subcoordinates2[0] * 10) &&
                    Math.round(subcoordinates1[1] * 10) ==
                      Math.round(subcoordinates2[1] * 10)
              ) {
                //Stockage des index et subindex des lines où les points se chevauchent
                overlapPoints.push(subcoordinates1);
              }
            }
          }
          return overlapPoints;
        }

        function getOverlapFeatureObjectIfAny(editdescription, newMergeFeature, layerName) {
          var overlapObjects = [];
          var mergedObjects = getMergedObjectsArrayByLayerName(editdescription ,layerName);
          for (var i = 0; i < mergedObjects.length; i++) {
            var currentLine = mergedObjects[i].feature.getGeometry();
            if (
              getOverlapPoints(currentLine, newMergeFeature.getGeometry())
                .length >= 2
            ) {
              overlapObjects.push(mergedObjects[i]);
            }
          }
          return overlapObjects;
        }

        /**
         * Vérifie que l'objet fusionné passé en paramètre à cette fonction ne chevauche pas un autre objet préalablement fusionné,
         * si c'est le cas il faut à nouveau fusionner cet objet et ceux qui le chevauche.
         * @param {type} featureToUpdate le ol.feature non encore sauvé, issu d'une premiere fusion mais qui se chevauche avec une autre feature precedement fusionnée present dans overlapFeatureObject.
         * @param {type} overlapFeatureObject l' autre feature precedement fusionné
         * @param {type} cfglayerObject l'objet configuration lié à la layer de l'objet fusionné.
         * @returns {undefined}
         */
        var mergeCanceledByUser = false;
        var allpopupInstances = [];

        function checkSecondaryMerges(featureToUpdate, cfglayerObject) {
          var deferred = $q.defer();
          if (mergeCanceledByUser) {
            deferred.resolve('CanceledByUser!');
            return deferred.promise;
          }

          var layer = getLayerByName(cfglayerObject.layerName, layers);

          var overlapFeatureObjects = getOverlapFeatureObjectIfAny(
            editdescription,
            featureToUpdate,
            cfglayerObject.layerName
          );

          //Si 'featureToUpdate' resultat d'une première fusion dans doMerge(),
          // ne superpose aucun autre objet precedement fusionné, on l'enregistre simplement.
          if (overlapFeatureObjects.length == 0) {
            //Visualisation
            gclayers
              .getDrawLayer()
              .getSource()
              .addFeature(featureToUpdate);
            //Enregistrement pour sauvegarde
            var objectToCreate = {
              shareObject: '',
              editType: EditTypesFactory.editTypes.add.name,
              feature: featureToUpdate,
              fti: layer.fti,
            };
            editdescription.relatedfeatures.push(objectToCreate);

            //Enregistrement dans un tableau secondaire pour acceder plus facilement aux objets déjà fusionnés lors des prochaines fusions.
            getMergedObjectsArrayByLayerName(editdescription, layer.fti.name).push(
              objectToCreate
            );

            deferred.resolve('terminé !');
          }
          //Sinon, fusion et enregistrement du resultat de la fusion des deux objets.
          else {
            var geoms = [];
            var f1geoJson = format.writeGeometryObject(
              featureToUpdate.getGeometry()
            );
            geoms.push(f1geoJson);
            for (var i = 0; i < overlapFeatureObjects.length; i++) {
              var overlapFeatureObject = overlapFeatureObjects[i];
              var f2geoJson = format.writeGeometryObject(
                overlapFeatureObject.feature.getGeometry()
              );
              geoms.push(f2geoJson);
            }

            //ATTENTION, SI UN DEUXIEME APPEL DE LA METHODE ARRIVE ICI ET QUE L'APPEL PRECEDENT N'EST PAS ENCORE TRAITE DANS LE HANDLER DE RESULTATS,
            // LE TABLEAU DE MERGED LINES NE SERA PAS à JOUR ET IL N'Y AURA PAS DE FUSION COMPLEMENTAIRE;
            GeometryFactory.union(geoms).then(
              function(result) {
                var mergedLine = format.readGeometry(result.data);

                //Enregistrement dans un tableau secondaire pour acceder plus facilement aux objets déjà fusionnés lors des prochaines fusions.
                //this temporary version will be replaced after
                let tempFeature = overlapFeatureObjects[0].feature;
                tempFeature.setGeometry(mergedLine);
                let objectToCreate = {
                  shareObject: '',
                  editType: EditTypesFactory.editTypes.add.name,
                  feature: tempFeature,
                  fti: layer.fti,
                };
                getMergedObjectsArrayByLayerName(editdescription, layer.fti.name).push(objectToCreate);

                var scope = $rootScope.$new(true);

                scope.editdescription = editdescription;
                scope.selectedFeatures = [];
                //On retire des tableaux les precedents objets fusionnés qui chevauchent le nouvel objet fusioné
                for (var i = 0; i < overlapFeatureObjects.length; i++) {
                  var overlapFeatureObject = overlapFeatureObjects[i];
                  var mergedObjects = getMergedObjectsArrayByLayerName(
                    editdescription, overlapFeatureObject.fti.name);
                  var index1 = mergedObjects.indexOf(overlapFeatureObject);
                  if (index1 >= 0) {
                    mergedObjects.splice(index1, 1);
                  }
                  var index2 = editdescription.relatedfeatures.indexOf(
                    overlapFeatureObject
                  );
                  if (index2 >= 0) {
                    editdescription.relatedfeatures.splice(index2, 1);
                  }

                  // //Retirer de la visualisation
                  // gclayers
                  //   .getDrawLayer()
                  //   .getSource()
                  //   .removeFeature(overlapFeatureObject.feature);

                  //Tratement pour le choix par l'utilisateur du feature à conserver
                  overlapFeatureObject.feature.currentProperties = overlapFeatureObject.feature.getProperties();
                  scope.selectedFeatures.push(overlapFeatureObject.feature);
                }

                /**
                 * Définition de la méthode qui poursuivra le traitement après le choix par défaut ou par l'utilisateur du feature dont les attributs sont à conserver.
                 * @param {type} featureChoosen
                 * @returns {undefined}
                 */
                scope.confirm = function(featureChoosen) {
                  if (featureChoosen == undefined) {
                    return;
                  }
                  if (scope.p) {
                    scope.p.destroy();
                  }

                  var newFeatureToUpdate = featureChoosen;
                  newFeatureToUpdate.setGeometry(mergedLine);

                  //Affectation de la valeur à la propriété longueur
                  if (cfglayerObject.lengthFieldName) {
                    //Recalcul de la longueur
                    var lineLength = 0;
                    if(mergedLine.getLineStrings){
                      //multiple lines (multiLineString)
                      angular.forEach(mergedLine.getLineStrings(), function(
                        value,
                        key
                      ) {
                        lineLength += getPreciseLineLength(value, map);
                      });
                    } else {
                      //single line (lineString)
                      lineLength = getPreciseLineLength(mergedLine, map);
                    }
                    var lengthFieldName = cfglayerObject.lengthFieldName;
                    var lengthProperties = {};
                    lengthProperties[lengthFieldName] =
                      Math.round(lineLength * 100) / 100;
                    newFeatureToUpdate.setProperties(lengthProperties);
                  }

                  var updatedObject = {
                    shareObject: '',
                    editType: EditTypesFactory.editTypes.add.name,
                    feature: newFeatureToUpdate,
                    fti: layer.fti,
                  };
                  //Enregistrement pour sauvegarde
                  editdescription.relatedfeatures.push(updatedObject);

                  //Enregistrement dans un tableau secondaire pour acceder plus facilement aux objets déjà fusionnés lors des prochaines fusions.
                  let tempFeatureIndex = mergedObjects.indexOf(tempFeature);
                  mergedObjects.splice(tempFeatureIndex, 1);
                  mergedObjects.push(updatedObject);

                  //Visualisation
                  gclayers
                    .getDrawLayer()
                    .getSource()
                    .addFeature(newFeatureToUpdate);

                  deferred.resolve('terminé !');
                };

                //Si FastMode est activé alors ON MET À JOUR LE PREMIER FEATURE AVEC LA NOUVELLE GEOMETRY ET ON SUPPRIME LE DEUXIÈME FEATURE
                if (editdescription.fastMode) {
                  scope.confirm(featureToUpdate);
                }
                //Sinon, l'utilisateur choisit dans une popup lequel est à conserver (celui pour lequel on garde les attributs)
                else {
                  scope.highLightFeature = function(f) {
                    gclayers.addhighLightFeature(f);
                    //ADD Timeout sur scope.removehighLightFeature si probleme pour retirer le highLight
                  };
                  scope.removehighLightFeature = function(f) {
                    gclayers.removehighLightFeatures(f);
                    //gclayers.clearhighLightFeatures();
                  };

                  //Choix de la feature dont les attributs sont à conserver
                  featureToUpdate.currentProperties = featureToUpdate.getProperties();
                  scope.selectedFeatures.push(featureToUpdate);
                  angular.forEach(scope.selectedFeatures, (feature) => {
                    if (!feature.getId()) {
                      feature.setId('Elément temporaire');
                    }
                  });
                  scope.intersectionlayer = layer;
                  scope.result = {};
                  scope.p = gcPopup.open({
                    template:
                      'js/XG/modules/edit/views/featuresChoice_popup.html',
                    scope: scope,
                    title: $filter('translate')(
                      'rules.mergeLines.scdMergeChoicePopupTitle'
                    ),
                    showClose: false,
                  });
                  allpopupInstances.push(scope.p);

                  scope.closeThisDialog = function(res) {
                    mergeCanceledByUser = true;
                    //Fermeture des popup
                    angular.forEach(allpopupInstances, function(value, key) {
                      value.destroy();
                    });

                    deferred.resolve('CanceledByUser!');
                  };
                }
              },
              function(error) {
                console.error(
                  'Erreur à l\'appel de   GeometryFactory.union,avec la couche:' +
                    cfglayerObject.layerName
                );
                deferred.resolve('terminé !');
              }
            );
          }
          return deferred.promise;
        }

        /**
         * Effectue la fusion des deux objets linéaires intersectés situé dans le tableau 'intersectedFeatures' passé en paramètre.
         * @param {type} deletedFeaturePoint
         * @param {type} intersectedFeatures
         * @param {type} cfglayerObject
         * @param {type} intersectionlayerLine
         * @returns {$q@call;defer.promise}
         */
        function doMerge(
          deletedFeaturePoint,
          intersectedFeatures,
          cfglayerObject,
          intersectionlayerLine
        ) {
          var deferred = $q.defer();

          if (mergeCanceledByUser) {
            deferred.resolve('CanceledByUser!');
            return deferred.promise;
          }

          var feature1 = intersectedFeatures[0];
          var feature2 = intersectedFeatures[1];

          // //change the 'multiLineString' in a 'LineString' because arcgis is throwing an error with 'multiLineString'
          // deletedFeaturePoint



          var geoms = [];
          var f1geoJson = format.writeGeometryObject(feature1.getGeometry());
          var f2geoJson = format.writeGeometryObject(feature2.getGeometry());
          geoms.push(f1geoJson);
          geoms.push(f2geoJson);

          //Appelle du service java de fusion
          GeometryFactory.union(geoms).then(
            //Fonction appellée lors du resultat de la fusion
            function(result) {
              //Geometrie resultat de la fusion
              var mergedLine = format.readGeometry(result.data);

              var scope = $rootScope.$new(true);

              /**
               * Définition de la méthode qui poursuivra le traitement après le choix par défaut ou par l'utilisateur du feature dont les attributs sont à conserver.
               * @param {type} featureChoosen
               * @returns {undefined}
               */
              scope.confirm = function(featureChoosen) {
                if (featureChoosen == undefined) {
                  return;
                }
                if (scope.p) {
                  scope.p.destroy();
                }

                var featureToCreate = featureChoosen.clone();

                featureToCreate.setGeometry(mergedLine);
                //Reprise des attributs dans la nouvelle ligne fusionnée: attributs dans feature deja present puisque seule la geometry a été remplacée dans le feature déja existant.

                //Affectation de la valeur à la propriété longueur
                if (cfglayerObject.lengthFieldName) {
                  //Recalcul de la longueur
                  var lineLength = 0;
                  if (mergedLine.getLineStrings) {
                    //multiple lines
                    angular.forEach(
                      mergedLine.getLineStrings(),
                      function (value, key) {
                        lineLength += getPreciseLineLength(value, map);
                      }
                    );
                  } else {
                    //multiple lines
                    lineLength = getPreciseLineLength(
                      mergedLine,
                      map
                    );
                  }
                  var lengthFieldName = cfglayerObject.lengthFieldName;
                  var lengthProperties = {};
                  lengthProperties[lengthFieldName] =
                    Math.round(lineLength * 100) / 100;
                  featureToCreate.setProperties(lengthProperties);
                }

                //Enregistrement des features
                //Objets à supprimer
                gclayers
                  .getDrawLayer()
                  .getSource()
                  .addFeature(feature1);
                var objectToDelete1 = {
                  shareObject: '',
                  editType: EditTypesFactory.editTypes.delete.name,
                  feature: feature1,
                  fti: intersectionlayerLine.fti,
                };
                recordObjectIfNotExists(editdescription, objectToDelete1);
                gclayers
                  .getDrawLayer()
                  .getSource()
                  .addFeature(feature2);
                var objectToDelete2 = {
                  shareObject: '',
                  editType: EditTypesFactory.editTypes.delete.name,
                  feature: feature2,
                  fti: intersectionlayerLine.fti,
                };
                recordObjectIfNotExists(editdescription, objectToDelete2);

                //Historisation eventuelle des objets lineaires modifié et supprimé:
                if (cfglayerObject.isHistoricize) {
                  //Enregistrement de la copie pour historisation
                  var objectToHistoricize1 = {
                    shareObject: '',
                    editType: EditTypesFactory.editTypes.tohistorize.name,
                    feature: feature1,
                    fti: intersectionlayerLine.fti,
                  };
                  recordObjectIfNotExists(
                    editdescription,
                    objectToHistoricize1
                  );
                  var objectToHistoricize2 = {
                    shareObject: '',
                    editType: EditTypesFactory.editTypes.tohistorize.name,
                    feature: feature2,
                    fti: intersectionlayerLine.fti,
                  };
                  recordObjectIfNotExists(
                    editdescription,
                    objectToHistoricize2
                  );
                }
                //Objet Créé

                // finish merging
                checkSecondaryMerges(featureToCreate, cfglayerObject).then(
                  function(result) {
                    deferred.resolve('terminé !');
                  },
                  function(error) {
                    console.error(
                      'Erreur à l\'appel de   checkSecondaryMerges, avec la couche:' +
                        intersectionlayerLine.fti.name
                    );
                    deferred.resolve('terminé !');
                  }
                );
              };

              //Si FastMode est activé alors ON MET À JOUR LE PREMIER FEATURE AVEC LA NOUVELLE GEOMETRY ET ON SUPPRIME LE DEUXIÈME FEATURE
              if (editdescription.fastMode) {
                scope.confirm(feature1);
              }
              //Sinon, l'utilisateur choisit dans une popup lequel est à conserver (celui pour lequel on garde les attributs)
              else {
                scope.highLightFeature = function(f) {
                  gclayers.addhighLightFeature(f);
                  //ADD Timeout sur scope.removehighLightFeature si probleme pour retirer le highLight
                };
                scope.removehighLightFeature = function(f) {
                  gclayers.removehighLightFeatures(f);
                  //gclayers.clearhighLightFeatures();
                };
                scope.editdescription = editdescription;
                feature1.currentProperties = feature1.getProperties();
                feature2.currentProperties = feature2.getProperties();
                scope.selectedFeatures = [feature1, feature2];
                scope.intersectionlayer = intersectionlayerLine;
                scope.result = {};
                scope.p = gcPopup.open({
                  template:
                    'js/XG/modules/edit/views/featuresChoice_popup.html',
                  scope: scope,
                  title: $filter('translate')(
                    'rules.mergeLines.choicePopupTitle'
                  ),
                  showClose: false,
                });
                allpopupInstances.push(scope.p);

                scope.closeThisDialog = function(res) {
                  mergeCanceledByUser = true;
                  //Fermeture des popup
                  angular.forEach(allpopupInstances, function(value, key) {
                    value.destroy();
                  });
                  deferred.resolve('CanceledByUser!');
                };
              }
            },
            function(error) {
              console.error(
                'Erreur à l\'appel de   GeometryFactory.union,avec la couche:' +
                  intersectionlayerLine.fti.name
              );
              deferred.resolve('terminé !');
            }
          );
          return deferred.promise;
        } //fin function utilitaire doMerge

        var layersObjects = ruleConf.parameters['layersObjects'];
        var promises = [];

        //Recherche des intersections avec l'objet principal édité 'editdescription.editedfeature' (Premier objet supprimé)
        angular.forEach(layersObjects, function(cfglayerObject, key) {

          // KIS-3502: si le composant n’a pas été trouvé, la suite du traitement ne doit pas avoir lieu
          if (layers.some(layer => layer.get('name') === cfglayerObject.layerName)) {
            let anOtherpromises = deleteAndMergeIntersectingLines(
                cfglayerObject,
                editdescription.editedfeature
            );
            promises = promises.concat(anOtherpromises);
          }
        });

        //Index de l'objet secondaire édité
        var relatedfeaturesIndex = 0;
        //Index de la layer d'intersection ligne courante
        var layersObjectsIndex = 0;
        //on mémorise le nombre initial d'objets secondaires car on va au cours de l'execution possiblement alimenter ce tableau.
        //Si la règle supprimait des objets ponctuels et que l'on voulait prendre en compte ces nouveaux objets,
        //on aurait comparé 'relatedfeaturesIndex' à 'editdescription.relatedfeatures.length' dans treatNextDeletedObject().
        var nbRelatedFeature = editdescription.relatedfeatures.length;

        /* if ( nbRelatedFeature > 1 ) {
                    console.info("more than one related feature");
                    deferred.resolve("terminé !");
                    return deferred.promise;
                } else {
                    console.info("less than one related feature");*/
        function treatNextDeletedObject() {
          if (layersObjectsIndex < layersObjects.length) {
            if (relatedfeaturesIndex < nbRelatedFeature) {
              var related =
                editdescription.relatedfeatures[relatedfeaturesIndex];
              if (
                related.editType == editdescription.editType &&
                related.fti.name == editdescription.fti.name
              ) {
                let anOtherpromises = deleteAndMergeIntersectingLines(
                  layersObjects[layersObjectsIndex],
                  related.feature
                );
                anOtherpromises.then(
                  function(res) {
                    relatedfeaturesIndex++;
                    treatNextDeletedObject();
                  },
                  function(error) {
                    relatedfeaturesIndex++;
                    treatNextDeletedObject();
                  }
                );
              } else {
                relatedfeaturesIndex++;
                treatNextDeletedObject();
              }
            }
            //si fin du parcours des objets secondaires ponctuels à intersecter avec la layer précisé par layersObjects[layersObjectsIndex]
            else {
              //On passe à l'objet de config (précisant une layer d'intersection) suivant.
              layersObjectsIndex++;
              relatedfeaturesIndex = 0;
              treatNextDeletedObject();
            }
          }
          //si fin du parcours des objets
          else {
            deferred.resolve('rule end !');
          }
        }

        treatNextDeletedObject();

        promises.push(deferred.promise);
        return $q.all(promises);
      }

      /**
       * Affecte la bonne valeur du champ orientation de l'objet ponctuel édité
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise|undefined}
       */
      function rotateSymbolPoint(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        if (ruleConf.hasOwnProperty('parameters') && editdescription.hasOwnProperty('fti')) {
          let error = $filter('translate')('rulecfg.rotatesymbolpoint.noAttributeCfgError');
          if (ruleConf.parameters['layerInfos'][editdescription.fti.name] === undefined) {
            error = $filter('translate')('rulecfg.rotatesymbolpoint.noFtiError');
            console.error(error);
            require('toastr').error(error);
            return;
          }

          if (ruleConf.parameters['layerInfos'][editdescription.fti.name].hasOwnProperty('rotateFieldName')) {
            const rotationAttribute = ruleConf.parameters['layerInfos'][editdescription.fti.name].rotateFieldName;
            const attributeDefExists = rotationAttribute !== null && rotationAttribute !== undefined && rotationAttribute.length > 0;
            const attributeExists = editdescription.fti.attributes.some(attr => attr.name === rotationAttribute);
            if (!attributeDefExists) {
              console.error(error);
              require('toastr').error(error);
              return;
            }
            if (!attributeExists) {
              error = $filter('translate')('rulecfg.rotatesymbolpoint.noFtiAttributeError');
              console.error(error);
              require('toastr').error(error);
              return;
            }
          } else {
            console.error(error);
            require('toastr').error(error);
            return;
          }
        }

        /**
         * Récupère un objet partagé ajouté à la session d'édition par une règlé executé juste avant,
         * et dont le nom est celui précisé dans l'objet configuration 'cfgShareObjects'.
         * @param {type} cfgShareObjects
         * @returns {undefined|type.shareObjects}
         */
        function getIntersectedShareObject(cfgShareObjects) {
          for (var j = 0; j < cfgShareObjects.length; j++) {
            for (var k = 0; k < editdescription.shareObjects.length; k++) {
              if (
                editdescription.shareObjects[k].shareObject ==
                  cfgShareObjects[j].name &&
                editdescription.shareObjects[k].fti.name ==
                  cfgShareObjects[j].layerName
              ) {
                return editdescription.shareObjects[k];
              }
            }
          }
          return undefined;
        }

        //Récupération de l'eventuel objet linéaire intersecté par le point édité.
        var cfgShareObjects = ruleConf.parameters['shareObjects'];
        var intersectedShareObject = getIntersectedShareObject(cfgShareObjects);
        //Si objet linéaire intersecté trouvé
        if (intersectedShareObject != undefined) {
          var rotateFieldName =
            ruleConf.parameters['layerInfos'][editdescription.fti.name][
              'rotateFieldName'
            ];
          //Récupération depuis la config d'une valeur de rotation à rajouter à celle qui sera calculée
          var rotationOffset = ruleConf.parameters['offset'];
          // Affectation de la valeur de l'angle dans l'attribut rotation de l'objet ponctuel édité
          // à partir du segment de la ligne sur lequel est posé le ponctuel,
          var intersectedFeatureLine = intersectedShareObject.feature;
          setPointRotationFromLineObject(
            editdescription.editedfeature,
            rotateFieldName,
            intersectedFeatureLine,
            rotationOffset
          );
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Affecte les valeurs attributaires representant les coordonnées X et Y de l'objet PRINCIPAL ponctuel édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setXYAttributes(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        //Si type POINT
        if (featureType.typeInfo == 'POINT') {
          var xFieldName = ruleConf.parameters['xFieldName'];
          var yFieldName = ruleConf.parameters['yFieldName'];
          var point = editdescription.editedfeature.getGeometry();
          var coords = point.getCoordinates();

          //Affectation des  valeurs
          var xProperties = {};
          xProperties[xFieldName] = coords[0];
          editdescription.editedfeature.setProperties(xProperties);
          var yProperties = {};
          yProperties[yFieldName] = coords[1];
          editdescription.editedfeature.setProperties(yProperties);

          if (editdescription.relatedfeatures.length > 0) {
            editdescription.relatedfeatures.map(function(feat) {
              if (
                feat.feature.getId() != undefined &&
                feat.feature.getId().split('.')[0] === editdescription.fti.name
              ) {
                feat.feature.setProperties(yProperties);
              }
            });
          }
        }
        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Affecte la valeur attributaire representant la longueur de l'objet PRINCIPAL linéaire édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setLengthAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        var lengthFieldName = ruleConf.parameters['lengthFieldName'];
        var line = editdescription.editedfeature.getGeometry();
        var isSimpleLine = line instanceof ol.geom.LineString;
        var isMultiLine = line instanceof ol.geom.MultiLineString;
        var lineLength = 0;

        var promise = getSimpleLineLength(line, map, featureType);
        promise.then(function(res) {
          lineLength += res.data[0].perimeter;
          lineLength = Math.round(lineLength * 1000) / 1000;
          var lengthProperties = {};
          lengthProperties[lengthFieldName] = lineLength;
          editdescription.editedfeature.setProperties(lengthProperties);
          if (editdescription.relatedfeatures.length > 0) {
            editdescription.relatedfeatures.map(function(feat) {
              // if ( feat.feature.getId() != undefined && feat.feature.getId().split('.')[0] === editdescription.fti.name )
              //  feat.feature.setProperties(lengthProperties);
              var lineRelated = feat.feature.getGeometry();
              var rules = feat.fti.rules;

              var lineLengthR = 0;
              for (var i = 0; i < rules.length; i++) {
                if (rules[i].name == 'SetLengthAttribute') {
                  var promise2 = getSimpleLineLength(
                    lineRelated,
                    map,
                    featureType
                  );
                  promise2.then(function(res2) {
                    lineLengthR += res2.data[0].perimeter;
                    lineLengthR = Math.round(lineLengthR * 1000) / 1000;
                    var lengthPropertiesR = {};
                    lengthPropertiesR[lengthFieldName] = lineLengthR;
                    feat.feature.setProperties(lengthPropertiesR);
                  });
                }
              }
            });
          }
        });

        //Si type LINE
        if (featureType.typeInfo == 'LINE') {
          //calculateLength();
          deferred.resolve('terminé !');
        }

        //Si type POLYGON  //TODO
        //                else  if (featureType.typeInfo == 'POLYGON'){
        //                    var perimeterFieldName = ruleConf.parameters['perimeterFieldName'];
        //                     var polygon = editdescription.feature.getGeometry();
        //                     var perimeter = 0;
        //                     //Affectation de la valeur
        //                    var perimeterProperties = {};
        //                    perimeterProperties[perimeterFieldName] = perimeter;
        //                    editdescription.feature.setProperties(perimeterProperties);
        //                }

        return $q.all([deferred.promise, promise]);
      }

      /**
       * format length output
       * @param {ol.geom.Polygon} polygon
       * @return {string}
       */
      function formatArea(area) {
        var output;
        if (area > 10000) {
          output = Math.round((area / 1000000) * 100) / 100 + ' ' + 'km²';
        } else {
          output = Math.round(area * 100) / 100 + ' ' + 'm²';
        }
        return output;
      }

      /**
       *  Affecte la valeur attributaire representant l'aire de l'objet PRINCIPAL polygonal édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setAreaAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        var promises = [deferred.promise];
        //Si type POLYGON
        if (featureType.typeInfo == 'POLYGON') {
          var areaFieldName = ruleConf.parameters['areaFieldName'];
          var polygon = editdescription.editedfeature.getGeometry();
          var promise = getSimplePolygonArea(polygon, map, featureType);
          promise.then(function(res) {
            if (res.data[0].area) {
              //var area = res.data[0].area;
              var area = Math.round(res.data[0].area * 100) / 100;
              //Affectation de la valeur
              var areaProperties = {};
              areaProperties[areaFieldName] = area;
              //areaProperties[areaFieldName] = formatArea(area);
              editdescription.editedfeature.setProperties(areaProperties);

              if (editdescription.relatedfeatures.length > 0) {
                editdescription.relatedfeatures.map(function(feat) {
                  // if ( feat.feature.getId() != undefined && feat.feature.getId().split('.')[0] === editdescription.fti.name )
                  // feat.feature.setProperties(areaProperties);
                  var polyRelated = feat.feature.getGeometry();
                  var rules = feat.fti.rules;
                  for (var i = 0; i < rules.length; i++) {
                    if (rules[i].name == 'SetAreaAttribute') {
                      var promise2 = getSimplePolygonArea(
                        polyRelated,
                        map,
                        featureType
                      );
                      promise2.then(function(res2) {
                        //var areaR = formatArea(res2.data[0].area);
                        //var areaR = res2.data[0].area;
                        var areaR = Math.round(res2.data[0].area * 100) / 100;
                        var areaProperties = {};
                        areaPropertiesR[areaFieldName] = areaR;
                        //areaPropertiesR[areaFieldName] = areaR;
                        feat.feature.setProperties(areaPropertiesR);
                      });
                    }
                  }
                });
              }
            }
          });
          promises.push(promise);
        }
        deferred.resolve('terminé !');
        return $q.all(promises);
      }

      /**
       * Affecte la valeur attributaire representant la longueur de l'objet PRINCIPAL linéaire édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setUserAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();

        var userFieldName = ruleConf.parameters['userFieldName'];
        var UserProperties = {};
        UserProperties[userFieldName] = $rootScope.xgos.user.login;
        editdescription.editedfeature.setProperties(UserProperties);
        if (editdescription.relatedfeatures.length > 0) {
          editdescription.relatedfeatures.map(function(feat) {
            if (
              feat.feature.getId() != undefined &&
              feat.feature.getId().split('.')[0] === editdescription.fti.name
            ) {
              feat.feature.setProperties(UserProperties);
            }
          });
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Affecte la valeur attributaire representant la longueur de l'objet PRINCIPAL linéaire édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setW3wAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        var promises = [deferred.promise];

        if (editdescription.editedfeature) {
          var w3wFieldName = ruleConf.parameters['w3wFieldName'];
          var coords = [];
          if (
            editdescription.editedfeature.getGeometry().getType() !== 'Point'
          ) {
            var geom = editdescription.editedfeature.getGeometry();
            coords = new ol.extent.getCenter(geom.getExtent());
          } else {
            coords = editdescription.editedfeature
              .getGeometry()
              .getCoordinates();
          }
          var coords_4326 = ol.proj.transform(
            coords,
            map
              .getView()
              .getProjection()
              .getCode(),
            'EPSG:4326'
          );
          //console.log(coords_4326);
          var promise = What3WordFactory.reverse(coords_4326);
          promise.then(function(res) {
            var words = res.data.words;
            var W3WProperties = {};
            W3WProperties[w3wFieldName] = words;
            editdescription.editedfeature.setProperties(W3WProperties);
            if (editdescription.relatedfeatures.length > 0) {
              editdescription.relatedfeatures.map(function(feat) {
                if (
                  feat.feature.getId() != undefined &&
                  feat.feature.getId().split('.')[0] ===
                    editdescription.fti.name
                ) {
                  feat.feature.setProperties(W3WProperties);
                }
              });
            }
          });
          promises.push(promise);
        }
        deferred.resolve('terminé !');
        return $q.all(promises);
      }

      /**
       * Affecte la valeur attributaire representant la longueur de l'objet PRINCIPAL linéaire édité.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function setDateAttribute(editdescription, ruleConf, featureType, map) {
        var deferred = $q.defer();
        var attributeType;
        var dateFieldName = ruleConf.parameters['dateFieldName'];
        for (var i = 0; i < featureType.attributes.length; i++) {
          if (featureType.attributes[i].name === dateFieldName) {
            attributeType = featureType.attributes[i].type;
            break;
          }
        }

        var dateProperties = {};
        var d = new Date();
        var dateFormat;
        switch (attributeType) {
          case 'java.lang.Date':
          case 'java.util.Date':
            dateFormat = 'yyyy-MM-ddT00:00:00.000Z';
            break;
          default:
            dateFormat = 'yyyy-MM-ddTHH:mm:ss.sssZ';
            break;
        }
        var n = $filter('date')(d, dateFormat);
        dateProperties[dateFieldName] = n;
        editdescription.editedfeature.setProperties(dateProperties);

        if (editdescription.relatedfeatures.length > 0) {
          editdescription.relatedfeatures.map(function(feat) {
            if (
              feat.feature.getId() != undefined &&
              feat.feature.getId().split('.')[0] === editdescription.fti.name
            ) {
              feat.feature.setProperties(dateProperties);
            }
          });
        }

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Merge current object with saved object
       * @param editdescription
       * @param ruleConf
       * @param featureType
       * @param map
       * @returns {*}
       */
      function copyPasteAttribute(editdescription, ruleConf, featureType, map) {
        let deferred = $q.defer();
        if (ruleConf.name === 'CopyPasteAttribute') {
          const currentObj = editdescription.editedfeature.getProperties();
          const uId = editdescription.fti.uid;
          let savedObj = CopyPasteAttributeFactory.GetStoredObject(uId);

          for (let prop in savedObj) {
            if (!currentObj.hasOwnProperty(prop)) {
              currentObj[prop] = savedObj[prop];
            }
          }
          deferred.resolve(currentObj);
          return deferred.promise;
        }
      }

      function replaceDeletedPoints(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();

        deferred.resolve('terminé !');
        return deferred.promise;
      }

      /**
       * Supprime les objets partagés portés par les objets linéaire édités.
       * Cette règle utilise le résultat de la règle 'getAllFeaturesIntersectingLine' qui récupère les objets portés par les lignes éditées.
       * @param {type} editdescription
       * @param {type} ruleConf
       * @param {type} featureType
       * @param {type} map
       * @returns {$q@call;defer.promise}
       */
      function deposeObjectsOnLine(
        editdescription,
        ruleConf,
        featureType,
        map
      ) {
        var deferred = $q.defer();
        if (featureType.typeInfo != 'LINE') {
          console.error(
            'Règle deleteObjectsOnLine non adaptée à la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }
        var cfgUpShareObjects = ruleConf.parameters['shareObjects'];

        if (
          cfgUpShareObjects == undefined &&
          !Array.isArray(cfgUpShareObjects)
        ) {
          console.error(
            'Règle deleteObjectsOnLine mal ou pas configurée pour la couche ' +
              featureType.name
          );
          deferred.resolve('terminé !');
          return deferred.promise;
        }

        var layers = gclayers.getOperationalLayer();

        function isThemeLayer(layer) {
          return layer.fti.theme == editdescription.theme;
        }

        function getThemeLayerUids() {
          var layerUids = '';
          var themeLayers = layers.filter(isThemeLayer);
          angular.forEach(themeLayers, function(l, key) {
            layerUids += l.fti.uid + ',';
          });
          layerUids.substring(0, layerUids.length - 1);
          return layerUids;
        }

        function getLayerUidsFromRule() {
          var layerUids = cfgUpShareObjects
            .map(function(x) {
              var fti = gclayers.getOperationalLayerByName(x.layerName);
              if (fti) {
                return fti.uid;
              }
            })
            .filter(function(x) {
              if (x) {
                return x;
              }
            });
          return layerUids.join(',');
        }

        //Récupère les features partagés stockés dans 'currenteditdescription' et dont les noms d'objets partagés sont dans l'objet de configuration 'cfgUpShareObjects'
        function getIntersectedShareObjects(currenteditdescription) {
          var shareObjects = [];

          if (currenteditdescription.shareObjects != undefined) {
            //Pour chaque layer d'intersection de la configuration
            for (var j = 0; j < cfgUpShareObjects.length; j++) {
              //Recherche dans les features récupérés les objets partagés du même nom.
              for (
                var k = 0;
                k < currenteditdescription.shareObjects.length;
                k++
              ) {
                if (
                  currenteditdescription.shareObjects[k].shareObject ==
                    cfgUpShareObjects[j].name &&
                  currenteditdescription.shareObjects[k].fti.name ==
                    cfgUpShareObjects[j].layerName
                ) {
                  shareObjects.push({
                    shareObject: currenteditdescription.shareObjects[k],
                    isHistoricize: cfgUpShareObjects[j].isHistoricize,
                  });
                }
              }
            }
          }
          return shareObjects;
        }

        function getFeatureCountInEditdescription(featureID) {
          var count = 0;
          angular.forEach(editdescription.shareObjects, function(
            shareObj,
            key
          ) {
            if (shareObj.feature.getId() == featureID) {
              count++;
            }
          });
          angular.forEach(editdescription.relatedfeatures, function(
            related,
            key
          ) {
            angular.forEach(related.shareObjects, function(shareObj, key) {
              if (shareObj.feature.getId() == featureID) {
                count++;
              }
            });
          });
          return count;
        }

        function isObjectRecordedInRelatedfeatures(featureID) {
          var isPresent = false;
          for (var i = 0; i < editdescription.relatedfeatures.length; i++) {
            if (
              featureID == editdescription.relatedfeatures[i].feature.getId()
            ) {
              isPresent = true;
              break;
            }
          }
          return isPresent;
        }

        function treatNextShareObject(shareObject, promises) {
          var shareObjectlayer = getLayerByName(
            shareObject.shareObject.fti.name,
            layers
          );
          if (shareObjectlayer == undefined) {
            return;
          }

          var intersectedGeom = shareObject.shareObject.feature.getGeometry();
          console.log(shareObject, editdescription);

          var geoJson = format.writeGeometryObject(intersectedGeom);

          //Buffer de la geometrie de l'objet ligne courant.
          var firstPromise = GeometryFactory.buffer(geoJson, 'ROUND', 1);
          promises.push(firstPromise);

          //Une fois le buffer de l'objet ligne effectué, on peut rechercher les intersections.
          var mainPromise = firstPromise.then(
            function(res) {
              var polygon = res.data;
              var lineCoordsString2 = '';

              if (polygon.type == 'Polygon') {
                var polygonGeom = format.readGeometry(polygon);
                angular.forEach(polygonGeom.getLinearRings(), function(
                  linearRing,
                  key
                ) {
                  var coords = linearRing.getCoordinates();
                  lineCoordsString2 += '(';
                  angular.forEach(coords, function(value, key) {
                    lineCoordsString2 += value[0] + ' ' + value[1] + ',';
                  });
                  lineCoordsString2 = lineCoordsString2.substring(
                    0,
                    lineCoordsString2.length - 1
                  );
                  lineCoordsString2 += ')';
                });
                var cql_filter =
                  'INTERSECTS(geom, Polygon(' + lineCoordsString2 + '))';

                //UID des layers du theme d'édition sur lequel on désire connaitre les intersections avec le shareObject courant.
                var uidSharedObj = shareObject.shareObject.fti.uid;
                var layersUids = editdescription.fti.uid + ',' + uidSharedObj;
                //var layersUids = "08f6b94b-5031-4c19-a370-161b0d0d4f6a,dd2ab6fb-2347-4d5b-b05e-e7fcbecfa28e";

                //La requete d'intersection de la featureLine avec les objets de la couche cfgShareObj.layerName.
                var promise = ogcFactory.getfeatures(
                  'GetFeature',
                  'WFS',
                  '1.0.0',
                  layersUids,
                  'json',
                  map
                    .getView()
                    .getProjection()
                    .getCode(),
                  cql_filter
                );
                //Préparation du traitement de la Reponse.
                var callBackSuccess = getCallbackSuccess(
                  shareObject,
                  shareObjectlayer
                );
                //promise2 sera resolue lorsque la callback aura fini d'être executée (car elle ne renvoie pas de promesse)
                var promise2 = promise.then(callBackSuccess, function(error) {
                  console.error(
                    'Erreur à l\'appel de ogc.data(), cql_filter:' + cql_filter
                  );
                });
                return promise2;
              } else {
                var deferred = $q.defer();
                console.error(
                  'Règle deleteObjectsOnLine featureLine after buffer operation is not a Polygon !'
                );
                deferred.resolve('terminé !');
                return deferred.promise;
              }
            },
            function(error) {
              console.error(
                'Erreur lors de la requete de bufferisation de l\'objet édité line de la couche ' +
                  currenteditdescription.fti.name
              );
              //Fin de la méthode de la recherche d'intesection pour l'objet édité featureLine (résolution de la mainPromise)
              return undefined;
            }
          );

          promises.push(mainPromise);
        }

        //Supprime les objets partagés de la ligne éditée presents dans 'currenteditdescription' correspondant à ceux configurés dans la règle.
        //Condition pour la suppression: l'objet partagé n'intersecte pas d'autres objets non supprimés.
        function treatAnEditedLine(currenteditdescription) {
          var promises = [];

          //Récupération des features intersectés par la ligne éditée dont les noms d'objets partagés sont dans cfgUpShareObjects.
          var shareObjects = getIntersectedShareObjects(currenteditdescription);

          //Pour chaque shareObject contenant le feature intersecté par la ligne éditée
          for (var j = 0; j < shareObjects.length; j++) {
            var shareObject = shareObjects[j];
            treatNextShareObject(shareObject, promises, currenteditdescription);
          }

          return $q.all(promises);
        }

        function getCallbackSuccess(shareObj, shareObjectlayer) {
          return function(result) {
            var intersectedFeatures = format.readFeatures(result.data);

            //Si la geométrie de l'objet n'intersecte que deux objets (l'objet lui-même et celui supprimé, alors on peut supprimer cet objet partagé)
            if (
              intersectedFeatures != undefined &&
              intersectedFeatures.length == 2
            ) {
              var newObject = {
                shareObject: shareObj.shareObject.name,
                editType: EditTypesFactory.editTypes.todepose.name,
                feature: shareObj.shareObject.feature,
                fti: shareObjectlayer.fti,
              };
              editdescription.relatedfeatures.push(newObject);

              //Historisation si configurée
              if (shareObj.isHistoricize) {
                var newObject = {
                  shareObject: shareObj.shareObject.name,
                  editType: EditTypesFactory.editTypes.tohistorize.name,
                  feature: shareObj.shareObject.feature,
                  fti: shareObjectlayer.fti,
                };
                editdescription.relatedfeatures.push(newObject);
              }
              //Mise en évidence du feature intersecté à supprimer
              gclayers
                .getselectSource()
                .addFeature(shareObj.shareObject.feature);
            }
            //Si le nombre d'intersections trouvées au niveau serveur est égale au nombre de fois que l'objet est present dans les shareobjets
            // (nombre de fois où l'objet est present sur l'ensemble des lignes à supprimer).
            //(+1 car le serveur considère l'intersection de la geometrie de l'objet avec lui-même)
            else if (
              intersectedFeatures.length ==
              1 +
                getFeatureCountInEditdescription(
                  shareObj.shareObject.feature.getId()
                )
            ) {
              if (
                !isObjectRecordedInRelatedfeatures(
                  shareObj.shareObject.feature.getId()
                )
              ) {
                var newObject = {
                  shareObject: shareObj.shareObject.name,
                  editType: EditTypesFactory.editTypes.delete.name,
                  feature: shareObj.shareObject.feature,
                  fti: shareObjectlayer.fti,
                };
                editdescription.relatedfeatures.push(newObject);

                //Mise en évidence du feature intersecté à supprimer
                gclayers
                  .getselectSource()
                  .addFeature(shareObj.shareObject.feature);

                //Historisation si configurée
                if (shareObj.isHistoricize) {
                  var newObject = {
                    shareObject: shareObj.shareObject.name,
                    editType: EditTypesFactory.editTypes.tohistorize.name,
                    feature: shareObj.shareObject.feature,
                    fti: shareObjectlayer.fti,
                  };
                  editdescription.relatedfeatures.push(newObject);
                }
              }
            }
          };
        }

        //Pour la ligne principale
        //Suppression des objets portés
        var promises = treatAnEditedLine(editdescription);

        //Pour les Les AUTRES lignes éditées (édition multiple de lignes)
        //Index de l'objet secondaire édité
        var relatedfeaturesIndex = 0;
        //on mémorise le nombre initial d'objets secondaires car on va au cours de l'execution possiblement alimenter ce tableau.
        //Si la règle ajoutait de nouveaux objets et que l'on voulait prendre en compte ces nouveaux objets,
        //on aurait comparé 'relatedfeaturesIndex' à 'editdescription.relatedfeatures.length' dans treatNextLineObject().
        var nbRelatedFeature = editdescription.relatedfeatures.length;

        function treatNextLineObject() {
          if (relatedfeaturesIndex < nbRelatedFeature) {
            var related = editdescription.relatedfeatures[relatedfeaturesIndex];
            if (
              related.editType == editdescription.editType &&
              related.fti.name == editdescription.fti.name
            ) {
              var anOtherpromises = treatAnEditedLine(related);

              anOtherpromises.then(
                function(res) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                },
                function(error) {
                  relatedfeaturesIndex++;
                  treatNextLineObject();
                }
              );
            } else {
              relatedfeaturesIndex++;
              treatNextLineObject();
            }
          }
          //si fin du parcours des objets
          else {
            deferred.resolve('rule end !');
          }
        }

        treatNextLineObject();

        //Crée une promesse qui sera résolue lorsque toutes les promesses du tableau seront résolues;
        //la valeur de résolution est un tableau contenant les valeurs des différentes promesses en paramètre.
        return $q.all([promises, deferred.promise]);
      }

      /**
       * Application de règle métier: vérification que les ponctuels auxquels on veut se connecter
       * ne sont pas déjà connectés à un nombre trop grand de linéaires de ce type
       * @param editdescription description de l'édition en cours
       * @param ruleConf configuration de la règle
       * @param featureType type de l'objet édité
       * @param map
       * @returns {*}
       */
      const checkNbConnectionOfSnappedPoint = (
        editdescription,
        ruleConf,
        featureType,
        map
      ) => {
        const deferred = $q.defer();
        const srid = map.getView().getProjection().getCode();
        const tol = 0.25;

        // Vérification du nombre d'objets intersectants en AMONT
        const amontPromises = [];
        const featuresAliasAmont = [];
        for (let shareObjectConfig of ruleConf.parameters.amont.shareObjects) {
          let shareObject = editdescription.shareObjects.find(shareObjectR => shareObjectR.shareObject === shareObjectConfig.name);
          if (shareObject) {
            const extent = [
              shareObject.feature.getGeometry().getCoordinates()[0] - tol, // Xmin
              shareObject.feature.getGeometry().getCoordinates()[1] - tol, // Ymin
              shareObject.feature.getGeometry().getCoordinates()[0] + tol, // Xmax
              shareObject.feature.getGeometry().getCoordinates()[1] + tol  // Ymax
            ];
            amontPromises.push(ogcFactory.findIntersectedFeaturesByRectangle(
              extent, srid, featureType.uid));
            featuresAliasAmont.push(shareObject.fti.alias);
          }
        }

        // Vérification du nombre d'objets intersectants en AVAL
        const avalPromises = [];
        const featuresAliasAval = [];
        for (let shareObjectConfig of ruleConf.parameters.aval.shareObjects) {
          let shareObject = editdescription.shareObjects.find(shareObjectR => shareObjectR.shareObject === shareObjectConfig.name);
          if (shareObject) {
            const extent = [
              shareObject.feature.getGeometry().getCoordinates()[0] - tol, // Xmin
              shareObject.feature.getGeometry().getCoordinates()[1] - tol, // Ymin
              shareObject.feature.getGeometry().getCoordinates()[0] + tol, // Xmax
              shareObject.feature.getGeometry().getCoordinates()[1] + tol  // Ymax
            ];
            avalPromises.push(ogcFactory.findIntersectedFeaturesByRectangle(
              extent, srid, featureType.uid));
            featuresAliasAval.push(shareObject.fti.alias);
          }
        }

        // Récupération des résultats
        $q.all(amontPromises).then((resultsAmont) => {
          let stopEdition = false;
          // AMONT
          for (let i in resultsAmont) {
            if (resultsAmont[i] && resultsAmont[i].data && resultsAmont[i].data.features
                && resultsAmont[i].data.features.length >= ruleConf.parameters.amont.nbMaxConnections) {
              deferred.reject();
              stopEdition = true;
              require('toastr').error('L\'objet ' + featuresAliasAmont[i]
                  + ' ne peut plus être connecté à un linéaire ' + featureType.alias);
            }
          }
          // AVAL
          $q.all(avalPromises).then((resultsAval) => {
            for (let i in resultsAval) {
              if (resultsAval[i] && resultsAval[i].data && resultsAval[i].data.features
                  && resultsAval[i].data.features.length >= ruleConf.parameters.aval.nbMaxConnections) {
                deferred.reject();
                stopEdition = true;
                require('toastr').error('L\'objet ' + featuresAliasAval[i]
                    + ' ne peut plus être connecté à un linéaire ' + featureType.alias);
              }
            }
            // Si la règle n'a pas arrêtée l'édition en cours, on continue
            if (!stopEdition) {
              deferred.resolve();
            }
          });
        });

        return deferred.promise;
      };

      function getSecondElement(data) {
        try {
          return data && data[1];
        }
        catch (err) {
          return false;
        }
      }

      return {
        initRules: {
          SnapOn: snapOn,
        },
        startRules: {
          GetInitialFeaturesIntersectingLine: getFeaturesIntersectingLine,
          GetAllFeaturesIntersectingInitialLine: getAllFeaturesIntersectingLine,
          MergeRemainingLines: mergeIntersectingLines,
          AllowDelete: EditRuleAllowDelete.allowDelete,
        },
        endRules: {
          CheckNbConnectionOfSnappedPoint: checkNbConnectionOfSnappedPoint,
          GetFeaturesIntersectingLine: getFeaturesIntersectingLine,
          GetAllFeaturesIntersectingLine: getAllFeaturesIntersectingLine,
          CutIntersectingLine: cutIntersectingLine,
          SetObjectOnExtremity: setObjectOnExtremity,
          GetFeaturesIntersectingPoint: getFeaturesIntersectingPoint,
          GetFeatureIntersectingPolygon: getFeatureIntersectingPolygon,
          MergeIntersectingLines: mergeIntersectingLines,
          MoveExtremityPoint: moveExtremityPoint,
          MoveObjectsOnLine: moveObjectsOnLine,
          DeleteObjectsOnLine: deleteObjectsOnLine,
          DeposeObjectsOnLine: deposeObjectsOnLine,
          SetAttribute: setAttribute,
          SetDefaultValue: setDefaultValue,
          SetDefaultValueByRole: EditRuleSetDefaultValueByRole.setDefaultValueByRole,
          RotateSymbolPoint: rotateSymbolPoint,
          SetXYAttributes: setXYAttributes,
          SetLengthAttribute: setLengthAttribute,
          SetAreaAttribute: setAreaAttribute,
          SetUserAttribute: setUserAttribute,
          SetW3wAttribute: setW3wAttribute,
          SetDateAttribute: setDateAttribute,
          SetRemoteAttribute: setRemoteAttribute,
          Historicize: historicize,
          VerifZoneSaisie: verifZoneSaisie,
          CopyPasteAttribute: copyPasteAttribute,
          MoveObjectOnEnd: EditRuleMoveObjectOnEnd.moveObjectOnEnd,
          ObjectNotAllowedAtIntersection: EditRuleObjectNotAllowedAtIntersection.checkIfObjectIsAllowedAtIntersection,
          StartLineSnappedOnPoint: EditRuleSnappedOnFeature.startLineSnappedOnPoint,
          EndLineSnappedOnPoint: EditRuleSnappedOnFeature.endLineSnappedOnPoint,
          StartLineSnappedOnLine: EditRuleSnappedOnFeature.startLineSnappedOnLine,
          EndLineSnappedOnLine: EditRuleSnappedOnFeature.endLineSnappedOnLine,
          MustNotSelfOverlap: MustNotSelfOverlapRule.mustNotSelfOverlap,
          MustNotSelfIntersect: MustNotSelfIntersectRule.mustNotSelfIntersect,
          GetFeaturesFromLineAssociation: GetFeaturesFromLineAssociationRule.getFeaturesFromLineAssociation,
          GetFeaturesFromPointAssociation: GetFeaturesFromLineAssociationRule.getFeaturesFromLineAssociation
        },
        postValidationRules: {
          SetAttributeByFormula: setAttributeByFormula,
          SetAttributeByConcat: setAttributeByConcat,
        },
        postSavingRules: {
          SetUniqueIdByConcat: setAttributeByConcat,
        },
      };
    },
  };

  EditRulesProvider.$get.$inject = [
    'gclayers',
    'gcWFS',
    '$q',
    'ogcFactory',
    'EditTypesFactory',
    'GeometryFactory',
    'gcPopup',
    '$rootScope',
    'SridFactory',
    'ConfigFactory',
    'ChartsFactory',
    '$filter',
    'What3WordFactory',
    '$timeout',
    'CopyPasteAttributeFactory',
    'gaJsGeneral',
    'gaDomUtils',
    'EditRuleAllowDelete',
    'EditRuleMoveObjectOnEnd',
    'EditRuleObjectNotAllowedAtIntersection',
    'EditRuleSetDefaultValueByRole',
    'EditRuleSnappedOnFeature',
    'ngDialog',
    'MustNotSelfIntersectRule',
    'MustNotSelfOverlapRule',
    'GetFeaturesFromLineAssociationRule',
    'bizeditProvider'
  ];
  return EditRulesProvider;
});
