/**
 *
 *        Report field management.
 *
 *    Report field may be a database field or not.
 *    Report field is configured in the widget description.
 *      Configuration example:
 *        {
 *         "field": "pj_tableau_recapitulatif",
 *         "mode": "write",
 *         "config": {"type": "report",
 *                    "exportFormat": "pdf",
 *                    "jasperFileName": "tableau_recap",
 *                    "parameters": [ {"name":"CURRENT_YEAR","value":"!CURRENT_YEAR!"} ],
 *                    "fieldAlias": "Tableau récapitulatif"
 *                   }
 *       }
 *
 *      field  :  name of field whether it exsts or not in the database
 *      mode   :  write or readonly  write -> report can be generated
 *                                 readonly -> only download of attached report is possible
 *                                 readOnlyWhenGenerated -> only download of attached report is possible when generated
 *      config.type           :  is "report" in order ro use this form field.
 *      config.exportFormat   :  report format can be "pdf", jpeg", "rtf", "xls".
 *      config.jasperFileName :  jasper report file name (without extension).
 *      config.parameters     :  list of parameters to send to the report
 *                               the value can directly be a basic value such as an integr
 *                               or a calculated value, the only calculated value availabe
 *                               today is "!CURRENT_YEAR!".
 *      config.fieldAlias     :  field name to display, when a database field is used,
 *                               this parameter is not necessary.
 *
 */
'use strict';
define(function() {
  var ReportFieldCtrl = function(
    $scope,
    ReportFactory,
    FeatureAttachmentFactory,
    GlobalServices,
    QueryFactory,
    gaJsUtils
  ) {
    $scope.getObjectId = function() {
      if (
        $scope.theField01.objectId != undefined &&
        $scope.theField01.featureName != undefined
      )
        return $scope.theField01.objectId;
      else return $scope.objectId;
    };

    $scope.getFeatureName = function() {
      if ($scope.theField01.featureName != undefined)
        return $scope.theField01.featureName;
      else return $scope.featureName;
    };

    $scope.getParamValue = function(value) {
      var res;

      //-- Query key word CURRENT_YEAR replaced by current full year value such as 2015.
      if (value == '!CURRENT_YEAR!') return '' + gaJsUtils.getCurrentYear();

      //-- Query key word CURRENT_YEAR and  + something.
      if (value.substr(0, 15) == '!CURRENT_YEAR!+')
        return '' + (gaJsUtils.getCurrentYear() + parseInt(value.substr(15)));

      //-- Query key word id.
      if (value == '!id!') return '' + $scope.getObjectId();

      //-- Query key word NULL is replaced by the null value.
      if (value == '!NULL!') return null;

      res = GlobalServices.getValueForKeyWord(value);
      if (res.done) return res.newValue;

      return '' + value;
    };

    $scope.getReportParameters = function() {
      var reportParams = {};
      var configParams = $scope.theField01.config.parameters;
      for (var ind = 0; ind < configParams.length; ind++)
        reportParams[configParams[ind].name] = $scope.getParamValue(
          configParams[ind].value
        );
      return reportParams;
    };

    $scope.setReadonlyIfGenerated = function() {
      $scope.readonly = $scope.generatedDocumentExists = false;
      FeatureAttachmentFactory.list(
        $scope.getFeatureName(),
        $scope.getObjectId()
      ).then(function(res) {
        var reportFileName;
        if ($scope.theField01.config.exportFileName != undefined)
          reportFileName =
            $scope.theField01.config.exportFileName +
            '.' +
            $scope.theField01.config.exportFormat;
        else
          reportFileName =
            $scope.theField01.config.jasperFileName +
            '.' +
            $scope.theField01.config.exportFormat;
        for (var ind = 0; ind < res.data.length; ind++) {
          if (res.data[ind] == reportFileName) {
            if (
              ($scope.theField01.config.mode == 'readOnlyWhenGenerated' &&
                $scope.theField01.objectField.value != null &&
                $scope.theField01.objectField.value != '') ||
              $scope.theField01.mode == 'readonly'
            )
              $scope.readonly = true;
            $scope.generatedDocumentExists = true;
            break;
          }
        }
      });
    };

    /********************************************
     *
     *
     *    Actions triggered  on user commands.
     *
     */

    $scope.generateAndOpen = function() {
      var config = $scope.theField01.config;
      ReportFactory.consultgeneratedreport(
        $scope.getReportParameters(),
        false,
        config.jasperFileName,
        $scope.theField01.datastoreName,
        config.exportFormat
      ).then(function(res) {
        if (res.data.strValeur != undefined) window.open(res.data.strValeur);
        else window.open(res.data);
      });
    };

    $scope.generateAndAttach = function() {
      var config = $scope.theField01.config;
      ReportFactory.consultgeneratedreport(
        $scope.getReportParameters(),
        false,
        config.jasperFileName,
        $scope.theField01.datastoreName,
        config.exportFormat
      ).then(function(res) {
        var newFileName, fn;
        if (config.exportFileName != undefined)
          newFileName = config.exportFileName + '.' + config.exportFormat;
        if (res.data.strValeur != undefined) fn = res.data.strValeur;
        else fn = res.data;
        FeatureAttachmentFactory.addServerFile(
          fn,
          $scope.getFeatureName(),
          $scope.getObjectId(),
          newFileName
        ).then(function(res) {
          //-- Store file name into object field value.
          var fn = res.data.fileName;
          var indSlash = fn.lastIndexOf('\\');
          if (indSlash == -1) indSlash = fn.lastIndexOf('/');
          if (indSlash != -1) fn = fn.substr(indSlash + 1);
          $scope.theField01.objectField.value = fn;

          //-- Run parent value changed method if it exists.
          if ($scope.valueChanged != undefined) $scope.valueChanged();
          $scope.setReadonlyIfGenerated();
        });
      });
    };

    $scope.openAttachedReport = function() {
      FeatureAttachmentFactory.getdownloadurl(
        $scope.theField01.objectField.value,
        $scope.getFeatureName(),
        $scope.getObjectId()
      ).then(function(res) {
        window.open(res.data);
      });
    };

    $scope.$on('gotRelatedObjects', function(event, data) {
      if (data.objects != undefined && data.objects.features != undefined) {
        $scope.objectId = QueryFactory.getFeatureId(data.objects.features[0]);
        $scope.featureName = QueryFactory.getFeatureName(
          data.objects.features[0]
        );
        $scope.setReadonlyIfGenerated();
      }
    });

    $scope.$on('gotFieldValue', function(event, data) {
      if (
        data.objectId != undefined &&
        data.theField.id == $scope.theField01.id
      ) {
        $scope.theField01.objectId = QueryFactory.getFeatureId({
          id: data.objectId,
        });
        $scope.theField01.featureName = QueryFactory.getFeatureName({
          id: data.objectId,
        });
        $scope.setReadonlyIfGenerated();
      }
    });

    //---  Initialization
    $scope.setReadonlyIfGenerated();
    GlobalServices.initialize();
  };

  ReportFieldCtrl.$inject = [
    '$scope',
    'ReportFactory',
    'FeatureAttachmentFactory',
    'GlobalServices',
    'QueryFactory',
    'gaJsUtils',
  ];
  return ReportFieldCtrl;
});
