'use strict';
define(function() {
  const attributeRestrictionsUtils = function(
    $filter
  ) {
    const DOMAIN = 'Domain';
    //  Domaines de restriction (type:'Domain'):
    //     -la traduction clé/valeur est faite en front-end
    //     -La clée est stocké directement dans res[attname] car dans le cas normal, le back-end doit recevoir la clée
    const TABLE = 'Tables';
    // Table de restriction (type:'Tables'):
    //     -la traduction clé/valeur est faite en front-end
    //     Dans les tables de restriction il y a 2 filtres: 
    //        -celui qui est configuré dans l'admin et qui est actif dès l'ouverture de la popup
    //        -celui qui sert à chercher une valeur dans la popup
    const USER = 'User';
    // Restriction liste utilisateurs (type: 'User'):
    //     -la traduction clé/valeur est faite en front-end
    //     -La clé est stocké directement dans res[attname]
    //     -la clé peut etre soit la login soit le fullname. selon de la config.
    const INPUT_SIZE = 'inputSize';
    const INTERVAL = 'Interval';

    const buildFullname = (name, vorname) => {
      return name + ' ' + vorname;
    }

    /**
     * get the 'key' corresponding to the 'value' in 'pickChoices' array
     * @param {*} pickChoices list of pairs value/key
     * @param {*} value 
     * @returns the key
     */
    const getKeyFromValue = function (pickChoices, value) {
      let key = null;
      if ((pickChoices !== undefined) && (Array.isArray(pickChoices)) && (value !== undefined)) {
        const indexOfPair = pickChoices.findIndex((restrictionPair, index) => {
            return restrictionPair.restrictedValue === value;

          });
        if (indexOfPair > -1) {
          key = pickChoices[indexOfPair].restrictedKey;
        }
      }
      if (key === undefined) {
        console.error("couldn't find key for given value");
      }
      return key;
    }

    /**
     * get the 'value' corresponding to the 'key' in 'pickChoices' array
     * @param {*} pickChoices list of pairs value/key
     * @param {*} key 
     * @returns  the value
     */
    const getValueFromKey = function (pickChoices, key) {
      let value = null;
      if ((pickChoices !== undefined) && (Array.isArray(pickChoices)) && (key !== undefined)) {
        const indexOfPair = pickChoices.findIndex(restrictionPair => restrictionPair.restrictedKey === key);
        if (indexOfPair > -1) {
          value = pickChoices[indexOfPair].restrictedValue;
        }
      }
      if (value === undefined) {
        console.error("couldn't find value for given key");
      }
      return value;
    }


    /**
     * fill dynamic where clause with data from variableData
     * replace '#$attribute' by the corresponding data in variableData
     *
     * @param {string} filterDefinition where clause with #$ attributes to replace
     *    ie: "commune = #$id_commune AND intervenant = #$intervenant"
     * @param {object} fti
     * @param {object} variableData properties of a variable
     * @param {object} elasticData data of the elasticFilter or undefined if we are not in an elastic filter
     * @returns Promise outputting an array contaning the filter completed
     */
    const prepareDynamicFilter = (filterDefinition, fti,
        variableData, elasticData, otherFiltersInfos) => {
      const resolveWithData = (resolve, filter) => {
        // if all dynamic fields have been replaced
        if (filter.indexOf('#$') < 0) {
          resolve(filter);
        } else {
          // don't send the filter if it's not finished
          resolve();
        }
      }
      //----------------------

      let promiseArray = [];
      let filterTranslated;
      if (typeof filterDefinition === 'string' && filterDefinition.trim().length > 0) {
        filterTranslated = filterDefinition;
        // identify groups like '#$code_insee'
        const regex = /(\#\$)([^\s']+)(\s*?)/gm;
        //get an array with fields with '#$'
        const dynamicFields = filterDefinition.match(regex);
        if (dynamicFields) {
          for (let dynamicField of dynamicFields) {
            let promise = new Promise((resolve, reject) => {
              // translate dynamicField and replace it in filterTranslated
              const fieldName = dynamicField.slice(2);
              const attribute = fti.attributes.find((attribute) => {return attribute.name === fieldName});
              let keyOrValue;
              // is it a normal form or an elastic filter ?
              if (!elasticData) {
                if (variableData[fieldName]) {
                  keyOrValue = variableData[fieldName];
                } else if (otherFiltersInfos && otherFiltersInfos[fieldName]
                  && otherFiltersInfos[fieldName].value) {
                  // Récupération des valeurs du filtre depuis les données d'une gcDatatable
                  keyOrValue = otherFiltersInfos[fieldName].value;
                }
              } else {
                //in this case, we are in <elastic-mrule>
                //and the data of the other attributes is not accessible through variableData
                const elasticRule = elasticData.find((rule) => {
                  return rule.name === fieldName
                });
                if(elasticRule) {
                  if (attribute && Array.isArray(attribute.restrictions)
                    && attribute.restrictions[0]) {
                    if (elasticRule.restrictedValue) {
                      keyOrValue = elasticRule.restrictedValue[fieldName];
                    }
                  } else {
                    keyOrValue = elasticRule.value;
                  }
                }
              }
              if (keyOrValue) {
                filterTranslated = filterTranslated.replace(dynamicField, keyOrValue);
                resolveWithData(resolve, filterTranslated);
              } else {
                // if we didn't find the key -> remove the clause
                // removeDynamicFieldFromfilter(fieldName);
                require('toastr').error( $filter('translate')('model.featuretypes.attributes.'
                + 'restrictions.table.couldNotFillDynamicFilter') + fieldName);
                reject();
              }
            });
            promiseArray.push(promise);
          }
        }
      } else {
        filterTranslated = '';
      }

      if (promiseArray.length <= 0) {
        promiseArray.push(Promise.resolve(filterTranslated));
      }
      return Promise.all(promiseArray);
    };

    /**
     * Vérifie dans le fti si l'attribut portant le nom <code>attributeName</code> possède 1 restriction
     * @param {object} fti fti de current contenant l'attribut à vérifier
     * @param {string} attributePath chemin de l'attribut dont on veut vérifier la présence d'une restriction
     * (ex. "current.properties.effluent" ou "effluent") <code>attributePath</code> contient le caractère '.' ou pas.
     *
     * @return {boolean} true si l'attribut a un élément dans son tableau de restrictions
     */
    const attributeHasRestriction = (fti, attributePath) => {
      if (attributePath) {
        if (fti && Array.isArray(fti.attributes)) {
          const attributeName = attributePath.split('.').pop();
          const ftiAttribute = fti.attributes.find(ftiAttr => ftiAttr.name === attributeName);
          if (ftiAttribute) {
            return Array.isArray(ftiAttribute.restrictions) && ftiAttribute.restrictions.length > 0;
          } else {
            console.error('attributeHasRestriction - Le fti sélectionné ne possède pas l\'attribut :'
                + ' attributeName = ' + attributeName);
            return false;
          }
        } else {
          console.error('attributeHasRestriction - Le fti est vide ou null : fti = ', fti);
        }
      } else {
        console.error('attributeHasRestriction - L\'attribut possédant une restriction'
            + ' n\'est pas défini : attributePath = ', attributePath);
      }
    };

    /**
     * Vérifie dans le fti si l'attribut portant le nom <code>attributeName</code> possède 1 table ou 1 domaine de restriction
     * @param {object} fti fti de current contenant l'attribut
     * @param {string} attributePath chemin de l'attribut dont on veut vérifier la présence d'une restriction
     * (ex. "current.properties.effluent" ou "effluent") <code>attributePath</code> contient le caractère '.' ou pas.
     *
     * @return {boolean} true si l'attribut possède une restriction de type table ou domaine dans son tableau de restrictions
     */
    const attributeHasTableOrDomainRestriction = (fti, attributePath) => {
      if (attributePath) {
        if (fti && Array.isArray(fti.attributes)) {
          const attributeName = attributePath.split('.').pop();
          const ftiAttribute = fti.attributes.find(ftiAttr => ftiAttr.name === attributeName);
          if (ftiAttribute !== undefined) {
            if (Array.isArray(ftiAttribute.restrictions) && ftiAttribute.restrictions.length > 0) {
              return ftiAttribute.restrictions[0].type === DOMAIN || ftiAttribute.restrictions[0].type === TABLE;
            }
          } else {
            console.error('attributeHasTableOrDomainRestriction - Le fti sélectionné ne possède pas'
                + ' l\'attribut ' + attributeName);
          }
        } else {
          console.error('attributeHasTableOrDomainRestriction - Le fti est vide ou null : ', fti);
        }
      } else {
        console.error('attributeHasTableOrDomainRestriction - L\'attribut possédant une restriction'
            + ' n\'est pas défini : attributePath = ', attributePath);
      }
      return false;
    };

    /**
     * Vérifie dans le fti si l'attribut portant le nom <code>attributeName</code> possède une restriction dont le type est fourni en paramètre
     * @param {object} fti fti de current contenant l'attribut
     * @param {string} attributePath chemin de l'attribut dont on veut vérifier la présence d'une restriction
     * (ex. "current.properties.effluent" ou "effluent") <code>attributePath</code> contient le caractère '.' ou pas.
     * @param {string} restrictionType type de restriction à rechercher dans l'attribut
     *
     * @return {boolean} true si l'attribut a un élément dans son tableau de restrictions
     */
    const attributeHasRestrictionType = (fti, attributePath, restrictionType) => {
      if (restrictionType) {
        if (attributePath) {
          if (fti && Array.isArray(fti.attributes)) {
            const attributeName = attributePath.split('.').pop();
            const ftiAttribute = fti.attributes.find(ftiAttr => ftiAttr.name === attributeName);
            if (ftiAttribute) {
              if (Array.isArray(ftiAttribute.restrictions) && ftiAttribute.restrictions.length > 0) {
                return ftiAttribute.restrictions[0].type === restrictionType;
              }
            } else {
              console.error('attributeHasRestrictionType - Le fti sélectionné ne possède pas ' +
                  'l\'attribut ' + attributeName);
            }
          } else {
            console.error('attributeHasRestrictionType - Le fti est vide ou null : ', fti);
          }
        } else {
          console.error('attributeHasRestrictionType - L\'attribut possédant une restriction'
              + ' n\'est pas défini : attributePath = ', attributePath);
        }
      } else {
        console.error('attributeHasRestrictionType - Le type de restriction à rechercher '
            + 'n\'est pas défini: restrictionType = ', restrictionType);
      }
      return false;
    };

    return {
      DOMAIN: DOMAIN,
      TABLE: TABLE,
      USER: USER,
      INPUT_SIZE: INPUT_SIZE,
      INTERVAL: INTERVAL,
      buildFullname: buildFullname,
      getKeyFromValue: getKeyFromValue,
      getValueFromKey: getValueFromKey,
      prepareDynamicFilter: prepareDynamicFilter,
      attributeHasRestriction: attributeHasRestriction,
      attributeHasTableOrDomainRestriction: attributeHasTableOrDomainRestriction,
      attributeHasRestrictionType: attributeHasRestrictionType
    };
  };

  attributeRestrictionsUtils.$inject = [
    '$filter'
  ];

  return attributeRestrictionsUtils;
});
